/* 
   XRView.m

   NSView for GNUstep GUI X/RAW Backend.

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: January 1996
   Author:  Ovidiu Predescu <ovidiu@net-community.com>
   Date: May 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>

#include <Foundation/NSArray.h>
#include <Foundation/NSThread.h>

#include <AppKit/NSColor.h>
#include <AppKit/PSMatrix.h>
#include <AppKit/GSTrackingRect.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <gnustep/xraw/XR.h>


#define DWZ 48									// initial drag window size in 
												// pixels, typically this is  
												// equal to that of std icons
//
// Backend structure for XRView
//
typedef struct _XRView_struct
{
	XRContext* xContext;						
	Display *xDisplay;
	int xScreen;

} XRView_struct;

#define XDISPLAY	(((XRView_struct *)be_view_reserved)->xDisplay)
#define XCONTEXT	(((XRView_struct *)be_view_reserved)->xContext)
#define XSCREEN		(((XRView_struct *)be_view_reserved)->xScreen)


extern void XRSetCanvasOrigin(NSPoint aPoint);




//*****************************************************************************
//
// 		PSMatrix (BackendMethods) category 
//
//*****************************************************************************

@implementation PSMatrix (BackendMethods)

- (void)set	
{ 
}

@end  /* PSMatrix (BackendMethods) */

//*****************************************************************************
//
// 		XRView 
//
//*****************************************************************************

@implementation XRView
															// Class variables
static NSString	*xrview_thread_key = @"NSViewThreadKey";

+ (void)initialize
{
	if (self == [XRView class])
		{
		NSDebugLog(@"Initialize XRView class\n");
		[self setVersion:1];						 
    	}
}

//
// Initialization
//
- (id)initWithFrame:(NSRect)frameRect
{														// Allocate back-end
	be_view_reserved = malloc(sizeof(XRView_struct));	// structure

	[super initWithFrame:frameRect];
												
	return self;
}

- (void)dealloc
{
	free(be_view_reserved);								// Free the back-end
														// structure
	[super dealloc];
}

//
// Dragging 
//
- (BOOL)dragFile:(NSString *)filename
		fromRect:(NSRect)rect
       	slideBack:(BOOL)slideFlag
	   	event:(NSEvent *)event
{
	return NO;
}

- (void)dragImage:(NSImage *)anImage					// initiate a dragging
			   at:(NSPoint)viewLocation					// session
			   offset:(NSSize)initialOffset
			   event:(NSEvent *)event
			   pasteboard:(NSPasteboard *)pboard
			   source:(id)sourceObject
			   slideBack:(BOOL)slideFlag
{
XRDragView *dragView = [XRDragView _sharedDraggingView]; 
XRWindow *dragWindow = [dragView window];
NSPoint dragWindowOrigin;								// translate source loc
								 						// to window's coords
	viewLocation = [[window contentView] convertPoint:viewLocation 
										 fromView:self];

	dragWindowOrigin = [window convertBaseToScreen:viewLocation];
	dragWindowOrigin.y += initialOffset.height;			// convert to screen
	dragWindowOrigin.x += initialOffset.width; 			// coords and add the
														// offset
	[dragWindow setFrameOrigin:dragWindowOrigin];
	[dragView setImage:anImage];
	[dragWindow orderFront:nil];
	[dragWindow _captureMouse:self];
	[dragView mouseDown:event];
}

- (void)registerForDraggedTypes:(NSArray *)newTypes
{
}

- (void)unregisterDraggedTypes
{
}

- (void)lockFocus										// not per spec FIX ME
{
NSMutableDictionary *dict = [[NSThread currentThread] threadDictionary];
NSMutableArray *stack = [dict objectForKey: xrview_thread_key];

	if(!window)									// do nothing if view is not in
		return;									// a window's view heirarchy

	PSgsave();											// Save graphics state

//	[frameMatrix set];
//  PSrectclip (0, 0, frame.size.width, frame.size.height);
//	[boundsMatrix set];

	if([self isFlipped])								// view has flipped
		{												// coordinates 
		NSPoint aPoint = NSMakePoint(0,0);

		if (super_view)									
			{
			float superSizeHeight = [super_view frame].size.height;
		    aPoint = [super_view convertPoint:frame.origin 
								 toView:[window contentView]];

			if(frame.size.height < superSizeHeight)
		    	aPoint.y = [window frame].size.height - aPoint.y - 
							frame.size.height;
			else
		    	aPoint.y = ([window frame].size.height - 
							(superSizeHeight + 
							2*[super_view frame].origin.y)) + aPoint.y;

#ifdef FAR_DEBUG
		fprintf (stderr,
"XRView: aPoint (%1.2f, %1.2f),frame.origin.y (%1.2f, window Frame ht%1.2f)\n",
				aPoint.x, aPoint.y, 
				frame.origin.y, [window frame].size.height);
			fprintf (stderr,
			"XRView: rect origin (%1.2f, %1.2f), size (%1.2f, %1.2f)\n",
				aPoint.x, aPoint.y, 
				[super_view frame].origin.y, [super_view frame].size.height);
#endif /* FAR_DEBUG */

			}
		else
		    aPoint.y = ([window frame].size.height + aPoint.y);

		XRSetCanvasOrigin (aPoint);			
		XRSetFlipped();
		}
	else												// view is not flipped
		{
		NSPoint aPoint;

		if (super_view)
            aPoint = [super_view convertPoint:frame.origin 
								 toView:[window contentView]];
		else
			aPoint = NSMakePoint(0,0);

		XRSetCanvasOrigin (aPoint);
		}

	if (stack == nil)									// create thread focus
		{												// stack if necessary
		stack = [[NSMutableArray alloc] initWithCapacity: 4];
		[dict setObject: stack forKey: xrview_thread_key];
		[stack release];
		}												// add self to the top
	[stack addObject: self];							// of current thread's
														// focus stack 
	XRFocusLock(self, window);
	XRSetCanvasSize ([window frame].size);

//	fprintf(stderr,"XRView lockFocus: focus stack size == %d\n",[stack count]);

	NSRectClip([self visibleRect]);						// set clipping path
}

- (void)unlockFocus
{
NSMutableDictionary *dict = [[NSThread currentThread] threadDictionary];
NSMutableArray *stack = [dict objectForKey: xrview_thread_key];
int top;

	if (!stack)
		{
		NSLog (@"XRView unlockFocus: current thread has no focus stack.");
		return;
		}

	PSgrestore();										// Restore the graphics

//	fprintf(stderr, "XRView unlockFocus: focus stack size == %d\n", 
//					[stack count]);
														// remove self from top
	if((top = [stack count]) > 0)						// of the focus stack
		{
		NSView *fView = [stack objectAtIndex:top - 1];
														// do so only if self
		if(fView == self)								// is at top of the 
			{											// stack (the norm)
			[stack removeObjectAtIndex:top - 1];
														// if focus stack is 
			if((top = [stack count]) > 0)				// not empty lock focus 
				{										// on next view in stac
				NSView *fView = [stack objectAtIndex:top - 1];
														
				XRFocusLock((XRView*)fView, (XRWindow*)[fView window]);
				}										// attempt to unlock an
			}											// unfocused view this
		else											// should not happen
			NSLog (@"XRView attempt to unlockFocus on an unfocused view.");
		}
														
//	fprintf(stderr, "XRView unlockFocus: focus stack size == %d\n", 
//					[stack count]);

///XRResetClipPath();

//XRDestroyPredefinedClipPath();
}														

- (void)drawRect:(NSRect)rect
{
	if(rect.size.width == 0 || rect.size.height == 0)	// FIX ME this should
		{												// not happen
		NSLog (@"XRView drawRect: attempted to draw a zero rect view.");
		return;		
		}									
 
#ifdef FAR_DEBUG
		fprintf (stderr,
	"XRView drawRect: rect origin (%1.2f, %1.2f), size (%1.2f, %1.2f)\n",
				rect.origin.x, rect.origin.y, 
				rect.size.width, rect.size.height);
#endif																		

	[super drawRect:rect];
}

@end /* XRView */

//*****************************************************************************
//
// 		XRDragView 
//
//*****************************************************************************

extern Window _findXWindow(Display* display, Window topwindow, 
							Window window_to_check, int x, int y);
extern int _sendXString(Display* display, Window window, const char* string);

@implementation XRDragView

// Class variables
static XRCell* dragCell = nil;
static XRDragView* sharedDragView = nil;					// dragging view

+ (void)initialize
{
NSImage* closeImage = [NSImage imageNamed:@"common_Close"];

	dragCell = [[XRCell alloc] initImageCell:closeImage];
	[dragCell setBordered:NO];
}

+ (XRDragView *)_sharedDraggingView
{
	if(!sharedDragView)							// class obj maintains a global
		{										// window for use in dragging
		NSRect winRect = {{0, 0}, {DWZ, DWZ}};	// ops.  alloc it if necessary.
		Display *xDisplay = [XRContext currentXDisplay];
		XSetWindowAttributes winattrs;
		unsigned long valuemask;
		XRWindow* sharedDragWindow = (XRWindow *)[NSWindow alloc];

		[sharedDragWindow initWithContentRect:winRect
						  styleMask:NSBorderlessWindowMask
						  backing:NSBackingStoreNonretained
						  defer:YES];

		valuemask = (CWSaveUnder|CWOverrideRedirect);
		winattrs.save_under = True;
		winattrs.override_redirect = True;
		XChangeWindowAttributes (xDisplay, (Window)[sharedDragWindow xWindow], 
								valuemask, &winattrs);
		sharedDragView = [XRDragView new];
		[sharedDragView setFrameOrigin:NSMakePoint(0, 0)];
		[sharedDragView setFrameSize:NSMakeSize (winRect.size.width, DWZ)];
		[[sharedDragWindow contentView] addSubview:sharedDragView];
		[[NSApplication sharedApplication] removeWindowsItem:sharedDragWindow];
		}

	return sharedDragView;
}

- (void)drawRect:(NSRect)rect
{												
	[dragCell drawWithFrame:rect inView:self];
}

- (void)mouseDown:(NSEvent*)theEvent
{
Display *xDisplay = [XRContext currentXDisplay];
int xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
unsigned int eventMask = NSLeftMouseDownMask | NSLeftMouseUpMask
							| NSLeftMouseDraggedMask | NSMouseMovedMask
							| NSPeriodicMask;
NSApplication *app = [NSApplication sharedApplication];
NSPoint point, lastPoint;
NSDate *theDistantFuture = [NSDate distantFuture];
NSEventType eventType;
Window xWindow = [window xWindow];
Window rootWindow, mouseWindow;
float wx, wy;
int init = 0;
NSRect dragWindowXFrame;
NSWindow *eWindow = [theEvent window];
													// get drag window's origin
	dragWindowXFrame = [window xFrame];				// in X screen coordinates	
	wx = dragWindowXFrame.origin.x;					
	wy = dragWindowXFrame.origin.y;					// set periodic events rate
													// to achieve max of ~30fps										
	[NSEvent startPeriodicEventsAfterDelay:0.02 withPeriod:0.03];
	[[NSRunLoop currentRunLoop] limitDateForMode:NSEventTrackingRunLoopMode];
													// convert cursor location
								 					// to X screen coordinates
	lastPoint = [[eWindow contentView] convertPoint:[theEvent locationInWindow] 
									   fromView:nil];
	lastPoint = [eWindow convertBaseToScreen:lastPoint];
	lastPoint.y = DisplayHeight(xDisplay, xScreen) - lastPoint.y;

	while ((eventType = [theEvent type]) != NSLeftMouseUp)				 
		{											// user is dragging window
		if (eventType != NSPeriodic)				// loop until left mouse up
			{
			point = [theEvent locationInWindow];
			if(init < 1)							// dump the first point 
				{									// because it is in coords
				init++;								// of the dragging source	
				point = lastPoint;					// view	 
				}
			else									// convert point to global
				{									// X screen coordinates
				point.y = dragWindowXFrame.size.height - point.y;
				point.y += wy;
				point.x += wx;
				}
			}										// at each periodic event
		else										// update the position of
			{										// drag window if necessary 
			if (point.x != lastPoint.x || point.y != lastPoint.y) 
				{
				wx += (point.x - lastPoint.x);				// Move drag window
				wy += (point.y - lastPoint.y);
				XMoveWindow (xDisplay, xWindow, (int)wx, (int)wy);	
				lastPoint = point;							
				}
			}
		theEvent = [app nextEventMatchingMask:eventMask
						untilDate:theDistantFuture 
						inMode:NSEventTrackingRunLoopMode
						dequeue:YES];
  		}												// left mouse button is
														// now up so clean up
	[NSEvent stopPeriodicEvents];						
	[window _releaseMouse:self];
	[window orderOut:nil];
	rootWindow = RootWindow(xDisplay, xScreen);			// find X window that 
														// cursor is over.  
														// return if not found
	if(!(mouseWindow = _findXWindow(xDisplay, rootWindow, rootWindow, wx, wy)))
		return;
	else										// hack so that drag Window
		{										// can query the Workspace 
		id browser = [app delegate];			// browser, DO NOT rely on this	
		BOOL exists = NO, is_dir = NO;			// FIX ME s/b moved to Workspac
		NSFileManager *fm = [NSFileManager defaultManager];
		BOOL altKeyDown = [theEvent modifierFlags] & NSAlternateKeyMask;
		NSMutableString *s = [[[NSMutableString alloc] 
								initWithString:[browser path]] autorelease];

		exists = [fm fileExistsAtPath: s isDirectory: &is_dir];
														// if ALT key was held 
		if ((exists) && (is_dir) && altKeyDown)			// down user wants to
			{											// change to dir
			[s insertString:@"cd " atIndex:0];
			[s appendString: @"\n"];					// send path string to
			}											// X window under the 
														// cursor 
		if(!_sendXString(xDisplay, mouseWindow, [s cString]))
			NSLog(@"XRDragWindow: error sending string to X window\n");
		}
}														

- (void)setImage:(NSImage *)anImage
{
	[dragCell setImage:anImage];

	[self lockFocus];
	[dragCell drawWithFrame:NSMakeRect(0,0,48,48) inView:self];
	[self unlockFocus];
}

@end /* XRDragView */
