/* 
   XRColor.m

   NSColor for GNUstep GUI X/RAW Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: January 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <stdlib.h>

#include <gnustep/xraw/XRColor.h>
#include <gnustep/xraw/XRContext.h>
#include <gnustep/xraw/XRScreen.h>

#include <gnustep/xraw/XRDrawingEngine.h>

//
// Backend structure for XRColor
//
typedef struct _XRColor_struct
{
  XColor color;
  BOOL is_alloced;
} XRColor_struct;

#define XRCOLOR (((XRColor_struct *)be_color_reserved)->color)
#define XRALLOC (((XRColor_struct *)be_color_reserved)->is_alloced)



@implementation XRColor (GNUstepXRAW)

- (XColor)xColor
{
  if (!XRALLOC && be_color_reserved)		  				// allocate a color 
	{													
	Display *xdis = [XRContext currentXDisplay];
	int xscreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
	Status ret;
	NSColor *c = self;

	if (([[c colorSpaceName] 
		compare:NSCalibratedRGBColorSpace] == NSOrderedSame) ||
		([[c colorSpaceName] compare:NSDeviceRGBColorSpace] == NSOrderedSame))
			{												// Create an XColor 
															// from the NSColor
			XRCOLOR.red = (unsigned short)(65535 * [c redComponent]);
			XRCOLOR.green = (unsigned short)(65535 * [c greenComponent]);
			XRCOLOR.blue = (unsigned short)(65535 * [c blueComponent]);
			}
		else 
			{
			if (([[c colorSpaceName] 
					compare: NSCalibratedWhiteColorSpace] == NSOrderedSame) || 
				([[c colorSpaceName] 
					compare:NSDeviceWhiteColorSpace] == NSOrderedSame))
				{											// Create an XColor 
															// from the NSColor
				XRCOLOR.red = (unsigned short)(65535 * [c whiteComponent]);
				XRCOLOR.green = (unsigned short)(65535 * [c whiteComponent]);
				XRCOLOR.blue = (unsigned short)(65535 * [c whiteComponent]);
				}
			else								// If not a colorspace we can 
				{								// easily convert then just 
				XRCOLOR.red = 32000;			// make the background gray.
				XRCOLOR.green = 32000;
				XRCOLOR.blue = 32000;
				}
			}

      													// Allocate the color
    ret = XAllocColor(xdis, XDefaultColormap(xdis, xscreen), &XRCOLOR);
    if (!ret)
		NSLog(@"Failed to allocate color\n");
	else											// Copy actual values back
		{			 								// to the NSColor variables
		RGB_component.red = ((float)XRCOLOR.red) / 65535;
		RGB_component.green = ((float)XRCOLOR.green) / 65535;
		RGB_component.blue = ((float)XRCOLOR.blue) / 65535;
	    XRALLOC = YES;
		}
    }

  	return XRCOLOR;
}

- (void)setXColor:(XColor)xcolor
{
  	XRCOLOR = xcolor;
}

@end

@implementation XRColor

//
// Class methods
//
+ (void)initialize
{
	if (self == [XRColor class])
		[self setVersion:1];								// Initial version
}

+ (NSColor *)colorWithCalibratedWhite:(float)white 
								alpha:(float)alpha
{
	if(white == NSLightGray)					// the drawing engine retains
		return XRLightGray();					// pointers to the most common
	else										// colors check if request is
		{										// for one of these.  Return
		if(white == NSBlack)					// the cached color if found
			return XRBlack();					// otherwise alloc and return
		else									// a new color
			{
			if(white == NSWhite)					 
				return XRWhite();
			else
				{
				if(white == NSDarkGray)					 
					return XRDarkGray();
		}	}	}

	return [super colorWithCalibratedWhite:white alpha:alpha];
}

+ (NSColor *)darkGrayColor
{
NSColor *cachedColor = nil;

	if((cachedColor = XRDarkGray()))			// if color is already cached, 		
		return cachedColor;						// return it.  otherwise alloc	

	return [super colorWithCalibratedWhite:NSDarkGray alpha: 1.0];
}

+ (NSColor *)lightGrayColor
{
NSColor *cachedColor = nil;

	if((cachedColor = XRLightGray()))			// if color is already cached, 		
		return cachedColor;						// return it.  otherwise alloc	

	return [super colorWithCalibratedWhite:NSLightGray alpha: 1.0];
}

+ (NSColor *)blackColor
{
NSColor *cachedColor = nil;

	if((cachedColor = XRBlack()))				// if color is already cached, 		
		return cachedColor;						// return it.  otherwise alloc	

	return [super colorWithCalibratedWhite:NSBlack alpha: 1.0];
}

+ (NSColor *)whiteColor
{
NSColor *cachedColor = nil;

	if((cachedColor = XRWhite()))				// if color is already cached, 		
		return cachedColor;						// return it.  otherwise alloc	

	return [super colorWithCalibratedWhite:NSWhite alpha: 1.0];
}

- init
{												// Allocate backend structure
	be_color_reserved = malloc(sizeof(XRColor_struct));
	XRALLOC = NO;
	
	[super init];
	
	return self;
}

- (void)dealloc
{
	free(be_color_reserved);					// Release backend structure

	[super dealloc];
}

- (void)set
{
    XRSetCurrentColor(self);					// set the drawing engine's 
}												// current color

@end
