/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "utils.h"


static void on_entry_dialog_ok (GtkWidget * widget,
				gpointer data);
static void on_entry_dialog_destroy (GtkWidget * widget,
				     gpointer data);

/*
 * Shows a simple message box with a label and an 'OK' button.
 * e.g. show_message_box ("Error saving file");
 */
void
show_message_box (gchar * message)
{
  GtkWidget *dialog, *label, *button;

  dialog = gtk_dialog_new ();
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_container_border_width (GTK_CONTAINER (dialog), 5);

  label = gtk_label_new (message);
  gtk_misc_set_padding (GTK_MISC (label), 20, 20);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label,
		      TRUE, TRUE, 0);
  gtk_widget_show (label);

  button = gtk_button_new_with_label (_("OK"));
  gtk_widget_set_usize (button, 80, -1);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), button,
		      FALSE, FALSE, 14);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (dialog);
}


/*
 * Shows a dialog box with a message and a number of buttons.
 * Signal handlers can be supplied for any of the buttons.
 * NOTE: The dialog is automatically destroyed when any buttons is clicked.
 * default_button specifies the default button, numbered from 1..
 * data is passed to the signal handler.
 * Example usage:
 gchar *buttons[] = { "Yes", "No", "Cancel" };
 GtkSignalFunc signal_handlers[] = { on_yes, on_no, on_cancel };
 show_dialog ("Do you want to save the current project?", 3, buttons, 3,
 signal_handlers, NULL);
 */
void
show_dialog (gchar * message,
	     gint nbuttons,
	     gchar * buttons[],
	     gint default_button,
	     GtkSignalFunc signal_handlers[],
	     gpointer data)
{
  GtkWidget *dialog, *hbox, *label, *button;
  int i;

  dialog = gtk_dialog_new ();
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_container_border_width (GTK_CONTAINER (dialog), 5);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 20);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox,
		      TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  label = gtk_label_new (message);
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
  gtk_widget_show (label);

  for (i = 0; i < nbuttons; i++)
    {
      button = gtk_button_new_with_label (buttons[i]);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), button,
			  FALSE, TRUE, 0);
      if (i == default_button - 1)
	{
	  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	  gtk_widget_grab_default (button);
	}
      gtk_widget_show (button);

      if (signal_handlers[i])
	gtk_signal_connect (GTK_OBJECT (button), "clicked", signal_handlers[i],
			    data);

      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (dialog));
    }

  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (dialog);
}


/* This shows a dialog box with a message and an Entry for entering a value.
 * When the OK button is pressed the handler of the specified widget will
 * be called with the value and the given data.
 * NOTE: The dialog is automatically destroyed when any button is clicked.

 Example usage:
 show_entry_dialog ("Name:", on_dialog_ok, "NewName");

 void
 on_dialog_ok (GtkWidget *widget, gchar *value, gpointer data)
 {
 ...

 */
void
show_entry_dialog (gchar * message,
		   gchar * initial_value,
		   GtkWidget * widget,
		   GbEntryDialogFunc signal_handler,
		   gpointer data)
{
  GtkWidget *dialog, *hbox, *label, *entry, *button;

  dialog = gtk_dialog_new ();
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_container_border_width (GTK_CONTAINER (dialog), 5);

  /* FIXME: Passing a function as a gpointer isn't compatable with ANSI. */
  gtk_object_set_data (GTK_OBJECT (dialog), "handler", signal_handler);
  gtk_object_set_data (GTK_OBJECT (dialog), "widget", widget);
  gtk_widget_ref (widget);

  hbox = gtk_hbox_new (FALSE, 10);
  gtk_container_border_width (GTK_CONTAINER (hbox), 20);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox,
		      TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  label = gtk_label_new (message);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
  gtk_widget_show (label);

  entry = gtk_entry_new ();
  if (initial_value)
    gtk_entry_set_text (GTK_ENTRY (entry), initial_value);
  gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
  gtk_widget_show (entry);
  gtk_widget_grab_focus (entry);
  gtk_object_set_data (GTK_OBJECT (dialog), "entry", entry);

  button = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), button,
		      FALSE, TRUE, 20);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_entry_dialog_ok),
		      data);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), button,
		      FALSE, TRUE, 20);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      GTK_SIGNAL_FUNC (on_entry_dialog_destroy), widget);
  gtk_widget_show (dialog);
}


static void
on_entry_dialog_destroy (GtkWidget * widget, gpointer data)
{
  gtk_widget_unref (GTK_WIDGET (data));
}


static void
on_entry_dialog_ok (GtkWidget * widget, gpointer data)
{
  GtkWidget *dialog = gtk_widget_get_toplevel (widget);
  GbEntryDialogFunc handler;
  GtkWidget *entry;
  gchar *text;
  gboolean close = TRUE;

  handler = (GbEntryDialogFunc) (gtk_object_get_data (GTK_OBJECT (dialog),
						      "handler"));
  widget = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (dialog), "widget"));
  entry = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (dialog), "entry"));
  g_return_if_fail (entry != NULL);
  text = gtk_entry_get_text (GTK_ENTRY (entry));

  if (handler)
    close = (*handler) (widget, text, data);
  if (close)
    gtk_widget_destroy (dialog);
}


/* Returns TRUE if string contains substring. */
gboolean
gb_strstr (gchar * string, gchar * substring)
{
  gchar first_char;
  gint len, i, j;
  gboolean found;

  if (!string || !substring)
    return FALSE;

  first_char = substring[0];
  len = strlen (substring);

  if (strlen (string) < len)
    return FALSE;
  for (i = 0; i <= strlen (string) - len; i++)
    {
      if (string[i] == first_char)
	{
	  found = TRUE;
	  for (j = 1; j < len; j++)
	    {
	      if (string[i + j] != substring[j])
		{
		  found = FALSE;
		  break;
		}
	    }
	  if (found == TRUE)
	    return TRUE;
	}
    }
  return FALSE;
}


gint
gb_box_get_pos (GtkBox * box, GtkWidget * widget)
{
  GList *children;
  GtkBoxChild *child;
  guint pos = 0;

  children = box->children;
  while (children)
    {
      child = children->data;
      if (widget == child->widget)
	return pos;
      pos++;
      children = children->next;
    }
  g_warning (_("Widget not found in box"));
  return -1;
}


GtkTableChild *
gb_table_find_child (GtkTable * table, GtkWidget * widget)
{
  GList *children;
  GtkTableChild *child;

  children = table->children;
  while (children)
    {
      child = children->data;
      if (widget == child->widget)
	return child;
      children = children->next;
    }
  g_warning (_("Widget not found in table"));
  return NULL;
}


#ifdef GLD_HAVE_GTK_1_1
GtkPackerChild *
gb_packer_find_child (GtkPacker * packer, GtkWidget * widget)
{
  GList *children;
  GtkPackerChild *child;

  children = packer->children;
  while (children)
    {
      child = children->data;
      if (widget == child->widget)
	return child;
      children = children->next;
    }
  g_warning (_("Widget not found in packer"));
  return NULL;
}
#endif
