/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <sys/stat.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>
#include <ctype.h>
#include <locale.h>

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "gbwidget.h"
#include "source.h"

#define GB_SOURCE_BUFFER_INCREMENT	8192
#define GB_SOURCE_BUFFER_ENSURE_SPACE	4096

#define MAX_IDENTIFIER_LEN		1024

#define TEXT_BUFFER_INCREMENT		1024

static GbStatusCode real_source_write (gchar * project_name,
				       gchar * directory);
static void destroy_standard_widget (gchar * key,
				     GtkWidget * widget,
				     gpointer data);

static void source_write_configure_in (GbWidgetWriteSourceData * data);
static void source_write_makefile_am (GbWidgetWriteSourceData * data);
static void source_write_makefile (GbWidgetWriteSourceData * data);

static void source_write_main_c_preamble (GbWidgetWriteSourceData * data);
static void source_write_main_h_preamble (GbWidgetWriteSourceData * data);
static void source_write_signal_c_preamble (GbWidgetWriteSourceData * data);
static void source_write_signal_h_preamble (GbWidgetWriteSourceData * data);
static void source_write_preamble (GbWidgetWriteSourceData * data,
				   FILE * fp);

static void source_write_component_create (GtkWidget * component,
					   GbWidgetWriteSourceData * data);
static void source_write_component (GtkWidget * component,
				    GbWidgetWriteSourceData * data);
static gboolean file_exists(gchar *filename);
static GbStatusCode backup_file_if_exists (gchar * filename);

static void add_to_buffer (gchar ** buffer,
			   gint * buffer_pos,
			   gint * buffer_space,
			   gchar * text);
static void add_char_to_buffer (gchar ** buffer,
				gint * buffer_pos,
				gint * buffer_space,
				gchar ch);

/* We need this so that numbers are written in C syntax rather than the
   current locale, which may use ',' instead of '.' and then the code
   will not compile. This code is from glibc info docs. */
GbStatusCode
source_write (gchar * project_name, gchar * directory)
{
  gchar *old_locale, *saved_locale;
  GbStatusCode status;
     
  old_locale = setlocale (LC_NUMERIC, NULL);
  saved_locale = g_strdup (old_locale);
  setlocale (LC_NUMERIC, "C");
  status = real_source_write (project_name, directory);
  setlocale (LC_NUMERIC, saved_locale);
  g_free (saved_locale);
  return status;
}


static GbStatusCode
real_source_write (gchar * project_name, gchar * directory)
{
  GbWidgetWriteSourceData data;

  data.status = GB_OK;
  data.project_name = project_name;
  data.main_c_filename = "gladesrc.c";
  data.main_h_filename = "gladesrc.h";
  data.signal_c_filename = "gladesig.c";
  data.signal_h_filename = "gladesig.h";
  data.set_widget_names = FALSE;
  data.use_widget_hash = TRUE;
  data.standard_widgets = g_hash_table_new (g_str_hash, g_str_equal);

  if (chdir (directory) == -1)
    return GB_CHDIR_ERROR;

  source_write_configure_in (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_makefile_am (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_makefile (&data);
  if (data.status != GB_OK)
    return data.status;

  /* Create all 4 files. Note that we will not overwrite signal files in the
     final version. */
  backup_file_if_exists (data.main_c_filename);
  backup_file_if_exists (data.main_h_filename);
  backup_file_if_exists (data.signal_c_filename);
  backup_file_if_exists (data.signal_h_filename);

  data.cfp = data.hfp = data.sigcfp = data.sighfp = NULL;
  data.cfp = fopen (data.main_c_filename, "w");
  if (data.cfp == NULL)
    return GB_FILE_OPEN_ERROR;
  data.hfp = fopen (data.main_h_filename, "w");
  if (data.hfp == NULL)
    return GB_FILE_OPEN_ERROR;
  data.sigcfp = fopen (data.signal_c_filename, "w");
  if (data.sigcfp == NULL)
    return GB_FILE_OPEN_ERROR;
  data.sighfp = fopen (data.signal_h_filename, "w");
  if (data.sighfp == NULL)
    return GB_FILE_OPEN_ERROR;

  source_write_main_c_preamble (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_main_h_preamble (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_signal_c_preamble (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_signal_h_preamble (&data);
  if (data.status != GB_OK)
    return data.status;

  data.buffer_space = GB_SOURCE_BUFFER_INCREMENT;
  data.buffer = g_new (gchar, data.buffer_space);
  data.buffer[0] = '\0';
  data.buffer_pos = 0;

  data.decl_buffer_space = GB_SOURCE_BUFFER_INCREMENT;
  data.decl_buffer = g_new (gchar, data.decl_buffer_space);
  data.decl_buffer[0] = '\0';
  data.decl_buffer_pos = 0;

  /* This outputs code in main() to create one of each component, just so that
     the user sees something after first building the project. */
  project_foreach_component ((GtkCallback) source_write_component_create,
			     &data);
  fprintf (data.sigcfp, "%s", data.decl_buffer);
  fprintf (data.sigcfp,
	   "\n"
	   "  gtk_set_locale ();\n"
	   "  gtk_init (&argc, &argv);\n"
	   "\n"
	   "  /*\n"
	   "   * The following code was added by Glade to create one of each component\n"
	   "   * (except popup menus), just so that you see something after building\n"
	   "   * the project. Delete any components that you don't want shown initially.\n"
	   "   */\n");
  fprintf (data.sigcfp, "%s", data.buffer);
  fprintf (data.sigcfp, "\n  gtk_main ();\n  return 0;\n}\n\n");

  /* This outputs the code to create the components and the signal handler
     prototypes. */
  project_foreach_component ((GtkCallback) source_write_component, &data);

  g_free (data.buffer);
  g_free (data.decl_buffer);
  g_hash_table_foreach (data.standard_widgets,
			(GHFunc) destroy_standard_widget, NULL);
  g_hash_table_destroy (data.standard_widgets);


  if (data.cfp)
    fclose (data.cfp);
  if (data.hfp)
    fclose (data.hfp);
  if (data.sigcfp)
    fclose (data.sigcfp);
  if (data.sighfp)
    fclose (data.sighfp);
  return data.status;
}


static void
destroy_standard_widget (gchar * key, GtkWidget * widget, gpointer data)
{
  gtk_widget_destroy (widget);
}


static void
source_write_configure_in (GbWidgetWriteSourceData * data)
{
  FILE *fp;

  /* FIXME: If configure.in exists, just leave it, for now. */
  if (file_exists ("configure.in"))
    return;

  fp = fopen ("configure.in", "w");
  if (fp == NULL)
    {
      data->status = GB_FILE_OPEN_ERROR;
      return;
    }
  /* FIXME: May want to use 1.1.0 for GTK version 1.1, especially if I
     write accelerator code. */
  fprintf (fp,
      "dnl Process this file with autoconf to produce a configure script.\n"
	   "\n"
	   "AC_INIT(%s)\n"
	   "\n"
	   "\nAM_INIT_AUTOMAKE(%s, 0.1)\n"
	   "\n"
	   "AC_PROG_CC\n"
	   "\n"
	   "AM_PATH_GTK(1.0.0,\n"
  "            [LIBS=\"$LIBS $GTK_LIBS\" CFLAGS=\"$CFLAGS $GTK_CFLAGS\"],\n"
      "            AC_MSG_ERROR(Cannot find GTK: Is gtk-config in path?))\n"
	   "\n"
	   "dnl Only use -Wall if we have gcc\n"
	   "if test \"x$GCC\" = \"xyes\"; then\n"
	   "  if test -z \"`echo \"$CFLAGS\" | grep \"\\-Wall\" 2> /dev/null`\" ; then\n"
	   "    CFLAGS=\"$CFLAGS -Wall\"\n"
	   "  fi\n"
	   "fi\n"
	   "\n"
	   "AC_OUTPUT(Makefile)\n",
	   data->main_c_filename, data->project_name);
  fclose (fp);
}


static void
source_write_makefile_am (GbWidgetWriteSourceData * data)
{
  FILE *fp;

  /* FIXME: If Makefile.am exists, just leave it, for now. */
  if (file_exists ("Makefile.am"))
    return;

  fp = fopen ("Makefile.am", "w");
  if (fp == NULL)
    {
      data->status = GB_FILE_OPEN_ERROR;
      return;
    }
  /* 134 is '\', but I have trouble if I use '\' in the string. */
  fprintf (fp,
	   "## Process this file with automake to produce Makefile.in\n"
	   "\n"
	   "bin_PROGRAMS = %s\n"
	   "\n"
	   "%s_SOURCES = \134\n"
	   "        %s \134\n"
	   "        %s\n"
	   "\n"
	   "noinst_HEADERS = \134\n"
	   "        %s \134\n"
	   "        %s\n\n",
	   data->project_name, data->project_name,
	   data->main_c_filename, data->signal_c_filename,
	   data->main_h_filename, data->signal_h_filename);
  fclose (fp);
}


/* Output a simple Makefile to run aclocal, automake & autoconf to create
   the real Makefiles. */
static void
source_write_makefile (GbWidgetWriteSourceData * data)
{
  FILE *fp;

  /* If Makefile exists, just return. */
  if (file_exists ("Makefile"))
    return;

  fp = fopen ("Makefile", "w");
  if (fp == NULL)
    {
      data->status = GB_FILE_OPEN_ERROR;
      return;
    }
  fprintf (fp,
	   "all:\n"
	   "\taclocal\n"
	   "\tautomake -a --foreign\n"
	   "\tautoconf\n"
	   "\t./configure\n"
	   "\tmake\n");
  fclose (fp);
}


static void
source_write_main_c_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->cfp);

  fprintf (data->cfp,
	   "#include <gtk/gtk.h>\n"
	   "#include <gdk/gdkkeysyms.h>\n"
	   "#include \"%s\"\n"
	   "#include \"%s\"\n\n",
	   data->signal_h_filename, data->main_h_filename);

  /* Write a function to get a widget from the component's hash. */
  if (data->use_widget_hash)
    {
      fprintf (data->cfp,
	       "GtkWidget*\n"
	       "get_widget                             (GtkWidget       *widget,\n"
	       "                                        gchar           *widget_name)\n"
	       "{\n"
	       "  GtkWidget *found_widget;\n"
	       "\n"
	       "  if (widget->parent)\n"
	       "    widget = gtk_widget_get_toplevel (widget);\n"
	       "  found_widget = (GtkWidget*) gtk_object_get_data (GTK_OBJECT (widget),\n"
	       "                                                   widget_name);\n"
	       "  if (!found_widget)\n"
	       "    g_warning (\"Widget not found: %%s\", widget_name);\n"
	       "  return found_widget;\n"
	       "}\n\n");
    }

  /* Write a real kludgey function to set the tab of an already-created
     notebook page. */
  fprintf (data->cfp,
	   "/* This is an internally used function to set notebook tab widgets. */\n"
	   "void\n"
	   "set_notebook_tab                       (GtkWidget       *notebook,\n"
	   "                                        gint             page_num,\n"
	   "                                        GtkWidget       *widget)\n"
	   "{\n"

	   "  GtkNotebookPage *page;\n"
	   "  GtkWidget *notebook_page;\n"
	   "\n"
	   "  page = (GtkNotebookPage*) g_list_nth (GTK_NOTEBOOK (notebook)->children, page_num)->data;\n"
	   "  notebook_page = page->child;\n"
	   "  gtk_widget_ref (notebook_page);\n"
	   "  gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), page_num);\n"
	   "  gtk_notebook_insert_page (GTK_NOTEBOOK (notebook), notebook_page,\n"
	   "                            widget, page_num);\n"
	   "  gtk_widget_unref (notebook_page);\n"
	   "}\n\n");
}


static void
source_write_main_h_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->hfp);

  if (data->use_widget_hash)
    {
      fprintf (data->hfp,
	       "/*\n"
	       " * This function returns a widget in a component created by Glade.\n"
	       " * Call it with the toplevel widget in the component (i.e. a window/dialog),\n"
	       " * or alternatively any widget in the component, and the name of the widget\n"
	       " * you want returned.\n"
	       " */\n"
	       "GtkWidget*\n"
	       "get_widget                             (GtkWidget       *widget,\n"
	       "                                        gchar           *widget_name);\n"
	       "\n"
	       "\n"
	       " /*\n"
	       "  * This is an internally used function for setting notebook tabs. It is only\n"
	       "  * included in this header file so you don't get compilation warnings\n"
	       "  */\n"
	       "void\n"
	       "set_notebook_tab                       (GtkWidget       *notebook,\n"
	       "                                        gint             page_num,\n"
	       "                                        GtkWidget       *widget);\n"
	       "\n");
    }
}


static void
source_write_signal_c_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->sigcfp);

  fprintf (data->sigcfp,
	   "#include <gtk/gtk.h>\n"
	   "#include \"%s\"\n"
	   "#include \"%s\"\n\n",
	   data->main_h_filename, data->signal_h_filename);

  fprintf (data->sigcfp,
	   "int\n"
	   "main (int argc, char *argv[])\n"
	   "{\n");
}


static void
source_write_signal_h_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->sighfp);
}


static void
source_write_preamble (GbWidgetWriteSourceData * data, FILE * fp)
{
  /* Write license info. Note this will eventually be editable in the
     user interface, with an option to include a few standard licenses, e.g.
     GPL, which are then edited by the user. */
  fprintf (fp,
	   "/*  Note: You are free to use whatever license you want.\n"
	   "    Eventually you will be able to edit it within Glade. */\n"
	   "\n"
	   "/*  <PROJECT NAME>\n"
	   " *  Copyright (C) <YEAR> <AUTHORS>\n"
	   " *\n"
	   " *  This program is free software; you can redistribute it and/or modify\n"
	   " *  it under the terms of the GNU General Public License as published by\n"
	   " *  the Free Software Foundation; either version 2 of the License, or\n"
	   " *  (at your option) any later version.\n"
	   " *\n"
	   " *  This program is distributed in the hope that it will be useful,\n"
	   " *  but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	   " *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	   " *  GNU General Public License for more details.\n"
	   " *\n"
	   " *  You should have received a copy of the GNU General Public License\n"
	   " *  along with this program; if not, write to the Free Software\n"
	   " *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.\n"
	   "*/\n\n");

}


static void
source_write_component_create (GtkWidget * component,
			       GbWidgetWriteSourceData * data)
{
  gchar *component_name = gtk_widget_get_name (component);

  /* Don't show popup menus. */
  if (GTK_IS_MENU (component))
    return;

  component_name = source_create_valid_identifier (component_name);
  source_add_decl (data, "  GtkWidget *%s;\n", component_name);
  source_add (data, "  %s = create_%s ();\n  gtk_widget_show (%s);\n",
	      component_name, component_name, component_name);
  g_free (component_name);
}


static void
source_write_component (GtkWidget * component, GbWidgetWriteSourceData * data)
{
  data->component = component;
  data->component_name = source_create_valid_identifier (gtk_widget_get_name (component));
  data->need_tooltips = FALSE;
  data->need_accel_group = FALSE;
  data->buffer_pos = 0;
  data->decl_buffer_pos = 0;
  data->parent = NULL;
  data->create_widget = TRUE;
  data->write_children = TRUE;

  fprintf (data->cfp,
	   "GtkWidget*\n"
	   "create_%s ()\n"
	   "{\n",
	   data->component_name);

  fprintf (data->hfp,
	   "GtkWidget* create_%s (void);\n",
	   data->component_name);

  gb_widget_write_source (component, data);

  fprintf (data->cfp, "%s", data->decl_buffer);

  if (data->need_accel_group)
    {
#ifdef GLD_HAVE_GTK_1_1
      fprintf (data->cfp, "  GtkAccelGroup *accel_group;\n");
#else
      fprintf (data->cfp, "  GtkAcceleratorTable *accelerator_table;\n");
#endif
    }

  if (data->need_tooltips)
    {
      fprintf (data->cfp,
	       "  GtkTooltips *tooltips;\n"
	       "\n"
	       "  tooltips = gtk_tooltips_new ();\n\n");
    }
  else
    {
      fprintf (data->cfp, "\n");
    }
  fprintf (data->cfp, "%s", data->buffer);
  fprintf (data->cfp, "  return %s;\n}\n\n", data->component_name);
  g_free (data->component_name);
}


void
source_add (GbWidgetWriteSourceData * data, gchar * fmt,...)
{
  va_list args;

  if (data->buffer_space - data->buffer_pos < GB_SOURCE_BUFFER_ENSURE_SPACE)
    {
      data->buffer_space += GB_SOURCE_BUFFER_INCREMENT;
      data->buffer = g_realloc (data->buffer, data->buffer_space);
    }

  va_start (args, fmt);
  vsprintf (data->buffer + data->buffer_pos, fmt, args);
  data->buffer_pos += strlen (data->buffer + data->buffer_pos);
  va_end (args);
}


void
source_add_decl (GbWidgetWriteSourceData * data, gchar * fmt,...)
{
  va_list args;

  if (data->decl_buffer_space - data->decl_buffer_pos
      < GB_SOURCE_BUFFER_ENSURE_SPACE)
    {
      data->decl_buffer_space += GB_SOURCE_BUFFER_INCREMENT;
      data->decl_buffer = g_realloc (data->decl_buffer,
				     data->decl_buffer_space);
    }

  va_start (args, fmt);
  vsprintf (data->decl_buffer + data->decl_buffer_pos, fmt, args);
  data->decl_buffer_pos += strlen (data->decl_buffer + data->decl_buffer_pos);
  va_end (args);
}


static gboolean
file_exists(gchar *filename)
{
  int status;
  struct stat filestat;

  status = stat(filename, &filestat);
  if (status == -1 && errno == ENOENT)
    return FALSE;
  return TRUE;
}


static GbStatusCode
backup_file_if_exists (gchar * filename)
{
  int status;
  gchar buffer[1024];

  /* FIXME: better error handling. */
  if (file_exists (filename))
    {
      sprintf (buffer, "%s.bak", filename);
      status = rename (filename, buffer);
      if (status == 0)
	return GB_OK;
    }
  return GB_FILE_OPEN_ERROR;
}


gchar *
source_create_valid_identifier (gchar * name)
{
  gchar buffer[MAX_IDENTIFIER_LEN];
  gint i;

  if ((name[0] >= 'a' && name[0] <= 'z')
      || (name[0] >= 'A' && name[0] <= 'Z')
      || name[0] == '_')
    buffer[0] = name[0];
  else
    buffer[0] = '_';

  for (i = 1; i < strlen (name); i++)
    {
      if ((name[i] >= 'a' && name[i] <= 'z')
	  || (name[i] >= 'A' && name[i] <= 'Z')
	  || (name[i] >= '0' && name[i] <= '9')
	  || name[i] == '_')
	buffer[i] = name[i];
      else
	buffer[i] = '_';
    }
  buffer[i] = '\0';

  return g_strdup (buffer);
}


/* This converts a string so that it can be output as part of the C source
 * code. It converts non-printable characters to escape codes.
 * Note that it uses one dynamically allocated buffer, so the result is only
 * valid until the next call to the function.
 */
gchar *
source_make_string (gchar * text)
{
  static gchar *buffer = NULL;
  static gint buffer_space = 0;
  static gint buffer_pos;

  gchar escape_buffer[16];
  gchar *p;

  g_return_val_if_fail (text != NULL, "");

  buffer_pos = 0;
  for (p = text; *p; p++)
    {
      switch (*p)
	{
	case '\n':
	  add_to_buffer (&buffer, &buffer_pos, &buffer_space, "\\n");
	  break;
	case '\r':
	  add_to_buffer (&buffer, &buffer_pos, &buffer_space, "\\r");
	  break;
	case '\t':
	  add_to_buffer (&buffer, &buffer_pos, &buffer_space, "\\t");
	  break;
	case '\\':
	  add_to_buffer (&buffer, &buffer_pos, &buffer_space, "\\\\");
	  break;
	case '"':
	  add_to_buffer (&buffer, &buffer_pos, &buffer_space, "\\\"");
	  break;
	default:
	  if (isprint (*p))
	    add_char_to_buffer (&buffer, &buffer_pos, &buffer_space, *p);
	  else
	    {
	      sprintf (escape_buffer, "\\%02o", (guchar) *p);
	      add_to_buffer (&buffer, &buffer_pos, &buffer_space,
			     escape_buffer);
	      break;
	    }
	}
    }
  add_char_to_buffer (&buffer, &buffer_pos, &buffer_space, '\0');

  return buffer;
}


static void
add_char_to_buffer (gchar ** buffer, gint * buffer_pos, gint * buffer_space,
		    gchar ch)
{
  if (*buffer_pos == *buffer_space)
    {
      *buffer_space += TEXT_BUFFER_INCREMENT;
      *buffer = g_realloc (*buffer, *buffer_space);
    }
  *(*buffer + *buffer_pos) = ch;
  *buffer_pos = *buffer_pos + 1;
}


static void
add_to_buffer (gchar ** buffer, gint * buffer_pos, gint * buffer_space,
	       gchar * text)
{
  gchar *pos;

  for (pos = text; *pos; pos++)
    add_char_to_buffer (buffer, buffer_pos, buffer_space, *pos);
}
