
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "../gb.h"

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
/* This widget has no icon as it doesn't appear in the palette */
/*#include "../graphics/" */

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */
static GbWidget gbwidget;

/* These are the special children of the widget. */
static gchar *ColorSelOKButton = "ColorSel:ok_button";
static gchar *ColorSelCancelButton = "ColorSel:cancel_button";
static gchar *ColorSelHelpButton = "ColorSel:help_button";

static gchar *Title = "ColorSelDialog|GtkWindow::title";
static gchar *Type = "ColorSelDialog|GtkWindow::type";
static gchar *Position = "ColorSelDialog|GtkWindow::position";
static gchar *Shrink = "ColorSelDialog|GtkWindow::allow_shrink";
static gchar *Grow = "ColorSelDialog|GtkWindow::allow_grow";
static gchar *AutoShrink = "ColorSelDialog|GtkWindow::auto_shrink";
static gchar *Policy = "ColorSelDialog|GtkColorSelection::policy";
static gchar *WMName = "ColorSelDialog|GtkWindow::wmclass_name";
static gchar *WMClass = "ColorSelDialog|GtkWindow::wmclass_class";

static gchar *GbTypeChoices[] =
{"Top Level", "Dialog", "Popup", NULL};
static gint GbTypeValues[] =
{
  GTK_WINDOW_TOPLEVEL,
  GTK_WINDOW_DIALOG,
  GTK_WINDOW_POPUP
};
static gchar *GbTypeSymbols[] =
{
  "GTK_WINDOW_TOPLEVEL",
  "GTK_WINDOW_DIALOG",
  "GTK_WINDOW_POPUP"
};

static gchar *GbPositionChoices[] =
{"None", "Center", "Mouse", NULL};
static gint GbPositionValues[] =
{
  GTK_WIN_POS_NONE,
  GTK_WIN_POS_CENTER,
  GTK_WIN_POS_MOUSE
};
static gchar *GbPositionSymbols[] =
{
  "GTK_WIN_POS_NONE",
  "GTK_WIN_POS_CENTER",
  "GTK_WIN_POS_MOUSE"
};

static gchar *GbPolicyChoices[] =
{"Continuous", "Discontinuous", "Delayed",
 NULL};
static gint GbPolicyValues[] =
{
  GTK_UPDATE_CONTINUOUS,
  GTK_UPDATE_DISCONTINUOUS,
  GTK_UPDATE_DELAYED
};
static gchar *GbPolicySymbols[] =
{
  "GTK_UPDATE_CONTINUOUS",
  "GTK_UPDATE_DISCONTINUOUS",
  "GTK_UPDATE_DELAYED"
};


/******
 * NOTE: To use these functions you need to uncomment them AND add a pointer
 * to the function in the GbWidget struct at the end of this file.
 ******/

/*
 * Creates a new GtkWidget of class GtkColorSelectionDialog, performing any specialized
 * initialization needed for the widget to work correctly in this environment.
 * If a dialog box is used to initialize the widget, return NULL from this
 * function, and call data->callback with your new widget when it is done.
 * If the widget needs a special destroy handler, add a signal here.
 */
GtkWidget *
gb_color_selection_dialog_new (GbWidgetNewData * data)
{
  GtkWidget *new_widget = gtk_color_selection_dialog_new (_("Select Color"));
  gtk_signal_connect (GTK_OBJECT (new_widget), "delete_event",
		      GTK_SIGNAL_FUNC (editor_close_window), NULL);

  gb_widget_create_from (GTK_COLOR_SELECTION_DIALOG (new_widget)->ok_button,
			 "ok_button");
  gtk_object_set_data (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (new_widget)->ok_button),
		       GB_CHILD_NAME_KEY, g_strdup (ColorSelOKButton));

  gb_widget_create_from (GTK_COLOR_SELECTION_DIALOG (new_widget)->cancel_button,
			 "cancel_button");
  gtk_object_set_data (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (new_widget)->cancel_button),
		       GB_CHILD_NAME_KEY, g_strdup (ColorSelCancelButton));

  gb_widget_create_from (GTK_COLOR_SELECTION_DIALOG (new_widget)->help_button,
			 "help_button");
  gtk_object_set_data (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (new_widget)->help_button),
		       GB_CHILD_NAME_KEY, g_strdup (ColorSelHelpButton));

  return new_widget;
}



/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_color_selection_dialog_create_properties (GtkWidget * widget, GbWidgetCreateArgData
					     * data)
{
  property_add_string (Title, _("Title:"), _("The title of the window"));
  property_add_choice (Type, _("Type:"),
		       _("The type of the window"),
		       GbTypeChoices);
  property_add_choice (Position, _("Position:"),
		       _("The initial position of the dialog"),
		       GbPositionChoices);
  property_add_bool (Shrink, _("Shrink:"), _("If the window can be shrunk"));
  property_add_bool (Grow, _("Grow:"), _("If the window can be enlarged"));
  property_add_bool (AutoShrink, _("Auto Shrink:"),
		     _("If the window shrinks automatically"));
  property_add_choice (Policy, _("Update Policy:"),
		       _("The update policy of the color selection scales"),
		       GbPolicyChoices);
  property_add_string (WMName, _("WM Name:"), _("The name to pass to the window manager"));
  property_add_string (WMClass, _("WM Class:"), _("The class name to pass to the window manager"));
}



/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_color_selection_dialog_get_properties (GtkWidget * widget, GbWidgetGetArgData
					  * data)
{
  GtkColorSelection *colorsel = GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG
						     (widget)->colorsel);
  gint i, type;

  gb_widget_output_string (data, Title, GTK_WINDOW (widget)->title);

  /* The type is stored in the object hash as we can't change it after the
     window is realized. It will default to 0, which is OK. */
  type = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), Type));
  gb_widget_output_choice (data, Type, type, GbTypeSymbols[type]);

  for (i = 0; i < sizeof (GbPositionValues) / sizeof (GbPositionValues[0]); i
       ++)
    {
      if (GbPositionValues[i] == GTK_WINDOW (widget)->position)
	gb_widget_output_choice (data, Position, i, GbPositionSymbols[i]);
    }
  gb_widget_output_bool (data, Shrink, GTK_WINDOW (widget)->allow_shrink);
  gb_widget_output_bool (data, Grow, GTK_WINDOW (widget)->allow_grow);
  gb_widget_output_bool (data, AutoShrink, GTK_WINDOW (widget)->auto_shrink);
  for (i = 0; i < sizeof (GbPolicyValues) / sizeof (GbPolicyValues[0]); i++)
    {
      if (GbPolicyValues[i] == colorsel->policy)
	gb_widget_output_choice (data, Policy, i, GbPolicySymbols[i]);
    }

  /* These are stored in the object hash since we can't set them after the
     window is realized. */
  gb_widget_output_string (data, WMName,
			 gtk_object_get_data (GTK_OBJECT (widget), WMName));
  gb_widget_output_string (data, WMClass,
			gtk_object_get_data (GTK_OBJECT (widget), WMClass));
}



/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_color_selection_dialog_set_properties (GtkWidget * widget, GbWidgetSetArgData
					  * data)
{
  GtkColorSelection *colorsel = GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG
						     (widget)->colorsel);
  gint i;
  gchar *title, *type, *position, *policy, *wmname, *wmclass;
  gboolean shrink, grow, auto_shrink, myApply;

  title = gb_widget_input_string (data, Title);
  if (data->apply)
    gtk_window_set_title (GTK_WINDOW (widget), title);

  type = gb_widget_input_choice (data, Type);
  if (data->apply)
    {
      for (i = 0; i < sizeof (GbTypeValues) / sizeof (GbTypeValues[0]); i++)
	{
	  if (!strcmp (type, GbTypeChoices[i])
	      || !strcmp (type, GbTypeSymbols[i]))
	    {
	      gtk_object_set_data (GTK_OBJECT (widget), Type, GINT_TO_POINTER
				   (i));
	      break;
	    }
	}
    }

  position = gb_widget_input_choice (data, Position);
  if (data->apply)
    {
      for (i = 0; i < sizeof (GbPositionValues) / sizeof (GbPositionValues[0]);
	   i++)
	{
	  if (!strcmp (position, GbPositionChoices[i])
	      || !strcmp (position, GbPositionSymbols[i]))
	    {
	      gtk_window_position (GTK_WINDOW (widget), GbPositionValues[i]);
	      break;
	    }
	}
    }

  shrink = gb_widget_input_bool (data, Shrink);
  myApply = data->apply;
  grow = gb_widget_input_bool (data, Grow);
  myApply |= data->apply;
  auto_shrink = gb_widget_input_bool (data, AutoShrink);
  if (data->apply || myApply)
    gtk_window_set_policy (GTK_WINDOW (widget), shrink, grow, auto_shrink);

  policy = gb_widget_input_choice (data, Policy);
  if (data->apply)
    {
      for (i = 0; i < sizeof (GbPolicyValues) / sizeof (GbPolicyValues[0]); i
	   ++)
	{
	  if (!strcmp (policy, GbPolicyChoices[i])
	      || !strcmp (policy, GbPolicySymbols[i]))
	    {
	      gtk_color_selection_set_update_policy (colorsel, GbPolicyValues
						     [i]);
	      break;
	    }
	}
    }

  wmname = gb_widget_input_string (data, WMName);
  if (wmname && wmname[0] == '\0')
    wmname = NULL;
  if (data->apply)
    {
      g_free (gtk_object_get_data (GTK_OBJECT (widget), WMName));
      gtk_object_set_data (GTK_OBJECT (widget), WMName, g_strdup (wmname));
    }

  wmclass = gb_widget_input_string (data, WMClass);
  if (wmclass && wmclass[0] == '\0')
    wmclass = NULL;
  if (data->apply)
    {
      g_free (gtk_object_get_data (GTK_OBJECT (widget), WMClass));
      gtk_object_set_data (GTK_OBJECT (widget), WMClass, g_strdup (wmclass));
    }
}



/*
 * Adds menu items to a context menu which is just about to appear!
 * Add commands to aid in editing a GtkColorSelectionDialog, with signals pointing to
 * other functions in this file.
 */
/*
   static void
   gb_color_selection_dialog_create_popup_menu(GtkWidget *widget, GbWidgetCreateMenuData *data)
   {

   }
 */



/*
 * Writes the source code needed to create this widget.
 * You have to output everything necessary to create the widget here, though
 * there are some convenience functions to help.
 */
static void
gb_color_selection_dialog_write_source (GtkWidget * widget, GbWidgetWriteSourceData
					* data)
{
  GtkColorSelection *colorsel = GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG
						     (widget)->colorsel);
  gint type, i;
  gchar *wmname, *wmclass;
  gchar *wname, *child_name;

  if (data->create_widget)
    {
      source_add (data, "  %s = gtk_color_selection_dialog_new (\"%s\");\n",
	      data->wname, source_make_string (GTK_WINDOW (widget)->title));
    }

  gb_widget_write_standard_source (widget, data);

  type = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), Type));
  if (type != GTK_WINDOW_TOPLEVEL)
    {
      /* Note: there is no gtk_window_set_type () */
      source_add (data, "  GTK_WINDOW (%s)->type = %s;\n",
		  data->wname, GbTypeSymbols[type]);
    }

  if (GTK_WINDOW (widget)->position != GTK_WIN_POS_NONE)
    {
      for (i = 0; i < sizeof (GbPositionValues) / sizeof (GbPositionValues[0]);
	   i++)
	{
	  if (GbPositionValues[i] == GTK_WINDOW (widget)->position)
	    source_add (data, "  gtk_window_position (GTK_WINDOW (%s), %s);\n",
			data->wname, GbPositionSymbols[i]);
	}
    }

  if (GTK_WINDOW (widget)->allow_shrink != FALSE
      || GTK_WINDOW (widget)->allow_grow != TRUE
      || GTK_WINDOW (widget)->auto_shrink != FALSE)
    source_add (data,
		"  gtk_window_set_policy (GTK_WINDOW (%s), %s, %s, %s);\n",
		data->wname,
		GTK_WINDOW (widget)->allow_shrink ? "TRUE" : "FALSE",
		GTK_WINDOW (widget)->allow_grow ? "TRUE" : "FALSE",
		GTK_WINDOW (widget)->auto_shrink ? "TRUE" : "FALSE");

  if (colorsel->policy != GTK_UPDATE_CONTINUOUS)
    {
      for (i = 0; i < sizeof (GbPolicyValues) / sizeof (GbPolicyValues[0]); i
	   ++)
	{
	  if (GbPolicyValues[i] == colorsel->policy)
	    source_add (data,
			"  gtk_color_selection_set_update_policy (GTK_COLOR_SELECTION (GTK_COLOR_SELECTION_DIALOG (%s)->colorsel), %s);\n",
			data->wname, GbPolicySymbols[i]);
	}
    }

  wmname = gtk_object_get_data (GTK_OBJECT (widget), WMName);
  wmclass = gtk_object_get_data (GTK_OBJECT (widget), WMClass);
  if (wmname || wmclass)
    source_add (data,
	    "  gtk_window_set_wmclass (GTK_WINDOW (%s), \"%s\", \"%s\");\n",
		data->wname, wmname ? source_make_string (wmname) : "",
		wmclass ? source_make_string (wmclass) : "");

  /* We output the source code for the buttons here, but we don't want them
     to be created. We need to remember the dialog's name since data->wname
     will be overwritten. */
  wname = g_strdup (data->wname);

  source_add (data, "\n");
  child_name = gtk_widget_get_name (GTK_COLOR_SELECTION_DIALOG (widget)->ok_button);
  child_name = source_create_valid_identifier (child_name);
  source_add (data, "  %s = GTK_COLOR_SELECTION_DIALOG (%s)->ok_button;\n",
	      child_name, wname);
  g_free (child_name);
  data->create_widget = FALSE;
  gb_widget_write_source (GTK_COLOR_SELECTION_DIALOG (widget)->ok_button,
			  data);

  child_name = gtk_widget_get_name (GTK_COLOR_SELECTION_DIALOG (widget)->cancel_button);
  child_name = source_create_valid_identifier (child_name);
  source_add (data, "  %s = GTK_COLOR_SELECTION_DIALOG (%s)->cancel_button;\n",
	      child_name, wname);
  g_free (child_name);
  data->create_widget = FALSE;
  gb_widget_write_source (GTK_COLOR_SELECTION_DIALOG (widget)->cancel_button,
			  data);

  child_name = gtk_widget_get_name (GTK_COLOR_SELECTION_DIALOG (widget)->help_button);
  child_name = source_create_valid_identifier (child_name);
  source_add (data, "  %s = GTK_COLOR_SELECTION_DIALOG (%s)->help_button;\n",
	      child_name, wname);
  g_free (child_name);
  data->create_widget = FALSE;
  gb_widget_write_source (GTK_COLOR_SELECTION_DIALOG (widget)->help_button,
			  data);

  g_free (wname);

  data->write_children = FALSE;
}



static GtkWidget *
gb_color_selection_dialog_get_child (GtkWidget * widget,
				     gchar * child_name)
{
  if (!strcmp (child_name, ColorSelOKButton))
    return GTK_COLOR_SELECTION_DIALOG (widget)->ok_button;
  else if (!strcmp (child_name, ColorSelCancelButton))
    return GTK_COLOR_SELECTION_DIALOG (widget)->cancel_button;
  else if (!strcmp (child_name, ColorSelHelpButton))
    return GTK_COLOR_SELECTION_DIALOG (widget)->help_button;
  else
    return NULL;
}



/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
GbWidget *
gb_color_selection_dialog_init ()
{
  /* Initialise the GTK type */
  gtk_color_selection_dialog_get_type ();

  /* Initialize the GbWidget structure */
  gb_widget_init_struct (&gbwidget);

  /* Fill in the pixmap struct & tooltip */
  gbwidget.pixmap_struct = NULL;
  gbwidget.tooltip = NULL;

  /* Fill in any functions that this GbWidget has */
  gbwidget.gb_widget_new = gb_color_selection_dialog_new;
  gbwidget.gb_widget_create_properties = gb_color_selection_dialog_create_properties;
  gbwidget.gb_widget_get_properties = gb_color_selection_dialog_get_properties;
  gbwidget.gb_widget_set_properties = gb_color_selection_dialog_set_properties;
  gbwidget.gb_widget_get_child = gb_color_selection_dialog_get_child;
  gbwidget.gb_widget_write_source = gb_color_selection_dialog_write_source;
/*
   gbwidget.gb_widget_create_popup_menu = gb_color_selection_dialog_create_popup_menu;
 */

  return &gbwidget;
}
