/* termcap.cxx
     $Id: termcap.cxx,v 1.19 1998/10/09 06:17:55 elf Exp $

   written by Marc Singer
   20 December 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   in a file called COPYING along with this program; if not, write to
   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
   02139, USA.

   -----------
   DESCRIPTION
   -----------

   Termcap and Terminfo interpretation.


   --------------------
   IMPLEMENTATION NOTES
   --------------------

   -- Termcap entries of '@'

   While the manual pages for termcap do not describe this feature, it
   appears to be implemented by ncurses.  The source code returns NULL
   when the termcap field is an '@' character.  We assume that this is
   a means to desable a termcap field and is equivalent to the field
   being absent from the termcap entry and any tc referenced entries.
   It is possible that some fields have defaults that must be disabled
   explicitly which we do so.

   -- Sloppy output

   We are not being careful to use output routines that work
   regardless of the string content.  We need to a) allow for explict
   specification of the output stream, and b) emit all characters in a
   string in the event that the user's terminal escape sequences
   include nul characters or (possibly) cooked control characters.

*/

#include "std.h"
//#include "preferences.h"

#include <stdarg.h>
#include <sys/stat.h>
#include <termios.h>		// ** FIXME need config for this
#include <sys/ioctl.h>
#include <errno.h>

#include "mman.h"
#include "termcap.h"

const char g_szPathTermcap[] = "/etc/termcap";
const char g_szPathTerminfo[] = "/usr/lib/terminfo";
const char g_szEnvTerm[] = "TERM";

bool Termcap::g_fValid;
bool Termcap::g_fActive;
bool Termcap::g_fTerminfo;
void* Termcap::g_pvTerminfo;
int Termcap::g_cLines;		// Screen size, zeroes when not a tty
int Termcap::g_cColumns;

#define __I(p)  #p
#define _I(p)   __I(p)
 
#if defined (USE_TERMCAPNAME)
# define __(id,sz,szT,ind,fl) { id, sz, fl, ind, _I(id) },
#else
# define __(id,sz,szT,ind,fl) { id, sz, fl, ind, "" },
#endif
TERMCAP_FIELD Termcap::g_rgTermcapField[] = {
# include "termcap.f"
  { 0 },	// Note that sentinel is marked by blank string, not zero id
};
#undef __


/* Termcap::activate

   performs initialization and restoration of terminal modes on
   activate/deactivation of the application.  It is customary to
   deactivate when the application is suspended and when it
   terminates.

*/

void Termcap::activate (bool fActivate)
{
  if (fActivate == g_fActive)
    return;

  char sz[80];
  int cch = 0;
  if (fActivate) {
    cch += compose (termcapInit, sz + cch);
    cch += compose (termcapInitCursorMotion, sz + cch);
    cch += compose (termcapEnableKeypad, sz + cch);
  }
  else {
    cch += compose (termcapDisableKeypad, sz + cch);
    cch += compose (termcapResetCursorMotion, sz + cch);
    cch += compose (termcapReset, sz + cch);
  }
  sz[cch] = 0;
  fprintf (stdout, sz);
  g_fActive = fActivate;
}


/* Termcap::compose 

   prepares and selects the appropriate composition function.

*/

int Termcap::compose (int field, char* pb, ...) const
{
  va_list ap;
  va_start (ap, pb);

  int result;
  if (g_fTerminfo)
    result = _compose_i (field, pb, ap);
  else
    result = _compose_c (field, pb, ap);

  va_end (ap);

  return result;
}


/* Termcap::_compose_c

   composes a method string from the parameters passed using termcap
   rules.  The return value is the number of characters copied to the
   buffer.  The output is NOT null terminated.

   The state values are:
     0) copy characters from the method definition to the output buffer.
     1) ^ escape encountered
     2) \ escape encountered
     3) % escape encountered
     4) \0 encountered

*/

int Termcap::_compose_c (int field, char* pb, va_list ap) const
{
  TERMCAP_FIELD* pField = locate (field);

  if (!pField
      || (pField->flags & termcapClassMask) != termcapClassMethod 
      || !pField->pv)
    return 0;

  int cchCopied = 0;
  const char* pbMethod = (const char*) pField->pv;

  int state = 0;
  int cIncrement = 0;
  for (; *pbMethod; ++pbMethod) {
    switch (state) {
    case 0:
      if (*pbMethod == '^')
	state = 1;
      else if (*pbMethod == '\\')
	state = 2;
      else if (*pbMethod == '%')
	state = 3;
      else {
	*pb++ = *pbMethod;
	++cchCopied;
      }
      break;

    case 1:
      *pb++ = toupper (*pbMethod) - '@';
      ++cchCopied;
      state = 0;
      break;

    case 2:
      if (*pbMethod == '0') {
	state = 4;
	break;
      }
      switch (*pbMethod) {
      case 'E':
	*pb++ = 27;		// escape
	break;
      case 'n':
	*pb++ = 10;		// newline
	break;
      case 'r':
	*pb++ = 13;		// carriage return
	break;
      case 't':
	*pb++ = 9;		// tab
	break;
      case 'b':
	*pb++ = 8;		// backspace
	break;
      case 'f':
	*pb++ = 12;		// form feed
	break;
      default:
	assert_ (0);
	break;
      }
      ++cchCopied;
      state = 0;
      break;

    case 3:
      switch (*pbMethod) {

      case 'i':			
	cIncrement = 2;		// Increment next two parameters
	break;
//      case 'r':			// single parameter capability?
//      case '+':

      case '2':
	{
	  int value = va_arg (ap, int) + (cIncrement ? (--cIncrement, 1) : 0);
	  int cch = sprintf (pb, "%02d", value);
	  pb += cch;
	  cchCopied += cch;
	}
	break;

      case 'd':
	{
	  int value = va_arg (ap, int) + (cIncrement ? (--cIncrement, 1) : 0);
	  int cch = sprintf (pb, "%03d", value);
	  pb += cch;
	  cchCopied += cch;
	}
	break;

      case '%':
	*pb++ = '%';
	++cchCopied;
	break;

      default:
	assert_ (0);
	break;
      }
      state = 0;
      break;

    case 4:
      if (!isdigit (*pbMethod) || *pbMethod == '8' || *pbMethod == '9') {
	*pb++ = '0';
	++cchCopied;
	break;
      }
      assert_ (isdigit (pbMethod[1]) && isdigit (pbMethod[2]));
      *pb++ = ((pbMethod[0] - '0') << 6)
	| ((pbMethod[1] - '0') << 3)
	|  (pbMethod[0] - '0');
      ++cchCopied;
      state = 0;
      break;
    }
  }

  return cchCopied;
}


/* Termcap::_compose_i

   composes a method string from the parameters passed using terminfo
   rules.  The return value is the number of characters copied to the
   buffer.  The output is NOT null terminated.

   The state values are:
     0) copy characters from the method definition to the output buffer.
     1) % escape encountered
     2) %p encountered
     3) %{ encountered


   The following is copied from the ncurses sources.  We do not
   support all of these, yet.

   %%        outputs `%'
   %d        print pop() like %d in printf()
   %2d       print pop() like %2d in printf()
   %02d      print pop() like %02d in printf()
   %3d       print pop() like %3d in printf()
   %03d      print pop() like %03d in printf()
   %2x       print pop() like %2x in printf()
   %02x      print pop() like %02x in printf()
   %3x       print pop() like %3x in printf()
   %03x      print pop() like %03x in printf()
   %c        print pop() like %c in printf()
   %s        print pop() like %s in printf()
	
   %p[1-9]   push ith parm
   %P[a-z]   set variable [a-z] to pop()
   %g[a-z]   get variable [a-z] and push it
   %'c'      push char constant c
   %{nn}     push integer constant nn
	
   %+ %- %* %/ %m
	     arithmetic (%m is mod): push(pop() op pop())
   %& %| %^  bit operations: push(pop() op pop())
   %= %> %<  logical operations: push(pop() op pop())
   %A %O     logical and & or operations for conditionals
   %! %~     unary operations push(op pop())
   %i        add 1 to first two parms (for ANSI terminals)
	
   %? expr %t thenpart %e elsepart %;
             if-then-else, %e elsepart is optional.
	     else-if's are possible ala Algol 68:
	     %? c1 %t b1 %e c2 %t b2 %e c3 %t b3 %e c4 %t b4 %e b5 %;
	
   For those of the above operators which are binary and not commutative,
   the stack works in the usual way, with
             %gx %gy %m
   resulting in x mod y, not the reverse.

*/

int Termcap::_compose_i (int field, char* pb, va_list ap) const
{
  TERMCAP_FIELD* pField = locate (field);

  if (!pField
      || (pField->flags & termcapClassMask) != termcapClassMethod 
      || !pField->pv)
    return 0;

  int cchCopied = 0;
  const char* pbMethod = (const char*) pField->pv;

  int rgParam[9];		// Parameters from the va_list
  int rgStack[9];		// Stack of pushed parameters
  int cParamLim = 0;		// Number of parameters pulled from va_list
  int cStackLim = 0;		// Current top (unused slot) of stack
  int iConstant = 0;

  int cWidth = 0;
  bool fLeadingZero = false;
  int cIncrement = 0;

  int state = 0;
  for (; *pbMethod; ++pbMethod) {
    switch (state) {
    case 0:
      if (*pbMethod == '%')
	state = 1;
      else {
	*pb++ = *pbMethod;
	++cchCopied;
      }
      break;

    case 1:
      switch (*pbMethod) {

      case 'c':
      case 'd':
      case 'x':
	assert_ (cStackLim);
	{
	  char szFmt[10];
	  char* pchFmt = szFmt;
	  *pchFmt++ = '%';
	  if (fLeadingZero)
	    *pchFmt++ = '0';
	  if (cWidth)
	    *pchFmt++ = cWidth + '0';
	  *pchFmt++ = *pbMethod;
	  *pchFmt = 0;
	  int cch = sprintf (pb, szFmt, rgStack[--cStackLim]);
	  pb += cch;
	  cchCopied += cch;
	}
	state = 0;
	break;

      case 'i':			
	cIncrement = 2;		// Increment next two parameters
	state = 0;
	break;

      case 'p':			// Push numbered parameter
	state = 2;
	break;

      case '0':
	assert_ (cWidth == 0);
	fLeadingZero = true;
	break;

      case '2':
	assert_ (cWidth == 0);
	cWidth = 2;
	break;

      case '3':
	assert_ (cWidth == 0);
	cWidth = 3;
	break;

      case '%':
	*pb++ = '%';
	++cchCopied;
	state = 0;
	break;

      case '{':
	iConstant = 0;
	state = 3;
	break;

      case '+':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] + rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case '-':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] - rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case '/':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] / rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case '*':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] * rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case 'm':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] % rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case '&':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] & rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case '|':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] | rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case '^':
	assert_ (cStackLim > 1);
	rgStack[cStackLim - 2] = 
	  rgStack[cStackLim - 1] ^ rgStack[cStackLim - 2];
	--cStackLim;
	state = 0;
	break;

      case '!':
	assert_ (cStackLim > 0);
	rgStack[cStackLim - 1] = !rgStack[cStackLim - 1];
	state = 0;
	break;

      case '~':
	assert_ (cStackLim > 0);
	rgStack[cStackLim - 1] = ~rgStack[cStackLim - 1];
	state = 0;
	break;

      default:
	assert_ (0);
	break;
      }
      break;

    case 2:
      assert_ (isdigit (*pbMethod));
      while (cParamLim < (*pbMethod - '0'))
	rgParam[cParamLim++] 
	  = va_arg (ap, int) + (cIncrement ? (--cIncrement, 1) : 0);
				// -- push
      assert_ (cStackLim < int (sizeof (rgStack)/sizeof (int)));
      rgStack[cStackLim++] = rgParam[*pbMethod - '1'];
      state = 0;
      break;
      
    case 3:
      assert_ (cStackLim < int (sizeof (rgStack)/sizeof (int)));
      if (*pbMethod == '}') {
	rgStack[cStackLim++] = iConstant;		// push
	state = 0;
      }
      else if (isdigit (*pbMethod))
	iConstant = iConstant*10 + *pbMethod - '0';	// accumulate
      else
	assert_ (0);
      break;
    }
  }

  return cchCopied;
}


void Termcap::init (void)
{
  TRACE ((T_TC_INFO, "Termcap::init for '%s'", getenv (g_szEnvTerm)));
  parse_terminfo (getenv (g_szEnvTerm)) ||
    parse_termcap (getenv (g_szEnvTerm));


  if (isatty (1)) {
    struct winsize size;

    errno = 0;
    do {
      if (ioctl(1, TIOCGWINSZ, &size) < 0
	  && errno != EINTR)
	break;
    } while (errno == EINTR);

    if (!errno) {
      g_cLines   = int (size.ws_row);
      g_cColumns = int (size.ws_col);
    }
  }


#if 0
    // A little piece of testarossa
    {
      fprintf (stderr, "Yow");
      char sz[80];
      int cch = compose (termcapClearScreen, sz);
      cch += compose (termcapMoveTo, sz + cch, 7, 12);
      sz[cch] = 0;
      strcat (sz, "Howdy");
      fprintf (stderr, sz);
    }
#endif

}


void Termcap::interpret_termcap_string (char* sz, const char* szSrc, int cch)
{
  int state = 0;
  while (cch--) {
    switch (state) {
    case 0:
      switch (*szSrc) {
      case '^':
	state = 1;
	break;
      case '\\':
	state = 2;
	break;
      default:
	*sz++ = *szSrc;
	break;
      }
      break;
    case 1:
      *sz++ = toupper (*szSrc) - '@';
      state = 0;
      break;
    case 2:
      switch (*szSrc) {
      case '\\':
	*sz++ = '\\';
	break;
      case 'E':
	*sz++ = '\e';
	break;
      default:
	*sz++ = *szSrc;
	break;
      }
      break;
    }
    ++szSrc;
  }
  *sz = 0;
}


/* Termcap::keyof

   returns a termcap id if there is a matching keystroke, termpcapNull
   if there is no possible keystroke match, or termcapCloseMatch if
   the given string matches a keystroke prefix.

*/

int Termcap::keyof (const char* rgb, size_t cch)
{
  for (TERMCAP_FIELD* pField = &g_rgTermcapField[0]; pField->sz[0]; ++pField) {
    if ((pField->flags & termcapClassMask) != termcapClassKey || !pField->pv)
      continue;
    if (memcmp (rgb, pField->pv, cch))
      continue;
    if (strlen ((const char*) pField->pv) == cch)
      return KEYCODE (pField->id);
    else
      return termcapCloseMatch;
  }
  return termcapNul;
}


/* Termcap::num

   returns the numeric field value.

*/

int Termcap::num (int field)
{
  TERMCAP_FIELD* pField = locate (field);
  if (!pField
      || (pField->flags & termcapClassMask) != termcapClassNumber
      || !pField->pv)
    return 0;

  switch (field) {
  case termcapNumLines:
    return g_cLines ? g_cLines : int (pField->pv);
  case termcapNumColumns:
    return g_cColumns ? g_cColumns : int (pField->pv);

  default:
    return int (pField->pv);
  }
}


/* Termcap::parse_field

   processes the data for a termcap field.  It sets booleans, converts
   numeric arguments, stores strings for methods and data items and
   keys, and it interprets the '@' inhibitor.
   
   The return value is the number of characters in the field.
*/

int Termcap::parse_field (void* pv)
{
  const char* pb = (const char*) pv;

  TERMCAP_FIELD* pField = g_rgTermcapField;
  for (; pField->sz[0]; ++pField)
    if (pField->sz[0] == pb[0] && pField->sz[1] == pb[1])
      break;
  if (!pField->sz[0])
    TRACE ((T_TC_WARN, "unrecognized termcap field '%c%c'", pb[0], pb[1]));
  else {
    if (pField->flags & termcapInhibit)
      TRACE ((T_TC_WARN, "inhibiting termcap field '%c%c'", pb[0], pb[1]));
    else
      switch (pField->flags & termcapClassMask) {

      case termcapClassBoolean:
	pField->pv = (void*) 1;
	TRACE ((T_TC_INFO, "%s true", pField->szName));
	break;

      case termcapClassNumber:
	if (pb[2] != '#') {
	  TRACE ((T_TC_WARN, "malformed numeric termcap field '%c%c'", 
	     pb[0], pb[1]));
	  break;
	}      
	pField->pv = (void*) atoi (pb + 3);
	TRACE ((T_TC_INFO, "%s %d", pField->szName, pField->pv));
	break;

      case termcapClassString:
      case termcapClassMethod:
      case termcapClassKey:
	if (pb[2] == '@' && pField->pv)
	  TRACE ((T_TC_INFO, "eliminating termcap field '%c%c' of '%s'",
	     pb[0], pb[1], (char*) pField->pv));
	if (pField->pv) {
	  TRACE ((T_TC_WARN, "replacing termcap field '%c%c'", pb[0], pb[1]));
	  free (pField->pv);
	  pField->pv = 0;
	}
	if (pb[2] == '@') {
	  pField->flags |= termcapInhibit;
	  break;
	}
	if (pb[2] != '=') {
	  TRACE ((T_TC_WARN, "malformed string termcap field '%c%c'", 
		  pb[0], pb[1]));
	  break;
	}      
	{
	  int cch = strcspn (pb + 3, ":");
	  pField->pv = malloc (cch + 1);
	  if ((pField->flags & termcapClassMask) == termcapClassKey)
	    interpret_termcap_string ((char*) pField->pv, pb + 3, cch);
	  else
	    memcpy (pField->pv, pb + 3, cch);
	  ((char*)(pField->pv))[cch] = 0;
	  TRACE ((T_TC_INFO, "%s (%c%c) '%s'", pField->szName, pb[0], pb[1], 
		  pField->pv));
	}
	break;
      }
  }

  return strcspn (pb, ":");
}


/* Termcap::parse_termcap

   parse the termcap entry for the named terminal type. 

   The state values are:
     0) at beginning of line
     1) searching for end of line
     2) searching for end of line terminators
     3) looking for a matching terminal name
     4) looking for beginning of next field after name matched
     5) parse entry

*/

bool Termcap::parse_termcap (const char* szTerm)
{
  TRACE ((T_TC_INFO, "Termcap::parse_termcap"));
  if (g_fValid)
    release_this ();

  int fh = -1;
  char* pbMap = NULL;
  struct stat stat;
  long cb = 0;
  int cchTerm = strlen (szTerm);

  if (!::stat (g_szPathTermcap, &stat) && (cb = stat.st_size)
      && ((fh = open (g_szPathTermcap, O_RDONLY)) != -1)
      && (pbMap = (char*) mmap (NULL, cb + 1, PROT_READ, 
				MAP_FILE | MAP_PRIVATE, fh, 0))) {
    TRACE ((T_TC_INFO, "Termcap::parse_termcap found"));
    do {
      char* pb = pbMap;
      int state = 0;
      while (pb - pbMap < cb && *pb) {
	switch (state) {
	case 0:
	  if (isspace (*pb) || *pb == '#')
	    state = 1;
	  else
	    state = 3;		// We've found an entry
	  break;

	case 1:
	  if (*pb != '\r' && *pb != '\n')
	    ++pb;
	  else
	    state = 2;
	  break;

	case 2:
	  if (*pb == '\r' || *pb == '\n')
	    ++pb;
	  else
	    state = 0;
	  break;

	case 3:
	  if (strncmp (pb, szTerm, cchTerm) == 0
	      && (pb[cchTerm] == '|' || pb[cchTerm] == ':'))
	    state = 4;
	  else {
	    pb += strcspn (pb, "|:");
	    if (*pb == '|')
	      ++pb;
	    else
	      state = 1;
	  }
	  break;

	case 4:
	  pb += strcspn (pb, "\n\r:");
	  if (*pb == ':') {
	    ++pb;
	    state = 5;
	  }
	  else
	    state = 1;
	  break;

	case 5:
	  if (isspace (*pb) || *pb == '\n') { 	// End of entries
	    state = 1;
	    break;
	  }
	  if (*pb == '\\' && *(pb + 1) == '\n') {	// Continuation
	    pb += strcspn (pb, ":") + 1;
	    break;
	  }
	  pb += parse_field (pb) + 1;
	  break;
	}
      }
      szTerm = NULL;		// Need to check for chaining
    } while (szTerm); 

    g_fValid = true;

  }
  if (pbMap)
    munmap (pbMap, stat.st_size);
  if (fh != -1)
    close (fh);
  return g_fValid;
}


bool Termcap::parse_terminfo (const char* szTerm)
{
  TRACE ((T_TC_INFO, "Termcap::parse_terminfo"));
  if (g_fValid)
    release_this ();

  char szPath[200];
  strcpy (szPath, g_szPathTerminfo);
  sprintf (szPath + strlen (szPath), "/%c/%s", *szTerm, szTerm);
  struct stat stat;
  if (::stat (szPath, &stat))	
    return false;		// No such file
  long cb = stat.st_size;
  TRACE ((T_TC_INFO, "Termcap::parse_terminfo found"));

				// -- Read the terminfo data
  assert_ (!g_pvTerminfo);
  g_pvTerminfo = malloc (cb + 1);
  char* pb = (char*) g_pvTerminfo;
  pb[cb] = 0;
  int fh = open (szPath, O_RDONLY);
  read (fh, pb, cb);
  close (fh);

				// -- Fill in the known terminfo items
  for (TERMCAP_FIELD* pField = g_rgTermcapField; pField->sz[0]; ++pField) {
    if (pField->index == -1)
      continue;

    switch (pField->flags & termcapClassMask) {

    case termcapClassBoolean:
      if (pField->index >= _tc_c_boolean ())
	continue;
      pField->pv = (void*) (_tc_boolean (pField->index) ? 1 : 0);
      TRACE ((T_TC_INFO, "%s %s", pField->szName,
	      pField->pv ? "true" : "false"));
      break;

    case termcapClassNumber:
      if (pField->index >= _tc_c_num ())
	continue;
      pField->pv = (void*) (int) _tc_num (pField->index);
      TRACE ((T_TC_INFO, "%s %d", pField->szName, pField->pv));
      break;

    case termcapClassMethod:
    case termcapClassString:
    case termcapClassStringMap:
    case termcapClassKey:
      if (pField->index >= _tc_c_strings ())
	continue;
      pField->pv = (void*) _tc_string (pField->index);
      TRACE ((T_TC_INFO, "%s '%s'", pField->szName,
	      pField->pv ? pField->pv : ""));
      break;

    case 0:
      break;

    default:
      assert_ (0);
      break;
    }
  }

  return g_fTerminfo = g_fValid = true;
}


void Termcap::release_this (void)
{
  for (TERMCAP_FIELD* pField = g_rgTermcapField; pField->sz[0]; ++pField) {
    int tc = (pField->flags & termcapClassMask);
    if (!g_fTerminfo
	&& (   tc == termcapClassString
	    || tc == termcapClassMethod
	    || tc == termcapClassKey))
      free (pField->pv);
    pField->pv = 0;
    pField->flags &= ~termcapInhibit;
  }      
  g_fTerminfo = false;
  if (g_pvTerminfo) {
    free (g_pvTerminfo);
    g_pvTerminfo = NULL;
  }
}


int Termcap::translate (int field, char* pb, const char* szSource)
{
  static char szMapDefault[] = 
    "'+"  // rhombus
    "+>"  // right arrow
    ",<"  // left arrow
    "-^"  // upper arrow
    ".v"  // down arrow
    "0#"  // full square
    "I#"  // latern
    "`+"  // diamond
    "a:"  // chess board
    "f'"  // degree
    "g#"  // plus-minus
    "h#"  // square
    "j+"  // right bottom corner
    "k+"  // right upper corner
    "l+"  // left upper corner
    "m+"  // left bottom corner
    "n+"  // cross
    "o-"  // upper horizontal line
    "q-"  // middle horizontal line
    "s_"  // bottom horizontal line
    "t+"  // left tee
    "u+"  // right tee
    "v+"  // bottom tee
    "w+"  // normal tee
    "x|"  // vertical line
    "~o";  // bullet (paragraph?)
      
  if (!szSource || !pb)
    return 0;

  TERMCAP_FIELD* pField = locate (field);
  if (!pField
      || (pField->flags & termcapClassMask) != termcapClassStringMap 
      || !pField->pv)
    return 0;

  int cb = 0;
  while (*szSource) {
    const char* pbMap;
    for (pbMap = (const char*) pField->pv; *pbMap; pbMap += 2) {
      if (*pbMap != *szSource)
	continue;
      *pb++ = pbMap[1];
      ++cb;
      break;
    }
    if (!pbMap) {
      for (pbMap = (const char*) szMapDefault; *pbMap; pbMap += 2) {
	if (*pbMap != *szSource)
	  continue;
	*pb++ = pbMap[1];
	++cb;
	break;
      }
    }
    ++szSource;
  }
  return cb;
}
