/* curves.h							-*- C++ -*-
     $Id: curves.h,v 1.34 1998/09/06 21:46:22 elf Exp $

   written by Marc Singer
   20 September 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   in a file called COPYING along with this program; if not, write to
   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
   02139, USA.

   -----------
   DESCRIPTION
   -----------

   Header for CurVeS containment class.  This class contains the state
   of the user's session.


   --------------------
   IMPLEMENTATION NOTES
   --------------------
   
   -- eClass

   The class has two purposes.  It is first an index into a
   color/attribute table such that the unique areas of the screen can
   have identifying color schemes.  It is second a rank for file types
   so that they sort in descending order of interest.

*/

#if !defined (_CURVES_H_)
#    define   _CURVES_H_

// #include "window.h"
#include "directory.h"
#include "cvs.h"
#include "blue6.h"


typedef enum {
  classNul			= 0,  // Unusable due to Curses (attrib. index)
  classEdited			= 1,  // User edited files
  classNewAdd			= 2,  // User added files
  classAdded			= 3,  // Added, but not committed
  classRemoved			= 4,  // Removed, but not committed
  classMerge			= 5,  // User edited with logfile update
  classPatch			= 6,  // Logfile update
  classCheckout			= 7,  // New logfile
  classConflict			= 8,  // Local edits conflict with repository
  classLost			= 9,  // User clobbered file
  classUpToDate			= 10,  // Controlled, but uninteresting
  classUncontrolled		= 11,  // Uncontrolled and really uninteresting
  classNormal			= 11,  //   <alias>
  classDir			= 12, // Attribute for directories

  classStatus			= 13, // Status bar
  classMenu			= 14, // Menu bar
  classCursor			= 15, // Tag and cursor
  classEnd			= 16, // One more than last entry
} eClass;


class LCurves {
protected:
  Termcap m_termcap;
  Window* m_pWindow;		// Main window

  Pane* m_pPaneMenu;		// Window for menu bar, maybe
  Pane* m_pPaneDir;		// Directory listing
  Pane* m_pPaneFile;		// File listing
  Pane* m_pPaneStatus;		// Status
  Pane* m_pPaneCommand;		// Command/menu

  Attribute  m_rgattrClass[classEnd];
  char m_rgClassMark[classEnd];

  unsigned m_fDirtyRoot		: 1;    // Root window dirty
  unsigned m_fDirtyMenu		: 1;	// Menu window dirty
  unsigned m_fDirtyDir		: 1;	// Directory window dirty
  unsigned m_fDirtyFile		: 1;	// File window dirty
  unsigned m_fDirtyStatus	: 1;	// Status window dirty
  unsigned m_fDirtyCommand	: 1;	// Command window dirty

  int m_dxDirectories;		// Width of directory pane

  LDirectory m_dir;		// Working directory
  LCVS m_cvs;			// CVS flags
  int m_cFiles;			// Number of files
  int m_cDirectories;		// Number of directories
  int m_cchFilenameMax;		// Width of longest filename
  int m_cchFileColumn;		// Width of each file display column
  int m_iDirScroll;		// Directory scroll position
  int m_iDirCursor;		// Directory cursor, current dir in list
  int m_iFileScroll;		// File scroll position
  int m_iFileCursor;		// File cursor, current dir in list
//  char m_szSortCriteria[4];	// Sort criteria as characters: acnolst

  int m_color_scheme;		// Color scheme, 0 for linux console

  int m_iPane;			// Current pane number (0 dir, 1 file)

  char** m_rgszMenu;		// Array of menu pointers
  int m_cMenuMax;		// Number of menu entries
  int m_iMenu;			// Index of current menu

  int m_cFilesTagged;		// Count of tagged files

  int _idx_file (int iFile) {
    return iFile + m_cDirectories; }
  int _idx_dir (int iDir) {
    return iDir; }
  FILE_INFO* _info_dir (int iDir) {
    return m_dir.file_info (_idx_dir (iDir)); }
  FILE_INFO* _info_file (int iFile) {
    return m_dir.file_info (_idx_file (iFile)); }

  static bool _explore (const char*, void*);
  const char* _dir (int iDir) {
				// Assertion lifted by looking through
				// the code.  It can happen when the
				// user is in an unreadable directory.
				// Note that the underlying call will
				// correctly return NULL when there is
				// no such directory.
    //    assert_ (iDir >= 0 && iDir < m_cDirectories);
    return m_dir.file (_idx_dir (iDir)); }
  const char* _file (int iFile) {
    assert_ (iFile >= 0 && iFile < m_cFiles);
    return m_dir.file (_idx_file (iFile)); }

  int _columns_of_files (void) {
    return m_pPaneFile->dx ()/m_cchFileColumn; }

  static int _classify (FILE_INFO* pInfo);
  static int _rank (char method, FILE_INFO* pInfo0, FILE_INFO* pInfo1);
  static int _sort_compare (const void* pv0, const void* pv1);

public:
  LCurves () {
    zero (); init (); }
  ~LCurves () {
    release_this (); }
  void zero (void) {
    memset (this, 0, sizeof (*this)); }
  void init (void);
  void init_colors (void);
  void init_menus (char** rgsz, int cMenuMax) {
    m_rgszMenu = rgsz, m_cMenuMax = cMenuMax; }
  void release_screen ();
  void release_this ();

  bool is_files (void) {
    return m_dir.count_files () != 0; }
  Pane* win_command (void) {
    return m_pPaneCommand; }
  Pane* win_menu (void) {
    return m_pPaneMenu; }

  void dirty_path (void) {
     m_fDirtyFile = m_fDirtyDir = m_fDirtyStatus = true; 
     m_iFileScroll = m_iFileCursor = 0; }

  int count_tagged_files (void) {
    return m_cFilesTagged; }
  void cursorpos_dir (int& x, int& y) {
    x = 0; y = m_iDirCursor - m_iDirScroll; }
  void cursorpos_file (int& x, int& y) { 
    x = (m_iFileCursor - m_iFileScroll)/m_pPaneFile->dy ()*m_cchFileColumn;
    y = (m_iFileCursor - m_iFileScroll)%m_pPaneFile->dy (); }
  void explore (void);		// Explore current path
  const char* file (void) {
    return m_cFiles ? _file (m_iFileCursor) : (const char*) NULL; }
  int getch (void);
//  const char* get_sort (void) {
//    return m_szSortCriteria; }
  char* name_tagged_files (bool fFilenameOnly = false) {
    return name_files (fFilenameOnly, flagTag, flagTag); }
  char* name_files (bool fFilenameOnly = false, 
		    int flagMask = 0, int flagSet = 0);
  int menu (void) {
    return m_iMenu; }
  int pane (void) {
    return m_iPane; }
  const char* path (void) {
    return m_dir.path (); }
  const char* path_abbreviate (int cch) {
    return m_dir.abbreviate (cch); }
  void refresh (bool f = false); // Redraw all dirty panes
  void render_dir (int iDir);
  void restore (bool fPrompt);
  void select_pane (int iPane);
  void save (void);
  void set_menu (int iMenu);
  void set_path (const char* szPath);  
  //  void set_sort (const char* szCriteria) {
  //    if (strlen (szCriteria) < sizeof (m_szSortCriteria) && *szCriteria)
  //      strcpy (m_szSortCriteria, szCriteria); }
  void sort (void);
  void tag (int iClass, bool fExclusive);
  void show_cursor (bool fShow); // Show/hide cursor mark
  void update (void) {		// Make changes to the user's screen
    m_pWindow->flush (); }

  void do_directory_move (int direction);
  void do_directory_change (void);
  void do_file_binarytoggle (void);
  void do_file_move (int direction);
  void do_file_tag (void);
  void do_search (const char* sz, bool fReverse);
  void do_update (const char* szFiles);
};

#endif /* _CURVES_H_ */
