/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate the stat cache
 */

#include <errno.h>
#include <ac/string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <archive.h>
#include <error_intl.h>
#include <fngrprnt.h>
#include <mem.h>
#include <option.h>
#include <stat.cache.h>
#include <symtab.h>
#include <trace.h>


typedef struct cache_ty cache_ty;
struct cache_ty
{
	time_t	oldest;
	time_t	newest;
};

static symtab_ty *symtab[2];


static void init _((int));

static void
init(n)
	int		n;
{
	trace(("init()\n{\n"/*}*/));
	if (!symtab[n])
	{
		symtab[n] = symtab_alloc(100);
		symtab[n]->reap = mem_free;
	}
	trace((/*{*/"}\n"));
}


static cache_ty *mem_copy_cache _((cache_ty *));

static cache_ty *
mem_copy_cache(st)
	cache_ty	*st;
{
	cache_ty	*result;

	result = mem_alloc(sizeof(cache_ty));
	*result = *st;
	return result;
}


/*
 * NAME
 *	stat_cache - stat() with caching
 *
 * SYNOPSIS
 *	int stat_cache(string_ty *path, struct stat *result);
 *
 * DESCRIPTION
 *	The stat_cache function is used to perform the same as the stat()
 *	system function, but the results are cached to avoid too many probes
 *	into the file system.  Files which do not exist are indicated by
 *	filling the result structure with zeros.
 *
 * RETURNS
 *	int; -1 on error, 0 on success
 *
 * CAVEAT
 *	Errors, other than ENOENT, result in a fatal diagnostic.
 */

static int stat_cache _((string_ty *, cache_ty *, int));

static int
stat_cache(path, cp, follow_links)
	string_ty	*path;
	cache_ty	*cp;
	int		follow_links;
{
	cache_ty	*data_p;
	int		err;
	struct stat	st;
	sub_context_ty	*scp;

	/*
	 * if we have previously stat()ed this file,
	 * return old information
	 */
	trace(("stat_cache(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (!symtab[follow_links])
		init(follow_links);
	data_p = symtab_query(symtab[follow_links], path);
	if (data_p)
	{
		*cp = *data_p;
		trace(("return 0;\n"));
		trace((/*{*/"}\n"));
		return 0;
	}

	/*
	 * new file, perform stat() for the first time
	 */
	trace(("stat(\"%s\")\n", path->str_text));
#ifdef S_IFLNK
	if (!follow_links)
		err = lstat(path->str_text, &st);
	else
#endif
		err = stat(path->str_text, &st);
	if (err && errno == ENOENT)
		err = archive_stat(path, &st);
	if (err)
	{
		switch (errno)
		{
		case ENOENT:
		case ENOTDIR:
			/*
			 * ENOENT occurs when a path element does not exist
			 * ENOTDIR occurs when a path element (except the last)
			 *		is not a directory.
			 * Either way, the file being "stat"ed does not exist.
			 */
			break;

		default:
			scp = sub_context_new();
			sub_errno_set(scp);
			sub_var_set(scp, "File_Name", "%S", path);
			error_intl(scp, i18n("stat $filename: $errno"));
			sub_context_delete(scp);
			trace(("return -1;\n"));
			trace((/*{*/"}\n"));
			return -1;
		}

		fp_delete(path);

		cp->newest = 0;
		cp->oldest = 0;
	}
	else
	{
		fp_ty	*fp;

		/*
		 * make sure the times of existing files
		 * are always positive
		 */
		if (st.st_mtime < 1)
			st.st_mtime = 1;
		cp->oldest = st.st_mtime;
		cp->newest = st.st_mtime;
		
		/*
		 * see if we have its fingerprint on file
		 */
		if (option_test(OPTION_FINGERPRINT))
		{
			fp = fp_search(path);
			if (fp)
			{
				/*
				 * we have seen this file before
				 */
				if (fp->newest != cp->newest)
				{
					fp_ty	data;

					/*
					 * but it has changed
					 * since we last saw it
					 */
					data.fingerprint = fp_fingerprint(path);
					if (!data.fingerprint)
						goto fp_not_useful;
					if (str_equal(fp->fingerprint, data.fingerprint))
					{
						/*
						 * the fingerprint is the same,
						 * so give the oldest mtime
						 * known
						 */
						data.oldest = fp->oldest;
						data.newest = cp->newest;
						cp->oldest = fp->oldest;
						if (option_test(OPTION_REASON))
						{
							struct tm	*tm;

							tm = localtime(&data.newest);
							scp = sub_context_new();
							sub_var_set
							(
								scp,
								"File_Name",
								"%S",
								path
							);
							sub_var_set
							(
								scp,
								"Number",
						 "%4d/%02d/%02d.%02d:%02d:%02d",
							     1900 + tm->tm_year,
								tm->tm_mon + 1,
								tm->tm_mday,
								tm->tm_hour,
								tm->tm_min,
								tm->tm_sec
							);
							error_intl
							(
								scp,
	  i18n("mtime(\"$filename\") was $number until fingerprinting (reason)")
							);
						}
					}
					else
					{
						/*
						 * the fingerprint differs
						 * do not lie about mtime
						 */
						data.oldest = cp->newest;
						data.newest = cp->newest;
					}
					fp_assign(path, &data);
					str_free(data.fingerprint);
				}
				else
				{
					/*
					 * file not modified since last seen
					 */
					cp->oldest = fp->oldest;
				}
			}
			else
			{
				fp_ty	data;

				/*
				 * never fingerprinted this file before
				 */
				data.oldest = cp->newest;
				data.newest = cp->newest;
				data.fingerprint = fp_fingerprint(path);
				if (!data.fingerprint)
				{
					fp_not_useful:
					fp_delete(path);
				}
				else
				{
					fp_assign(path, &data);
					str_free(data.fingerprint);
				}
			}
		}
	}

	/*
	 * remember the stat information
	 */
	symtab_assign(symtab[follow_links], path, mem_copy_cache(cp));
	trace(("return 0;\n"));
	trace((/*{*/"}\n"));
	return 0;
}


time_t
stat_cache_newest(path, follow_links)
	string_ty	*path;
	int		follow_links;
{
	cache_ty	cache;
	sub_context_ty	*scp;

	if (stat_cache(path, &cache, !!follow_links))
		return -1;

	/*
	 * trace the last-modified time
	 */
	if (option_test(OPTION_REASON))
	{
		if (!cache.newest)
		{
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%S", path);
			error_intl
			(
				scp,
				i18n("mtime(\"$filename\") == ENOENT (reason)")
			);
			sub_context_delete(scp);
		}
		else
		{
			struct tm	*tm;

			tm = localtime(&cache.newest);
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%S", path);
			sub_var_set
			(
				scp,
				"Number",
				"%4d/%02d/%02d.%02d:%02d:%02d",
				1900 + tm->tm_year,
				tm->tm_mon + 1,
				tm->tm_mday,
				tm->tm_hour,
				tm->tm_min,
				tm->tm_sec
			);
			if
			(
				option_test(OPTION_FINGERPRINT)
			&&
				cache.newest != cache.oldest
			)
			{
				error_intl
				(
					scp,
			 i18n("newest mtime(\"$filename\") == $number (reason)")
				);
			}
			else
			{
				error_intl
				(
					scp,
				i18n("mtime(\"$filename\") == $number (reason)")
				);
			}
		}
	}
	return cache.newest;
}


time_t
stat_cache_oldest(path, follow_links)
	string_ty	*path;
	int		follow_links;
{
	cache_ty	cache;
	sub_context_ty	*scp;

	if (stat_cache(path, &cache, !!follow_links))
		return -1;

	/*
	 * trace the last-modified time
	 */
	if (option_test(OPTION_REASON))
	{
		if (!cache.oldest)
		{
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%S", path);
			error_intl
			(
				scp,
				i18n("mtime(\"$filename\") == ENOENT (reason)")
			);
			sub_context_delete(scp);
		}
		else
		{
			struct tm	*tm;

			tm = localtime(&cache.oldest);
			scp = sub_context_new();
			sub_var_set(scp, "File_Name", "%S", path);
			sub_var_set
			(
				scp,
				"Number",
				"%4d/%02d/%02d.%02d:%02d:%02d",
				1900 + tm->tm_year,
				tm->tm_mon + 1,
				tm->tm_mday,
				tm->tm_hour,
				tm->tm_min,
				tm->tm_sec
			);
			if
			(
				option_test(OPTION_FINGERPRINT)
			&&
				cache.newest != cache.oldest
			)
			{
				error_intl
				(
					scp,
			 i18n("oldest mtime(\"$filename\") == $number (reason)")
				);
			}
			else
			{
				error_intl
				(
					scp,
				i18n("mtime(\"$filename\") == $number (reason)")
				);
			}
		}
	}
	return cache.oldest;
}


void
stat_cache_set(path, when, fp2)
	string_ty	*path;
	time_t		when;
	int		fp2;
{
	cache_ty	*data_p;
	cache_ty	cache;
	sub_context_ty	*scp;

	/*
	 * clear the don't-follow-links cache
	 */
	trace(("stat_cache_set(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (symtab[0])
		symtab_delete(symtab[0], path);

	if (!symtab[1])
		init(1);
	data_p = symtab_query(symtab[1], path);
	if (data_p)
	{
		if
		(
			!data_p->oldest
		||
			!option_test(OPTION_FINGERPRINT)
		||
			when < data_p->oldest
		)
			data_p->oldest = when;
		data_p->newest = when;
	}
	else
	{
		cache.oldest = when;
		cache.newest = when;
		symtab_assign(symtab[1], path, mem_copy_cache(&cache));
		data_p = &cache;
	}

	/*
	 * Update the fingerprint.
	 * (Important not to lie here, the fp2 flags
	 * says is immediately following a utime call.)
	 */
	if (fp2 && option_test(OPTION_FINGERPRINT))
	{
		fp_ty		*fp;

		fp = fp_search(path);
		if (fp)
		{
			fp_ty		data;

			data.oldest = fp->oldest;
			data.newest = when;
			data.fingerprint = str_copy(fp->fingerprint);
			fp_assign(path, &data);
			str_free(data.fingerprint);
		}
	}

	/*
	 * emit a trace
	 */
	if (option_test(OPTION_REASON))
	{
		struct tm	*tm;

		tm = localtime(&when);
		scp = sub_context_new();
		sub_var_set(scp, "File_Name", "%S", path);
		sub_var_set
		(
			scp,
			"Number",
			"%4d/%02d/%02d.%02d:%02d:%02d",
			1900 + tm->tm_year,
			tm->tm_mon + 1,
			tm->tm_mday,
			tm->tm_hour,
			tm->tm_min,
			tm->tm_sec
		);
		if
		(
			option_test(OPTION_FINGERPRINT)
		&&
			data_p->oldest != data_p->newest
		)
		{
			error_intl
			(
				scp,
			  i18n("newest mtime(\"$filename\") = $number (reason)")
			);
		}
		{
			error_intl
			(
				scp,
				i18n("mtime(\"$filename\") = $number (reason)")
			);
		}
	}
	trace((/*{*/"}\n"));
}


void
stat_cache_clear(path)
	string_ty	*path;
{
	trace(("stat_cache_clear(path =\"%s\")\n{\n"/*}*/, path->str_text));
	if (symtab[0])
		symtab_delete(symtab[0], path);
	if (symtab[1])
		symtab_delete(symtab[1], path);
	trace((/*{*/"}\n"));
}
