/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to implement the builtin functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 *
 * Only a limited set of this are candidates for builtin functions,
 * these are 
 *	- string manipulation [dirname, stringset, ect ]
 *	- environment manipulation [getenv(3), etc]
 *	- stat(3) related functions [exists, mtime, pathname, etc]
 *	- launching OS commands [execute, collect]
 * The above list is though to be exhaustive.
 *
 * This explicitly and forever excluded from being a builtin function
 * is anything which known or understands the format of some secific 
 * class of files.
 *
 * Access to stdio(3) has been thought of, and explicitly avoided.
 * Mostly because a specialist program used through [collect]
 * will almost always be far faster.
 */

#include <ac/stdlib.h>
#include <ac/string.h>

#include <builtin/find_command.h>
#include <error_intl.h>
#include <expr/position.h>
#include <os.h>
#include <trace.h>


/*
 * NAME
 *	builtin_find_command - find pathname commands
 *
 * SYNOPSIS
 *	int builtin_find_command(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	Find_command is a built-in function of cook, described as follows:
 *	This function requires one or more arguments.
 *
 * RETURNS
 *	A word list containing the expanded pathname of the named commands
 *	given as arguments.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_find_command(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;
	string_ty	*paths;
	char		*cp;

	trace(("find command\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	cp = getenv("PATH");
	if (!cp || !*cp)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_var_set(scp, "Name", "PATH");
		error_with_position
		(
			pp,
			scp,
			i18n("the $name environment variable is not set")
		);
		sub_context_delete(scp);
		return -1;
	}
	paths = str_from_c(cp);
	for (j = 1; j < args->nstrings; ++j)
	{
		string_ty	*s;
		string_ty	*s2;

		s = args->string[j];
		if (strchr(s->str_text, '/'))
		{
			if (os_exists(s))
			{
				s2 = os_pathname(s);
				string_list_append(result, s2);
				str_free(s2);
			}
			else
				string_list_append(result, str_false);
		}
		else
		{
			string_ty	*f;
			int		n;

			for (n = 0; ; ++n)
			{
				f = str_field(paths, ':', n);
				if (!f)
				{
					string_list_append(result, str_false);
					break;
				}
				s2 = str_format("%S/%S", f, s);
				str_free(f);
				switch (os_exists(s2))
				{
				case -1:
					str_free(s2);
					return -1;

				case 0:
					str_free(s2);
					continue;

				case 1:
					f = os_pathname(s2);
					str_free(s2);
					string_list_append(result, f);
					str_free(f);
					break;
				}
				break;
			}
		}
	}
	str_free(paths);
	return 0;
}
