/* run-parts: run a bunch of scripts in a directory
 *
 * Debian Linux run-parts program
 * Copyright (C) 1996 Jeff Noxon <jeff@router.patch.net>,
 * Copyright (C) 1996-1998 Guy Maor <maor@debian.org>
 *
 * This is free software; see the GNU General Public License version 2
 * or later for copying conditions.  There is NO warranty.
 *
 * Based on run-parts.pl version 0.2, Copyright (C) 1994 Ian Jackson.
 *
 * Revision History
 *
 * 01/07/96 - JEN - v0.3 - First version; released with miscutils-1.3-7
 *
 * 04/11/96 - GM  - v0.4 - Made it skip over directories (fixes Bug#2244)
 *                         Better error checking
 *
 * 06/01/96 - AND - v0.5 - Added --verbose option to print the name of each
 *                         script before executing it
 *
 * 01/20/98 - GM  - v0.6 - Added --report option.
 *
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>

#define VERSION "1.10"

int test_mode = 0;
int verbose_mode = 0;
int report_mode = 0;
int exitstatus = 0;

void error (char *format,...)
{
  va_list ap;
  
  fprintf (stderr, "run-parts: ");

  va_start (ap, format);
  vfprintf (stderr, format, ap);
  va_end (ap);

  fprintf (stderr, "\n");
}


void version ()
{
  fprintf (stderr, "Debian GNU/Linux run-parts program, version " VERSION
"\nCopyright (C) 1994 Ian Jackson, Copyright (C) 1996 Jeff Noxon.
Copyright (C) 1996,1997,1998 Guy Maor
This is free software; see the GNU General Public License version 2
or later for copying conditions.  There is NO warranty.
");
  exit (0);
}


void usage ()
{
  fprintf (stderr, "Usage: run-parts [OPTION]... DIRECTORY
  --test          print script names which would run, but don't run them.
  --verbose       print script names before running them.
  --report        print script names if they produce output.
  --umask=UMASK   sets umask to UMASK (octal), default is 022.
  --version       output version information and exit.
  --help          display this help and exit.
");
  exit(0);
}


/* The octal conversion in libc is not foolproof; it will take the 8 and 9
 * digits under some circumstances.  We'll just have to live with it.
 */
void set_umask ()
{
  int mask, result;

  result = sscanf (optarg, "%o", &mask);
  if ((result != 1) || (mask > 07777) || (mask < 0)) {
    error ("bad umask value");
    exit (1);
  }

  umask (mask);
}


/* True or false? Is this a valid filename (upper/lower alpha, digits,
 * underscores, and hyphens only?)
 */
int valid_name (char *filename)
{
  while (*filename) {
    if (!(((*filename >= 'a') && (*filename <= 'z')) ||
	  ((*filename >= 'A') && (*filename <= 'Z')) ||
	  ((*filename >= '0') && (*filename <= '9')) ||
	  (*filename == '_') ||
	  (*filename == '-')))
      return 0;
    ++filename;
  }
  
  return 1;
}


void set_fl (int fd, int flags)
{
  int val;
  if ((val = fcntl(fd, F_GETFL, 0)) < 0) {
    error("fcntl F_GETFL: %s", strerror(errno));
    exit(1);
  }
  val |= flags;
  if (fcntl(fd, F_SETFL, val) < 0) {
    error("fcntl F_SETFL: %s", strerror(errno));
    exit(1);
  }
}


/* Execute a file */
void run_part (char *progname)
{
  int result;
  int pid;
  int pout[2], perr[2];

  if (report_mode && (pipe(pout) || pipe(perr))) {
    error("pipe: %s", strerror(errno));
    exit(1);
  }
  if ((pid=fork()) < 0) {
    error("failed to fork: %s", strerror(errno));
    exit(1);
  }
  else if (!pid) {
    if (report_mode) {
      if (dup2(pout[1], STDOUT_FILENO) == -1 ||
	  dup2(perr[1], STDERR_FILENO) == -1) {
	error("dup2: %s", strerror(errno));
	exit(1);
      }
      close(pout[0]);
      close(perr[0]);
      close(pout[1]);
      close(perr[1]);
    }
    execl (progname, progname, 0);
    error ("failed to exec %s: %s", progname, strerror (errno));
    exit (1);
  }

  if (report_mode) {
    fd_set set;
    int max, r, printflag;
    ssize_t c;
    char buf[4096];
    
    close(pout[1]);
    close(perr[1]);
    set_fl(pout[0], O_NONBLOCK);
    set_fl(perr[0], O_NONBLOCK);
    max = pout[0] > perr[0] ? pout[0]+1 : perr[0]+1;
    printflag = 0;
    for (;;) {
      if (pout[0] < 0 && perr[0] < 0) {
	r = waitpid(pid, &result, 0);
	if (r > 0)
	  break;
	else {
	  error("waitpid: %s", strerror(errno));
	  exit(1);
	}
      }

      FD_ZERO(&set);
      if (pout[0] >= 0) FD_SET(pout[0], &set);
      if (perr[0] >= 0) FD_SET(perr[0], &set);
      r = select(max, &set, 0, 0, 0);
      if (r > 0) {
	if (pout[0] >= 0 && FD_ISSET(pout[0], &set)) {
	  c = read(pout[0], buf, sizeof(buf));
	  if (c > 0) {
	    if (!printflag) {
	      printf("%s:\n", progname);
	      fflush(stdout);
	      printflag = 1;
	    }
	    write(STDOUT_FILENO, buf, c);
	  }
	  else if (c == 0) {
	    close(pout[0]);
	    pout[0] = -1;
	  }
	}
	if (perr[0] >= 0 && FD_ISSET(perr[0], &set)) {
	  c = read(perr[0], buf, sizeof(buf));
	  if (c > 0) {
	    if (!printflag) {
	      fprintf(stderr, "%s:\n", progname);
	      fflush(stderr);
	      printflag = 1;
	    }
	    write(STDERR_FILENO, buf, c);
	  }
	  else if (c == 0) {
	    close(perr[0]);
	    perr[0] = -1;
	  }
	}
      }
      else if (r < 0) {
	error("select: %s", strerror(errno));
	exit(1);
      }
      else {			/* should never happen */
	error("internal error: select() returned 0 !");
	exit(1);
      }
    }
  }
  else {
    waitpid(pid, &result, 0);
  }
  
  if (WIFEXITED (result) && WEXITSTATUS(result)) {
    error ("%s exited with return code %d", progname, WEXITSTATUS(result));
    exitstatus = 1;
  }
  else if (WIFSIGNALED (result)) {
    error ("%s exited because of uncaught signal %d", progname,
	   WTERMSIG(result));
    exitstatus = 1;
  }
}


/* Find the parts to run & call run_part() */
void run_parts (char *dirname)
{
  struct dirent **namelist;
  char filename[PATH_MAX];
  int entries, i, result;
  struct stat st;

  /* scandir() isn't POSIX, but it makes things easy. */
  entries = scandir (dirname, &namelist, 0, alphasort);
  if (entries < 0)
    error ("failed to open directory %s: %s", dirname, strerror (errno));

  for (i = 0; i < entries; i++) {
    if (valid_name (namelist[i]->d_name)) {
      strcpy (filename, dirname);
      strcat (filename, "/");
      strcat (filename, namelist[i]->d_name);

      result = stat (filename, &st);
      if (result < 0) {
	error ("failed to stat component %s: %s", filename,
	       strerror (errno));
	exit (1);
      }
      if (S_ISREG(st.st_mode) && !access (filename, X_OK)) {
	if (test_mode)
	  printf ("run-parts would run %s\n", filename);
	else {
	  if (verbose_mode)
	    fprintf(stderr, "run-parts: executing %s\n", filename);
	  run_part (filename);
	}
      }
      else if (!S_ISDIR(st.st_mode)) {
	printf ("run-parts: component %s is not an executable plain file\n",
		filename);
	exitstatus = 1;
      }

    }
    free (namelist[i]);
  }
  free (namelist);
}

/* Process options */
int main (int argc, char *argv[])
{
  umask (022);
  
  for (;;) {
    int c;
    int option_index = 0;

    static struct option long_options[] = {
      {"test", 0, &test_mode, 1},
      {"verbose", 0, &verbose_mode, 1},
      {"report", 0, &report_mode, 1},
      {"umask", 1, 0, 1},
      {"help", 0, 0, 2},
      {"version", 0, 0, 3},
      {0, 0, 0, 0}
    };

    opterr = 0;
    c = getopt_long (argc, argv, "", long_options, &option_index);
    if (c == EOF)
      break;
    switch (c) {
    case 0:
      break;
    case 1:
      set_umask ();
      break;
    case 2:
      usage();
    case 3:
      version ();
    default:
      error("unrecognized option `%s'", argv[optind-1]);
      fprintf(stderr, "Try `run-parts --help' for more information.\n");
      exit (1);
    }
  }
  
  /* We require exactly one argument: the directory name */
  if (optind != (argc - 1)) {
    error("missing operand");
    fprintf(stderr, "Try run-parts --help' for more information.\n");
    exit (1);
  }

  run_parts (argv[optind]);

  return exitstatus;
}
