#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <elf.h>

static const char *curfile;

/* Returns -1 on error, 1 if not dynamic executable, 
   2 if no problem, 0 otherwise.  */
static int
fixfile(int fd, const Elf32_Ehdr *ehdr)
{
  Elf32_Off dyn_offset;
  enum { num_load = 5 };
  struct {
    char *data;
    Elf32_Addr base;
    Elf32_Word length, fsz;
    Elf32_Off off;
  } loaded[num_load];
  int numloaded = 0;
  int i;
  Elf32_Addr sym_offset, string_offset, hash_offset;

  if (lseek(fd, ehdr->e_phoff, SEEK_SET) == -1)
    {
      fprintf (stderr, "%s: positioning for sections: %m\n", curfile);
      return -1;
    }
  
  dyn_offset = 0;
  for (i = 0; i < ehdr->e_phnum; i++)
    {
      Elf32_Phdr phdr;
      if (read(fd, &phdr, sizeof(phdr)) != sizeof(phdr))
	{
	  fprintf (stderr, "%s: reading section header %d: %m\n",
		   curfile, i);
	  return -1;
	}
      if (phdr.p_type == PT_DYNAMIC)
	if (dyn_offset == 0)
	  dyn_offset = phdr.p_offset;
	else
	  {
	    fprintf(stderr, "%s: seems to have multiple dynamic sections.\n",
		    curfile);
	    return -1;
	  }
      else if (phdr.p_type == PT_LOAD)
	{
	  if (numloaded == num_load)
	    {
	      fprintf(stderr, "%s: too many loadable sections.\n",
		      curfile);
	      return -1;
	    }
	  loaded[numloaded].base = phdr.p_vaddr;
	  loaded[numloaded].off = phdr.p_offset;
	  loaded[numloaded].length = phdr.p_memsz;
	  loaded[numloaded].fsz = phdr.p_filesz;
	  numloaded++;
	}
    }
  
  if (dyn_offset == 0)
    {
      fprintf(stderr, "%s: no dynamic section.\n", curfile);
      return 1;
    }
  
  if (lseek(fd, dyn_offset, SEEK_SET) == -1)
    {
      fprintf (stderr, "%s: positioning for dynamic: %m\n", curfile);
      return -1;
    }
  sym_offset = string_offset = hash_offset = 0;
  for (;;)
    {
      Elf32_Dyn dent;
      if (read(fd, &dent, sizeof(dent)) != sizeof(dent))
	{
	  fprintf (stderr, "%s: reading .dynamic entry: %m\n",
		   curfile);
	  return -1;
	}
      switch (dent.d_tag)
	{
	case DT_SYMTAB:
	  sym_offset = dent.d_un.d_ptr;
	  break;
	case DT_STRTAB:
	  string_offset = dent.d_un.d_ptr;
	  break;
	case DT_HASH:
	  hash_offset = dent.d_un.d_ptr;
	  break;
	case DT_NULL:
	  goto endloop;
	default:
	  break;
	}
    }
endloop:
  if (sym_offset == 0 || string_offset == 0 || hash_offset == 0)
    {
      fprintf(stderr, "%s: No strings or no symbols!\n", curfile);
      return -1;
    }

  for (i = 0; i < numloaded; i++)
    {
      loaded[i].data = alloca(loaded[i].length);
      memset(loaded[i].data, 0, loaded[i].length);

      if (lseek(fd, loaded[i].off, SEEK_SET) == -1)
	{
	  fprintf (stderr, "%s: seeking for segment %d: %m\n", curfile, i);
	  return -1;
	}
      if (read(fd, loaded[i].data, loaded[i].fsz) != (ssize_t)loaded[i].fsz)
	{
	  fprintf (stderr, "%s: reading segment %d: %m\n", curfile, i);
	  return -1;
	}
    }

#define findlocn(addr) ({ \
      int t; \
      char *result = 0; \
      for (t = 0; t < numloaded; t++) \
	if (loaded[t].base <= (addr) \
	    && loaded[t].base + loaded[t].length > (addr)) \
	  result = loaded[t].data + ((addr) - loaded[t].base); \
      if (result == 0) \
	{ \
	  fprintf (stderr, "%s: unloaded address %08x\n", curfile, addr); \
	  return -1; \
	} \
      result; \
    })
#define badsym(name) (strstr(name, "sxx") != NULL \
		      || strncmp(name, "xx", 2) == 0)

  {
    Elf32_Word nchain;
    Elf32_Sym *syms;
    Elf32_Word nbucket;
    Elf32_Addr *bucket, *chain;
    Elf32_Word c;
    char *name, *namep;

    nchain = *(Elf32_Word *) findlocn(hash_offset + 4);
    syms = (Elf32_Sym *) findlocn(sym_offset);
    for (c = 0; c < nchain; c++)
      if (syms[c].st_name != 0)
	{
	  name = findlocn(string_offset + syms[c].st_name);
	  if (namep = strstr(name, "A0"))
	    {
	      namep[0] = '!'; namep[1] = ',';
	    }
	  else if (namep = strstr(name, "A1"))
	    {
	      namep[0] = ','; namep[1] = '!';
	    }
	}

    /* Rebuild the hash table.  */
    nbucket = *(Elf32_Word *) findlocn(hash_offset);
    bucket = (Elf32_Addr *) findlocn(hash_offset + 8);
    chain = bucket + nbucket;
    memset(bucket, 0, sizeof(Elf32_Addr)*nbucket);
    memset(chain, 0, sizeof(Elf32_Addr)*nchain);
    
    for (c = 1; c < nchain; c++)
      if (syms[c].st_name != 0)
	{
	  unsigned long h=0, g, j;
	  
	  name = findlocn(string_offset + syms[c].st_name);
	  for (j = 0; name[j]; j++)
	    {
	      h = (h << 4) + name[j];
	      if (g = h & 0xf0000000)
		h ^= g >> 24;
	      h &= ~g;
	    }
	  h %= nbucket;
	  if (bucket[h] == 0)
	    bucket[h] = c;
	  else
	    {
	      for (h = bucket[h]; chain[h] != 0; h = chain[h])
		;
	      chain[h] = c;
	    }
	}
  }  

  /* We now need to write out all the segments.  */
  for (i = 0; i < numloaded; i++)
    {
      if (lseek(fd, loaded[i].off, SEEK_SET) == -1)
	{
	  fprintf (stderr, "%s: seeking for segment %d: %m\n", curfile, i);
	  return -1;
	}
      if (write(fd, loaded[i].data, loaded[i].fsz) != (ssize_t)loaded[i].fsz)
	{
	  fprintf (stderr, "%s: writing segment %d: %m\n", curfile, i);
	  return -1;
	}
    }
  return 0;
}

void
fixit(const char *whichfile)
{
  ssize_t numread;
  Elf32_Ehdr ehdr;
  int fd;

  curfile = whichfile;
  fd = open(curfile, O_RDWR);
  if (fd == -1)
    {
      fprintf (stderr, "%s: open failed: %m\n", curfile);
      return;
    }
  
  numread = read(fd, &ehdr, sizeof(ehdr));
  if (numread == -1)
    {
      fprintf (stderr, "%s: header read failed: %m\n", curfile);
      goto donefile;
    }
  else if (numread < (ssize_t)sizeof(ehdr))
    {
      fprintf(stderr, "%s: too short.\n", curfile);
      goto donefile;
    }

  if (*(unsigned *)ehdr.e_ident != *(const unsigned *)ELFMAG ||
      ehdr.e_ident[EI_CLASS] != ELFCLASS32 ||
      ehdr.e_ident[EI_DATA] != ELFDATA2MSB ||
      ehdr.e_ident[EI_VERSION] != EV_CURRENT ||
      ehdr.e_type != ET_DYN ||
      ehdr.e_machine != EM_PPC)
    {
      fprintf(stderr, "%s: not a PPC shared library.\n", curfile);
      goto donefile;
    }

  if (ehdr.e_phentsize != sizeof(Elf32_Phdr))
    {
      fprintf(stderr, "%s: section size was read as %d, not %d!\n",
	      curfile, ehdr.e_phentsize, sizeof(Elf32_Phdr));
      goto donefile;
    }

  fixfile(fd, &ehdr);

donefile:
  close(fd);
}

int
main(int argc, char **argv)
{
  if (argc != 2)
    {
      printf("usage: %s lib-to-mangle\n", argv[0]);
      exit(1);
    }

  fixit(argv[1]);
  return 0;
}
