/*
 * $Id: queue.c,v 1.18 1998/09/21 14:32:39 gregm Exp $
 * GXSNMP -- An snmp mangament application
 * Copyright (C) 1998 Gregory McLean
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * Queue handling functions.
 */
#include "main.h"
#include "queue.h"

/*
 * Local variables
 */
static GList    *w_queue = NULL;  /* pending */
static GList    *q_item  = NULL;  /* current queue item */
static int      next_id  = 0;     
/*
 * Forward declarations
 * static void            create_queue_panel        (void);
 * static gint            queue_panel_delete_cb     (GtkWidget   *widget,
 * 						  GdkEvent    *e,
 * 						  gpointer    data);
 * static void            queue_panel_close_cb      (GtkWidget   *widget,
 *						  gpointer    data);
 */
static queue_entry     *find_queue_entry         (int         id,
						  GList       *queue);

/*
 * Local functions
 */
static queue_entry *
find_queue_entry (int id, GList *queue)
{
  GList        *item;
  queue_entry  *data;

  if (g_list_length (queue) <= 0)
    return NULL;                         /* not found */
  item = g_list_first (queue);
  while (item)
    {
      data = (queue_entry *)item->data;
      item = g_list_next (item);
      if ( (int)data->id == id)
	return data;
    }
  return NULL;                           /* not found */
}

/*
 * Module UI Functions
 * static void
 * create_queue_panel ()
 * {
 * }
 *
 * static gint
 * queue_panel_delete_cb (GtkWidget *widget, GdkEvent *e, gpointer data)
 * {
 *   return 0;
 * }
 *
 * static void
 * queue_panel_close_cb (GtkWidget *widget, gpointer data)
 * {
 * }
 */

/*
 * Global functions
 */

int
get_next_queue_id ()
{
  next_id++;
  return next_id;
}

int
add_queue_item (queue_entry *entry)
{
  queue_entry      *cp;

  g_return_val_if_fail (entry != NULL, -1);
  
  if (find_queue_entry (entry->id, w_queue))
    {
      g_print ("Queue entry with that id (%d) already exists\n", entry->id);
      return -1;
    }
  cp = (queue_entry *)g_malloc (sizeof (queue_entry));
  memcpy (cp, entry, sizeof (queue_entry));
  w_queue = g_list_append (w_queue, cp);
  return cp->id;
}

int
remove_queue_item (int id)
{
  queue_entry   *rem;

  if ( (rem = find_queue_entry (id, w_queue)))
    {
      w_queue = g_list_remove (w_queue, rem);
      return TRUE;
    }
  return -1;
}

/*
 * This will walk the queue processing one element
 */
int 
run_queue (gpointer data)
{
  queue_entry   *current;

  if (q_item == NULL)
      q_item = g_list_first (w_queue);

  if (q_item)
    {
      current = (queue_entry *)q_item->data;
#ifdef QUEUE_DEBUG
      g_print ("Proccessing queue item %d\n", current->id);
#endif
      current->time_last = current->time;
      current->time      = time (NULL);
      if (current->queue_callback)
	{
	  current->queue_callback (current, current->data);
	}
      if (q_item->next == NULL)
	{
	  q_item = g_list_first (w_queue);
#ifdef QUEUE_DEBUG
	  g_print ("End of queue reached roll around to the begining..\n");
#endif
	  return TRUE;
	  /* Hit the end, roll back to the begining */
	}
      q_item = q_item->next;
      current = (queue_entry *)q_item->data;
#ifdef QUEUE_DEBUG
      g_print ("next _should_ be %d\n", current->id);
#endif
      return TRUE;
    }
  else 
    q_item = NULL;
  return TRUE;
}

/*
 * Standard queue callback functions
 */
void
test_callback (queue_entry *queue, gpointer data)
{
  hosts     *entry;
  time_t    current_time;

  entry = (hosts *)data;
  if (entry) 
    {
      current_time = time (NULL);
      if ( (entry->hl_queue_last_time == 0) ||
	   (current_time >= (entry->hl_queue_last_time + 
	    entry->hl_poll_interval)))
	{ 
	  queue_get_load (queue, data);
	}
    }
}
void
q_add_null_var (GSList **objs, GSList **entry, char *what, 
		    char *identifier, gpointer variable)
{
  struct _SNMP_OBJECT    *obj;
  q_oidentry     *var;

  obj = g_malloc (sizeof (struct _SNMP_OBJECT));

  obj->request  = 0;
  obj->type     = SNMP_NULL;
  obj->id_len   = SNMP_SIZE_OBJECTID;
  obj->id       = g_malloc (SNMP_SIZE_OBJECTID);

  if (!read_objid (what, obj->id, &obj->id_len))
    {
      g_print ("Eh? (%s)\n",what);
      g_free (obj);
      return;
    }
  var         = g_malloc (sizeof (q_oidentry));
  var->id_len = obj->id_len;
  g_memmove (var->id, obj->id, obj->id_len * sizeof (gulong));
  var->name = g_strdup (identifier);
  var->hl_variable = variable;
  *objs  = g_slist_append (*objs, obj);
  *entry = g_slist_append (*entry, var);
}

/*
 * Poll the load off the server (ethernet)
 * This could be routed through the collector if that ever gets working 
 * again.
 *
 */
void
queue_get_load (queue_entry *queue, gpointer data)
{
  hosts     *entry;
  GSList    *objs;
  GSList    *magic;

  entry = (hosts *)data;
  if (entry)
    {
      if (entry->hl_queue_request)
	{
	  g_print ("Queue has a request running to %s\n", entry->hl_disp);
	  return;
	}
      objs = NULL;
      magic = NULL;
      q_add_null_var (&objs, &magic, 
		      "interfaces.ifTable.ifEntry.ifInOctets.1", 
		      "in_octets", &entry->hl_last_in);
      q_add_null_var (&objs, &magic,
		      "interfaces.ifTable.ifEntry.ifOutOctets.1",
		      "out_octets", &entry->hl_last_out);
      q_add_null_var (&objs, &magic,
		      "interfaces.ifTable.ifEntry.ifSpeed.1",
		      "if_speed", NULL);
      entry->hl_snmp.magic = magic;
      entry->hl_snmp.done_callback = queue_snmp_update_load;
      entry->hl_queue_request = g_async_get (&entry->hl_snmp, objs);
      
    }
}

/*
 * SNMP netload callback (from the snmp lib)
 */
gboolean
queue_snmp_update_load (host_snmp *host, void *magic, SNMP_PDU *spdu, 
			GSList *objs)
{
  hosts               *entry;
  GSList              *myobj;
  GSList              *myid;
  GSList              *id;
  q_oidentry          *var;
  gulong              foo;
  gulong              last;
  gulong              max_bytes;
  gulong              io_bytes;
  double              time_diff;
  time_t              current_time;
  SNMP_OBJECT         *obj;
  
  entry = find_host_by_name (host->name);
  foo = 0;
  time_diff = 0.0;
  if (entry)
    {
      entry->hl_queue_request = NULL;     /* request complete */
      id = (GSList *)magic;
      myobj = objs;
      while (myobj) 
	{
	  obj = (struct _SNMP_OBJECT *)myobj->data;
	  switch (obj->type)
	    {
	    case SNMP_INTEGER:
	      foo = (glong)obj->syntax.lng;
	      break;
	    case SNMP_COUNTER:
	    case SNMP_GAUGE:
	      foo = (gulong)obj->syntax.ulng;
	      break;
	    default:
	      break;
	    }
	  current_time = time(NULL);
	  if (entry->hl_queue_last_time != 0)
	    time_diff = (current_time - entry->hl_queue_last_time);
	  myid = id;
	  while (myid)
	    {
	      var = (q_oidentry *)myid->data;
	      if (!memcmp(obj->id, var->id, var->id_len * sizeof (gulong)))
		{		  
		  if (var->hl_variable)
		    {
		      if (time_diff > 0)
			{
			  last = *var->hl_variable;
			  io_bytes = (foo - last) / time_diff;
			}
		      
		      *var->hl_variable = foo;
		      if (strcasecmp (var->name, "in_octets") == 0)
			{
			  if (entry->hl_ifspeed)
			    entry->hl_in_load = ((io_bytes * 1.0) / 
					       (entry->hl_ifspeed * 1.0));
			}
		      if (strcasecmp (var->name, "out_octets") == 0)
			{
			  if (entry->hl_ifspeed)
			    entry->hl_out_load = ((io_bytes * 1.0) /
						(entry->hl_ifspeed * 1.0));
			}
		    }
		  if (strcasecmp (var->name, "if_speed") == 0)
		    {
		      max_bytes = foo / 8;
		      entry->hl_ifspeed = max_bytes;
		    }
		}
	      myid = g_slist_next (myid);
	    }
	  myobj = g_slist_next (myobj);
	}
      entry->hl_queue_last_time = current_time;
      entry->hl_lastsnmp = current_time;
      g_slist_free (id);
      update_host_load ();
    }
  return TRUE;
}

/*
 * Standard global panel functions
 */
void 
open_queue_panel ()
{
}

void
hide_queue_panel ()
{
}

void
destroy_queue_panel ()
{
}

void 
reset_queue_panel ()
{
}

/* EOF */
