#include <stdio.h>
#include <strings.h>
#include <signal.h>
#include <errno.h>
#include <sys/reboot.h>
#include <sys/types.h>
#include <fcntl.h>
#include <newt.h>
#include "dinstall.h"
#include INCLINGUA

#define LOCALBUFSIZE (256)

#ifdef LOG

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

void dinstall_log(const char * pattern, ...) {
   int     fd;
   FILE *  con = 0;
   va_list arguments;

   if ( ((fd = open("/dev/tty4",O_WRONLY|O_NOCTTY)) < 0)
     ||  ((con = fdopen(fd, "w")) == NULL) )
     return;

   va_start(arguments, pattern);
   vfprintf(con, pattern, arguments);
   va_end(arguments);
   fclose(con);
}
#endif

int stderrToTTY(int tty) {
	static int fd=-1;
	char dev[10];

	/* stderr redirect only works when tty is not /dev/console */
	/* FIXME: we need to figure out why it fails on /dev/console
	 *        In the meantime skip it. */
	if (strcmp( ttyname(0), "/dev/console" )==0)
		return 0;
	if (fd > 0) 
		close(fd);
	snprintf(dev,10,"/dev/tty%d",tty);
	if ((fd = open(dev, O_RDWR|O_NOCTTY)) < 0)
		return 1;
	if (-1 == dup2(fd,2))
		return 1;
	return 0;
}

void boxResume(void) {
	stderrToTTY(3);
	newtResume();
}

void boxSuspend(void) {
	newtSuspend();
	stderrToTTY(1);
}

void boxPopWindow(void) {
	newtPopWindow();
}

void boxFinished(void) {
	newtFinished();
	stderrToTTY(1);
}

static void suspend(void) {
	boxSuspend();
	raise(SIGTSTP);
	boxResume();
}

void initMonoScreen (void) {
/* Print 24 blank lines right before erasing the screen. That way,
 * the user may see the boot messages using Shift-PgUp. */
	printf("\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");
	setenv("NEWT_MONO","1",1);
	newtInit();
	newtSetSuspendCallback(suspend);
	newtPushHelpLine(MSG_MAIN_KEY);
	stderrToTTY(3);
}

/* A lot of this functionality is in newt-0.25 windows.c .
 * It would be nice to merge both works...
 */
int pleaseWaitBox(const char *text) {
    int height;
    newtComponent f1, t1;
    newtOpenWindow (10, 8, 60, 5, MSG_PLEASE_WAIT);
    f1 = newtForm(NULL, NULL, 0);
    t1 = newtTextbox (1, 1, 59, 4, NEWT_FLAG_WRAP);
    newtTextboxSetText(t1,text);
    height=newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtFormAddComponent (f1 , t1);
    newtDrawForm(f1);
    newtRefresh();
    newtFormDestroy(f1);
    return 0;
}

int problemBox(const char *text, const char *title) {
    newtComponent f1, t1, b1;
    int height;

    t1 = newtTextbox (1, 1, 60, 10, NEWT_FLAG_WRAP);
    newtTextboxSetText(t1, text);
    height=newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtOpenWindow (10, 9-height/2, 61, height+3, title);
    b1 = newtCompactButton ( 23, height+2, MSG_CONTINUE );
    f1 = newtForm(NULL, NULL, 0);
    newtFormAddComponents (f1, t1, b1, NULL);
    newtRunForm(f1);
    newtPopWindow();
    newtFormDestroy(f1);
    return 0;
}

// This one should be merged into problemBox...
int wideMessageBox(const char *text, const char *title) {
    newtComponent f1, t1, b1;

    t1 = newtTextbox (1, 0, 76, 20, NEWT_FLAG_WRAP | NEWT_FLAG_SCROLL);
    newtTextboxSetText(t1, text);
    newtOpenWindow (1, 1, 78, 22, title);
    b1 = newtCompactButton ( 32, 21, MSG_CONTINUE );
    f1 = newtForm(NULL, NULL, 0);
    newtFormAddComponents (f1, t1, b1, NULL);
    newtRunForm(f1);
    newtPopWindow();
    newtFormDestroy(f1);
    return 0;
}

int perrorBox(const char *text) {
    char buf[LOCALBUFSIZE];
    snprintf(buf,LOCALBUFSIZE,"%s: %s",text,strerror(errno));
    problemBox(buf,MSG_ERROR);
    return 0;
}

int twoButtonBox(const char *text, const char *title, const char *button1, const char* button2) {
    newtComponent f1, t1, b1, b2, ans;
    int height;

    t1 = newtTextbox (1, 1, 60, 10, NEWT_FLAG_WRAP);
    newtTextboxSetText(t1, text);
    height=newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtOpenWindow (10, 9-height/2, 61, height+3, title);
    b1 = newtCompactButton ( 20, height+2, button1 );
    b2 = newtCompactButton ( 32, height+2, button2 );
    f1 = newtForm(NULL, NULL, 0);
    newtFormAddComponents (f1, t1, b1, b2, NULL);
    ans = newtRunForm(f1);
    newtPopWindow();
    newtFormDestroy(f1);
    if (ans == b1) return -1;
    return 0;
}

int yesNoBox(const char *text, const char *title) {
    return twoButtonBox(text, title, MSG_YES, MSG_NO);
}

char *inputBox(const char *text, const char *title, const char *proto) { 
    newtComponent f1, t1, i1, b1, b2, ans;
    char *val, *res;
    int height;

    t1 = newtTextbox (1, 1, 60, 10, NEWT_FLAG_WRAP);
    newtTextboxSetText(t1, text);
    height=newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtOpenWindow (10, 9-height/2, 61, height+5, title);
    i1 = newtEntry(1, height+2, proto, 59, &val, 
                     NEWT_FLAG_SCROLL | NEWT_FLAG_RETURNEXIT);
    b1 = newtCompactButton ( 20, height+4, MSG_OK );
    b2 = newtCompactButton ( 32, height+4, MSG_CANCEL );
    f1 = newtForm(NULL, NULL, 0);
    newtFormAddComponents (f1, t1, i1, b1, b2, NULL);
    ans = newtRunForm(f1);
    res = strdup(val);
    newtPopWindow();
    newtFormDestroy(f1);
    if (ans != b2) return res;
    free(res);
    return NULL;
}

#define _MAX_LIST_HEIGHT  (15)
#define _MIN_WIDTH  (60)
int menuBox (const char* text, const char* title,
         struct d_choices* choices, int nchoices, int cancel) {
  char buf[LOCALBUFSIZE];
  char format1[40],format2[40];
  newtComponent form, textbox, listbox, button, ans;
  int txheight, height, ix, string, tag, width;
  int mstring1=0, mstring2=0, mtag=0, lwidth=0;
  int result, *rs, *entry=calloc(nchoices,sizeof(int));

  for (ix=0;ix<nchoices;ix++) {
      if (choices[ix].tag)
        tag=strlen(choices[ix].tag);
      else
	tag=0;
      if(tag>mtag)
        mtag=tag;
      if (choices[ix].string)
        string=strlen(choices[ix].string);
      else
	string=0;
      if (tag) {
	if (string > mstring2)
          mstring2=string;
	if (lwidth < mstring2+mtag+3)
	  lwidth=mstring2+mtag+3;
      } else {
	if (string > mstring1)
          mstring1=string;
	if (lwidth < mstring1+1)
	  lwidth=mstring1+1;
      }
  }
  if (mtag > 0)
    snprintf(format2,40,"%%-%ds: %%-%ds ",mtag,mstring2);
  snprintf(format1,40,"%%-%ds ",mstring1);

  lwidth+=7;
  width=((lwidth>_MIN_WIDTH) ? lwidth : _MIN_WIDTH);

  textbox = newtTextbox (1, 1, width, 5, NEWT_FLAG_WRAP);
  newtTextboxSetText(textbox, text);
  txheight = newtTextboxGetNumLines(textbox);
  newtTextboxSetHeight(textbox,txheight);

  height = _MAX_LIST_HEIGHT - txheight;
  height = ((nchoices<height) ? nchoices : height) + 2;
  listbox = newtListbox((width-lwidth)/2+1, txheight+2, height,
            NEWT_FLAG_DOBORDER | NEWT_FLAG_RETURNEXIT);
  height += txheight+1;

  newtCenteredWindow(width+2, height+2+cancel, title);

  for (ix=0;ix<nchoices;ix++) {
    if (mtag > 0) {
      if (choices[ix].string) {
        if (choices[ix].tag) {
          snprintf(buf,LOCALBUFSIZE,format2,choices[ix].tag,choices[ix].string);
        } else {
          snprintf(buf,LOCALBUFSIZE,format1,choices[ix].string);
        }
      } else {
        if (choices[ix].tag) {
          snprintf(buf,LOCALBUFSIZE,format2,choices[ix].tag,"");
        } else {
          snprintf(buf,LOCALBUFSIZE,"");
        }
      }
    } else {
      if (choices[ix].string) {
        snprintf(buf,LOCALBUFSIZE,format1,choices[ix].string);
      } else {
        snprintf(buf,LOCALBUFSIZE,"");
      }
    }
    entry[ix]=ix;
    if (strlen(buf))
      newtListboxAddEntry(listbox,buf,&entry[ix]);
    else
      newtListboxAddEntry(listbox,buf,NULL);
  }

  form = newtForm(NULL, NULL, 0);
  newtFormAddComponents(form, textbox, listbox, NULL);

  sprintf(buf,MSG_CANCEL);
  button = newtCompactButton ( width/2-strlen(buf)/2, height+2, buf );

  if (cancel) {
    newtFormAddComponent(form, button);
  }

  do {
    ans = newtRunForm(form);
    rs = newtListboxGetCurrent(listbox);
  } while ((rs == NULL) && (ans != button));

  if (ans == button) 
    result=-1;
  else
    result=*rs;

  newtPopWindow();
  newtFormDestroy(form);
  free(entry);

  return result;
}

int scaleBox(const char *text, const char *title, long long value, int action){
  static newtComponent tx=NULL, scale=NULL, f=NULL;

  switch (action) {
    case SCALE_CREATE:
      if (f) return -1;
      newtOpenWindow(15, 10, 50, 6, title);
      f = newtForm(NULL, NULL, 0);
      tx= newtTextbox(1, 1, 49, 2, NEWT_FLAG_WRAP);
      newtTextboxSetText(tx, text);
      scale = newtScale(10, 4, 30, value);
      newtFormAddComponents(f, tx, scale, NULL);
      newtDrawForm(f);
      newtRefresh();
      break;;
    case SCALE_REFRESH:
      if (!f) return -1;
      newtScaleSet(scale, value);
      newtRefresh();
      break;;
    case SCALE_DELETE:
      if (!f) return -1;
      newtPopWindow();
      newtFormDestroy(f);
      f=NULL;
      break;;
  }
  return 0;
}
	  
#define _ButtonH 1

int check_box(const char* text, const char* title, int height, int width,
          char** choices, char** values, int nchoices)
{
  newtComponent form,tbox,okay,cancel,sform;
  newtComponent answer;
  newtComponent *chk;
  newtComponent sb=NULL;
  int rc=DLG_OKAY;
  int top,ix;
  char *result;

  int lheight=(nchoices>height - _ButtonH - 6)
    ? height - _ButtonH - 6 : nchoices;

  chk=calloc(nchoices,sizeof(newtComponent));
  if(!chk)
    return DLG_ERROR;

  result=calloc(nchoices+1,sizeof(char));
  if(!result)
    return DLG_ERROR;
  else
    result[nchoices]='\0';

  newtOpenWindow (39-width/2, 12-height/2, width, height, title);

  form=newtForm(NULL, NULL, 0);
  tbox=newtTextbox(1, 0, width-2, 10, NEWT_FLAG_WRAP);
  newtTextboxSetText(tbox, text);
  top=newtTextboxGetNumLines(tbox);
  newtTextboxSetHeight(tbox,top);
  
  if(lheight!=nchoices) /* We need a scrollbar. */
    {
      sb=newtVerticalScrollbar(width - 4, top + 1,
                   lheight,
                               NEWT_COLORSET_CHECKBOX,
                               NEWT_COLORSET_ACTCHECKBOX);
      newtFormAddComponent(form,sb);
    }

  sform=newtForm(sb,NULL,0);
  newtFormSetBackground(sform, NEWT_COLORSET_CHECKBOX);
  for(ix=0;ix<nchoices;ix++)
    {
      chk[ix]=newtCheckbox(4 , top + 1 + ix, choices[ix],
               (*values)[ix],NULL,&result[ix]);
      newtFormAddComponent(sform, chk[ix]);
    }

  newtFormSetHeight(sform, lheight);
  newtFormSetWidth(sform,width - 10);
  okay = newtCompactButton ( (width-18)/3, height-_ButtonH, MSG_OK);
  cancel = newtCompactButton ( (width-18)/3*2+9, height-_ButtonH, MSG_CANCEL);
  newtFormAddComponents(form,tbox,sform,okay,cancel,NULL);
  
  answer=newtRunForm(form);

  if(answer==cancel)
    rc=DLG_CANCEL;
  else
    strcpy(*values,result);

  newtPopWindow();
  newtFormDestroy(form);

  free(result);
  free(chk);
  return rc;
}

#if defined(_TESTING_) || defined(_BOXTESTING_)
#include <slang.h>
void initScreen (char *msg) {
	SLtt_Use_Ansi_Colors = 1;
	newtSetThreeD(1);
	newtInit();
	newtCls();
	newtDrawRootText(0, 0, msg);
	newtOpenWindow(7, 2, 65, 18, "Root Window");
	newtPushHelpLine(NULL);
}

void finishScreen(void) {
	newtPopWindow();
	newtFinished();
}
#endif

#ifdef _BOXTESTING_
/* To test, compile using: make boxes_test */
void main (void) {
	struct d_choices opt[3];
	int rs;
	char buf[128];

	initScreen("Boxes test program");

// Testing menuBox
	opt[0].tag="test one";
	opt[0].string="one string...";
	opt[1].tag="test two";
	opt[1].string=NULL;
	opt[2].tag=NULL;
	opt[2].string="This is a somewhat large string. Let's test resizable boxes";
	rs=menuBox("This is a little text","I'm the title",opt,3,1);

// Testing problemBox
	sprintf(buf,"You've chosen option %d",rs);
	problemBox(buf,"Answer");

	finishScreen();
}
#endif
