// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: widget.cc,v 1.9 1998/10/24 22:46:30 jgg Exp $
/* ######################################################################

   Widget - Base class for the internal widget set.

   The widgets are stored in a tree structure. Each widget has a Parent
   and a set of children. One of the main jobs of the base class is to 
   manage this tree. The widget with no parent is the root widget and
   is usually the main window.

   Some layout aides are supplied to help clients like MenuBar order 
   their children. These organize the children in a single line, 
   horizontal or vertical.
   
   Grab support is handled by allowing multiple widgets (and all of their
   children) to aquire a mouse grab. This allows things like menus to
   work correctly.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#ifdef __GNUG__
#pragma implementation "deity/widget.h"
#endif  
#include <deity/widget.h>
#include <iostream.h>
#include <typeinfo>
									/*}}}*/

Widget *Widget::LastWidget = 0;
Widget::GrabItem *Widget::GrabList = 0;
int Widget::GrabSize = 0;
Widget *Widget::Root = 0;
void *Widget::Type = (void *)&Widget::Type;

// Widget::Widget - Construct this widget as the child of another	/*{{{*/
// ---------------------------------------------------------------------
/* */
Widget::Widget(Widget *Parent)
{
   if (Parent == 0)
      Parent = LastWidget;
   
   this->Parent = 0;
   Child = 0;
   Brother = 0;
   Realized = false;
   FocusWidget = 0;
   BorderX = 0;
   BorderY = 0;
   Flags = 0;
   ActionFlags = 0;
   ExtentXFlags = 0;
   ExtentYFlags = 0;
   
   ChangeParent(Parent);
}
									/*}}}*/
// Widget::~Widget - Destructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Widget::~Widget()
{
   UnRealize();
   ChangeParent(0);
   while (Child != 0)
      delete Child;
}
									/*}}}*/
// Widget::ExtentFamily - Extent the widgets family			/*{{{*/
// ---------------------------------------------------------------------
/* Extenting computes the size of the widgets. Things like lables generaly
   extent to the minimum size required to store the text. */
void Widget::ExtentFamily()
{
   Extent();
   for (Widget *I = Child; I != 0; I = I->Brother)
      I->ExtentFamily();
}
									/*}}}*/
// Widget::ExtentMode - Set the X/Y extent mode				/*{{{*/
// ---------------------------------------------------------------------
/* */
void Widget::ExtentMode(unsigned long X,unsigned long Y)
{
   // Remove the previous options
   ExtentXFlags &= ~(ExtentModeMask | ExtentDone);
   ExtentYFlags &= ~(ExtentModeMask | ExtentDone);

   // Set the new options
   ExtentXFlags |= X & ExtentModeMask;
   ExtentYFlags |= Y & ExtentModeMask;
}
									/*}}}*/
// Widget::ExtentDo - Set the size based on the extent mode		/*{{{*/
// ---------------------------------------------------------------------
/*  This possibly sets the real size to the ideal size depending on the
    extent flags. */
long Widget::ExtentDo(long New,long Old,unsigned short &Flags)
{
   if ((Flags & ExtentModeMask) == ExtentNever)
      return Old;
   
   // We always set the size on the first call
   if ((Flags & ExtentDone) != ExtentDone)
      return New;
   
   switch (Flags & ExtentModeMask)
   {
      case ExtentOnce:
      return Old;

      case ExtentAlways:
      return New;

      case ExtentLarger:
      if (New > Old)
	 return New;
      return Old;
   }   
   return Old;
}
									/*}}}*/
// Widget::Extent - Set the widgets size				/*{{{*/
// ---------------------------------------------------------------------
/* */
void Widget::Extent()
{
   Point Ideal = IdealSize();
   Rect NewSz = Rect(Pos.x,Pos.y,ExtentDo(Ideal.x,Pos.w,ExtentXFlags),
		     ExtentDo(Ideal.y,Pos.h,ExtentYFlags));
   Resize(NewSz);
}
									/*}}}*/
// Widget::Realize - Realize this widget				/*{{{*/
// ---------------------------------------------------------------------
/* Once realize the widget is drawable and the user can see and interact
   with it. Realization of a parent does not imply realization of the
   children.*/
void Widget::Realize()
{
   // Already realized
   if (Realized == true)
      return;

   // Parent isnt realized
   if (Parent != 0 && Parent->Realized == false)
       return;

   Extent();
   
   Realized = true;
   if (Parent != 0 && Parent->FocusWidget == 0)
      Parent->FocusWidget = this;

   /* It is assumed that if a region is created then it will be drawn 
      when the GUI decides it's time (typically after an event loop) */
   if (IsFlag(Region) == true)
   {
      // Find the first regioned parent
      Widget *I = Parent;
      for (; I != 0 && I->IsFlag(Region) == false;
	   I = I->Parent);
      
      if (I != 0)
	 RegionData = GenGC::GC->CreateRegion(this,I->RegionData);
      else
	 RegionData = GenGC::GC->CreateRegion(this,0);
      if (IsFlag(Region) == true)
	 Flag(Building);
   }
   
   // Create region can unset the flag if it does not want to make a region
   if (IsFlag(Region) == false)
      Damage();
}
									/*}}}*/
// Widget::UnRealize - UnRealize this widget				/*{{{*/
// ---------------------------------------------------------------------
/* Once UnRealized the widget and all of it's children will no longer be
   visiable to the user. */
void Widget::UnRealize()
{
   if (Realized == false)
      return;
   
   Realized = false;
   for (Widget *I = Child; I != 0; I = I->Brother)
      I->UnRealize();

   if (IsFlag(Region) == true)
      GenGC::GC->DeleteRegion(RegionData);
   
   if (Parent != 0 && IsFlag(Region) == false)
      Parent->Damage();
}
									/*}}}*/
// Widget::RealizeFamily - Realize the widget and children		/*{{{*/
// ---------------------------------------------------------------------
/* This recursively realizes the widget and all of it's children. It is
   probably the realize function you want to be calling most of the time. */
void Widget::RealizeChildren()
{
   for (Widget *I = Child; I != 0; I = I->Brother)
   {
      if (I->IsFlag(DelayRealize) == true)
	 continue;
      I->Realize();
      I->RealizeChildren();
   }   
}

void Widget::RealizeFamily()
{
   Realize();
   if (Realized == true)
      RealizeChildren();
}
									/*}}}*/
// Widget::Resize - Change the size of the widget			/*{{{*/
// ---------------------------------------------------------------------
/* This sets the given size to the internal size. Derived widgets may
   mutate the size in any way, possibly even reject the size change. 
   Once the size is changed the children are notified so that they can
   re-extent */
void Widget::Resize(Rect NewSize)
{
   if (NewSize == Pos)
      return;
   
   /* ResizeRegion must cause Draw() to be called on the region after
      the windowing system has resized it */
   if (IsFlag(Region) == true && Realized == true)
      GenGC::GC->ResizeRegion(this,RegionData,NewSize);
   else
      Damage();

   ExtentXFlags |= ExtentDone;
   ExtentYFlags |= ExtentDone;
   
   Rect OldSize = Pos;
   Trigger(Nt_Resize,&NewSize);
   Pos = NewSize;
   
   // Position changes do not warrent a re-extent.
   if (OldSize.w == Pos.w && OldSize.h == Pos.h)
      return;
   
   for (Widget *I = Child; I != 0; I = I->Brother)
      if (I->IsFlag(AutoExtent) == true)
	 I->Extent();

   /* Now we perform layout calculations on the widget, this reorganizes
      the children. We should also call the parents layout function
      but that will need some form of recursion protection */
   Layout();
}
									/*}}}*/
// Widget::AbsoluteRect - Make an absolute rectangle for the widget	/*{{{*/
// ---------------------------------------------------------------------
/* */
Rect Widget::AbsoluteRect(bool ToRegion)
{
   if (ToRegion == true && IsFlag(Region) == true)
      return Rect(0,0,Pos.w,Pos.h);
   
   /* Compute the origin and cliping region, We never apply clipping 
      from the root region */
   Rect P = Pos;
   if (Parent != 0 && Parent->Parent != 0)
   {
      Rect Par = Parent->Pos;
      Par.w -= 2*Parent->BorderX;
      Par.h -= 2*Parent->BorderY;
      
      P.Clip(Par);
   }
   
   Widget *I;
   for (I = Parent; I != 0 && (ToRegion == false || 
        I->IsFlag(Region) == false); I = I->Parent)
   {
      P.x += I->Pos.x + I->BorderX;
      P.y += I->Pos.y + I->BorderY;
   }
   if (I != 0)
   {
      P.x += I->BorderX;
      P.y += I->BorderY;
   }

   return P;
}
									/*}}}*/
// Widget::Draw - Redraw the widget					/*{{{*/
// ---------------------------------------------------------------------
/* We chain to the children since no clipping is done in the internal 
   drawing routines. We also setup the proper origin before calling
   Render and goto the 0,0 point in the new coord system */
void Widget::Draw()
{
   if (Realized == false)
      return;

   // Render the widget
   CombinedGC Tmp;
   if (ConnectGC(Tmp) == false)
      return;
   Render(Tmp);

   ActionFlags &= ~(NeedsRender | RedrawPending);
   
   // Do the children.
   for (Widget *I = Child; I != 0; I = I->Brother)
      if (I->Realized == true && I->IsFlag(Region) == false)
	 I->Draw();
}
									/*}}}*/
// Widget::ConnectGC - Connect the GC to the widget			/*{{{*/
// ---------------------------------------------------------------------
/* This connects to the proper region and sets the clipping and offseting
   so drawing can work. It returns true if it is okay to draw in the
   GC. */
bool Widget::ConnectGC(CombinedGC &GC)
{
   if (Realized == false)
      return false;
   
   // Locate the first region widget
   Widget *I = this;
   for (; I != 0 && I->IsFlag(Region) == false; I = I->Parent);

   // Attach the GC to the region
   if (I != 0)
   {
      /* This means the GUI is still constructing the widget so we can't
         draw in it. */
      if (I->IsFlag(Building) == true)
	 return false;
      GC->AttachRegion(I->RegionData);
   }

   Rect P = AbsoluteRect(true);
   GC->EraseClippings();
   GC->AddClipping(P);
   return true;
}
									/*}}}*/
// Widget::DoActions - Process all pending actions for the widget tree	/*{{{*/
// ---------------------------------------------------------------------
/* This does things like deferred drawing and syncronization for the
   widget tree. I think this should be optmized with some form of
   dirty table. The Event system will invoke it after handling each
   event. */
void Widget::DoActions()
{
   if (Realized == false)
      return;

   // We have to call render after sync because sync will damage widgets
   DoSync();
   DoRender(false);
}
									/*}}}*/
// Widget::DoSync - Locate and sync all widgets that need it		/*{{{*/
// ---------------------------------------------------------------------
/* */
void Widget::DoSync()
{
   // See if a sync operation is pending
   if ((ActionFlags & NeedsSync) == NeedsSync)
   {
      // Signal the client that the widget should be syncd
      Trigger(Nt_Sync,0);
      ActionFlags &= ~NeedsSync;
   }
   
   // Do all the children too
   for (Widget *I = Child; I != 0; I = I->Brother)
      I->DoSync();
}
									/*}}}*/
// Widget::DoRender - Locate and render all widgets that need it	/*{{{*/
// ---------------------------------------------------------------------
/* NoDraw is used to control the RedrawPending mode. If RenderPending is
   set then drawing in that region is blocked. */
void Widget::DoRender(bool NoDraw)
{
   // Unset the flag if this is a new region
   if (IsFlag(Region) == true)
      NoDraw = false;
   
   // Disable drawing if a render command is going to arrive soon
   if ((ActionFlags & RedrawPending) == RedrawPending)
      NoDraw = true;
   
   // See if a render operation is pending for the widget
   if ((ActionFlags & NeedsRender) == NeedsRender)
   {
      // And the parent region is not RedrawPending..
      if (NoDraw == false)
      {
	 Draw();
	 
	 /* This is necessary to get proper render order. If a widget
	    below others was damaged then all of its upwards siblings 
	    need rendering */	    
	 for (Widget *I = this; I != 0; I = I->Brother)
	    if (I->IsFlag(Region) == false)
	       I->Damage();	       

	 ActionFlags &= ~NeedsRender;
      }
   }
   
   // Render the children
   for (Widget *I = Child; I != 0; I = I->Brother)
      I->DoRender(NoDraw);
}
									/*}}}*/
// Widget::Focus - Returns true if this widget has focus		/*{{{*/
// ---------------------------------------------------------------------
/* We check all parents to ensure they have focus */
bool Widget::Focus()
{
   // Handle forced focus (menus, combos and things)
   if (IsFlag(ForceFocus) == true)
      return true;
   
   Widget *I;
   for (I = this; I->Parent != 0 && I->Parent->FocusWidget == I; I = I->Parent);
   if (I->Parent == 0)
      return true;
   return false;
}
									/*}}}*/
// Widget::GiveFocus - Adjusts the focuses for the parent widgets	/*{{{*/
// ---------------------------------------------------------------------
/* Walk upwards and change the focus of each widget in turn.
 
   This is really poor, it calls draw too much */
void Widget::GiveFocus()
{
   // Give focus to us (unrealized)
   if (Realized == false)
   {
      if (Parent == 0)
	 return;
	 
      // Give focus to the parent
      if (Parent->Focus() == false)
      {
	 Parent->FocusWidget = this;
	 Parent->GiveFocus();
	 return;
      }
      
      // Remove focus from the old focus widgets
      for (Widget *I = Parent->FocusWidget; I != 0; I = I->FocusWidget)
	 I->FocusLost(this);
      Parent->FocusWidget = this;
      Parent->Damage();
      return;
   }
      
      
   Widget *I;
   Widget *OldFocus = 0;
   Widget *NewFocus = 0;
   while (1)
   {
      // Find the deviation
      for (I = this; I->Parent != 0 && I->Parent->FocusWidget == I;
	   I = I->Parent);
      if (I->Parent == 0)
	 break;
      
      // Change focus
      OldFocus = I->Parent->FocusWidget;
      I->Parent->FocusWidget = I;
      NewFocus = I;
      if (OldFocus != 0)
	 OldFocus->Damage();
   }   

   // Determine who last had focus
   Widget *OldFWidget = OldFocus;
   for (;OldFWidget != 0 && OldFWidget->FocusWidget != 0;
	OldFWidget = OldFWidget->FocusWidget);
   
   // Walk down the OldFocus tree and notify
   for (Widget *J = OldFocus; J != 0; J = J->FocusWidget)
      J->FocusLost(this);
   
   // Notify widgets the got focus
   if (NewFocus != 0)
      for (Widget *J = NewFocus; J != 0; J = J->FocusWidget)
	 J->FocusGained(OldFWidget);
      
   // Walk down the tree and draw
   for (I = NewFocus; I != 0; I = I->FocusWidget)
      if (I->IsFlag(ShowFocus) == true)
	 I->Damage();
}
									/*}}}*/
// Widget::FocusNext - Give Focus to the next widget			/*{{{*/
// ---------------------------------------------------------------------
/* Simply use FocusWidgets brother pointer */
void Widget::FocusNext()
{
   if (Child == 0)
      return;

   if (FocusWidget == 0)
   {
      Child->GiveFocus();
      return;
   }
   
   for (Widget *I = FocusWidget->Brother; I != FocusWidget;)
   {      
      if (I != 0 && I->IsFlag(Focusable) == true && I->Realized == true)
      {
	 I->GiveFocus();
	 return;
      }

      if (I == 0)
	 I = Child;
      else
	 I = I->Brother;
   }
}
									/*}}}*/
// Widget::FocusLast - Give Focus to the last widget			/*{{{*/
// ---------------------------------------------------------------------
/* Similar idea to FocusNext but we scan till we hit the current focus 
   and then give the last hit focus */
void Widget::FocusLast()
{
   if (Child == 0)
      return;

   if (FocusWidget == 0)
   {
      Child->GiveFocus();
      return;
   }
   
   Widget *Last = 0;
   for (Widget *I = FocusWidget->Brother; I != FocusWidget;)
   {      
      if (I != 0 && I->IsFlag(Focusable) == true && I->Realized == true)
	 Last = I;

      if (I == 0)
	 I = Child;
      else
	 I = I->Brother;
   }

   if (Last != 0)
      Last->GiveFocus();
}
									/*}}}*/
// Widget::RouteKey - Input keys into the proper widget			/*{{{*/
// ---------------------------------------------------------------------
/* We walk down the Focus tree and then walk back up trying to pass off the
   key. We also check children along each step for ParentKeys */
void Widget::RouteKey(const KeyEvent &Key)
{
   Widget *I;
   for (I = this; I->FocusWidget != 0 && I->FocusWidget->Realized == true; 
	I = I->FocusWidget);
   Widget *Source = I;
   for (;I != 0; I = I->Parent)
   {
      if (I->Key(Key,Source) == true)
	 return;
      for (Widget *I2 = I->Child; I2 != 0; I2 = I2->Brother)
      {
	 if (I2->IsFlag(ParentKeys) == true && I2 != I->FocusWidget && 
	     I2->Realized == true)
	    if (I2->Key(Key,Source) == true)
	       return;
      }      
   }
}
									/*}}}*/
// Widget::RouteMouse - Input Mouse events into the proper widget	/*{{{*/
// ---------------------------------------------------------------------
/* This locates the proper widget to receive mouse events. The position
   of the event is offset to be relative to the canvas area of the
   widget (-BoarderX,-BoarderY) */
void Widget::RouteMouse(MouseEvent &Event,bool Direct)
{
   // Hm, button depressed. Consider the grab list
   if (Direct == false && GrabList != 0 && GrabList[0].Target != 0 &&
       ((GrabList[0].IsFlag(GrabAllGrab) == true && GrabList[0].Target->Realized == true) ||
       Event.ButtonState != 0))
   {
      // Convert to absolute coordinates
      Point P = Event.Pos;
      for (Widget *I = this; I != 0; I = I->Parent)
      {
	 P.x += I->Pos.x + I->BorderX;
	 P.y += I->Pos.y + I->BorderY;
      }
      
      // We look for a widget that contains this event. Overlapping doesnt work
      for (GrabItem *I = GrabList; I != GrabList + GrabSize; I++)
      {
	 if (I->Target == 0)
	    break;
	 if (I->Target->Realized == false)
	    continue;

	 if (I->IsFlag(GrabWhenFocused) == true && I->Target->Focus() == false)
	    continue;

	 Rect R = I->Target->AbsoluteRect(false);
	 if (R.Inside(P) == true)
	 {
	    Event.Pos.x = P.x - R.x;
	    Event.Pos.y = P.y - R.y;

	    // Recurse to handle children
	    if (I->IsFlag(GrabChildren) == true)
	       I->Target->RouteMouse(Event,true);
	    else
	    {
	       Event.Pos.x -= I->Target->BorderX;
	       Event.Pos.y -= I->Target->BorderY;
	       I->Target->Mouse(Event);
	    }
	    
	    return;
	 }
      }
      
      // Send the event to the first grabbed widget
      Rect R = GrabList[0].Target->AbsoluteRect(false);
      Event.Pos.x = P.x - R.x - GrabList[0].Target->BorderX;
      Event.Pos.y = P.y - R.y - GrabList[0].Target->BorderY;
      
      // Recurse to handle children
      if (GrabList[0].IsFlag(GrabChildren) == true)
	 GrabList[0].Target->RouteMouse(Event,true);
      else
      {
	 Event.Pos.x -= GrabList[0].Target->BorderX;
	 Event.Pos.y -= GrabList[0].Target->BorderY;
	 GrabList[0].Target->Mouse(Event);
      }
      
      return;
   }
   else
   {
      // Dump the grab list.
      if (Direct == false)
      {
	 // Compress the grab list, remove all non permanent items
	 GrabItem *Replace = GrabList;
	 for (GrabItem *I = GrabList; I != GrabList + GrabSize; I++)
	 {
	    if (I->Target != 0 && I->IsFlag(GrabPermanent) == true && 
		I->Target->Realized == true)
	    {
	       *Replace = *I;
	       if (I != Replace)
		  I->Target = 0;
	       Replace++;
	       continue;
	    }
	    I->Target = 0;
	 }
      }
   }
   
   // Outside our scope
   if (Direct == false && (Pos.w < Event.Pos.x || Pos.h < Event.Pos.y))
      return;
   
   // Offset the event by our offset for processing
   Event.Pos.x += Pos.x;
   Event.Pos.y += Pos.y;
   
   Rect HitRect = Pos;
   Widget *Hit = this;
   Widget *Best = this;
   while (Hit != 0)
   {
      // Componsate for borders
      HitRect.x += Hit->BorderX;
      HitRect.y += Hit->BorderY;
      HitRect.w -= Hit->BorderX;
      HitRect.h -= Hit->BorderY;
      
      // Move to the hit
      Widget *I = Hit->Child;
      Best = Hit;
      Hit = 0;
      Rect LastRect = HitRect;
      for (; I != 0; I = I->Brother)
      {
	 if (I->Realized == false || I->IsFlag(MouseTransparent) == true)
	    continue;
	 
	 Rect P = I->Pos;
	 P.Clip(LastRect);
	 P.x += LastRect.x;
	 P.y += LastRect.y;
	 
	 if (P.Inside(Event.Pos) == true)
	 {
	    HitRect = P;
	    Hit = I;
	 }
      }
   }

   // Best is now the lowest widget that contains the point
   Event.Pos.x = Event.Pos.x - HitRect.x;
   Event.Pos.y = Event.Pos.y - HitRect.y;

   // We give focus to the widget if it is a click
   if (Event.IsDn() == true && 
       Best->IsFlag(Focusable) == true)
      Best->GiveFocus();

   if (Event.IsDn() == true)
   {
      // Check if there is a grab that is enabled
      GrabItem *I;
      for (I = GrabList; I != GrabList + GrabSize; I++)
      {
	 if (I->Target == 0)
	    break;
	 if (I->Target->Realized == false)
	    continue;
	 if (I->IsFlag(GrabWhenFocused) == true && 
	     I->Target->Focus() == false)
	    continue;
	 break;
      }      

      // Add a new one
      if (I == GrabList + GrabSize || I->Target == 0)
	 Best->GrabMouse(0,true);
   }

   Best->Mouse(Event);
}
									/*}}}*/
// Widget::NumChildren - Return the number of children widgets		/*{{{*/
// ---------------------------------------------------------------------
/* */
unsigned int Widget::NumChildren()
{
   unsigned int I = 0;
   for (Widget *W = Child; W != 0; W = W->Brother)
      I++;
   return I;
}
									/*}}}*/
// Widget::ChangeParent - Safely change the parent of a widget		/*{{{*/
// ---------------------------------------------------------------------
/* */
void Widget::ChangeParent(Widget *New)
{
   // Unlink
   if (Parent != 0)
   {
      if (Parent->FocusWidget == this)
	 Parent->FocusWidget = 0;
      
      Widget **Last = &Parent->Child;
      Widget *I = Parent->Child;
      for (; I != 0 && I != this; I = I->Brother)
	 Last = &I->Brother;

      // Force a seg fault if this list is invalid..
      *Last = I->Brother;
      I->Brother = 0;
      Parent = 0;
   }

   // Relink
   if (New != 0)
   {
      Parent = New;
      Widget **Last = &New->Child;
      for (Widget *I = New->Child; I != 0; I = I->Brother)
	 Last = &I->Brother;
      
      *Last = this;
   }
}
									/*}}}*/
// Widget::GrabMouse - Grab the mouse					/*{{{*/
// ---------------------------------------------------------------------
/* This simply adds an entry to the grab list. RouteMouse does all the
   Deep Magic */
void Widget::GrabMouse(unsigned long Flags,bool Primary)
{
   if ((Flags & GrabAllGrab) == GrabAllGrab)
   {
      Primary = true;
      if (GrabList != 0 && GrabList[0].Target != 0 && 
	  GrabList[0].IsFlag(GrabAllGrab) == true)
      {
	 if (GrabList[0].Target->Realized == false)
	 {
	    GrabList[0].Target = this;
	    GrabList[0].Flags = Flags;
	 }
	 
	 return;
      }
      
/*      for (Widget *I = this; I != 0 && I->IsFlag(Region) == false); 
           I = I->Parent);
      GenGC::GC->GrabMouse();*/
   }
      
   // Allocate a new grab list
   if (GrabList == 0)
   {
      GrabSize = 10;
      GrabList = (GrabItem *)calloc(sizeof(*GrabList),GrabSize);
   }

   // Find an empty entry
   for (GrabItem *I = GrabList; I != GrabList + GrabSize; I++)
   {
      if (I->Target == 0)
      {
	 // Insert at the front
	 if (Primary == true)
	 {
	    memmove(GrabList + 1, GrabList,(I - GrabList)*sizeof(*GrabList));
	    GrabList[0].Target = this;
	    GrabList[0].Flags = Flags;
	    return;
	 }
	 
	 I->Target = this;
	 I->Flags = Flags;
	 return;
      }

      /* Hm, dup! We or the flags otherwise permanent items can be removed
         when an automatic grab is generated. */
      if (I->Target == this)
      {
	 I->Flags |= Flags;

	 if (Primary == true)
	 {
	    GrabItem Old = GrabList[0];
	    GrabList[0] = *I;
	    *I = Old;
	 }
	 return;
      }      
   }
   
   // Allocate more space
   GrabSize += 10;
   GrabList = (GrabItem *)realloc(GrabList,sizeof(*GrabList)*GrabSize);
   memset(GrabList + GrabSize - 10,0,sizeof(*GrabList)*10);

   // Insert at the front
   if (Primary == true)
   {
      memmove(GrabList + 1,GrabList,(GrabSize - 10)*sizeof(*GrabList));
      GrabList[0].Target = this;
      GrabList[0].Flags = Flags;
      return;
   }
   
   GrabList[GrabSize - 10].Target = this;
   GrabList[GrabSize - 10].Flags = Flags;
}
									/*}}}*/
// Widget::ToFront - Move the widget to the end of the child list	/*{{{*/
// ---------------------------------------------------------------------
/* Because the child is is stored in draw order moving the widget to the
   last link will make it drawn last. Note that overlapping widgets have
   unusual effects with region and non regioned widgets. */
void Widget::ToFront()
{
   // No brothers
   if (Parent == 0)
      return;
   
   // Find us
   Widget **Last = &Parent->Child;
   Widget *I;
   for (I = *Last; I != 0 && I != this; I = I->Brother)
      Last = &I->Brother;

   // Unlink us (segfault if the linked list is invalid)
   *Last = I->Brother;

   // Link us to the end
   Last = &Parent->Child;
   for (I = *Last; I != 0; I = I->Brother)
      Last = &I->Brother;
   *Last = this;
   Brother = 0;
   Parent->Damage();
}
									/*}}}*/
// Widget::ToBack - Move the widget to the back of the draw list	/*{{{*/
// ---------------------------------------------------------------------
/* */
void Widget::ToBack()
{
   // No brothers
   if (Parent == 0)
      return;
   
   // Find us
   Widget **Last = &Parent->Child;
   Widget *I;
   for (I = *Last; I != 0 && I != this; I = I->Brother)
      Last = &I->Brother;

   // Unlink us (segfault if the linked list is invalid)
   *Last = I->Brother;

   // Link us to the front of the list
   Brother = Parent->Child;
   Parent->Child = this;
   Parent->Damage();
}
									/*}}}*/
// Widget::Describe - Print a quick summary of the widget to a stream	/*{{{*/
// ---------------------------------------------------------------------
/* This uses RTTI to determine the real name of the derived type. If
   necessary the typeid could be replaced with "Widget".*/
void Widget::Describe(ostream &stream) const
{
   if (this == Root)
      stream << typeid(*this).name() << '(' << (void *)this << " == Widget::Root)";
   else
      stream << typeid(*this).name() << '(' << (void *)this << ')';
}
									/*}}}*/
