// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: graphics.cc,v 1.7 1998/09/15 05:57:19 jgg Exp $
/* ######################################################################

   Graphics - The graphics System
   GenGC - Generic Graphics Context
   TextGC - Textual GC (Character cells)
   GraphicsGC - Graphical GC

   Point and Rect provide some basic point and rectangle manipulations.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#ifdef __GNUG__
#pragma implementation "deity/graphics.h"
#endif  
#include <deity/graphics.h>
#include <deity/widget.h>
#include <deity/utils.h>

#include <iostream.h>
									/*}}}*/

GenGC *GenGC::GC = 0;
TextGC *TextGC::GC = 0;
GraphicGC *GraphicGC::GC = 0;

// GenGC::GenGC - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
GenGC::GenGC()
{
   EraseClippings();
}
									/*}}}*/
// GenGC::CreateRegion - Create a region for the widget			/*{{{*/
// ---------------------------------------------------------------------
/* Default action is to not support regions and to unset the region flag
   in the widget and return 0. */
void *GenGC::CreateRegion(Widget *For,void *)
{
   For->Flag(0,Widget::Region);
   return 0;
}
									/*}}}*/
// GenGC::EraseClippings - Erase all of the clipping rectangles		/*{{{*/
// ---------------------------------------------------------------------
/* AddClipping should be called after this to set a proper clipping 
   region for the region. */
void GenGC::EraseClippings()
{
   ClipStack[0] = Rect();
   ClipDepth = 0;
}
									/*}}}*/
// GenGC::AddClipping - Add a new clipping rectangle			/*{{{*/
// ---------------------------------------------------------------------
/* This pushes a new clipping rectangle onto the stack. Each rectangle 
   interescts with the previous to produce a single clip rect. */
void GenGC::AddClipping(Rect Win)
{
   if (ClipDepth == _count(ClipStack))
      return;

   ClipStack[ClipDepth] = Win;
   ClipDepth++;
   GenerateClip();
}
									/*}}}*/
// GenGC::PopClipping - Pull one clip region off the stack		/*{{{*/
// ---------------------------------------------------------------------
/* There should always be at least one clipping region set to be able
   to draw on the region */
void GenGC::PopClipping()
{
   if (ClipDepth == 0)
      return;
   ClipDepth--;
   ClipStack[ClipDepth] = Rect();
   GenerateClip();
}
									/*}}}*/
// GenGC::GenerateClip - Compute the single clipping rectangle		/*{{{*/
// ---------------------------------------------------------------------
/* This computes the final interesected clipping region. Clipping regions 
   also include coordinate translatations */
void GenGC::GenerateClip()
{
   if (ClipDepth == 0)
      return;
   
   Rect Clip = ClipStack[ClipDepth - 1];
   for (int I = ClipDepth - 2; I >= 0; I--)
   {
      Clip.Clip(ClipStack[I]);
      Clip.x += ClipStack[I].x;
      Clip.y += ClipStack[I].y;
   }

   // Tell the real GC about the new region
   ClipRect(Clip);
}
									/*}}}*/
// GenGC::DrawText - Converstion to Rect,Point draw text		/*{{{*/
// ---------------------------------------------------------------------
/* This computes the point arugment based on the bounding rectangle. */
void GenGC::DrawText(Rect Pos,const char *Start,unsigned int Len,
		     unsigned long Flags)
{
   Point P = Point(0,0);

   // Compute the anchor point.
   if ((Flags & XCenter) == XCenter)
      P.x = Pos.w/2;
   if ((Flags & XRight) == XRight)
      P.x = Pos.w;
   if ((Flags & YCenter) == YCenter)
      P.y = Pos.h/2;
   if ((Flags & YBottom) == YBottom)
      P.y = Pos.h;

   DrawText(Pos,P,Start,Len,Flags);
}
									/*}}}*/
// GenGC::FlagPosition - Compute a new rectangle corner			/*{{{*/
// ---------------------------------------------------------------------
/* As input pass a rectangle with the ul corner as the user parameter and
   the w/h as the object size. Flags indicates the meaning of the ul corner
   (Center, left right, up down, etc). This will compute a new ul corner 
   so that everything is nice and easy to draw. */
void GenGC::FlagPosition(Rect &P,unsigned long Flags)
{
   // Reposition the text
   if ((Flags & XCenter) == XCenter)
      P.x -= P.w/2;
   if ((Flags & XRight) == XRight)
      P.x -= P.w;
   if ((Flags & YCenter) == YCenter)
      P.y -= P.h/2;
   if ((Flags & YBottom) == YBottom)
      P.y -= P.h;   
}
									/*}}}*/
// GenGC::ExtentWrappedText - Compute the extents of the text		/*{{{*/
// ---------------------------------------------------------------------
/* The input box size is used to compute the wrapping. The output is the
   ideal wrap size to get minimum overrun. */
Point GenGC::ExtentWrappedText(Rect Loc,string S,bool Blank)
{  
   // Loop over the string until all the lines have been drawn
   long Height = ExtentText(" ").h;
   Point P = Point(0,0);   
   for (string::const_iterator I = S.begin(); I < S.end();)
   {
      string::const_iterator J;
      WrapStep(S,I,J,Loc.w);
      P.x = MAX(P.x,ExtentText(I,J-I).w);
      I = J;
      P.y += Height;
   }

   if (S.empty() == true || Blank == true)
      P.y += Height;
   return P;
}
									/*}}}*/
// GenGC::DrawWrappedText - Draws text into a box			/*{{{*/
// ---------------------------------------------------------------------
/* This simply warps the text at its longest point into a rectangle. */
long GenGC::DrawWrappedText(Rect Loc,string S,bool Blank)
{  
   // Loop over the string untill all the lines have been drawn
   long Height = ExtentText(" ").h;
   Point P = Point(Loc.x,Loc.y);
   for (string::const_iterator I = S.begin(); I < S.end();)
   {
      string::const_iterator J;
      
      // Out of horizontal space
      if (P.y + Height > Loc.y + Loc.h)
	 return P.y;
      
      WrapStep(S,I,J,Loc.w);

      DrawText(P,I,J - I);
      I = J;
      P.y += Height;
   }
   
   if (S.empty() == true || Blank == true)
      P.y += Height;
   return P.y;
}
									/*}}}*/
// GenGC::WrapStep - Helper function for the linewrapping functions	/*{{{*/
// ---------------------------------------------------------------------
/* This computes the next drawable portion of the string. */
void GenGC::WrapStep(string &S,string::const_iterator &Start,
		     string::const_iterator &End,long Width)
{
   for (; *Start == ' ' && Start < S.end(); Start++);
      
   // Compute the wrap point
   unsigned int Dist = GC->WrapText(Start,S.end() - Start,Width);
   
   // Look for an explicit line break
   string::const_iterator J = Start;
   for (; J < Start + Dist; J++)
      if (*J == '\n')
	 Dist = J - Start;
   
   // Backtrack to the nearest separator
   J = Start + Dist;
   for (; J > Start && *J != ' ' && *J != '-' && *J != '\n' && J != S.end(); J--);
   if (J == Start)
      J = Start + Dist;
   else
      if (J != S.end())
	 J++;

   // Just in case.
   if (J == Start && J < S.end())
      J++;
   
   End = J;
}
									/*}}}*/
// GenGC::MaskFill - Draw the border of a rectangle			/*{{{*/
// ---------------------------------------------------------------------
/* This function is used to partiall fill the background of a widget
   excluding the region it is going to draw in. It takes an outer
   and inner rectangle, the fill does not touch the inner rectangle. */
void GenGC::MaskFill(Rect Outer,Rect Inner)
{
   // Fill the remainder of the rectangle not covered by the text..
   Inner.Clip(Outer);
   if (Inner.x != 0)
      Fill(Rect(Outer.x,Outer.y,Inner.x,Outer.h));
   if (Inner.y != 0)
      Fill(Rect(Outer.x,Outer.y,Outer.w,Inner.y));
   if (Inner.x + Inner.w != Outer.w)
      Fill(AbsRect(Inner.x + Inner.w + Outer.x,Outer.y,
		   Outer.x + Outer.w,Outer.y + Outer.h));
   if (Inner.y + Inner.h != Outer.h)
      Fill(AbsRect(Outer.x,Inner.y + Inner.h + Outer.y,
		   Outer.x + Outer.w,Outer.y + Outer.h));
}
									/*}}}*/

// TextGC::~TextGC - Provides a base for the class			/*{{{*/
// ---------------------------------------------------------------------
/* */
TextGC::~TextGC()
{
}
									/*}}}*/
// TextGC::ExtentText - Return the size of a text string		/*{{{*/
// ---------------------------------------------------------------------
/* This is simple for text mode, number of chars and 1 */
Rect TextGC::ExtentText(const char *,unsigned int Len) 
{
   return Rect(0,0,Len,1);
}
									/*}}}*/
// GraphicGC::~GraphicGC - Provides a base for the class		/*{{{*/
// ---------------------------------------------------------------------
/* */
GraphicGC::~GraphicGC()
{
}
									/*}}}*/
// GraphicGC::DrawBitmap - Converstion to Rect,Point draw bitmap	/*{{{*/
// ---------------------------------------------------------------------
/* This computes the point arugment based on the bounding rectangle. */
void GraphicGC::DrawBitmap(Rect Pos,XPMImage &Image,unsigned long Flags)
{
   Point P = Point(0,0);

   // Compute the anchor point.
   if ((Flags & XCenter) == XCenter)
      P.x = Pos.w/2;
   if ((Flags & XRight) == XRight)
      P.x = Pos.w;
   if ((Flags & YCenter) == YCenter)
      P.y = Pos.h/2;
   if ((Flags & YBottom) == YBottom)
      P.y = Pos.h;

   DrawBitmap(Pos,P,Image,Flags);
}
									/*}}}*/

// Color::Color - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Color::Color() : RedLevel(0), GreenLevel(0), BlueLevel(0), Text(None)
{
}
									/*}}}*/
// SimpleFont::SimpleFont - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
SimpleFont::SimpleFont() : Weight(Normal), Slant(Upright)
{
}
									/*}}}*/

// Point::Point - Constructor  						/*{{{*/
// ---------------------------------------------------------------------
/* */
Point::Point()
{
   x = 0;
   y = 0;
}
									/*}}}*/
// operator <<(Point) - Stream output for a point			/*{{{*/
// ---------------------------------------------------------------------
/* */
ostream &operator << (ostream &stream,const Point &P)
{
   stream << "Point(" << P.x << ',' << P.y << ')';
   return stream;
}
									/*}}}*/

// Rect::Rect - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Rect::Rect()
{
   x = 0;
   y = 0;
   w = 0;
   h = 0;
}
									/*}}}*/
// Rect::Clip - Clips this rectangle so it is inside the other		/*{{{*/
// ---------------------------------------------------------------------
/* this is a subrectangle of other. The origin of other is irrelivant
   to the clipping algorithm. */
void Rect::Clip(Rect Other)
{
   if (x > Other.w)
      x = Other.w;
   if (y > Other.h)
      y = Other.h;
   if (x < 0)
   {
      w += x;
      x = 0;
   }
   if (y < 0)
   {
      h += y;
      y = 0;
   }
   if (x + w > Other.w)
      w = Other.w - x;
   if (y + h > Other.h)
      h = Other.h - y;
   if (w < 0)
      w = 0;
   if (h < 0)
      h = 0;
}
									/*}}}*/
// Rect::Inside - Determine if a point is inside a rectangle		/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Rect::Inside(Point P)
{
   if (P.x < x || P.y < y)
      return false;
   if (P.x >= x + w || P.y >= y + h)
      return false;
   return true;
}
									/*}}}*/
// Rect::Inside - Determine if a whole rectangle is within another	/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Rect::Inside(Rect R)
{
   if (R.Inside(Point(x,y)) && R.Inside(Point(x+w,y+h)))
      return true;
   return false;
}
									/*}}}*/
// Rect::Intersecting - Determine if two rectangles intersect		/*{{{*/
// ---------------------------------------------------------------------
/* If R is re-defined in terms of a sub-rectanlge of this then clipped to
   this a non-zero clipping region means they interesct. */
bool Rect::Intersecting(Rect R)
{
   R.x -= x;
   R.y -= y;
   R.Clip(*this);
   if (R.w == 0 && R.h == 0)
      return false;
   return true;
}
									/*}}}*/
// operator <<(Rect) - Stream output for a rect				/*{{{*/
// ---------------------------------------------------------------------
/* */
ostream &operator << (ostream &stream,const Rect &R)
{
   stream << "Rect(" << R.x << ',' << R.y << ',' << R.w << ',' << R.h << ')';
   return stream;
}
									/*}}}*/

// AbsRect::AbsRect - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
AbsRect::AbsRect()
{
   x1 = 0;
   y1 = 0;
   x2 = 0;
   y2 = 0;
}
									/*}}}*/
// operator <<(AbsRect) - Stream output for a rect			/*{{{*/
// ---------------------------------------------------------------------
/* */
ostream &operator << (ostream &stream,const AbsRect &R)
{
   stream << "AbsRect(" << R.x1 << ',' << R.y1 << ',' << 
      R.x2 << ',' << R.y2 << ')';
   return stream;
}
									/*}}}*/
