/*
Copyright (c) 1996-1997 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: cppparse.hpp,v 1.20 1997/09/15 16:44:26 srjohnso Exp $
*/

#ifndef _cppparse_hpp_
#define _cppparse_hpp_ 1

/*
The classes declared here provide a framework for building ISL parser
backends based on the OMG's IDL to C++ mapping. The ILU C++ stubber is
one such backend.

Namespace CppParse declares C++ classes (eg, Object, Procedure, etc)
representing syntactic elements of ISL. ISL parsing causes instances
of these objects to be arranged into an abstract syntax tree, with a
structure reflecting that of the input (ISL) file.

The CppParse classes are knowledgeable of the OMG mapping, and provide
functionality useful for any backend based on that mapping (eg, the
C++ name of an argument).

A backend can be built on the CppParse classes by extending them to
the purpose at hand (eg, the C++ stubber can add a method to the
CppParse "Type" class to generate code that will marshall values of
that type). By implementing and "registering" a "Factory", the backend
can tell the parser how to create a structure of it's specialized
types.
*/



// G++2.7.1 Compiler Note #1

#ifdef __GNUC__  // G++2.7.1 Compiler Note #2
#pragma interface "cppparse.hpp"
#endif  // __GNUC__

#include "iluptypecpp.hpp"
#include "cppparsenarrow.hpp"
#include "cppparsename.hpp"
#include "cppportability.hpp"


template <class T, class FROM> class CppParse_ListIterT;
template <class T, class FROM> class CppParse_ListT;


template <class T, class FROM>
class CppParse_FactoryT {
  public:  // Make/Destroy Ts
    static T * findOrCreate (FROM);
    static void release (T *);
    static void set (CppParse_FactoryT<T, FROM> * f);
    static CppParse_FactoryT<T, FROM> * current ();
  private:  // Override for custom implementation of T
    virtual T * _create (FROM) const;
    virtual void _destroy (T *) const;
    virtual void _createList (::iluptype::list from, ::iluptype::list * to) const;
      // Make a  list of T from a list of FROM.
      // MSVC4.1 Compiler Note #1
  #ifdef _MSC_VER  // MSVC4.1 Compiler Note #2
    public:
      friend class T;
      static FROM noteAllocated(T *, FROM);
  #else
    private:  // Used by T
      friend class T;
      static FROM noteAllocated(T *, FROM);
  #endif  // _MSC_VER
  private:  // Used by CppParse_ListT<T, FROM>
    friend class CppParse_ListT<T, FROM>;
    static void createList (::iluptype::list from, ::iluptype::list * to);
  private:
    static CppParse_FactoryT<T, FROM> * _current;
    static CppParse_FactoryT<T, FROM> _default;
    static ::iluptype::list _allocated;  // list of T
};

#if (defined(__GNUC__) || defined(_MSC_VER))
// G++2.7.1 Compiler Note #3
// MSVC4.1 Compiler Note #3

  template <class FROM>
  class CppParse_CreatableT {
    public:  // TEMP: for debugging; should be private
      const FROM _from;
    protected:
      CppParse_CreatableT (FROM);
    public:
      int _refCount;
  };

#elif __SUNPRO_CC
// SunPro4.1 Compiler Note #1
// SunPro4.1 Compiler Note #2

  template <class FROM>
  class CppParse_CreatableT {
    public:  // TEMP: for debugging; should be private
      FROM _from;
    protected:
      CppParse_CreatableT (FROM);
      void reInit (FROM);
    public:
      int _refCount;
  };

#else

  template <class FROM>
  class CppParse_CreatableT {
    public:  // TEMP: for debugging; should be private
      const FROM _from;
    protected:
      CppParse_CreatableT (FROM);
    private:
      template <class T, class FROM> friend CppParse_FactoryT;
      int _refCount;
  };

#endif  // (defined(__GNUC__) || defined(_MSC_VER))


template <class T, class FROM>
class CppParse_ListT {
  friend class CppParse_ListIterT<T, FROM>;
  public:
    CppParse_ListT (const ::iluptype::list);
    CppParse_ListT (const CppParse_ListT<T, FROM> &);
    ~CppParse_ListT ();
    unsigned long count () const;
  protected:
    unsigned CppParse_ListT<T, FROM>:: * _ptr_to_ref_count;
    unsigned _ref_count;
    ::iluptype::list _list;
};

template <class T, class FROM>
class CppParse_ListIterT {
  public:
    CppParse_ListIterT (const CppParse_ListT<T, FROM> &);
    void reset ();
    const T * next ();
  private:
    const CppParse_ListT<T, FROM> * _list;
    ::iluptype::listElement * _current;
};

const ::iluptype::TypeKind CppParse_Last_iluptype_TypeKind = iluptype::enumeration_Type;


#define CppParse_(name) NAME_INSIDE_SCOPE(CppParse, name)
#define CppParse(name) NAME_OUTSIDE_SCOPE(CppParse, name)

BEGIN_NAMESPACE(CppParse)


    // Scalar Types


            // CppParse name       iluptype struct   iluptype type kind             base name        ILU name
            // -------------       ---------------   ------------------             ---------        --------

    #define CppParse_ENUMERATE_CORBA_SCALAR_TYPES(ACTION)                                                            \
        ACTION(Byte,               ::iluptype::Type, iluptype::byte_Type,           Octet,           Byte)           \
        ACTION(Boolean,            ::iluptype::Type, iluptype::boolean_Type,        Boolean,         Boolean)        \
        ACTION(ShortCharacter,     ::iluptype::Type, iluptype::shortcharacter_Type, Char,            ShortCharacter) \
        ACTION(ShortInteger,       ::iluptype::Type, iluptype::shortinteger_Type,   Short,           ShortInteger)   \
        ACTION(Integer,            ::iluptype::Type, iluptype::integer_Type,        Long,            Integer)        \
        ACTION(ShortCardinal,      ::iluptype::Type, iluptype::shortcardinal_Type,  UShort,          ShortCardinal)  \
        ACTION(Cardinal,           ::iluptype::Type, iluptype::cardinal_Type,       ULong,           Cardinal)       \
        ACTION(Real,               ::iluptype::Type, iluptype::real_Type,           Double,          Real)           \
        ACTION(ShortReal,          ::iluptype::Type, iluptype::shortreal_Type,      Float,           ShortReal)

    #define CppParse_ENUMERATE_NON_CORBA_SCALAR_TYPES(ACTION)                                                      \
        ACTION(Character,          ::iluptype::Type, iluptype::character_Type,      iluCharacter,    Character)    \
        ACTION(LongInteger,        ::iluptype::Type, iluptype::longinteger_Type,    iluLongInteger,  LongInteger)  \
        ACTION(LongCardinal,       ::iluptype::Type, iluptype::longcardinal_Type,   iluLongCardinal, LongCardinal) \
        ACTION(LongReal,           ::iluptype::Type, iluptype::longreal_Type,       iluLongReal,     LongReal)

    #define CppParse_ENUMERATE_SCALAR_TYPES(ACTION)       \
        CppParse_ENUMERATE_CORBA_SCALAR_TYPES(ACTION)     \
        CppParse_ENUMERATE_NON_CORBA_SCALAR_TYPES(ACTION)

    
    // Sequence Types

    #define CppParse_ENUMERATE_SEQUENCE_TYPES(ACTION)                                                        \
        ACTION(Sequence,           ::iluptype::Type, iluptype::sequence_Type,       XXX,           Sequence) \
        ACTION(ObjectSequence,     ::iluptype::Type, iluptype::sequence_Type,       XXX,           Sequence) \
        ACTION(StringSequence,     ::iluptype::Type, iluptype::sequence_Type,       XXX,           Sequence) \
        ACTION(WideStringSequence, ::iluptype::Type, iluptype::sequence_Type,       XXX,           Sequence) \
        ACTION(String,             ::iluptype::Type, iluptype::sequence_Type,       char*,         Sequence) \
     /* ACTION(WideString,         ::iluptype::Type, iluptype::sequence_Type,       wchar_t*,      Sequence) */   \
        ACTION(WideString,         ::iluptype::Type, iluptype::sequence_Type,       iluCharacter*, Sequence)  
    // SunPro4.1 Compiler Note #3


    // Constructed Types

    #define CppParse_ENUMERATE_CONSTRUCTED_TYPES(ACTION)                                              \
        ACTION(Object,             ::iluptype::Type, iluptype::object_Type,         XXX, Object)      \
        ACTION(Record,             ::iluptype::Type, iluptype::record_Type,         XXX, Record)      \
        ACTION(Union,              ::iluptype::Type, iluptype::union_Type,          XXX, Union)       \
        ACTION(Array,              ::iluptype::Type, iluptype::array_Type,          XXX, Array)       \
        ACTION(Enumeration,        ::iluptype::Type, iluptype::enumeration_Type,    XXX, Enumeration) \
        ACTION(Alias,              ::iluptype::Type, iluptype::alias_Type,          XXX, Alias)       \
        ACTION(Optional,           ::iluptype::Type, iluptype::optional_Type,       XXX, Optional)


    // All Types

    #define CppParse_ENUMERATE_TYPES(ACTION)         \
        CppParse_ENUMERATE_SCALAR_TYPES(ACTION)      \
        CppParse_ENUMERATE_SEQUENCE_TYPES(ACTION)    \
        CppParse_ENUMERATE_CONSTRUCTED_TYPES(ACTION)


    // All Creatables

    #define CppParse_ENUMERATE_CREATABLES(ACTION)                             \
        CppParse_ENUMERATE_TYPES(ACTION)                                      \
        ACTION(Type,               ::iluptype::Type,           XXX, XXX, XXX) \
        ACTION(ConstantValue,      ::iluptype::ConstantValue,  XXX, XXX, XXX) \
        ACTION(UnionArm,           ::iluptype::Argument,       XXX, XXX, XXX) \
        ACTION(Argument,           ::iluptype::Argument,       XXX, XXX, XXX) \
        ACTION(Member,             ::iluptype::Argument,       XXX, XXX, XXX) \
        ACTION(ArrayDimension,     long unsigned,              XXX, XXX, XXX) \
        ACTION(Enumerator,         ::iluptype::EnumField,      XXX, XXX, XXX) \
        ACTION(Exception,          ::iluptype::Exception,      XXX, XXX, XXX) \
        ACTION(IDLException,       ::iluptype::Exception,      XXX, XXX, XXX) \
        ACTION(ExceptionValue,     ::iluptype::Exception,      XXX, XXX, XXX) \
        ACTION(Procedure,          ::iluptype::Procedure,      XXX, XXX, XXX) \
        ACTION(Interface,          ::iluptype::Interface,      XXX, XXX, XXX) \
        ACTION(ReturnVal,          ::iluptype::Procedure,      XXX, XXX, XXX) \
        ACTION(Import,             ::iluptype::Imported,       XXX, XXX, XXX)


    class CppParse_(Bool) {

    public:

        static
        const char *
        trueValue () {
            return _current->_trueValue();
        };

        static
        const char *
        falseValue () {
            return _current->_falseValue();
        };

        static
        void
        set (CppParse_(Bool) * newCurrent);

    private:

        virtual
        const char *
        _trueValue ();

        virtual
        const char *
        _falseValue ();
        
        static
        CppParse_(Bool) * _current;
    };


    enum CppParse_(Indirection) {  // TMP 7/7: NEW
        CppParse_(direct),        // no indirection; eg, CORBA::Long  // TMP 7/7: NEW
        CppParse_(pointer),       // indirection via "*"; eg, char *  // TMP 7/7: NEW
        CppParse_(pseudoPointer)  // indirection via typedef; eg, MyAlias, const_MyAlias  // TMP 7/7: NEW
    };  // TMP 7/7: NEW




    
  class CppParse_(Type);
  class CppParse_(Alias);
  class CppParse_(Optional);
  
  typedef ILUCPP_BOOL (*CppParse_(TypeEnumProc)) (const CppParse_(Type) *, void * state);  // TMP 7/23: NEW
      // Callback proc, used to visit types contained in the definition of some other type.
      // State is caller-supplied state information.
      // Return ILUCPP_TRUE to continue enumeration, ILUCPP_FALSE to halt.


  class CppParse_(Type):
      public CppParse_CreatableT< ::iluptype::Type>,
      public virtual CppParseName(TypeName),
      public virtual CppParseNarrow_Root {
          CppParseNarrow_FUNCS0(CppParse_(Type));
          friend CppParse_FactoryT<CppParse_(Type), ::iluptype::Type>;
          friend CppParseName(TypeName);
          friend CppParse_(Alias);
          friend CppParse_(Optional);
          public:
              ILUCPP_BOOL builtIn () const;
/* TMP 7/7
              virtual ILUCPP_BOOL isPtr () const;
              virtual ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const; 
*/
              virtual CppParse_(Indirection) indirection () const;  // TMP 7/7: NEW
              virtual CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
              virtual ILUCPP_BOOL variableLength () const;
              virtual const CppParse_(Type) * urType () const;
              virtual const char * typeKindName (ILUCPP_BOOL mixedCase = ILUCPP_FALSE) const;  // TMP 8/4: NEW
	          const char * uid () const;
              virtual void enumerateContained (CppParse_(TypeEnumProc) proc, void * state) const;  // TMP 7/23: NEW
                  // Visit the closure of contained types, calling proc with state.
                  // A no-op for non-container types.
                  // Recursive calls have no effect.
          private:  // Override methods from CppParseName::TypeName
              const CppParseName(AffixSet) * affixes () const;
              const CppParseName(AffixSet) * optionalAffixes () const;  // TMP 6/28: NEW
          protected:
            CppParse_(Type) (::iluptype::Type);
            #ifdef __SUNPRO_CC  // SunPro4.1 Compiler Note #2
                void reInit(::iluptype::Type);
            #endif  // __SUNPRO_CC
  };

   // MSVC4.1 Compiler Note #4
  typedef CppParse_FactoryT<CppParse_(Type), ::iluptype::Type>  CppParse_(TypeFactory);
  typedef CppParse_ListT<CppParse_(Type), ::iluptype::Type>     CppParse_(TypeList);
  typedef CppParse_ListIterT<CppParse_(Type), ::iluptype::Type> CppParse_(TypeListIter);
  
/*
    Note on Factories for TypeKinds: The default TypeFactory will call the create and destroy members of the Factory for the TypeKind corresponding to the type_kind of the underlying iluptype::Type.
*/


/* TMP 7/20
    #define _cppparse_hpp_DEF_SCALAR_TYPE(T, FROM, UNUSED1, UNUSED2, UNUSED3) \
        class CppParse_(T): public virtual CppParse_(Type) {                  \
            CppParseNarrow_FUNCS1(CppParse_(T), CppParse_(Type));             \
            friend CppParse_FactoryT<CppParse_(T), ::iluptype::Type>;         \
            protected:                                                        \
                CppParse_(T) (FROM);                                          \
        };
*/

    #define _cppparse_hpp_DEF_CORBA_SCALAR_TYPE(T, FROM, UNUSED1, UNUSED2, UNUSED3)                               \
        class CppParse_(T): public virtual CppParse_(Type) {                                                      \
            CppParseNarrow_FUNCS1(CppParse_(T), CppParse_(Type));                                                 \
            friend CppParse_FactoryT<CppParse_(T), ::iluptype::Type>;                                             \
        public:  /* Override method from CppParseName::TypeName */                                                \
            const char * usageName (CppParseName(TypeUsage), const CppParseName(Scope) * context = NULL) const;   \
        protected:                                                                                                \
            CppParse_(T) (FROM);                                                                                  \
        };

    #define _cppparse_hpp_DEF_NON_CORBA_SCALAR_TYPE(T, FROM, UNUSED1, UNUSED2, UNUSED3) \
        class CppParse_(T): public virtual CppParse_(Type) {                            \
            CppParseNarrow_FUNCS1(CppParse_(T), CppParse_(Type));                       \
            friend CppParse_FactoryT<CppParse_(T), ::iluptype::Type>;                   \
        protected:                                                                      \
            CppParse_(T) (FROM);                                                        \
        };

    #define _cppparse_hpp_DEF_FACTORY(T, FROM, UNUSED1, UNUSED2, UNUSED3)      \
        typedef CppParse_FactoryT<CppParse_(T), FROM> CppParse_(T ## Factory);

    #define _cppparse_hpp_DEF_LIST(T, FROM, UNUSED1, UNUSED2, UNUSED3)           \
        typedef CppParse_ListT<CppParse_(T), FROM> CppParse_(T ## List);         \
        typedef CppParse_ListIterT<CppParse_(T), FROM> CppParse_(T ## ListIter);


// TMP 7/20    CppParse_ENUMERATE_SCALAR_TYPES(_cppparse_hpp_DEF_SCALAR_TYPE)
    CppParse_ENUMERATE_CORBA_SCALAR_TYPES(_cppparse_hpp_DEF_CORBA_SCALAR_TYPE)
    CppParse_ENUMERATE_NON_CORBA_SCALAR_TYPES(_cppparse_hpp_DEF_NON_CORBA_SCALAR_TYPE)
    CppParse_ENUMERATE_SCALAR_TYPES(_cppparse_hpp_DEF_FACTORY)
    CppParse_ENUMERATE_SCALAR_TYPES(_cppparse_hpp_DEF_LIST)


  class CppParse_(NamedValue): public CppParseName(Name) {

  public:
      CppParseName(TypeUsage) usage () const;
      const CppParse_(Type) * type () const;

  public:  // override CppParseName::Name method  // TMP 8/28: NEW
      virtual const char * simpleName (CppParseName(Key) = NULL) const;  // TMP 8/28: NEW

  private:
      const CppParseName(TypeUsage) _usage;
      const CppParse_(Type) * _type;

  protected:
      CppParse_(NamedValue) (::iluptype::Procedure);
          // Procedure return value
      CppParse_(NamedValue) (::iluptype::Argument);
          // Procedure arguments
      CppParse_(NamedValue) (::iluptype::Argument, CppParseName(TypeUsage));
          // Record & union members
      CppParse_(NamedValue) (::iluptype::Exception);
          // ExceptionValue's
      CppParse_(NamedValue) (const CppParseName(Name)&, CppParseName(TypeUsage), const CppParse_(Type) *);
          // Generic.
  };

    
    class CppParse_(Record);


    class CppParse_(Member):
    public virtual CppParse_(NamedValue),
        public CppParse_CreatableT< ::iluptype::Argument>,
        public virtual CppParseNarrow_Root {
        CppParseNarrow_FUNCS0(CppParse_(Member));
        friend CppParse_FactoryT<CppParse_(Member), ::iluptype::Argument>;
        friend CppParse_(Record);
    protected:
        CppParse_(Member) (::iluptype::Argument);
    private:  // Override method from CppParseName::TypeName)
        const CppParseName(AffixSet) * affixes () const;
    private:  // Reintroduce CppParseName::Name::resetName, to give access to CppParse::Record and CppParse::Union
        void resetName (const CppParseName(Scope) * definedIn) {
            CppParseName(Name)::resetName(definedIn);
        };
    };

  typedef CppParse_FactoryT<CppParse_(Member), ::iluptype::Argument>   CppParse_(MemberFactory);
  typedef CppParse_ListT<CppParse_(Member), ::iluptype::Argument>      CppParse_(MemberList);
  typedef CppParse_ListIterT<CppParse_(Member), ::iluptype::Argument>  CppParse_(MemberListIter);


  class CppParse_(Enumeration);

  class CppParse_(Enumerator): public CppParse_CreatableT< ::iluptype::EnumField>, public CppParseName(Name), public virtual CppParseNarrow_Root {
      CppParseNarrow_FUNCS0(CppParse_(Enumerator));
      friend CppParse_FactoryT<CppParse_(Enumerator), ::iluptype::EnumField>;
      friend CppParse_(Enumeration);
      public:
          int id () const;
      private:  // Reintroduce CppParseName::Name::resetName, to give access to CppParse::Enumeration
        void resetName (const CppParseName(Scope) * definedIn) {
            CppParseName(Name)::resetName(definedIn);
        };
      protected:
          CppParse_(Enumerator) (::iluptype::EnumField);
  };

  
  typedef CppParse_FactoryT<CppParse_(Enumerator), ::iluptype::EnumField>   CppParse_(EnumeratorFactory);
  typedef CppParse_ListT<CppParse_(Enumerator), ::iluptype::EnumField>      CppParse_(EnumeratorList);
  typedef CppParse_ListIterT<CppParse_(Enumerator), ::iluptype::EnumField>  CppParse_(EnumeratorListIter);


  class CppParse_(Enumeration): public virtual CppParse_(Type) {
      CppParseNarrow_FUNCS1(CppParse_(Enumeration), CppParse_(Type));	  
      friend CppParse_FactoryT<CppParse_(Enumeration), ::iluptype::Type>;
  protected:
      CppParse_(Enumeration) (::iluptype::Type from);
  public:
      const CppParse_(EnumeratorList) & enumerators () const;
/* TMP 7/14: appears to be the same as default
  private:  // Override method from CppParseName::TypeName
      virtual const CppParseName(AffixSet) * affixes () const;
*/
  private:
      const CppParse_(EnumeratorList) _enums;
  };

  typedef CppParse_FactoryT<CppParse_(Enumeration), ::iluptype::Type>   CppParse_(EnumerationFactory);
  typedef CppParse_ListT<CppParse_(Enumeration), ::iluptype::Type>      CppParse_(EnumerationList);
  typedef CppParse_ListIterT<CppParse_(Enumeration), ::iluptype::Type>  CppParse_(EnumerationListIter);
  

  class CppParse_(Sequence): public virtual CppParse_(Type) {
      CppParseNarrow_FUNCS1(CppParse_(Sequence), CppParse_(Type));	  
      friend CppParse_FactoryT<CppParse_(Sequence), ::iluptype::Type>;
  public:
      long unsigned limit () const;
          // 0 => unbounded sequence, else bounded
      const CppParse_(Type) * elementType () const;
  public:  // Override methods from CppParse::Type
// TMP 7/7      ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const;
      virtual CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
      virtual ILUCPP_BOOL variableLength () const;
      void enumerateContained (CppParse_(TypeEnumProc) proc, void * state) const;  // TMP 7/23: NEW
  private:  // Override method from CppParseName::TypeName
      virtual const CppParseName(AffixSet) * affixes () const;
  protected:
      CppParse_(Sequence) (::iluptype::Type from);
      #ifdef __SUNPRO_CC  // SunPro4.1 Compiler Note #2
          void reInit (::iluptype::Type);
      #endif  // __SUNPRO_CC
  private:
      const CppParse_(Type) * _elementType;
  };

  typedef CppParse_FactoryT<CppParse_(Sequence), ::iluptype::Type>   CppParse_(SequenceFactory);
  typedef CppParse_ListT<CppParse_(Sequence), ::iluptype::Type>      CppParse_(SequenceList);
  typedef CppParse_ListIterT<CppParse_(Sequence), ::iluptype::Type>  CppParse_(SequenceListIter);

  
  class CppParse_(ObjectSequence): public virtual CppParse_(Sequence) {
      CppParseNarrow_FUNCS1(CppParse_(ObjectSequence), CppParse_(Sequence));	  
      friend CppParse_FactoryT<CppParse_(ObjectSequence), ::iluptype::Type>;
  protected:
      CppParse_(ObjectSequence) (::iluptype::Type from);
  };

  typedef CppParse_FactoryT<CppParse_(ObjectSequence), ::iluptype::Type>   CppParse_(ObjectSequenceFactory);
  typedef CppParse_ListT<CppParse_(ObjectSequence), ::iluptype::Type>      CppParse_(ObjectSequenceList);
  typedef CppParse_ListIterT<CppParse_(ObjectSequence), ::iluptype::Type>  CppParse_(ObjectSequenceListIter);
  

  class CppParse_(StringSequence): public virtual CppParse_(Sequence) {
      CppParseNarrow_FUNCS1(CppParse_(StringSequence), CppParse_(Sequence));	  
      friend CppParse_FactoryT<CppParse_(StringSequence), ::iluptype::Type>;
  protected:
      CppParse_(StringSequence) (::iluptype::Type from);
  };

  typedef CppParse_FactoryT<CppParse_(StringSequence), ::iluptype::Type>   CppParse_(StringSequenceFactory);
  typedef CppParse_ListT<CppParse_(StringSequence), ::iluptype::Type>      CppParse_(StringSequenceList);
  typedef CppParse_ListIterT<CppParse_(StringSequence), ::iluptype::Type>  CppParse_(StringSequenceListIter);
  
  class CppParse_(WideStringSequence): public virtual CppParse_(Sequence) {
      CppParseNarrow_FUNCS1(CppParse_(WideStringSequence), CppParse_(Sequence));	  
      friend CppParse_FactoryT<CppParse_(WideStringSequence), ::iluptype::Type>;
  protected:
      CppParse_(WideStringSequence) (::iluptype::Type from);
  };

  typedef CppParse_FactoryT<CppParse_(WideStringSequence), ::iluptype::Type>   CppParse_(WideStringSequenceFactory);
  typedef CppParse_ListT<CppParse_(WideStringSequence), ::iluptype::Type>      CppParse_(WideStringSequenceList);
  typedef CppParse_ListIterT<CppParse_(WideStringSequence), ::iluptype::Type>  CppParse_(WideStringSequenceListIter);
  

  
  class CppParse_(String): public CppParse_(Sequence) {
      CppParseNarrow_FUNCS1(CppParse_(String), CppParse_(Sequence));	  
      friend CppParse_FactoryT<CppParse_(String), ::iluptype::Type>;
  public:  // Override method from CppParse::Type
/* TMP 7/7
      ILUCPP_BOOL isPtr () const;
      ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const;
          // TEMP: Although CORBA does not specify mapping for record members and array/sequence elements,
          // strings in those contexts *behave* like char*; thus CppParseName::recordMember and
          // CppParseName::element return true.
*/
      CppParse_(Indirection) indirection () const;  // TMP 7/7: NEW
      CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
          // TEMP: Although CORBA does not specify mapping for record members and array/sequence elements,  // TMP 7/7: NEW
          // strings in those contexts *behave* like char*; thus CppParseName::recordMember and  // TMP 7/7: NEW
          // CppParseName::element return CppParseName::pointer.  // TMP 7/7: NEW
  public:  // Override method from CppParseName::TypeName
      virtual const char * simpleName (CppParseName(Key) key = NULL) const;
  private:  // Override method from CppParseName::TypeName
      virtual const CppParseName(AffixSet) * affixes () const;
  protected:
      CppParse_(String) (::iluptype::Type from);
  };

  typedef CppParse_FactoryT<CppParse_(String), ::iluptype::Type>   CppParse_(StringFactory);
  typedef CppParse_ListT<CppParse_(String), ::iluptype::Type>      CppParse_(StringList);
  typedef CppParse_ListIterT<CppParse_(String), ::iluptype::Type>  CppParse_(StringListIter);
  
  class CppParse_(WideString): public CppParse_(Sequence) {
      CppParseNarrow_FUNCS1(CppParse_(WideString), CppParse_(Sequence));	  
      friend CppParse_FactoryT<CppParse_(WideString), ::iluptype::Type>;
  public:  // Override method from CppParse::Type
/* TMP 7/7
      ILUCPP_BOOL isPtr () const;
      ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const;
          // TEMP: Although CORBA does not specify mapping for record members and array/sequence elements,
          // strings in those contexts *behave* like wchar_t*; thus CppParseName::recordMember and
          // CppParseName::element return true.
*/
      CppParse_(Indirection) indirection () const;  // TMP 7/7: NEW
      CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
          // TEMP: Although CORBA does not specify mapping for record members and array/sequence elements,  // TMP 7/7: NEW
          // strings in those contexts *behave* like wchar_t*; thus CppParseName::recordMember and  // TMP 7/7: NEW
          // CppParseName::element return CppParseName::pointer.  // TMP 7/7: NEW
  public:  // Override method from CppParseName::TypeName
      virtual const char * simpleName (CppParseName(Key) key = NULL) const;
  private:  // Override method from CppParseName::TypeName
      virtual const CppParseName(AffixSet) * affixes () const;
  protected:
      CppParse_(WideString) (::iluptype::Type from);
  };

  typedef CppParse_FactoryT<CppParse_(WideString), ::iluptype::Type>   CppParse_(WideStringFactory);
  typedef CppParse_ListT<CppParse_(WideString), ::iluptype::Type>      CppParse_(WideStringList);
  typedef CppParse_ListIterT<CppParse_(WideString), ::iluptype::Type>  CppParse_(WideStringListIter);
  

  class CppParse_(Record): public virtual CppParse_(Type), public CppParseName(Scope) {
      CppParseNarrow_FUNCS2(CppParse_(Record), CppParse_(Type), CppParseName(Scope));
      friend CppParse_FactoryT<CppParse_(Record), ::iluptype::Type>;
  public:  // Override methods from CppParse::Type
// TMP 7/7      ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const;
      CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
      virtual ILUCPP_BOOL variableLength () const;
      void enumerateContained (CppParse_(TypeEnumProc) proc, void * state) const;  // TMP 7/23: NEW
  private:  // Override method from CppParseName::TypeName
      virtual const CppParseName(AffixSet) * affixes () const;
  protected:
      CppParse_(Record) (::iluptype::Type from);
  public:
      const CppParse_(MemberList) & members () const;
  private:
      const CppParse_(MemberList) _members;
// TMP 7/5      const ILUCPP_BOOL _variableLength;
  public:  // TMP 7/20
      void * operator new (size_t, ::iluptype::Type);  // TMP 7/20
  };

  typedef CppParse_FactoryT<CppParse_(Record), ::iluptype::Type>   CppParse_(RecordFactory);
  typedef CppParse_ListT<CppParse_(Record), ::iluptype::Type>      CppParse_(RecordList);
  typedef CppParse_ListIterT<CppParse_(Record), ::iluptype::Type>  CppParse_(RecordListIter);


  class CppParse_(Union);

  class CppParse_(ConstantValue): public CppParse_CreatableT< ::iluptype::ConstantValue>, public virtual CppParseNarrow_Root {
      CppParseNarrow_FUNCS0(CppParse_(ConstantValue));	  
      friend CppParse_FactoryT<CppParse_(ConstantValue), ::iluptype::ConstantValue>;
	  friend CppParse_(Union);  // TMP 9/7: NEW

  public:
      const char * str (const CppParseName(Scope)* context = NULL) const;
	      // The constant's value, converted to a string
	      // When the constant's type is an enumeration, str is the enumerator's
	      // name, qualified per context, ala CppParseName::Name.
          // string belongs to ConstantValue
	  const char * typeKindName (ILUCPP_BOOL mixedCase = ILUCPP_FALSE) const;

  protected:
      CppParse_(ConstantValue) (::iluptype::ConstantValue);
  private:  // TMP 9/7: NEW
	  const CppParse_(Type) * _type;  // TMP 9/7: NEW
  };

  typedef CppParse_FactoryT<CppParse_(ConstantValue), ::iluptype::ConstantValue>   CppParse_(ConstantValueFactory);
  typedef CppParse_ListT<CppParse_(ConstantValue), ::iluptype::ConstantValue>      CppParse_(ConstantValueList);
  typedef CppParse_ListIterT<CppParse_(ConstantValue), ::iluptype::ConstantValue>  CppParse_(ConstantValueListIter);


  class CppParse_(UnionArm): public CppParse_(Member) {
      CppParseNarrow_FUNCS1(CppParse_(UnionArm), CppParse_(Member));
      friend CppParse_FactoryT<CppParse_(UnionArm), ::iluptype::Argument>;
	  friend CppParse_(Union);
  public:
	  long unsigned index () const;
	      // Rank in list of Union's arms (0-based).
      const CppParse_(ConstantValueList) & values () const;
          // Not valid when this is a default arm
      const CppParse_(ConstantValue) * firstValue () const;
          // Returns first value specified for this list
          // default arm => returns lowest value not specified in other arms

/* TMP 8/28
    public:  // override CppParseName::Name methods  // TMP 8/27: NEW
        virtual const char * simpleName (CppParseName(Key) = NULL) const;  // TMP 8/27: NEW
TMP 8/28 */

  private:
      const CppParse_(ConstantValueList) _values;
	  const long unsigned _index;
  protected:
      CppParse_(UnionArm) (::iluptype::Argument);
  };

  typedef CppParse_FactoryT<CppParse_(UnionArm), ::iluptype::Argument>   CppParse_(UnionArmFactory);
  typedef CppParse_ListT<CppParse_(UnionArm), ::iluptype::Argument>      CppParse_(UnionArmList);
  typedef CppParse_ListIterT<CppParse_(UnionArm), ::iluptype::Argument>  CppParse_(UnionArmListIter);


  class CppParse_(Union): public virtual CppParse_(Record) {
      CppParseNarrow_FUNCS1(CppParse_(Union), CppParse_(Record));	  
      friend CppParse_FactoryT<CppParse_(Union), ::iluptype::Type>;
      friend CppParse_(UnionArm);
  public:
      const CppParse_(Type) * discriminatorType () const;
      const CppParse_(MemberList) & arms () const {return members();}
      const CppParse_(UnionArm) * defaultArm () const;
      ILUCPP_BOOL otherDiscriminatorValuesOK () const;
          // true => discriminator may legally have a value
          // not associated with an arm
  protected:
      CppParse_(Union) (::iluptype::Type from);
  private:
      const CppParse_(Type) * _discriminatorType;
      const CppParse_(UnionArm) * _defaultArm;
      const ILUCPP_BOOL _otherDiscriminatorValuesOK;
      const CppParse_(ConstantValue) * _firstUnspecifiedValue;
  public:  // TMP 7/20
      void * operator new (size_t, ::iluptype::Type);  // TMP 7/20
  };

  typedef CppParse_FactoryT<CppParse_(Union), ::iluptype::Type>   CppParse_(UnionFactory);
  typedef CppParse_ListT<CppParse_(Union), ::iluptype::Type>      CppParse_(UnionList);
  typedef CppParse_ListIterT<CppParse_(Union), ::iluptype::Type>  CppParse_(UnionListIter);


  class CppParse_(Alias):
      public virtual CppParse_(Type),
      public CppParseName(AliasName) {
      CppParseNarrow_FUNCS1(CppParse_(Alias), CppParse_(Type));	  
      friend CppParse_FactoryT<CppParse_(Alias), ::iluptype::Type>;

  public:

      virtual
      ILUCPP_BOOL
      variableLength () const {
          return urType()->variableLength();
      };

/* TMP 7/7
      virtual
      ILUCPP_BOOL
      isPtr () const {
          return urType()->isPtr();
      };

      virtual
      ILUCPP_BOOL
      isPtr (CppParseName(TypeUsage) usage) const {
          return urType()->isPtr(usage);
      };
*/

      virtual  // TMP 7/7: NEW
      CppParse_(Indirection)  // TMP 7/7: NEW
      indirection () const {  // TMP 7/7: NEW
          CppParse_(Indirection) rtn = urType()->indirection();  // TMP 7/7: NEW
          return ((rtn == CppParse_(pointer)) ? CppParse_(pseudoPointer) : rtn);  // TMP 7/7: NEW
      };  // TMP 7/7: NEW

      virtual  // TMP 7/7: NEW
      CppParse_(Indirection)  // TMP 7/7: NEW
      indirection (CppParseName(TypeUsage) usage) const {  // TMP 7/7: NEW
          CppParse_(Indirection) rtn = urType()->indirection(usage);  // TMP 7/7: NEW
          return ((rtn == CppParse_(pointer)) ? CppParse_(pseudoPointer) : rtn);  // TMP 7/7: NEW
      };  // TMP 7/7: NEW

      virtual
      const CppParse_(Type) *
      urType () const;

      virtual  // TMP 7/23: NEW
      void
      enumerateContained (CppParse_(TypeEnumProc) proc, void * state) const;

  private:  // Override method from CppParseName::TypeName

      virtual
      const CppParseName(AffixSet) *
      affixes () const {
          return urType()->affixes();
      };

      virtual  // TMP 6/28: NEW
      const CppParseName(AffixSet) *
      optionalAffixes () const {
          return urType()->optionalAffixes();
      };

  protected:
      CppParse_(Alias) (::iluptype::Type from);
  private:
      const CppParse_(Type) * _urType;
  };

  typedef CppParse_FactoryT<CppParse_(Alias), ::iluptype::Type>   CppParse_(AliasFactory);
  typedef CppParse_ListT<CppParse_(Alias), ::iluptype::Type>      CppParse_(AliasList);
  typedef CppParse_ListIterT<CppParse_(Alias), ::iluptype::Type>  CppParse_(AliasListIter);


  class CppParse_(ArrayDimension): public CppParse_CreatableT<long unsigned>, public virtual CppParseNarrow_Root {
      CppParseNarrow_FUNCS0(CppParse_(ArrayDimension));
      friend CppParse_FactoryT<CppParse_(ArrayDimension), long unsigned>;
  public:
      long unsigned length () const;
  protected:
      CppParse_(ArrayDimension) (long unsigned);
  };

  typedef CppParse_FactoryT<CppParse_(ArrayDimension), long unsigned>   CppParse_(ArrayDimensionFactory);
  typedef CppParse_ListT<CppParse_(ArrayDimension), long unsigned>      CppParse_(ArrayDimensionList);
  typedef CppParse_ListIterT<CppParse_(ArrayDimension), long unsigned>  CppParse_(ArrayDimensionListIter);

  class CppParse_(Array): public virtual CppParse_(Type) {
      CppParseNarrow_FUNCS1(CppParse_(Array), CppParse_(Type));	  
      friend CppParse_FactoryT<CppParse_(Array), ::iluptype::Type>;
  public:
      const CppParse_(ArrayDimensionList) & dimensions () const;
      const CppParse_(Type) * elementType () const;
  public:  // Override methods from CppParse::Type
// TMP 7/7      ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const;
      CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
      virtual ILUCPP_BOOL variableLength () const;
      void enumerateContained (CppParse_(TypeEnumProc) proc, void * state) const;  // TMP 7/23: NEW
  private:  // Override method from CppParseName::TypeName
      virtual const CppParseName(AffixSet) * affixes () const;
      virtual const CppParseName(AffixSet) * optionalAffixes () const;  // TMP 7/15: NEW
  protected:
      CppParse_(Array) (::iluptype::Type from);
  private:
      const CppParse_(ArrayDimensionList) _dimensions;
      const CppParse_(Type) * _elementType;
// TMP 7/5      const ILUCPP_BOOL _variableLength;
  };

  typedef CppParse_FactoryT<CppParse_(Array), ::iluptype::Type>   CppParse_(ArrayFactory);
  typedef CppParse_ListT<CppParse_(Array), ::iluptype::Type>      CppParse_(ArrayList);
  typedef CppParse_ListIterT<CppParse_(Array), ::iluptype::Type>  CppParse_(ArrayListIter);
  

  class CppParse_(Optional): public virtual CppParse_(Type) {
      CppParseNarrow_FUNCS1(CppParse_(Optional), CppParse_(Type));	  
      friend CppParse_FactoryT<CppParse_(Optional), ::iluptype::Type>;
  public:
      const CppParse_(Type) * valueType () const;
  public:  // Override methods from CppParse::Type
/* TMP 7/7
      ILUCPP_BOOL isPtr () const;
      ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const;
*/
      CppParse_(Indirection) indirection () const;  // TMP 7/7: NEW
      CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
      virtual ILUCPP_BOOL variableLength () const;
          // returns ILUCPP_TRUE
      void enumerateContained (CppParse_(TypeEnumProc) proc, void * state) const;  // TMP 7/23: NEW
  private:  // Override method from CppParseName::TypeName
      virtual const CppParseName(AffixSet) * affixes () const;
      virtual const CppParseName(AffixSet) * optionalAffixes () const;  // TMP 6/28: NEW
  protected:
      CppParse_(Optional) (::iluptype::Type from);
  private:
      const CppParse_(Type) * _valueType;
  };

  typedef CppParse_FactoryT<CppParse_(Optional), ::iluptype::Type>   CppParse_(OptionalFactory);
  typedef CppParse_ListT<CppParse_(Optional), ::iluptype::Type>      CppParse_(OptionalList);
  typedef CppParse_ListIterT<CppParse_(Optional), ::iluptype::Type>  CppParse_(OptionalListIter);


  class CppParse_(TMP) {  // TMP 5/27: NEW
  public:  // TMP 5/27: NEW
      CppParse_(TMP) () {};  // TMP 5/27: NEW
      CppParse_(TMP) (void *) {};  // TMP 5/27: NEW
  };  // TMP 5/27: NEW

    class CppParse_(ExceptionValue):
    public virtual CppParse_(TMP),  // TMP 5/27: NEW
    public virtual CppParse_(NamedValue),
    public virtual CppParseNarrow_Root,
    public CppParse_CreatableT< ::iluptype::Exception> {
        CppParseNarrow_FUNCS0(CppParse_(ExceptionValue));
    public:
        CppParse_(ExceptionValue) (::iluptype::Exception);
    };
    
    typedef CppParse_FactoryT<CppParse_(ExceptionValue), ::iluptype::Exception>   CppParse_(ExceptionValueFactory);
    typedef CppParse_ListT<CppParse_(ExceptionValue), ::iluptype::Exception>      CppParse_(ExceptionValueList);
    typedef CppParse_ListIterT<CppParse_(ExceptionValue), ::iluptype::Exception>  CppParse_(ExceptionValueListIter);
    

    class CppParse_(Exception):
    public CppParse_CreatableT< ::iluptype::Exception>,
        public CppParseName(Scope),
        public virtual CppParseNarrow_Root {
        CppParseNarrow_FUNCS0(CppParse_(Exception));
        friend CppParse_FactoryT<CppParse_(Exception), ::iluptype::Exception>;
    public:
        const CppParse_(ExceptionValue) * value () const;
    protected:
        CppParse_(Exception) (::iluptype::Exception);
    private:
        const CppParse_(ExceptionValue) * _value;        
    };

    typedef CppParse_FactoryT<CppParse_(Exception), ::iluptype::Exception>   CppParse_(ExceptionFactory);
    typedef CppParse_ListT<CppParse_(Exception), ::iluptype::Exception>      CppParse_(ExceptionList);
    typedef CppParse_ListIterT<CppParse_(Exception), ::iluptype::Exception>  CppParse_(ExceptionListIter);


  class CppParse_(IDLException): public virtual CppParse_(TMP), public virtual CppParse_(Exception) {
      CppParseNarrow_FUNCS1(CppParse_(IDLException), CppParse_(Exception));
      friend CppParse_FactoryT<CppParse_(IDLException), ::iluptype::Exception>;
      public:
// TMP 8/29          const CppParse_(Record) * record () const { return CppParse_(Record)::narrow(value()->type()); };
          const CppParse_(Record) * record () const { return CppParse_(Record)::narrow(value()->type()->urType()); };  // TMP 8/29: NEW
      protected:
        CppParse_(IDLException) (::iluptype::Exception);
      public:  // TMP 5/30
          void * operator new (size_t, ::iluptype::Exception);  // TMP 5/30
  };

  typedef CppParse_FactoryT<CppParse_(IDLException), ::iluptype::Exception>   CppParse_(IDLExceptionFactory);
  typedef CppParse_ListT<CppParse_(IDLException), ::iluptype::Exception>      CppParse_(IDLExceptionList);
  typedef CppParse_ListIterT<CppParse_(IDLException), ::iluptype::Exception>  CppParse_(IDLExceptionListIter);


  class CppParse_(ReturnVal);
  class CppParse_(Procedure);  // TMP 7/3: NEW

  class CppParse_(Argument):
      public CppParse_CreatableT< ::iluptype::Argument>,
      public virtual CppParse_(NamedValue),
      public virtual CppParseNarrow_Root {
      CppParseNarrow_FUNCS0(CppParse_(Argument));
      friend CppParse_(Procedure);  // TMP 7/3
      friend CppParse_FactoryT<CppParse_(Argument), ::iluptype::Argument>;
  public:
      ILUCPP_BOOL sibling () const;
  private:  // Reintroduce CppParseName::Name::resetName, to give access to CppParse::Procedure  // TMP 7/3 NEW
      void resetName (const CppParseName(Scope) * definedIn) {  // TMP 7/3 NEW
          CppParseName(Name)::resetName(definedIn);  // TMP 7/3 NEW
      };  // TMP 7/3 NEW
  protected:
      CppParse_(Argument) (::iluptype::Argument);  // function arguments
  };

  typedef CppParse_FactoryT<CppParse_(Argument), ::iluptype::Argument>   CppParse_(ArgumentFactory);
  typedef CppParse_ListT<CppParse_(Argument), ::iluptype::Argument>      CppParse_(ArgumentList);
  typedef CppParse_ListIterT<CppParse_(Argument), ::iluptype::Argument>  CppParse_(ArgumentListIter);


  class CppParse_(ReturnVal):
  public CppParse_CreatableT< ::iluptype::Procedure>,
      public virtual CppParse_(NamedValue),
      public virtual CppParseNarrow_Root {
      CppParseNarrow_FUNCS0(CppParse_(ReturnVal));
      friend CppParse_(Procedure);  // TMP 7/3: NEW
      friend CppParse_FactoryT<CppParse_(ReturnVal), ::iluptype::Procedure>;
  private:  // Reintroduce CppParseName::Name::resetName, to give access to CppParse::Procedure  // TMP 7/3 NEW
      void resetName (const CppParseName(Scope) * definedIn) {  // TMP 7/3 NEW
          CppParseName(Name)::resetName(definedIn);  // TMP 7/3 NEW
      };  // TMP 7/3 NEW
  protected:
      CppParse_(ReturnVal) (::iluptype::Procedure);
  };

  typedef CppParse_FactoryT<CppParse_(ReturnVal), ::iluptype::Procedure>   CppParse_(ReturnValFactory);
  typedef CppParse_ListT<CppParse_(ReturnVal), ::iluptype::Procedure>      CppParse_(ReturnValList);
  typedef CppParse_ListIterT<CppParse_(ReturnVal), ::iluptype::Procedure>  CppParse_(ReturnValListIter);


  class CppParse_(Object);
  typedef CppParse_FactoryT<CppParse_(Object), ::iluptype::Type>   CppParse_(ObjectFactory);
  typedef CppParse_ListT<CppParse_(Object), ::iluptype::Type>      CppParse_(ObjectList);
  typedef CppParse_ListIterT<CppParse_(Object), ::iluptype::Type>  CppParse_(ObjectListIter);
  

  class CppParse_(Procedure):
    public CppParse_CreatableT< ::iluptype::Procedure>,
// TMP 7/3    public CppParseName(Name),
    public CppParseName(Scope),  // TMP 7/3 NEW
    public virtual CppParseNarrow_Root {
      CppParseNarrow_FUNCS0(CppParse_(Procedure));
      friend CppParse_FactoryT<CppParse_(Procedure), ::iluptype::Procedure>;
      public:
        const CppParse_(ArgumentList) & arguments () const;
        const CppParse_(ReturnVal) * returnVal () const;
        const CppParse_(ExceptionList) & exceptions () const;
        int protocolId () const;
		const CppParse_(Object) * object () const;
		ILUCPP_BOOL functional () const;
		ILUCPP_BOOL asynch () const;
      protected:
        CppParse_(Procedure) (::iluptype::Procedure);
      private:
        const CppParse_(ArgumentList) _arguments;
        const CppParse_(ReturnVal) * _returnVal;
        const CppParse_(ExceptionList) _exceptions;
        const CppParse_(Object) * _object;
    };

  typedef CppParse_FactoryT<CppParse_(Procedure), ::iluptype::Procedure>  CppParse_(ProcedureFactory);
  typedef CppParse_ListT<CppParse_(Procedure), ::iluptype::Procedure>     CppParse_(ProcedureList);
  typedef CppParse_ListIterT<CppParse_(Procedure), ::iluptype::Procedure> CppParse_(ProcedureListIter);

  class CppParse_(Object):
    public virtual CppParse_(Type),
    public CppParseName(Scope) {
// TMP 7/4      CppParseNarrow_FUNCS1(CppParse_(Object), CppParse_(Type));
      CppParseNarrow_FUNCS2(CppParse_(Object), CppParse_(Type), CppParseName(Scope));
      friend CppParse_FactoryT<CppParse_(Object), ::iluptype::Type>;
      public:
        const CppParse_(TypeList) & superclasses () const;
		  // List of Objects and/or Aliases
		const char * brand () const;
		const char * singleton () const;
		ILUCPP_BOOL collectible () const;
		ILUCPP_BOOL optional () const;
		const char * authentication () const;
		const char * docString () const;
        const CppParse_(ProcedureList) & procedures () const;
      public:  // Override methods from CppParse::Type
/*
        ILUCPP_BOOL isPtr (CppParseName(TypeUsage)) const;
          // TEMP: Although CORBA does not specify mapping for record members and array/sequence elements,
          // objects in those contexts *behave* like pointers; thus CppParseName::recordMember and
          // CppParseName::element return true.
*/
        CppParse_(Indirection) indirection (CppParseName(TypeUsage)) const;  // TMP 7/7: NEW
          // TEMP: Although CORBA does not specify mapping for record members and array/sequence elements,  // TMP 7/7: NEW
          // objects in those contexts *behave* like _ptr's; thus CppParseName::recordMember and  // TMP 7/7: NEW
          // CppParseName::element return CppParse::pseudoPtr.  // TMP 7/7: NEW
        ILUCPP_BOOL variableLength () const;
      private:  // Override methods from CppParseName::TypeName
        const CppParseName(AffixSet) * affixes () const;
        const CppParseName(AffixSet) * optionalAffixes () const;  // TMP 6/28: NEW
      protected:
        CppParse_(Object) (::iluptype::Type);
      private:
// TMP 9/7        const CppParse_(ObjectList) _superclasses;
        const CppParse_(TypeList) _superclasses;  // TMP 9/7: NEW
        const CppParse_(ProcedureList) _procedures;
      public:  // TMP 7/3: NEW
          void * operator new (size_t, ::iluptype::Type);  // TMP 7/3: NEW
  };

    
    class CppParse_(Import):  // TMP 6/12: NEW
        public CppParse_CreatableT< ::iluptype::Imported>,  // TMP 6/12: NEW
        public CppParseName(Name),  // TMP 6/12: NEW
        // TEMP: perhaps Import should derive from Interface, instead of CppParseName::Name ?  // TMP 6/12: NEW
        public CppParseNarrow_Root {  // TMP 6/12: NEW
            CppParseNarrow_FUNCS0(CppParse_(Import));  // TMP 6/12: NEW
            friend CppParse_FactoryT<CppParse_(Import), ::iluptype::Imported>;  // TMP 6/12: NEW
    public:  // TMP 6/12: NEW
        const char * filename () const;  // TMP 6/12: NEW
    protected:  // TMP 6/12: NEW
        CppParse_(Import) (::iluptype::Imported);  // TMP 6/12: NEW
    };  // TMP 6/12: NEW

  typedef CppParse_FactoryT<CppParse_(Import), ::iluptype::Imported>  CppParse_(ImportFactory);  // TMP 6/12: NEW
  typedef CppParse_ListT<CppParse_(Import), ::iluptype::Imported>     CppParse_(ImportList);  // TMP 6/12: NEW
  typedef CppParse_ListIterT<CppParse_(Import), ::iluptype::Imported> CppParse_(ImportListIter);  // TMP 6/12: NEW


  class CppParse_(Interface):
    public CppParse_CreatableT< ::iluptype::Interface>,
    public CppParseName(Scope) {
      CppParseNarrow_FUNCS0(CppParse_(Interface));
      friend CppParse_FactoryT<CppParse_(Interface), ::iluptype::Interface>;
      public:
		const char * brand () const;
        const CppParse_(ImportList) & imports () const;  // TMP 6/12:  NEW
            // All interfaces imported by this interface
        const CppParse_(TypeList) & types () const;
            // All types (except objects) defined in this interface.
        const CppParse_(ObjectList) & objects () const;
            // All objects defined in this interface.
        const CppParse_(ExceptionList) & exceptions () const;
            // All exceptions defined in this interface.
        const CppParse_(TypeList) & importedTypes () const;
            // All types (except objects) imported by this interface.
        const CppParse_(ObjectList) & importedObjects () const;
            // All objects imported by this interface.
        const CppParse_(ExceptionList) & importedExceptions () const;
            // All exceptions imported by this interface.
      protected:
        CppParse_(Interface) (::iluptype::Interface);
      public:  // TMP 8/13
        void * operator new (size_t, ::iluptype::Interface);  // TMP 8/13
      private:
        const CppParse_(ImportList) _imports;
        const CppParse_(TypeList) _types;
        const CppParse_(ObjectList) _objects;
        const CppParse_(ExceptionList) _exceptions;
        const CppParse_(TypeList) _importedTypes;
        const CppParse_(ObjectList) _importedObjects;
        const CppParse_(ExceptionList) _importedExceptions;
    };


  typedef CppParse_FactoryT<CppParse_(Interface), ::iluptype::Interface>  CppParse_(InterfaceFactory);
  typedef CppParse_ListT<CppParse_(Interface), ::iluptype::Interface>     CppParse_(InterfaceList);
  typedef CppParse_ListIterT<CppParse_(Interface), ::iluptype::Interface> CppParse_(InterfaceListIter);


  
  // TMP 4/21 We may eventually want to make this a method of Type or Interface
  NS_EXTERN
  const CppParse_(Interface) *
  CppParse_(InterfaceForType) (::iluptype::Type type);

    
END_NAMESPACE;


#ifdef _MSC_VER
    class CppParse_(_MSVCNote2Workaround) {
      private:
        CppParse_(_MSVCNote2Workaround) ();
        const CppParse_ListT<CppParse_(Interface), ::iluptype::Interface> interfaceList;
    };
#endif  // _MSC_VER


#endif  // _cppparse_hpp_


/*

    G++2.7.1 Compiler Note #1 - The compiler sometimes requires a leading "::" for names from namespace "iluptype" in order to recognize equivalent types. At other times, it prefers *not* to see a leading "::"

    G++2.7.1 Compiler Note #2 - "pragma interface" is part of the G++ mechanism for declaring that a template specialization is "external"; ie, it will be instantiated in another compilation unit which the current unit will link to.

    G++2.7.1 Compiler Note #3 - The compiler does not support the ANSI C++ (April, 1995 Working Draft; X3J16/95-0087) syntax for specifying a template class as a friend; thus, we make the _refCount member public so that CppParseFactoryT<T, FROM> can access it.

    SunPro4.1 Compiler Note #1 - The compiler does not support the ANSI C++ (April, 1995 Working Draft; X3J16/95-0087) syntax for specifying a template class as a friend; thus, we make the _refCount member public so that CppParseFactoryT<T, FROM> can access it.

    SunPro4.1 Compiler Note #2 - In cases involving virtual inheritance and virtual functions, the compiler will trash the base class' data members before the derived class' constructor completes. Thus, we add a "reInit" member function to re-initialize data members after construction. (Note that this also requires that any "const" data members be made non-const).

    SunPro4.1 Compiler Note #3 - SunPro defines wchar_t to be equivalant to long, which is ambiguous for overloading purposes.

    MSVC4.1 Compiler Note #1 - A more natural signature for createList (ie, "::iluptype::list createList (::iluptype::list)") was rejected with warning C4661 ("no suitable definition provided for explicit template instantiation request").

    MSVC4.1 Compiler Note #2 - The compiler apparently ignores "friend class T", issuing error C2248 ("'noteAllocated' : cannot access private member ..."); thus, we make "noteAllocated" public.

    MSVC4.1 Compiler Note #3 - The compiler does not support the ANSI C++ (April, 1995 Working Draft; X3J16/95-0087) syntax for specifying a template class as a friend; thus, we make the _refCount member public so that CppParseFactoryT<T, FROM> can access it.

    MSVC4.1 Compiler Note #4 - The compiler requires the *Factory, *List, and *ListIter typedefs to occur after the classes to which they apply (it would be convenient to define them before their associated classes so that they could be used in the class' friend declarations).

    TEMP: MSVC4.1 Compiler Note #99 - (Code provoking this error has been removed, but we want to keep the note) The compiler is confused by the use of a scoped name used to initialize a virtual base class in a derived class (errors C2512 ("'Type::Type" : no appropriate default constructor available") and C2614 ("illegal member initialization: 'Type' is not a base or member"). It also confused by the use of such a name when calling a static member function (error C2352: "... illegal call of nonstatic member function"). In such cases, an unscoped typedef'd equivalent can be used.

*/
