/*  Copyright (c) 1996 John E. Davis (davis@space.mit.edu)
 *  All rights reserved.
 */
#include "config.h"
#include "features.h"
#include "server.h"

#if SLRN_HAS_NNTP_SUPPORT
# include "clientlib.c"
# include "nntp.c"
#endif

#if SLRN_HAS_SPOOL_SUPPORT
# include "spool.c"
#endif

#include "misc.h"

int Slrn_Server_Id;
int Slrn_Post_Id;

Slrn_Server_Obj_Type *Slrn_Server_Obj;
Slrn_Post_Obj_Type *Slrn_Post_Obj;

#if SLRN_HAS_INEWS_SUPPORT
static FILE *Fp_Inews;
char *Slrn_Inews_Pgm;

static int inews_start_post (void)
{
   /* pipe a message to inews. Its done this way because inews expects the
    * article on stdin. Inews errors WILL mess up the screen.
    */
   /* !HACK! should we use slrn_popen() and slrn_pclose()?
    * They stop the screen getting messed up, but the error messages aren't
    * very appropriate
    */
   if (NULL == (Fp_Inews = popen (Slrn_Inews_Pgm, "w")))
     {
	slrn_error ("Couldn't open pipe to inews! -- Article not posted.");
	return -1;
     }
   return 0;
}

static int inews_end_post (void)
{
   int res=0;
   if (pclose(Fp_Inews)==-1)
     {
	slrn_error ("pclose() failed -- check if article was posted"); /* !HACK! can we do better? */
	res=-1;
     }
   Fp_Inews=NULL;
   return res;
}

static int inews_puts (char *s)
{
   char *p;
   /* write out buffer skipping \r characters, 'cos inews wants \n only */
   while (NULL != (p = slrn_strchr (s, '\r')))
     {
	fwrite( s, p-s, 1, Fp_Inews );
	s = p + 1;
     }
   /* and write out the rest */
   fputs (s, Fp_Inews );
   return 0;
}

static int inews_printf (char *fmt, ...)
{
   va_list ap;
   char buf[512];
   va_start (ap, fmt);
   vsprintf (buf, fmt, ap);
   return inews_puts( buf ); /* obvious tail-recursion for compiler to spot */
}


static Slrn_Post_Obj_Type Inews_Post_Obj;

static int inews_init_objects (void)
{
   Inews_Post_Obj.start = inews_start_post;
   Inews_Post_Obj.end = inews_end_post;
   Inews_Post_Obj.printf = inews_printf;
   Inews_Post_Obj.puts = inews_puts;
   Inews_Post_Obj.can_post = 1;
   Slrn_Inews_Pgm = slrn_make_startup_string (SLRN_INEWS_PROGRAM);
   return 0;
}

static int inews_select_post_object (void)
{
   Slrn_Post_Obj = &Inews_Post_Obj;
   return 0;
}

static void inews_usage (void)
{
   fputs ("--inews options:\n\
",
	  stdout);
   exit (0);
}

static int inews_parse_args (char **argv, int argc)
{
   int i;
   
   for (i = 0; i < argc; i++)
     {
	if (!strcmp (argv[i], "--help"))
	  inews_usage ();
	else break;
     }
   
   return i;
}

#endif				       /* HAS_INEWS_SUPPORT */


int slrn_init_objects (void)
{
#if SLRN_HAS_NNTP_SUPPORT
   if (-1 == nntp_init_objects ())
     return -1;
#endif
#if SLRN_HAS_SPOOL_SUPPORT
   if (-1 == spool_init_objects ())
     return -1;
#endif
#if SLRN_HAS_INEWS_SUPPORT
   if (-1 == inews_init_objects ())
     return -1;
#endif
   
   return 0;
}

#if !SLRN_HAS_INEWS_SUPPORT
# undef SLRN_FORCE_INEWS
# define SLRN_FORCE_INEWS 0
#endif

int slrn_select_post_object (int id)
{
   switch (id)
     {
#if SLRN_HAS_INEWS_SUPPORT
      case SLRN_POST_ID_INEWS:
	return inews_select_post_object ();
#endif
   
#if !SLRN_FORCE_INEWS
# if SLRN_HAS_NNTP_SUPPORT
      case SLRN_POST_ID_NNTP:
	return nntp_select_post_object ();
# endif
#endif
      default:
	slrn_error ("Object %d is not a supported posting agent.", id);
     }
   
   return -1;
}

int slrn_select_server_object (int id)
{
   switch (id)
     {
#if SLRN_HAS_NNTP_SUPPORT
      case SLRN_SERVER_ID_NNTP:
	return nntp_select_server_object ();
#endif
	
#if SLRN_HAS_SPOOL_SUPPORT
      case SLRN_SERVER_ID_SPOOL:
	return spool_select_server_object ();
#endif
   
      default:
	slrn_error ("server(%d) is not a supported server object.", id);
     }
   return -1;
}

int slrn_parse_object_args (char *name, char **argv, int argc)
{
   int num_parsed = -1;

   if (name == NULL) return -1;
   
#if SLRN_HAS_NNTP_SUPPORT
   if (!strcmp (name, "nntp"))
     {
	num_parsed = nntp_parse_args (argv, argc);
	if (Slrn_Server_Id == 0)
	  Slrn_Server_Id = SLRN_SERVER_ID_NNTP;
# if !SLRN_FORCE_INEWS
	if (Slrn_Post_Id == 0)
	  Slrn_Post_Id = SLRN_POST_ID_NNTP;
# endif
	return num_parsed;
     }
#endif
   
#if SLRN_HAS_SPOOL_SUPPORT
   if (!strcmp (name, "spool"))
     {
	num_parsed = spool_parse_args (argv, argc);
	Slrn_Server_Id = SLRN_SERVER_ID_SPOOL;
	return num_parsed;
     }
#endif
   
#if SLRN_HAS_INEWS_SUPPORT
   if (!strcmp (name, "inews"))
     {
	num_parsed = inews_parse_args (argv, argc);
	Slrn_Post_Id = SLRN_POST_ID_INEWS;
	return num_parsed;
     }
#endif

   return num_parsed;
}
