/* Copyright 1993-95 by Carl Harris, Jr. Copyright 1996 by Eric S. Raymond
 * All rights reserved.
 * For license terms, see the file COPYING in this directory.
 */

/***********************************************************************
  module:       poprc.c
  project:      popclient
  programmer:   Carl Harris, ceharris@mal.com
  description:  .poprc file functionality

 ***********************************************************************/

#include <config.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include "poprc_y.h"
#include "popclient.h"
#include "bzero.h"

extern int yydebug;
extern FILE *yyin;

int prc_errflag;

/* .poprc records are passed in this structure type */
struct prc_server {
  char *server;
  int protocol;
  char *remotename;
  char *password;
  char *remotefolder;
  char *userfolder;
  char *mda;
  int keep;
  int flush;
  int fetchall;
  int rewrite;
  struct prc_server *next;
};

/* Server parameters for current .poprc record */
static struct prc_server sp;

/* Head and tail of server list */
static struct prc_server *sp_head, *sp_tail;

/******************************************************************
  function:	prc_filecheck
  description:	Check that a configuration file is secure
  arguments:
    pathname	pathname for the configuration file

  ret. value:	error code.
  globals:	none
 *****************************************************************/

int prc_filecheck(pathname)
char *pathname;
{
  struct stat statbuf;

  /* the .poprc file must have the same uid as the REAL uid of this process, 
     it must have permissions no greater than 600, and it must not be a 
     symbolic link.  We check these conditions here. */

  if (lstat(pathname, &statbuf) < 0) {
    if (errno == ENOENT) 
      return(0);
    else {
      perror(pathname);
      return(PS_IOERR);
    }
  }

  if ((statbuf.st_mode & S_IFLNK) == S_IFLNK) {
    fprintf(stderr, "File %s must not be a symbolic link.\n", pathname);
    return(PS_AUTHFAIL);
  }

  if (statbuf.st_mode & ~(S_IFREG | S_IREAD | S_IWRITE)) {
    fprintf(stderr, "File %s must have no more than -rw------ permissions.\n", 
            pathname);
    return(PS_AUTHFAIL);
  }

  if (statbuf.st_uid != getuid()) {
    fprintf(stderr, "File %s must be owned by you.\n", pathname);
    return(PS_AUTHFAIL);
  }

  return(0);
}

/******************************************************************
  function:	prc_parse_file
  description:	Read the contents of the .poprc file, storing 
		each parsed record in a linked list.
  arguments:
    pathname	pathname for the .poprc file

  ret. value:	error code.
  globals:	writes sp_head, writes sp_tail, writes yyin,
		writes poprcfile, writes prc_errflag.
  calls:	prc_reset, yyparse.
 *****************************************************************/

prc_parse_file (pathname)
char *pathname;
{
  prc_errflag = 0;
  sp_head = sp_tail = (struct prc_server *) 0;
  prc_reset();

  /* Check that the file is secure */
  if ((prc_errflag = prc_filecheck(pathname)) != 0)
      return(prc_errflag);

  /* Open the poprc and feed it to the lexer. */
  if ((yyin = fopen(pathname,"r")) == (FILE *) 0) {
    perror(pathname);
    return(PS_IOERR);
  }

  yyparse();		/* parse entire file */

  fclose(yyin);

  if (prc_errflag) 
    return(PS_SYNTAX);
  else
    return(0);
}



/******************************************************************
  function:	prc_reset
  description:	clear the global sp record (server parameters)
		used by the parser.
  arguments:	none.
  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_reset()
{
  sp.server = (char *) 0;
  sp.protocol = DEF_PROTOCOL;
  sp.remotename = (char *) 0;
  sp.password = (char *) 0;
  sp.remotefolder = (char *) 0;
  sp.userfolder = (char *) 0;
  sp.mda = (char *) 0;
}



/******************************************************************
  function:	prc_register
  description:	register the parsed server params by appending
		them to a linked list of server param records.
  arguments:	none.
  ret. value:	none.
  globals:	reads sp.
  calls:	none.
 *****************************************************************/

prc_register()
{
  struct prc_server *node;

  /* initialize new node */
  node = (struct prc_server *) xmalloc(sizeof(struct prc_server));
  node->next = (struct prc_server *) 0;
  bcopy(&sp, node, sizeof(sp));
  
  /* append to end of list */
  if (sp_tail != (struct prc_server *) 0)
    /* list contains at least one element */
    sp_tail->next = node;
  else
    /* list is empty */
    sp_head = node;
  sp_tail = node;

}
 


/******************************************************************
  function:	prc_setserver
  description:	stub function to record parsed server name string
		into global sp record.
  arguments:	
    s		parsed server name

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setserver(s)
char *s;
{
  sp.server = s;
}


/******************************************************************
  function:	prc_setproto
  description:	stub function to record parsed protocol identifier
		into global sp record.
  arguments:	
    p		protocol identifier		

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setproto(p)
int p;
{
  sp.protocol = p;
}


/******************************************************************
  function:	prc_setremotename
  description:	stub function to record parsed username string
		into global sp record.
  arguments:	
    s		remotename string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_remotename(s)
char *s;
{
  sp.remotename = s;
}


/******************************************************************
  function:	prc_setpassword
  description:	stub function to record parsed password string
		into global sp record.
  arguments:	
    s		password string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setpassword(s)
char *s;
{
  sp.password = s;
}


/******************************************************************
  function:	prc_setremote
  description:	stub function to record parsed remote folder string
		into global sp record.
  arguments:	
    s		remote foldername string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setremote(s)
char *s;
{
  sp.remotefolder = s;

}


/******************************************************************
  function:	prc_setlocal
  description:	stub function to record parsed local folder string
		into global sp record.
  arguments:	
    s		local foldername string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setlocal(s)
char *s;
{
  sp.userfolder = s;
}



/******************************************************************
  function:	prc_setmda
  description:	stub function to record parsed local folder string
		into global sp record.
  arguments:	
    s		local mda string

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setmda(s)
char *s;
{
  sp.mda = s;
}



/******************************************************************
  function:	prc_setkeep
  description:	stub function to record keep option flag
		into global sp record.
  arguments:	
    n		keep option value

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setkeep(n)
int n;
{
  sp.keep = n;
}



/******************************************************************
  function:	prc_setflush
  description:	stub function to record flush option flag
		into global sp record.
  arguments:	
    n		flush option value

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setflush(n)
int n;
{
  sp.flush = n;
}



/******************************************************************
  function:	prc_setfetchall
  description:	stub function to record fetchall option flag
		into global sp record.
  arguments:	
    n		fetchall option value

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setfetchall(n)
int n;
{
  sp.fetchall = n;
}



/******************************************************************
  function:	prc_setrewrite
  description:	stub function to record rewrite option flag
		into global sp record.
  arguments:	
    n		rewrite option value

  ret. value:	none.
  globals:	writes sp.
  calls:	none.
 *****************************************************************/

prc_setrewrite(n)
int n;
{
  sp.rewrite = n;
}



/******************************************************************
  function:	prc_mergeoptions
  description:	merge command-line options, default options, and
		.poprc options into a single options record.
  arguments:
    servername	name of the server to search in the .poprc
    cmd_opts 	command-line options.
    def_opts 	default options.
    mrg_opts	record to receive the merged options.

  ret. value:	none.
  globals:	reads sp_head.
  calls:	none.
 *****************************************************************/

int prc_mergeoptions (servername, cmd_opts, def_opts, mrg_opts)
char *servername;
struct hostrec *cmd_opts, *def_opts, *mrg_opts;
{
  struct prc_server *sp_node;

  /* copy those options which aren't subject to merging */
  bcopy(cmd_opts, mrg_opts, sizeof(*mrg_opts));

  if (!cmd_opts->localname[0])
      strcpy(mrg_opts->localname, def_opts->localname);
  if (!cmd_opts->remotename[0])
      strcpy(mrg_opts->remotename, def_opts->localname);
  if (!cmd_opts->userfolder[0])
      strcpy(mrg_opts->userfolder, def_opts->userfolder);
  if (!cmd_opts->mda[0])
      strcpy(mrg_opts->mda, def_opts->mda);
  if (!cmd_opts->protocol)
      mrg_opts->protocol = DEF_PROTOCOL;
  if (!cmd_opts->output)
      mrg_opts->output = def_opts->output;

  for (sp_node = sp_head; sp_node != NULL; sp_node = sp_node->next)
    if (strcmp(sp_node->server, servername) == 0
	|| strcmp(sp_node->server, "defaults") == 0)
    {
#define STR_OPTION_MERGE(fld, len)	\
				if (*(cmd_opts->fld) != '\0') \
				  strcpy(mrg_opts->fld, cmd_opts->fld); \
				else if (sp_node->fld != (char *) 0) \
				   strncpy(mrg_opts->fld,sp_node->fld,len-1);

      /* Determine protocol to use */
      if (cmd_opts->protocol != 0) 
	mrg_opts->protocol = cmd_opts->protocol; 
      else if (sp_node->protocol != 0)
	mrg_opts->protocol = sp_node->protocol;

      STR_OPTION_MERGE(remotename, USERNAMELEN);
      STR_OPTION_MERGE(password, PASSWORDLEN);
      STR_OPTION_MERGE(userfolder, FOLDERLEN);
      STR_OPTION_MERGE(remotefolder, FOLDERLEN);
      STR_OPTION_MERGE(mda, MDALEN);
#undef STR_OPTION_MERGE

      /* merge in options */
#define FLAG_OPTION_MERGE(fld)	\
				if (!mrg_opts->fld && sp_node->fld) \
				  mrg_opts->fld = 1;
      FLAG_OPTION_MERGE(keep);
      FLAG_OPTION_MERGE(flush);
      FLAG_OPTION_MERGE(fetchall);
      FLAG_OPTION_MERGE(rewrite);
#undef FLAG_OPTION_MERGE

      if (cmd_opts->output != 0)
	mrg_opts->output = cmd_opts->output;
      else if (sp_node->mda)
	mrg_opts->output = TO_MDA;
      else if (sp_node->userfolder)
	mrg_opts->output = TO_FOLDER;
    }

  if (mrg_opts->output == TO_FOLDER && strcmp(mrg_opts->userfolder,"-") == 0)
    mrg_opts->output = TO_STDOUT;
  if (mrg_opts->password[0] == '\0' && !versioninfo)
  {
    char prompt[BUFSIZ];

    (void) sprintf(prompt, "Enter password for host %s: ", servername);
    strncpy(mrg_opts->password, (char *) getpassword(prompt), PASSWORDLEN-1);
  }
}

/******************************************************************
  function:	apopend_server_names
  description:	append all server names from the .poprc file to arg vector
  arguments:
     pargc      address of arg count
     argv       argument vector to append to.

  ret. value:	none.
  globals:	reads sp_head.
  calls:	none.
 *****************************************************************/

void append_server_names(int *pargc, char **argv)
{
  struct prc_server *sp_node;

  for (sp_node = sp_head; sp_node != NULL; sp_node = sp_node->next) {
    if (strcmp(sp_node->server, "defaults"))
      argv[(*pargc)++] = sp_node->server;
  }
     
}
