#include <sys/types.h>
#include <stdio.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <termios.h>
#include <unistd.h>
#include <linux/ax25.h>

#include "config.h"
#include "call.h"

static int debug_setting  = FALSE;
static int window_setting = 0;
static int port_setting   = 0;

int interrupted = FALSE;
int fd;

void convert_cr_lf(char *buf, int len)
{
	while (len--)
	{
		if (*buf == '\r')
			*buf = '\n';
		buf++;
	}
}

void convert_lf_cr(char *buf, int len)
{
	while (len--)
	{
		if (*buf == '\n')
			*buf = '\r';
		buf++;
	}
}

static int connect_to(char *address[])
{
	int fd;
  	int addrlen;
  	struct full_sockaddr_ax25 addr;
  	int one = debug_setting;

  	if ((fd = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0)
  	{
  		perror("socket");
		return(-1);
	}
	  
  	if (debug_setting && setsockopt(fd, SOL_SOCKET, SO_DEBUG, &one, sizeof(one)) == -1)
  	{
  		perror("SO_DEBUG");
  		close(fd);
  		return(-1);
  	}
  	
  	if (window_setting == 0)
  		window_setting = config_get_window(port_setting);
  		
  	if (setsockopt(fd, SOL_AX25, AX25_WINDOW, &window_setting, sizeof(window_setting)) == -1)
	{
 		perror("AX25_WINDOW");
  		close(fd);
 		return(-1);
	}

  	addrlen = convert_call(config_get_addr(port_setting), &addr);
    
  	if (bind(fd, (struct sockaddr *)&addr, addrlen) == -1)
  	{
  		perror("bind");
  		close(fd);
  		return(-1);
  	}

  	if ((addrlen = convert_call_arglist(address, &addr)) == -1)
  	{
  		close(fd);
  		return(-1);
  	}

	printf("Trying...\r");
	fflush(stdout);
  
  	if (connect(fd, (struct sockaddr *)&addr, addrlen))
  	{
  		printf("\n");
		perror("connect");
		close(fd);
		return(-1);
	}
  
  	printf("***Connected to %s\n", address[0]);

  	return(fd);
}

void cmd_intr(int sig)
{
	signal(SIGQUIT, cmd_intr);
	interrupted = TRUE;
}

void cmd_call(char *call[])
{
	fd_set sock_read;
	fd_set sock_write;
	char buf[512];
	int bytes;
	int logfile    = -1;
	int uploadfile = -1;
 	long uplsize=0;
 	long uplpos=0;
 	char uplbuf[128];	/* Upload buffer */
 	int upldp=0;
 	int upllen=0;
 	char *c, *t;

	if ((fd = connect_to(call)) == -1)
		return;

	interrupted = FALSE;
	signal(SIGQUIT, cmd_intr);
	signal(SIGINT, SIG_IGN);
	signal(SIGTSTP, SIG_IGN);
	
	fcntl(fd, F_SETFL, O_NDELAY);
	fcntl(STDIN_FILENO, F_SETFL, O_NDELAY);
	
	while (TRUE)
	{
		FD_ZERO(&sock_read);
		FD_SET(STDIN_FILENO, &sock_read);
		FD_SET(fd, &sock_read);
		FD_ZERO(&sock_write);
		
		if (uploadfile != -1)
			FD_SET(fd, &sock_write);
		
		if (select(fd + 1, &sock_read, &sock_write, NULL, NULL) == -1)
		{
			if (!interrupted && errno == EAGAIN)
				continue;
			if (!interrupted)
				perror("select");
			break;
		}
		
		while ((bytes = read(fd, buf, 511)) > 0)
		{
			convert_cr_lf(buf, bytes);
			
			if (logfile != -1)
			{
				if (write(logfile, buf, bytes) != bytes)
				{
					close(logfile);
					logfile = -1;
					printf("\nError while writing log. Log closed.\n");
				}
			}

			write(STDOUT_FILENO, buf, bytes);
		}
		
		if (bytes == -1 && errno != EWOULDBLOCK && errno != EAGAIN)
		{
			if (errno != ENOTCONN)
				perror("read");
			break;
		}
		
		bytes = read(STDIN_FILENO, buf, 511);
		
		if (bytes > 1 && *buf == '~')
		{
			buf[bytes] = 0;
			
			switch (buf[1])
			{
				case '.':
					bytes = 0;
					interrupted = TRUE;
					break;
				case '!':
					if (buf[2] != '\0' && buf[2] != '\n')
					{
						c = buf + 2;
						if ((t = strchr(c, '\n')) != NULL)
							*t = '\0';
					}
					else
					{
						if ((c = getenv("SHELL")) == NULL)
							c = "/bin/sh";
					}
					
					fcntl(STDIN_FILENO, F_SETFL, 0);
					printf("\n[Spawning subshell]\n");
					system(c);
					printf("\n[Returned to connect]\n");
					fcntl(STDIN_FILENO, F_SETFL, O_NDELAY);
					continue;
				case 'z':
				case 'Z':
				case 'Z' - 64:
					fcntl(STDIN_FILENO, F_SETFL, 0);
					kill(getpid(), SIGSTOP);
					printf("\n[Resumed]\n");
					fcntl(STDIN_FILENO, F_SETFL, O_NDELAY);
					continue;
				case '?':
				case 'h':
				case 'H':
					printf("\nTilde escapes:\n.  close\n!  shell\nZ  suspend\ns Stop upload\no  Open log\nc  Close log\nu  Upload\nyd  YAPP Download\nyu  YAPP Upload\n");
					continue;
				case 'S':
				case 's':
					if (uploadfile != -1)
					{
						printf("\n[Upload file closed]\n");
						close(uploadfile);
						uploadfile = -1;
					}
					else
					{
						printf("\n[No upload in progress]\n");
					}
					continue;
				case 'u':
				case 'U':
					if (uploadfile != -1)
					{
						printf("\n[Already uploading]\n");
						continue;
					}
					
					if ((t = strchr(buf, '\n')) != NULL)
						*t = '\0';
					t = buf + 2;
					while (*t != '\0' && isspace(*t))
						t++;
					if (*t == '\0')
					{
						printf("\n[Upload requires a filename - eg ~u hello.txt]\n");
						continue;
					}
					uploadfile = open(t, O_RDONLY);
					if (uploadfile == -1)
					{
						printf("\n[Unable to open upload file]\n");
						continue;
					}
					if (lseek(uploadfile, 0L, SEEK_END) !=-1)
						uplsize = lseek(uploadfile, 0L, SEEK_CUR);
					else
						uplsize = 0;
					lseek(uploadfile, 0L, SEEK_SET);
					uplpos  = 0;
					upldp   = -1;
					upllen  = 0;
					if (uplsize != -1)
						printf("\n[Uploading %ld bytes from %s]\n", uplsize, t);
					else
						printf("\n[Uploading from %s]\n", t);
					continue;
				case 'O':
				case 'o':
					if ((t=strchr(buf, '\n')) != NULL)
						*t = '\0';
					if (logfile != -1)
					{
						close(logfile);
						logfile = -1;
					}	
					t = buf + 2;
					while (*t != '\0' && isspace(*t))
						t++;
					if (*t == '\0')
						t = "logfile.txt";
					if ((logfile = open(t, O_RDWR | O_APPEND | O_CREAT, 0666)) == -1)
						printf("\n[Unable to open %s]\n", buf + 2);
					continue;
				case 'C':
				case 'c':
					if (logfile != -1)
					{
						close(logfile);
						logfile = -1;
					}
					else
					{
						printf("\n[Log file not open]\n");
					}
					continue;
				case 'Y':
				case 'y':
					cmd_yapp(buf + 2, bytes - 2);
					continue;
				case '~':
					bytes--;
					memmove(buf, buf + 1, strlen(buf));
					break;
				default:
					printf("\nUnknown '~' escape. Type ~h for a list.\n");
					continue;
			}
		}
					
		if (bytes == 0 || (bytes == -1 && errno != EWOULDBLOCK && errno != EAGAIN))
		{
			if (!interrupted)
				perror("input");
			break;
		}
		
		if (bytes > 0)
		{
			if (uploadfile != -1)
			{
				printf("\n[Ignored. Type ~s to stop upload]\n");
				continue;
			}

			convert_lf_cr(buf, bytes);
			
			if (write(fd, buf, bytes) == -1)
			{
				perror("write");
				break;	
			}
		}

		if (uploadfile != -1)
		{
			if (uplsize == 0)
			{
				close(uploadfile);
				uploadfile = -1;
				printf("\n[Upload complete: 0 bytes\n");
				continue;
			}
			
			if (upldp == -1)
			{
				upllen = read(uploadfile, uplbuf, 128);
				
				if (upllen == 0)
				{
					close(uploadfile);
					uploadfile = -1;
					printf("\n[Upload complete: %ld bytes]\n", uplpos);
					continue;
				}
				
				if (upllen == -1)
				{
					close(uploadfile);
					uploadfile = -1;
					printf("\n[Error reading upload file: upload aborted at %ld bytes]\n", uplpos);
					continue;
				}

				convert_lf_cr(uplbuf, upllen);

				upldp = 0;
			}
			
			bytes = write(fd, uplbuf + upldp, upllen - upldp);

			if ((bytes == 0 || bytes == -1) && errno != EWOULDBLOCK && errno != EAGAIN)
			{
				printf("\n[Write error during upload. Connection lost]\n");
				perror("write");
				break;
			}
			
/*			if (uplpos / 1024 != (uplpos + bytes) / 1024)
			{ */
				printf("\r%ld bytes sent    ", uplpos + bytes);
/*			} */

			uplpos += bytes;
			upldp  += bytes;

			if (upldp >= upllen)
				upldp = -1;
		}	
	}

	close(fd);

	if (logfile != -1)
	{
		close(logfile);
		logfile = -1;
	}
	
	fcntl(STDIN_FILENO, F_SETFL, 0);
	
	signal(SIGQUIT, SIG_IGN);
	signal(SIGINT, SIG_DFL);
	
	printf("*** Cleared\n");
}
	
	
int main(int argc, char **argv)
{
	int p = 1;
	int n;
	
	if (config_load_ports() == -1)
	{
		fprintf(stderr, "No AX.25 port data configured.\n");
		return(1);
	}

	if (argv[p] != NULL && strcmp(argv[p], "-p") == 0)
	{
		p++;

		if (argv[p] == NULL || (n = atoi(argv[p])) == 0)
		{
			fprintf(stderr, "%s: option '-p' requires numeric argument.\n", argv[0]);
			return(1);
		}
		
		if (n < 1 || n > config_num_ports())
		{
			fprintf(stderr, "%s: port must be between 1 and %d.\n", argv[0], config_num_ports());
			return(1);
		}

		p++;

		port_setting = n - 1;
	}

	if (argv[p] != NULL && strcmp(argv[p], "-w") == 0)
	{
		p++;
		
		if (argv[p] == NULL || (n = atoi(argv[p])) == 0)
		{
			fprintf(stderr, "%s: option '-w' requires numeric argument.\n", argv[0]);
			return(1);
		}
		
		if (n < 1 || n > 7)
		{
			fprintf(stderr, "%s: window must be between 1 and 7 frames.\n", argv[0]);
			return(1);
		}

		p++;

		window_setting = n;
	}
		
	if (argv[p] != NULL && strcmp(argv[p], "-d") == 0)
	{
		debug_setting = 1;
		p++;
	}

	if (argc < p + 1)
	{
		fprintf(stderr, "%s [-p port] [-w window] [-d] callsign [[via] digpieaters...]\n", argv[0]);
		return(1);
	}
	
	printf("GW4PTS AX.25 Connect v1.08\n");
	
	cmd_call(argv + p);

	return(0);
}
