#include "wily.h"
#include "data.h"
#include <dirent.h>

typedef struct Descriptor Descriptor;
struct Descriptor {
	Path			path;
	enum { Failure, Newfile, Oldfile, Directory }	status;
	struct stat	buf;
	int			fd;
	DIR*			dirp;
};

static Descriptor	getdesc(char*path);
static void		data_load(Data *d, Descriptor desc);

/* Set 'd's tag appropriately */
static void
data_settag(Data*d)
{
	Path	buf;
	
	sprintf (buf, "%s %s %s ",
			d->label, 
			d->names? dirtools : filetools,
			tag_match(d->path));

	tag_set(d->tag, buf);
}

static Data *
data_alloc(void)
{
	Data	*d;
	static Id id;
	
	d = NEW(Data);
	d->id = id++;
	d->next = dataroot;
	dataroot = d;
	d->t = text_alloc(d, true);
	d->tag = text_alloc(d, false);
	d->names = 0;
	d->backupto = 0;
	d->fd = 0;
	d->emask = 0;
	
	return d;
}

/*
 * Open a new 'data' to represent 'path'.  Either read 'path', or if
 * it doesn't exist and 'create' is set, return an empty window to
 * represent where we _will_ write it.
 *
 * Return (View*)0 if 'path' exists but we can't read it, or it doesn't
 * exist and we don't want to create it.
 */
View*
data_open(char*path, Bool create)
{
	Descriptor desc;
	Data	*d;
	
	desc = getdesc(path);
	if (desc.status == Failure || (desc.status == Newfile && !create) )
		return 0;

	d = data_alloc();

	data_load(d, desc);
	data_settag(d);
	
	win_new(path, d->tag, d->t);
	return text_view(d->t);
}

/*
 * Read file or directory 'path' into 'd'. 'path' might not exist yet.
 * Return 0 for success.
 */
int
data_get(Data *d, char *path)
{
	Path	p;
	
	if(data_backup(d))
		return -1;		/* If we did this, we'd lose data */

	pathexpand(path?path:d->label, 0, p);
	data_load(d, getdesc(p));
	tag_setlabel(d->tag, d->label);
	return 0;
}

/* Load the file or directory represented by 'desc' into 'd'
 */
static void
data_load(Data *d, Descriptor desc)
{
	extern Bool	utfHadNulls;
	char	*backup;

	pathcontract(d->label, desc.path);
	strcpy(d->path, desc.path);
	
	cursorswitch(&boxcursor);
	bflush();
	
	dirnames_free(d->names);
	d->names = 0;
	undo_reset(d->t);

	/* Read in some data */
	switch(desc.status) {
	case Newfile:	break;
	case Oldfile: text_read(d->t, desc.fd, desc.buf.st_size); break;
	case Directory:
		tag_addtool(d->tag, "Get");
		d->names = dirnames(desc.dirp, d->path);
		text_formatdir(d->t, d->names);
		break;
	case Failure:
		text_replaceutf(d->t, range(0,0), "COULD NOT LOAD");
		break;
	}

	tag_rmtool(d->tag, "Put");
	
	/* What should we backup to now? */
	switch(desc.status) {
	case Newfile:	backup = d->path; break;
	case Oldfile:
		if (utfHadNulls) {
			diag(d->path, "removed nulls from %s", d->path);
			backup = 0;
		} else {
			backup = d->path;
		}
		break;
	default:	backup = 0; break;
	}
	data_setbackup(d, backup);
	
	undo_start(d->t);
	undo_mark(d->t);
	cursorswitch(cursor);
}

static Descriptor
getdesc(char*path)
{
	Descriptor 	d;

	assert(path && strlen(path));

	strcpy(d.path, path);
	
	if(stat(path,&d.buf)) {				/* doesn't exist -- yet */
		d.status = Newfile;
	} else if ( S_ISDIR(d.buf.st_mode) ){
		add_slash(d.path);
		d.dirp = opendir(path);
		if(d.dirp) {
			d.status = Directory;
		} else {
			d.status = Failure;
			diag(path, "couldn't opendir %s", path);
		}
	} else {
		d.fd = open(path, OREAD);
		if(d.fd>=0) {
			d.status = Oldfile;
		} else {
			d.status = Failure;
			diag(path, "couldn't open %s", path);
		}
	}
	return d;
}

