/*----------------------------------------------------------------------
 * Modified for Linux-PAM by Al Longyear <longyear@netcom.com> 96/5/5
 *----------------------------------------------------------------------
 * Copyright (c) 1983, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <endian.h>

#include <sys/types.h>
#include <sys/uio.h>
#include <string.h>
#define __USE_BSD
#include <unistd.h>
#include <stdlib.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <arpa/inet.h>

#define __USE_BSD
#ifndef MAXDNAME
#define MAXDNAME  256
#endif

#include <stdarg.h>
#include <ctype.h>

#include <net/if.h>
#include <linux/sockios.h>

#include <paths.h>
#include <pwd.h>
#include <sys/file.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <sys/syslog.h>

#define PAM_SM_AUTH  /* only defines this management group */

#ifndef LINUX    /* AGM added this as of 0.2 */
#include <security/pam_appl.h>
#endif           /* ditto */
#define _PAM_EXTERN_FUNCTIONS
#include <security/pam_modules.h>

static int _pam_auth_rhosts (pam_handle_t *pamh, 
			     int flags, int argc, 
			     const char **argv);

static int _pam_auth_rhosts (pam_handle_t *pamh,
			     int flags, 
			     int argc,
			     const char **argv);

int pam_get_rhost(pam_handle_t *pamh, const char **rhost, const char *prompt);
int pam_get_rhost(pam_handle_t *pamh, const char **rhost, const char *prompt)
{
    const char *use_prompt;
    int retval;
    struct pam_message msg, *pmsg;
    struct pam_response *resp;
    struct pam_conv *conv;
    const char   *current;

    retval = pam_get_item (pamh, PAM_RHOST, (const void **)&current);
    if (retval != PAM_SUCCESS)
        return retval;

    if (current == NULL) {
        use_prompt = prompt;
	if (use_prompt == NULL) {
	    use_prompt = "RemoteHost name: ";
	}

	/* converse with application -- prompt user for a username */
	
	pmsg          = &msg;
	msg.msg_style = PAM_PROMPT_ECHO_ON;
	msg.msg       = use_prompt;
	resp          = NULL;

	retval = pam_get_item (pamh, PAM_CONV, (const void **)&conv);
	if (retval == PAM_SUCCESS) {
	    retval = conv->conv (1,
				 (const struct pam_message **) &pmsg,
				 &resp,
				 conv->appdata_ptr);
	}

	if (retval == PAM_SUCCESS) {
	    current = resp->resp;
	    free (resp);
	}
    }
    *rhost = current;

    return retval;        /* pass on any error from conversation */
}

int pam_get_ruser(pam_handle_t *pamh, const char **ruser, const char *prompt);
int pam_get_ruser(pam_handle_t *pamh, const char **ruser, const char *prompt)
{
    const char *use_prompt;
    int retval;
    struct pam_message msg, *pmsg;
    struct pam_response *resp;
    struct pam_conv *conv;
    const char   *current;

    retval = pam_get_item (pamh, PAM_RUSER, (const void **)&current);
    if (retval != PAM_SUCCESS)
        return retval;

    if (current == NULL) {
        use_prompt = prompt;
	if (use_prompt == NULL) {
	    use_prompt = "RemoteUser name: ";
	}

	/* converse with application -- prompt user for a username */
	
	pmsg          = &msg;
	msg.msg_style = PAM_PROMPT_ECHO_ON;
	msg.msg       = use_prompt;
	resp          = NULL;

	retval = pam_get_item (pamh, PAM_CONV, (const void **)&conv);
	if (retval == PAM_SUCCESS) {
	    retval = conv->conv (1,
				 (const struct pam_message **) &pmsg,
				 &resp,
				 conv->appdata_ptr);
	}

	if (retval == PAM_SUCCESS) {
	    current = resp->resp;
	    free (resp);
	}
    }
    *ruser = current;

    return retval;        /* pass on any error from conversation */
}

/*
 * Options for this module
 */

struct _options {
    int  opt_no_hosts_equiv;
    int  opt_no_rhosts;
    int  opt_debug;
    int  opt_nowarn;
    int  opt_disallow_null_authtok;
    int  opt_silent;
    const char *last_error;
};

/*
 * Returns 0 if match, 1 if no match.
 */

static int
__icheckhost (struct _options *opts, u_long raddr, register char *lhost)
{
    struct hostent *hp;
    u_long laddr;
    char **pp;
    
    /* Try for raw ip address first. */
    if (isdigit(*lhost) && (long)(laddr = inet_addr(lhost)) != -1)
	return (!! (raddr ^ laddr));

    /* Better be a hostname. */
    hp = gethostbyname(lhost);
    if (hp == NULL)
	return (1);
    
    /* Spin through ip addresses. */
    for (pp = hp->h_addr_list; *pp; ++pp)
	if (!memcmp (&raddr, *pp, sizeof (unsigned long)))
	    return (0);

    /* No match. */
    return (1);
}

/*
 * Returns 0 if match, 1 if not ok.
 */

static int
__ivaliduser (struct _options *opts, FILE *hostf, u_long raddr,
	      const char *luser, const char *ruser)
{
    register const char *user;
    register char *p;
    int ch;
    char buf[MAXHOSTNAMELEN + 128];		/* host + login */

    while (fgets(buf, sizeof (buf), hostf)) {
        p = buf;
	/* Skip lines that are too long. */
	if (strchr(p, '\n') == (char *) 0) {
	    ch = getc(hostf);
	    while (ch != '\n' && ch != EOF)
		ch = getc(hostf);

	    continue;
	}

	while (*p != '\n' && *p != ' ' && *p != '\t' && *p != '\0') {
	    if ((unsigned int) (unsigned short) (*(unsigned char *) p) < 256)
	        *p = tolower(*p);
	    p++;
	}

	if (*p == ' ' || *p == '\t') {
	    *p++ = '\0';
	    while (*p == ' ' || *p == '\t')
	        p++;

	    user = p;
	    while (*p != '\n' && *p != ' ' && *p != '\t' && *p != '\0')
	        p++;
	} else 
	    user = p;

	*p = '\0';

	if (__icheckhost (opts, raddr, buf) == 0) {
	    if (! (*user))
	        user = luser;
	    if (strcmp(ruser, user) == 0)
	        return (0);
	}
    }

  return (1);
}

/*
 * New .rhosts strategy: We are passed an ip address. We spin through
 * hosts.equiv and .rhosts looking for a match. When the .rhosts only
 * has ip addresses, we don't have to trust a nameserver.  When it
 * contains hostnames, we spin through the list of addresses the nameserver
 * gives us and look for a match.
 *
 * Returns 0 if ok, -1 if not ok.
 */

static int
iruserok(struct _options *opts, u_long raddr, int superuser,
	 const char *ruser, const char *luser)
{
    const char *cp;
    struct stat sbuf;
    struct passwd *pwd;
    FILE *hostf;
    uid_t uid;
    int answer;
    char pbuf[MAXPATHLEN];

    if (superuser || ! opts->opt_no_hosts_equiv) {
	hostf = fopen (_PATH_HEQUIV, "r");
	if (hostf) {
	    answer = __ivaliduser(opts, hostf, raddr, luser, ruser);
	    (void) fclose(hostf);
	    if (answer == 0)
		return(0);
	}
    }
    
    if (opts->opt_no_rhosts || superuser)
	return (1);

    pwd = getpwnam(luser);
    if (pwd == NULL)
	return(1);

    (void)strcpy(pbuf, pwd->pw_dir);
    (void)strcat(pbuf, "/.rhosts");

    /*
     * Change effective uid while opening .rhosts.  If root and
     * reading an NFS mounted file system, can't read files that
     * are protected read/write owner only.
     */

    uid = geteuid();
    (void)seteuid(pwd->pw_uid);
    hostf = fopen(pbuf, "r");
    (void)seteuid(uid);
	
    if (hostf == NULL)
	return(1);

    /*
     * If not a regular file, or is owned by someone other than
     * user or root or if writeable by anyone but the owner, quit.
     */

    cp = NULL;
    if (lstat(pbuf, &sbuf) < 0 || !S_ISREG(sbuf.st_mode))
	cp = ".rhosts not regular file";

    if (fstat(fileno(hostf), &sbuf) < 0)
	cp = ".rhosts fstat failed";
    else
	if (sbuf.st_uid && sbuf.st_uid != pwd->pw_uid)
	    cp = "bad .rhosts owner";
	else
	    if (sbuf.st_mode & (S_IWGRP|S_IWOTH))
		cp = ".rhosts writeable by other than owner";

    /* If there were any problems, quit. */
    if (cp) {
	opts->last_error = cp;
	fclose(hostf);
	return(1);
    }

    answer = __ivaliduser (opts, hostf, raddr, luser, ruser);
    (void) fclose(hostf);
    return (answer);
}

static int
pam_ruserok (struct _options *opts, const char *rhost, int superuser,
	     const char *ruser, const char *luser)
{
    struct hostent *hp;
    int answer = 1;
    u_long addr;
    char **ap;

    opts->last_error = (char *) 0;
    hp               = gethostbyname(rhost);

    if (hp != NULL) {
	ap = hp->h_addr_list;
	while (*ap) {
	    memcpy (&addr, *ap, sizeof(addr));
	    if (iruserok(opts, addr, superuser, ruser, luser) == 0) {
		answer = 0;
		break;
	    }
	    ++ap;
	}
    }

    return answer;
}

static void set_option (struct _options *opts, const char *arg)
{
    do {
	if (strcmp (arg, "no_hosts_equiv") == 0) {
	    opts->opt_no_hosts_equiv = 1;
	    break;
	}

	if (strcmp (arg, "no_rhosts") == 0) {
	    opts->opt_no_rhosts = 1;
	    break;
	}

	if (strcmp (arg, "debug") == 0) {
	    opts->opt_debug = 1;
	    break;
	}

	if (strcmp (arg, "nowarn") == 0) {
	    opts->opt_nowarn = 1;
	    break;
	}
/*
 * All other options are ignored for the present time.
 */
	break;
    } while (0);
}

static void set_parameters (struct _options *opts, int flags,
			    int argc, const char **argv)
{
    opts->opt_silent                = flags & PAM_SILENT;
    opts->opt_disallow_null_authtok = flags & PAM_DISALLOW_NULL_AUTHTOK;

    while (argc-- > 0) {
	set_option (opts, *argv);
	++argv;
    }
}

static int _pam_auth_rhosts (pam_handle_t *pamh,
			     int flags, 
			     int argc,
			     const char **argv) 
{
    int retval;
    char *luser;
    const char *ruser,*rhost;
    struct _options opts;
/*
 * Look at the options and set the flags accordingly.
 */
    memset (&opts, 0, sizeof (opts));
    set_parameters (&opts, flags, argc, argv);
/*
 * Obtain the parameters for the various items
 */
    do {
	/* get the remotehost */
	retval = pam_get_rhost(pamh, &rhost, NULL);
	(void) pam_set_item(pamh, PAM_RHOST, rhost);
	if (retval != PAM_SUCCESS)
	    break;

	/* get the remote user */
	retval = pam_get_ruser(pamh, &ruser, NULL);
	(void) pam_set_item(pamh, PAM_RUSER, ruser);
	if (retval != PAM_SUCCESS)
	    break;

	/* get the local user */
	retval = pam_get_user(pamh, &luser, NULL);
	(void) pam_set_item(pamh, PAM_USER, luser);
	if (retval != PAM_SUCCESS)
	    break;
/*
 * Validate the account information.
 */
	if (pam_ruserok (&opts, rhost, 0, ruser, luser) != 0)
	    retval = PAM_AUTH_ERR;
      } while (0);

    return (retval);
}

/* --- authentication management functions --- */

PAM_EXTERN
int pam_sm_authenticate (pam_handle_t *pamh, 
			 int flags,
			 int argc, 
			 const char **argv)
{
    return _pam_auth_rhosts (pamh, flags, argc, argv);
}

PAM_EXTERN
int pam_sm_setcred(pam_handle_t *pamh,int flags,int argc,
		   const char **argv)
{
    return PAM_SUCCESS;
}

/* end of module definition */


#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_rhosts_auth_modstruct = {
    "pam_rhosts_auth",
    pam_sm_authenticate,
    pam_sm_setcred,
    NULL,
    NULL,
    NULL,
    NULL,
};

#endif
