------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                SYSTEM.COMPILER_EXCEPTIONS.MACHINE_SPECIFICS              --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.4 $                             --
--                         (SunOS, Solaris Version)                         --
--                                                                          --
--   Copyright (C) 1991,1992,1993,1994,1995 Free Software Foundation, Inc.  --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --

--  This file performs the system-dependent translation between machine
--  exceptions and the Ada exceptions, if any, that should be raised when
--  when they occur. This version works for SunOS and Solaris.

with Interfaces.C; use Interfaces.C;

with Interfaces.C.POSIX_RTE;

package body System.Compiler_Exceptions.Machine_Specifics is

   package RTE renames Interfaces.C.POSIX_RTE;

   ------------------------
   -- Identify_Exception --
   ------------------------

   --  This function identifies the Ada exception to be raised using
   --  the information when the system received a synchronous signal.
   --  Since this function is machine and OS dependent, different code
   --  has to be provided for different target.

   --  Following code is intended for SunOS/Solaris on Sparcstation.

   function Identify_Exception
     (Which              : System.Task_Primitives.Machine_Exceptions;
      Info               : System.Task_Primitives.Error_Information;
      Modified_Registers : Pre_Call_State) return Ada.Exceptions.Exception_Id
   is

      SPARC_MAXREGWINDOW : constant := 31;

      type sc_spbuf_t is array (1 .. SPARC_MAXREGWINDOW) of System.Address;

      type sc_wbuf_t is array (1 .. SPARC_MAXREGWINDOW, 1 .. 16) of int;

      type sigcontext is record
         sc_onstack : int;              -- sigstack state to restore
         sc_mask    : int;              -- signal mask to restore
         sc_sp      : System.Address;   -- sp to restore
         sc_pc      : System.Address;   -- pc to restore
         sc_npc     : System.Address;   -- next pc to restore
         sc_psr     : int;              -- psr to restore
         sc_g1      : int;              -- register that must be restored
         sc_o0      : int;
         sc_wbcnt   : int;              -- number of outstanding windows
         sc_spbuf   : sc_spbuf_t;       -- sp's for each wbuf (in C is char *)
         sc_wbuf    : sc_wbuf_t;        -- window save buf
      end record;

      type sigcontext_ptr is access sigcontext;

      --  The above operations will be available as predefined operations on
      --  the modula Address type in GNARL, since this package is a child of
      --  System.

      FPE_INTOVF_TRAP   : constant int := 16#1#;  -- Int overflow
      FPE_STARTSIG_TRAP : constant int := 16#2#;  -- process using fp
      FPE_INTDIV_TRAP   : constant int := 16#14#; -- Int divide by zero
      FPE_FLTINEX_TRAP  : constant int := 16#c4#; -- floating inexact result
      FPE_FLTDIV_TRAP   : constant int := 16#c8#; -- floating divide by zero
      FPE_FLTUND_TRAP   : constant int := 16#cc#; -- floating underflow
      FPE_FLTOPERR_TRAP : constant int := 16#d0#; -- floating operand error
      FPE_FLTOVF_TRAP   : constant int := 16#d4#; -- floating overflow

      --  Following is SIGILL generated by trap 5 instruction

      ILL_CHECK_TRAP    : constant int := 16#80# + 16#05#;

      function Pre_Call_To_Context is new
        Unchecked_Conversion (Pre_Call_State, sigcontext_ptr);


      Current_Exception : Ada.Exceptions.Exception_Id;

      context : sigcontext_ptr :=
                  Pre_Call_To_Context (Modified_Registers);

      sig     : RTE.Signal := RTE.Signal (Which);

   begin

      --  As long as we are using a longjmp to return control to the
      --  exception handler on the runtime stack, we are safe. The original
      --  signal mask (the one we had before coming into this signal catching
      --  function) will be restored by the longjmp. Therefore, raising
      --  an exception in this handler should be a safe operation.

      case sig is

         when RTE.SIGFPE =>

            case Info.si_code is

               when FPE_INTDIV_TRAP   |
                    FPE_FLTINEX_TRAP  |
                    FPE_FLTDIV_TRAP   |
                    FPE_FLTUND_TRAP   |
                    FPE_FLTOVF_TRAP   |
                    FPE_FLTOPERR_TRAP |
                    FPE_INTOVF_TRAP   =>

                  Current_Exception := Constraint_Error_Id;

               when others =>
                  pragma Assert (False, "Unexpected SIGFPE signal");
                  null;

            end case;

         when RTE.SIGILL =>

            case Info.si_code is

               when ILL_CHECK_TRAP =>
                  Current_Exception := Constraint_Error_Id;

               when others =>

                  pragma Assert (false, "Unexpected SIGILL signal");
                  null;
            end case;

         when RTE.SIGSEGV =>

         --  If the address that caused the error was in the first page, this
         --  was caused by accessing a null pointer.

            if context.sc_o0 >= 0 and context.sc_o0 < 16#2000# then
               Current_Exception := Constraint_Error_Id;

            else
               Current_Exception := Storage_Error_Id;
            end if;

         when others =>

            pragma Assert (false, "Unexpected signal");
            null;
      end case;

      return Current_Exception;

   end Identify_Exception;

end System.Compiler_Exceptions.Machine_Specifics;
