%{
/*############################################################################*/
/*      LaTeX - SYNTAX ANALIZER and SEMANTICS ANALIZER                        */
/*############################################################################*/

#include        "latex2lyx.h"
#include        "latexdefs.c"

Node		*latex_tree;

%}

/*----------------------------------------------------------------------------*/
/*      Nonterminals and Terminals type                                       */
/*----------------------------------------------------------------------------*/

%union
  {
    char        *ptr;      
    Node        *node;
    int		spec;
  }   

/*----------------------------------------------------------------------------*/
/*      Nonterminals - Type declarations                                      */
/*----------------------------------------------------------------------------*/

%type <node> 	latex
%type <node>  	head,		documentstyle,	preambule,	document
%type <node>	restricted_list,special_list,	word_list
%type <node>	optional_spec,	skip_space
%type <node>	gen_word,	special_word,	word
%type <node>	command


/*----------------------------------------------------------------------------*/
/*      Terminals - Constants definitions, Type declarations                  */
/*----------------------------------------------------------------------------*/

%token <ptr> L_documentstyle   1 L_pagenumbering  2 L_pagestyle   3 L_maketitle   4 
%token <ptr> L_tableofcontents 5 L_begin	  6 L_end	  7 L_item	  8 
%token <ptr> L_bibitem	       9 L_label	 10 L_ref	 11 L_cite	 12 
%token <ptr> L_noindent	      13 L_input	 14 L_include	 15 L_includeonly 16 
%token <ptr> L_centerline     17 L_mbox		 18 L_fbox	 19 L_framebox	 20 
%token <ptr> L_sbox	      21 L_savebox	 22 L_newsavebox 23 L_usebox	 24 
%token <ptr> L_newpage	      25 L_clearpage	 26 L_samepage	 27 L_linebreak	 28 
%token <ptr> L_nolinebreak    29 L_sloppy	 30 
%token <ptr> L_hline	      31 L_cline	 32 L_multicolumn 33 L_caption	 34 
%token <ptr> L_kill	      35 
%token <ptr> L_vspace	      36 L_hspace	 37 L_vfill	 38 L_hfill	 39 
%token <ptr> L_put	      40 L_multiput	 41 L_makebox	 42 L_dashbox	 43 
%token <ptr> L_line	      44 L_vector	 45 L_shortstack 46 L_circle	 47 
%token <ptr> L_oval           48 L_frame	 49 L_thinlines	 50 L_thicklines 51 
%token <ptr> L_sqrt           52 L_frac	 	 53 L_overline	 54 L_appendix 	 55
%token <ptr> L_tabset	      56 L_tabskip       57 L_pmod	 58 L_vec        59
%token <ptr> L_stackrel	      60 L_centering     61
%token <ptr> L_lbracket	      62 L_rbracket	 63 L_lcurly	 64 L_rcurly	 65 
%token <ptr> L_word           66 L_linefeed	 67
%token <ptr> L_bibliographystyle 68 L_newline	 69 L_smallspace 70 L_today	 71
%token <ptr> L_footnote	      72 L_subscript	 73 L_superscript 74
%token <ptr> L_thinspace      75 L_negspace      76 L_mediumspace 77 L_thickspace 78
%token <ptr> L_plus           79 L_div           80 L_equal      81
%token <spec> L_mathbegin     82 L_mathend	 83
%token <ptr> L_less	      84 L_more	 	 85
%token <ptr> L_amper	      86 
%token <ptr> L_marginpar      87
%token <ptr> L_lparen	      88 L_rparen	 89
%token <ptr> L_tilde	      90 
%token <spec> L_dollar	      91 
%token <ptr> L_backslash      92 L_mathspace	 93 L_box        94 L_textwidth  95
%token <ptr> L_bibliography   96
%token <ptr> L_linespacing    97 L_pagebreak     98 L_nopagebreak 99
%token <spec> L_Heading	     100 L_Heading_star 112
%token <spec> L_FontShape    101
%token <spec> L_FontSize     102
%token <spec> L_Environment  103
%token <spec> L_DocStyle     104
%token <spec> L_MathGreek    105 L_MathSymbol   106 L_MathDelim 107
%token <spec> L_accent	     108 L_LatexMisc	109 L_space	110
%token <spec> L_document     111
%token <ptr> L_pageref       113 L_rule         114 L_bezier    115

%start latex

/******************************************************************************/
/*      LaTeX grammar definition and respective actions	                      */
/******************************************************************************/
%%

latex		: head documentstyle preambule document
                        {
			    int i=2;

			    if ($1 != NULL) {
				push_node ($1);
				i++;
			    }
			    push_node ($2);
			    if ($3 != NULL) {
				push_node ($3);
				i++;
			    }
			    push_node ($4);
			    push_node (create_node(L_latex));
			    $$ = build_tree(i);
			    latex_tree = $$;
			}
                ;

head		: special_list
                        {   $$ = $1;						}
                | /* epsilon */
                        {   $$ = NULL;						}
                ;

documentstyle	: L_documentstyle optional_spec L_DocStyle
                        {
			    if ($2 != NULL) {
				push_node ($2);
				push_node (create_node(L_documentstyle));
				$$ = build_tree (1);
			    } else {
				$$ = create_node(L_documentstyle);
			    }
			    $$->data.spec = $3;
			}
                ;

preambule	: special_list
                        {   $$ = $1;						}
                | /* epsilon */
                        {   $$ = NULL;						}
                ;

document	: L_begin L_document word_list
                        {   $$ = $3;						}
                ;




restricted_list	: restricted_list gen_word
                        {   $$ = make_list (L_word_list, $1, $2);		}
                | gen_word
                        {   $$ = $1;						}
                ;

special_list	: special_list special_word
                        {   $$ = make_list (L_word_list, $1, $2);		}
                | special_word
                        {   $$ = $1;						}
                ;

word_list	: word_list word 
                        {   $$ = make_list (L_word_list, $1, $2);		}
                | word
                        {   $$ = $1;						}
                ;




optional_spec	: L_lbracket word_list L_rbracket
                        {
			    push_node ($2);
			    push_node (create_node(L_optional_spec));
			    $$ = build_tree (1);
                        }
                | /* epsilon */
                        {   $$ = NULL;						}
                ;

skip_space	: skip_space L_linefeed
                        {   $$ = NULL;						}
                | skip_space L_space
                        {   $$ = NULL;						}
                | /* epsilon */
                        {   $$ = NULL;						}
                ;

gen_word	: L_word
                        {
			    $$ = create_node (L_word);
			    $$->data.ptr = $1;
			}
                | command
                        {   $$ = $1;						}
                | L_linefeed
                        {
			    $$ = create_node (L_linefeed);
			}
                | L_space
                        {
			    $$ = create_node (L_space);
			    $$->data.spec = $1;
			}
                | L_amper
                        {   $$ = create_node(L_amper);				}
                | L_tilde
                        {   $$ = create_node(L_tilde);				}
                | L_plus
                        {   $$ = create_node(L_plus);				}
                | L_div
                        {   $$ = create_node(L_div);				}
                | L_equal
                        {   $$ = create_node(L_equal);				}
                | L_less
                        {   $$ = create_node(L_less);				}
                | L_more
                        {   $$ = create_node(L_more);				}
                ;




special_word 	: gen_word
                        {   $$ = $1;						}
                | L_lcurly word_list L_rcurly
                        {   
			    push_node ($2);
			    push_node (create_node(L_lcurly));
			    $$ = build_tree (1);
		        }
                | L_lparen
                        {   $$ = create_node(L_lparen);				}
                | L_rparen
                        {   $$ = create_node(L_rparen);				}
                | L_lbracket
                        {   $$ = create_node(L_lbracket);			}
                | L_rbracket
                        {   $$ = create_node(L_rbracket);			}
                ;




word            : gen_word
                        {   $$ = $1;						}
                | L_lbracket word_list L_rbracket
                        {
			    push_node ($2);
			    push_node (create_node(L_optional_spec));
			    $$ = build_tree (1);
                        }
                | L_begin skip_space L_Environment word_list L_end skip_space L_Environment
                        {
			    push_node ($4);
			    push_node (create_node (L_Environment));
			    $$ = build_tree (1);
			    $$->data.spec = $3;
			}
                | L_end L_document
                        {
			    $$ = create_node (L_end);
			    $$->data.spec = $2;
			}
                | L_lcurly word_list L_rcurly
                        {
			    push_node ($2);
			    push_node (create_node(L_lcurly));
			    $$ = build_tree (1);
			}
                | L_mathbegin word_list L_mathend
                        {
			    push_node ($2);
			    push_node (create_node(L_dollar));	
			    $$ = build_tree (1);
			    $$->data.spec = $1;
			}
                | L_dollar special_list L_dollar
                        {
			    push_node ($2);
			    push_node (create_node(L_dollar));	
			    $$ = build_tree (1);
			    $$->data.spec = '(';
			}
                | L_lparen
                        {   $$ = create_node(L_lparen);				}
                | L_rparen
                        {   $$ = create_node(L_rparen);				}
                ;




command		: L_MathGreek
                        {
			    $$ = create_node(L_MathGreek);
			    $$->data.spec = $1;
			}
                | L_MathSymbol
                        {
			    $$ = create_node(L_MathSymbol);
			    $$->data.spec = $1;
			}
                | L_MathDelim
                        {
			    $$ = create_node(L_MathDelim);
			    $$->data.spec = $1;
			}
                | L_Heading optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($2 == NULL) {
				push_node ($4);
				push_node (create_node(L_Heading));
				$$ = build_tree (1);
			    } else {
				push_node ($2);
				push_node ($4);
				push_node (create_node(L_Heading));
				$$ = build_tree (2);
			    }
			    $$->data.spec = $1;
			}
                | L_Heading_star L_lcurly word_list L_rcurly
                        {
			    push_node ($3);
			    push_node (create_node(L_Heading_star));
			    $$ = build_tree (1);
			    $$->data.spec = $1;
			}
                | L_FontShape
                        {
			    $$ = create_node(L_FontShape);
			    $$->data.spec = $1;
			}
                | L_FontSize
                        {
			    $$ = create_node(L_FontSize);
			    $$->data.spec = $1;
			}
                | L_accent skip_space L_lcurly word_list L_rcurly
                        {
			    push_node ($4);
			    push_node (create_node(L_accent));
			    $$ = build_tree (1);
			    $$->data.spec = $1;
			}
                | L_LatexMisc
                        {
			    $$ = create_node(L_LatexMisc);
			    $$->data.spec = $1;
			}



                | L_appendix
                        {   $$ = pack_latex_nop(L_appendix);			}
                | L_box
                        {   $$ = pack_latex_nop(L_box);				}
                | L_clearpage
                        {   $$ = pack_latex_nop(L_clearpage);			}
                | L_maketitle
                        {   $$ = pack_latex_nop(L_maketitle);			}
                | L_newpage
                        {   $$ = pack_latex_nop(L_newpage);			}
                | L_samepage
                        {   $$ = pack_latex_nop(L_samepage);			}
                | L_sloppy
                        {   $$ = pack_latex_nop(L_sloppy);			}
                | L_textwidth
                        {   $$ = pack_latex_nop(L_textwidth);			}
                | L_today
                        {   $$ = pack_latex_nop(L_today);			}
                | L_centering
                        {   $$ = pack_latex_nop(L_centering);			}
                | L_noindent
                        {   $$ = pack_latex_nop(L_noindent);			}
                | L_vfill
                        {   $$ = pack_latex_nop(L_vfill);			}
                | L_hfill
                        {   $$ = pack_latex_nop(L_hfill);			}



                | L_bibliography skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_bibliography);		}
                | L_bibliographystyle skip_space L_lcurly gen_word L_rcurly
                        {   $$ = pack_latex_1op ($4, L_bibliographystyle);	}
                | L_input skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_input);			}
                | L_fbox skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_fbox);			}
                | L_footnote skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_footnote);		}
                | L_marginpar skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_marginpar);		}
                | L_pageref skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_pageref);		}
                | L_label skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_label);			}
                | L_mbox skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_mbox);			}
                | L_newsavebox skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_newsavebox);		}
                | L_pagenumbering skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_pagenumbering);		}
                | L_pagestyle skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_pagestyle);		}
                | L_ref skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_ref);			}
                | L_sqrt skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_sqrt);			}
                | L_subscript word
                        {   $$ = pack_latex_1op ($2, L_subscript);		}
                | L_superscript word
                        {   $$ = pack_latex_1op ($2, L_superscript);		}
                | L_usebox skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_usebox);			}
                | L_hspace skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_hspace);			}
                | L_vspace skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_vspace);			}
                | L_cite skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op ($4, L_cite);			}
                | L_circle skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_circle);			}
                | L_frame skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_frame);			}
                | L_centerline skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_centerline);		}
                | L_pmod skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_pmod);			}
                | L_vec skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_vec);			}
                | L_overline skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_overline);		}
                | L_linespacing skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_linespacing);		}
                | L_cline skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_cline);			}
                | L_include skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_include);			}
                | L_includeonly skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_1op($4, L_includeonly);		}
                | L_oval L_lparen restricted_list L_rparen
                        {   $$ = pack_latex_1op($3, L_oval);			}





                | L_frac skip_space L_lcurly word_list L_rcurly skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_2op ($4, $8, L_frac);		}
                | L_stackrel skip_space L_lcurly word_list L_rcurly skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_2op ($4, $8, L_stackrel);		}
                | L_sbox skip_space L_lcurly word_list L_rcurly skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_2op ($4, $8, L_sbox);		}
                | L_line skip_space L_lparen restricted_list L_rparen skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_2op ($4, $8, L_line);		}
                | L_vector skip_space L_lparen restricted_list L_rparen skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_2op ($4, $8, L_vector);		}
                | L_put skip_space L_lparen restricted_list L_rparen skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_2op ($4, $8, L_put);		}



                | L_multicolumn skip_space L_lcurly word_list L_rcurly skip_space L_lcurly word_list L_rcurly skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_3op ($4, $8, $12, L_multicolumn);	}



                | L_multiput skip_space L_lparen restricted_list L_rparen skip_space L_lparen restricted_list L_rparen skip_space L_lcurly word_list L_rcurly skip_space L_lcurly word_list L_rcurly
                        {   $$ = pack_latex_4op ($4, $8, $12, $16, L_multiput);	}
                | L_bezier L_lcurly word_list L_rcurly skip_space L_lparen restricted_list L_rparen skip_space L_lparen restricted_list L_rparen skip_space L_lparen restricted_list L_rparen
                        {   $$ = pack_latex_4op ($3, $7, $11, $15, L_bezier);	}




                | L_bibitem skip_space optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($3 == NULL)
			        $$ = pack_latex_1op ($5, L_bibitem);
			    else
			        $$ = pack_latex_2op ($3, $5, L_bibitem);
			}
                | L_shortstack skip_space optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($3 == NULL)
			        $$ = pack_latex_1op ($5, L_shortstack);
			    else
			        $$ = pack_latex_2op ($3, $5, L_shortstack);
			}
                | L_caption skip_space optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($3 == NULL)
			        $$ = pack_latex_1op ($5, L_caption);
			    else
			        $$ = pack_latex_2op ($3, $5, L_caption);
			}



                | L_makebox skip_space L_lparen restricted_list L_rparen optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($6 == NULL)
			        $$ = pack_latex_2op ($4, $8, L_makebox);
			    else
			        $$ = pack_latex_3op ($4, $6, $8, L_makebox);
			}
                | L_framebox skip_space L_lparen restricted_list L_rparen optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($6 == NULL)
			        $$ = pack_latex_2op ($4, $8, L_framebox);
			    else
			        $$ = pack_latex_3op ($4, $6, $8, L_framebox);
			}
                | L_rule optional_spec L_lcurly word_list L_rcurly skip_space L_lcurly word_list L_rcurly
                        {
			    if ($2 == NULL)
			        $$ = pack_latex_2op ($4, $8, L_rule);
			    else
			        $$ = pack_latex_3op ($2, $4, $8, L_rule);
			}



                | L_dashbox skip_space L_lcurly word_list L_rcurly L_lparen restricted_list L_rparen optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($9 == NULL)
			        $$ = pack_latex_3op ($4, $7, $11, L_dashbox);
			    else
			        $$ = pack_latex_4op ($4, $7, $9, $11, L_dashbox);
			}
                | L_savebox skip_space L_lcurly word_list L_rcurly L_lparen restricted_list L_rparen optional_spec L_lcurly word_list L_rcurly
                        {
			    if ($9 == NULL)
			        $$ = pack_latex_3op ($4, $7, $11, L_savebox);
			    else
			        $$ = pack_latex_4op ($4, $7, $9, $11, L_savebox);
			}




                | L_newline
                        {   $$ = create_node(L_newline);			}
                | L_item
                        {   $$ = create_node(L_item);				}
                | L_linebreak
                        {   $$ = create_node(L_linebreak);			}
                | L_nolinebreak
                        {   $$ = create_node(L_nolinebreak);			}
                | L_pagebreak
                        {   $$ = create_node(L_pagebreak);			}
                | L_nopagebreak
                        {   $$ = create_node(L_nopagebreak);			}
                | L_kill
                        {   $$ = create_node(L_kill);				}
                | L_hline
                        {   $$ = create_node(L_hline);				}
                | L_smallspace
                        {   $$ = create_node(L_smallspace);			}
                | L_thinspace
                        {   $$ = create_node(L_thinspace);			}
                | L_negspace
                        {   $$ = create_node(L_negspace);			}
                | L_mediumspace
                        {   $$ = create_node(L_mediumspace);			}
                | L_thickspace
                        {   $$ = create_node(L_thickspace);			}
                | L_mathspace
                        {   $$ = create_node(L_mathspace);			}
                | L_tabset
                        {   $$ = create_node(L_tabset);				}
                | L_tabskip
                        {   $$ = create_node(L_tabskip);			}
                | L_thinlines
                        {   $$ = create_node(L_thinlines);			}
                | L_thicklines
                        {   $$ = create_node(L_thicklines);			}
                | L_tableofcontents
                        {   $$ = create_node(L_tableofcontents);		}
                | L_backslash L_word
                        {
			    Node *t;
			    
			    t = create_node(L_word);
			    t->data.ptr = $2;
			    push_node (t);
			    push_node (create_node(L_backslash));
			    $$ = build_tree(1);
			}
                ;

%%

/******************************************************************************/
/*      Lexical Analizer routines                                             */
/******************************************************************************/

#include "lex.yy.c"


/******************************************************************************/
/*      Node manipulation routines                                            */
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/*      Definition of a stack for nodes                                       */
/*----------------------------------------------------------------------------*/

#define  STACK_SIZE 1000
Node     *nodestack [STACK_SIZE];
int      top = 0;                       /* pointer to top of stack     */


/*----------------------------------------------------------------------------*/
/*      push_node - Put given node on top of stack                            */
/*----------------------------------------------------------------------------*/

inline Node	*push_node (Node *n)
{ 
    nodestack [top] = n;
    top ++;
    if (top == STACK_SIZE)
    {
        fprintf (stderr, "NODE STACK OVERFLOW\n\n");
        exit (1);
    }
    return n;
}


/*----------------------------------------------------------------------------*/
/*      pop_node - Take node on top of stack and return it                    */      
/*----------------------------------------------------------------------------*/

inline Node     *pop_node (void)
{ 
    top --;
    if (top < 0)
    {
        fprintf (stderr, "NODE STACK UNDERFLOW\n\n");
        exit (1);
    }
    return nodestack [top];
}


/*----------------------------------------------------------------------------*/
/*      create_node - Allocate memory and initialize fields of node structure */
/*----------------------------------------------------------------------------*/

inline Node     *create_node (int c)
{
    Node        *n;
  
    n = (Node *) malloc (sizeof (Node));
    if (n == NULL)
      {
        fprintf (stderr, "OUT OF MEMORY\n\n");
	exit (1);
      }
    n->last = 1;
    n->next = NULL;
    n->down = NULL;
    n->lson = NULL;
    n->token = c;
    n->data.spec = 0;
    return n;
}


/*----------------------------------------------------------------------------*/
/*      append_node - Put given node as the last son of node on top of stack  */
/*----------------------------------------------------------------------------*/

inline Node     *append_node (Node *n)
{
    Node        *list;
    Node        *p;
   
    list = pop_node();
    p = list->lson;
    n->last = 1;
    n->next = list;
    p->last = 0;
    p->next = n;
    list->lson = n;
    return list;
}


/*----------------------------------------------------------------------------*/
/*      insert_node - Put given node as the first son of node on top of stack */
/*----------------------------------------------------------------------------*/

inline Node     *insert_node (Node *n)
{
    Node        *p;
   
    p = pop_node();   
    n->last = 0;
    n->next = p->down; 
    p->down = n;     
    return p;
}


/*----------------------------------------------------------------------------*/
/*      merge_nodes - Put all sons of given node as sons of node on top of stk*/
/*----------------------------------------------------------------------------*/

inline Node     *merge_nodes (Node *n)
{	
    Node        *p, *q;

    q = pop_node();
    p = q->lson;
    p->last = 0;
    p->next = n->down;
    q->lson = n->lson;
    n = n->lson;
    n->next = q;
    return q;
}


/*----------------------------------------------------------------------------*/
/*      build_tree - Take top of stack as root of tree and 'n' others as sons */
/*----------------------------------------------------------------------------*/
 
inline Node     *build_tree (int i)
{
    Node        *root, *leaf, *n;
    int         flag;

    root = pop_node();
    n = root;
    flag = 1;

    for (  ; i>0; i--)
    {
        leaf = pop_node ();
        leaf->last = flag;
	leaf->next = n;
	if (flag == 1)
	    root->lson = leaf;
        n = leaf;
        flag = 0;
    }
    root->down = n;
    return root;
}


/*----------------------------------------------------------------------------*/
/*      make_list - Create a new list or append the node in the given list    */
/*----------------------------------------------------------------------------*/

inline Node     *make_list (int token, Node *list, Node *n)
{
    if (list->token == token)
    {
        push_node (list);
        return append_node(n);
    } else {
        push_node (list);
        push_node (n);
        push_node (create_node(token));
        return build_tree(2);
    }
}



/*----------------------------------------------------------------------------*/
/*      print_tree - Display contents of tree                                 */
/*----------------------------------------------------------------------------*/

Node            *print_tree (Node *n, int d)
{
    int         i;
 
    for (i=d; i>0; i--)
        fprintf (stderr, "   ");  

    /****** Put printing routine here */
    fprintf (stderr, "%d data: %d\n", n->token, n->data.spec);

    if (n->down != NULL)
	print_tree (n->down, d+1);
    if (n->last == 0)
    {
        n = n -> next;
        print_tree (n, d);
    }
}



/*----------------------------------------------------------------------------*/
/*      pack_latex_nop - Create a tree for latex commands with no arguments   */
/*----------------------------------------------------------------------------*/

inline Node	*pack_latex_nop (int spec)
{
    Node	*tree;

    tree = create_node(L_latex_nop);
    tree->data.spec = spec;
    return tree;
}



/*----------------------------------------------------------------------------*/
/*      pack_latex_1op - Create a tree for latex commands with 1 argument     */
/*----------------------------------------------------------------------------*/

inline Node	*pack_latex_1op (Node *data, int spec)
{
    Node	*tree;

    push_node (data);
    push_node (create_node(L_latex_1op));
    tree = build_tree (1);
    tree->data.spec = spec;
    return tree;
}



/*----------------------------------------------------------------------------*/
/*      pack_latex_2op - Create a tree for latex commands with 2 arguments    */
/*----------------------------------------------------------------------------*/

inline Node	*pack_latex_2op (Node *data1, Node *data2, int spec)
{
    Node	*tree;

    push_node (data1);
    push_node (data2);
    push_node (create_node(L_latex_2op));
    tree = build_tree (2);
    tree->data.spec = spec;
    return tree;
}



/*----------------------------------------------------------------------------*/
/*      pack_latex_3op - Create a tree for latex commands with 3 arguments    */
/*----------------------------------------------------------------------------*/

inline Node	*pack_latex_3op (Node *data1, Node *data2, Node *data3, int spec)
{
    Node	*tree;

    push_node (data1);
    push_node (data2);
    push_node (data3);
    push_node (create_node(L_latex_3op));
    tree = build_tree (3);
    tree->data.spec = spec;
    return tree;
}



/*----------------------------------------------------------------------------*/
/*      pack_latex_4op - Create a tree for latex commands with 4 arguments    */
/*----------------------------------------------------------------------------*/

inline Node	*pack_latex_4op (Node *data1, Node *data2, Node *data3, Node *data4, 
				 int spec)
{
    Node	*tree;

    push_node (data1);
    push_node (data2);
    push_node (data3);
    push_node (data4);
    push_node (create_node(L_latex_4op));
    tree = build_tree (4);
    tree->data.spec = spec;
    return tree;
}



/******************************************************************************/
/*      Error reporting and parser recovery routine                           */
/******************************************************************************/

int             yyerror ()
{
    fprintf(stderr, "****** Error line: %d, context: %s\n", linecounter, yytext);
    errorcounter ++;
    return 0;
}


/******************************************************************************/
/*      Main program - Parse invocation                                       */
/******************************************************************************/

FILE		*fout = NULL;
int  		linecounter = 1; /* To indicate location of erros             */
int     	errorcounter = 0;/* Nunber of error detected during compilation*/


int             main (int argc, char **argv)
{
    int         errorlevel;

    if (argc == 1)
    { 
	yyin = stdin;
	fout = stdout;
    } else if (argc == 2) {
	if (*argv[1] == '-')
	    yyin = stdin;
	else
	    yyin = fopen (argv[1], "r");
	fout = stdout;
    } else if (argc == 3) {
	if (*argv[1] == '-')
	    yyin = stdin;
	else
	    yyin = fopen (argv[1], "r");
	fout = fopen (argv[2], "w");
   }
    if (yyin == NULL || fout == NULL)
    {
        fprintf (stderr, "******************************************\n");
        fprintf (stderr, "usage: %s [<file in> [<file out>]]\n", argv[0]);
        fprintf (stderr, "or:    %s - [<file out>]\n", argv[0]);
        return (1);
     }

    errorlevel = yyparse ();
    if (errorcounter != 0)
        fprintf (stderr, "...... %2d errors encountered during LaTeX parsing\n", errorcounter);
    else
        fprintf (stderr, "...... No errors found during LaTeX parsing\n");

    if (errorcounter == 0) {
        print_lyx (fout, latex_tree, latex_tree);
	fprintf (stderr, "\n...... Conversion completed\n\n");
    }

    return errorlevel;
  }






