/*
 * triangulate.c - triangulation in VRwave
 * Copyright (c) 1997 IICM
 *
 * created: kwagen, 19970607
 *
 * changed: kwagen, 19970703
 * changed: mpichler, 19970807
 *
 * $Id: triangulate.c,v 1.3 1997/08/08 10:01:48 mpichler Exp $
 */
/* triangulation done with "ear clipping" algorithm */
/* Lit.: O'Rourke, "Computational Geometry in C", ch. 1.6 */


#ifdef __cplusplus
extern "C" {
#endif
#include "Builder.h"
#include "IntArray.h"
#include "FloatArray.h"
#ifdef __cplusplus
} // C++
#endif


#ifdef macintosh
# include <vectors.h>
# include <ge3d.h>
# define M_PI _PI
# define PACKAGE iicm_vrml_vrwave_
#else
# include <ge3d/vectors.h>
# include <ge3d/ge3d.h>
#endif

#include "jutils.h"

#include <math.h>  /* M_PI */
#include <stdio.h>  /* debugging */


/* usage: Debug ((stderr, "format", arguments)) */
#if 0
#define Debug(body)  fprintf body
#else
#define Debug(body)  /**/
#endif


typedef struct vertex_str
{ /* node of double linked list */
  /* for representing polygon vertices during triangulation */
  point2D pos;
  int index;
  int texindex;
  struct vertex_str *next, *prev;
} vertex;



/* area2 : twice area of a triangle. area2 > 0 if CCW */
static double area2 (const point2D* a, const point2D* b, const point2D* c)
{
  return
    a->x * b->y - a->y * b->x +
    a->y * c->x - a->x * c->y +
    b->x * c->y - b->y * c->x;
}

/* twice area of a polygon. area > 0 if CCW */
static double area (vertex* polygon, int num)
{
  double sum = 0.0;
  vertex* tmp1 = polygon->next;
  vertex* tmp2 = tmp1->next;

  while (tmp2 != polygon)
  {
    sum += area2 (&polygon->pos, &tmp1->pos, &tmp2->pos);
    tmp1 = tmp2;
    tmp2 = tmp2->next;
  }

  return sum;
}

/* checks if a polygon is convex */
static int is_convex (vertex* polygon, int num, int ccw)
{
  int i;
  vertex* tmp = polygon;
  double a2;

  if (num > 3)
    for (i = 0; i < num; i++)
    {
      a2 = area2 (&tmp->prev->pos, &tmp->pos, &tmp->next->pos);
      if (ccw && a2 < 0.0)
	return 0;
      if (!ccw && a2 > 0.0)
	return 0;

      tmp = tmp->next;
    }

  return 1;
}

/* checks if a point is left of a line */
static int left (const vector3D* line, const vector2D* pos)
{
  return (line->x * pos->x + line->y * pos->y + line->z > -1e-6);
}    

/* checks if a point is right of a line */
static int right (const vector3D* line, const vector2D* pos)
{
  return (line->x * pos->x + line->y * pos->y + line->z < 1e-6);
}    

/* check if the line segment between va and vb is a diagonal */
static int diagonal (vertex* va, vertex* vb, int ccw)
{
  vertex* vc = va->next;
  vertex* v;
  vector3D ac, cb, ba;

  if (ccw && area2 (&va->pos, &vc->pos, &vb->pos) < 1e-6)
    return 0;
  if (!ccw && area2 (&va->pos, &vc->pos, &vb->pos) > -1e-6)
    return 0;

  ba.x = vb->pos.y - va->pos.y;
  ba.y = va->pos.x - vb->pos.x;
  ba.z = -ba.x * va->pos.x - ba.y * va->pos.y;
  ac.x = va->pos.y - vc->pos.y;
  ac.y = vc->pos.x - va->pos.x;
  ac.z = -ac.x * vc->pos.x - ac.y * vc->pos.y;
  cb.x = vc->pos.y - vb->pos.y;
  cb.y = vb->pos.x - vc->pos.x;
  cb.z = -cb.x * vb->pos.x - cb.y * vb->pos.y;

  for (v = vb->next; v != va; v = v->next)
  {
    if (ccw && left (&ba, &v->pos) && left (&ac, &v->pos) && left (&cb, &v->pos))
      return 0;
    if (!ccw && right (&ba, &v->pos) && right (&ac, &v->pos) && right (&cb, &v->pos))
      return 0;
  }  

  return 1;
}

/* add triangle to new vertexindex */
static void add_triangle (vertex* v1, vertex* v2, vertex* v3, jn_int32* ncoordinds, int* nnum, jn_int32* ntexcoordinds)
{
  /* printf ("triangle: %d %d %d\n", v1->index, v2->index, v3->index); */

  ncoordinds[*nnum] = v1->index;
  ncoordinds[*nnum+1] = v2->index;
  ncoordinds[*nnum+2] = v3->index;
  ncoordinds[*nnum+3] = -1;
  if (ntexcoordinds)
  {
    ntexcoordinds[*nnum] = v1->texindex;
    ntexcoordinds[*nnum+1] = v2->texindex;
    ntexcoordinds[*nnum+2] = v3->texindex;
    ntexcoordinds[*nnum+3] = -1;
  }
  (*nnum) += 4;
}

/* add unchanged vertexindices of a convex polygon */
static void add_convex (vertex* polygon, int num, jn_int32* ncoordinds, int* nnum, jn_int32* ntexcoordinds)
{
  int i;
  vertex* tmp = polygon;

  for (i = 0; i < num; i++)
  {
    ncoordinds[*nnum] = tmp->index;
    if (ntexcoordinds)
      ntexcoordinds[*nnum] = tmp->texindex;
    (*nnum)++;
    tmp = tmp->next;
  }

  ncoordinds[*nnum] = -1;
  (*nnum)++;
}

/* remove ear tip vertex from vertexlist */
static void clip_ear (vertex* polygon, vertex* v, jn_int32* ncoordinds, int* nnum, jn_int32* ntexcoordinds)
{
  add_triangle (v->prev, v, v->next, ncoordinds, nnum, ntexcoordinds);

  /* if (v == polygon)
    polygon = v->next; */
  v->prev->next = v->next;
  v->next->prev = v->prev;

  free (v);
}

/* triangulate polygon (2D) if not convex */
static int triangulate (vertex* polygon, int num, jn_int32* ncoordinds, int* nnum, jn_int32* ntexcoordinds)
{
  /* int i; */
  int fn;
  int ccw = 1;
  vertex* tmp = polygon;

  Debug ((stderr, "  triangulate polygon with %d vertices\n", num));

  /* for (i = 0; i < num; i++)
  {
    printf ("vertex %d: %f, %f\n", i, tmp->pos.x, tmp->pos.y);
    tmp = tmp->next;
  } */

  if (area (polygon, num) < 0)
  {
    /* printf ("NOTE: polygon is defined clockwise.\n"); */
    ccw = 0;
  }

  if (is_convex (polygon, num, ccw))
  {
    Debug ((stderr, "  polygon is already convex. unchanged.\n"));
    add_convex (polygon, num, ncoordinds, nnum, ntexcoordinds);
    return 1;
  }
  Debug ((stderr, "  non-convex polygon, must triangulate. ccw: %d\n", ccw));

  fn = 0;
  while (num > 3)
  {
    tmp = polygon;

    /* fprintf (stderr, "    diagonal (%d) ", num); */
    while (!diagonal (tmp, tmp->next->next, ccw))
    {
      tmp = tmp->next;
      /* fprintf (stderr, "."); */
      if (tmp == polygon)
      {
        fprintf (stderr, "triangulation error: no ear diagonal found. weird input data.\n");
#if 0
// do {
//   fprintf (stderr, "%10g, %10g\n", tmp->pos.x, tmp->pos.y);
//   tmp = tmp->next;
// } while (tmp != polygon);
#endif
        break;
      }
    }

    /* fprintf (stderr, " clip ear\n"); */
    clip_ear (polygon, tmp->next, ncoordinds, nnum, ntexcoordinds);

    fn++;
    num--;
  }

  add_triangle (polygon, polygon->next, polygon->prev, ncoordinds, nnum, ntexcoordinds);
  fn++;

  Debug ((stderr, "  polygon decomposed into %d triangles.\n", fn));

  return fn;
} /* triangulate */

/* convexify faces of IndexedFaceSet */
void name2(PACKAGE,Builder_convexify) (struct name3(H,PACKAGE,Builder)* handle,
  HArrayOfFloat* coordsh,
  HArrayOfInt* coordindsh,
  jn_int32 numcoordinds,
  HArrayOfFloat* facenormalsh,
  HArrayOfInt* colorindsh,
  jn_int32 matb,  /* 2: color per face, 1: per face indexed, 0: any other (overall/per vertex) */
  HArrayOfInt* texcoordindsh,
  struct Hiicm_vrml_pwutils_IntArray* ncoordindsh,
  struct Hiicm_vrml_pwutils_FloatArray* nfacenormalsh,
  struct Hiicm_vrml_pwutils_IntArray* ncolorindsh,  /* to be set when matb != 0 */
  struct Hiicm_vrml_pwutils_IntArray* ntexcoordindsh
)
{
  point3D* coords = (point3D*) unhand (coordsh)->body;
  jn_int32* coordinds = (jn_int32*) unhand (coordindsh)->body;
  vector3D* facenormals = (vector3D*) unhand (facenormalsh)->body;
  jn_int32* ncoordinds = (jn_int32*) unhand (unhand (ncoordindsh)->data_)->body;
  vector3D* nfacenormals = (vector3D*) unhand (unhand (nfacenormalsh)->data_)->body;
  jn_int32* colorinds = colorindsh ? (jn_int32*) unhand (colorindsh)->body : 0;
  jn_int32* ncolorinds = ncolorindsh ? (jn_int32*) unhand (unhand (ncolorindsh)->data_)->body : 0;
  jn_int32* texcoordinds = texcoordindsh ? (jn_int32*) unhand (texcoordindsh)->body : 0;  
  jn_int32* ntexcoordinds = ntexcoordindsh ? (jn_int32*) unhand (unhand (ntexcoordindsh)->data_)->body : 0;

  int i, j, n, dir = 0;
  int fnum = 0;
  int vi = -1;
  int ti = 0;
  int num = 0;
  int nnum = 0;
  int orgfn = 0;
  vertex* tmp;
  vertex* polygon = NULL;  
  float normalmaxcomp;
  vector3D* fnormal = facenormals;
  vector3D* nnormal = nfacenormals;
  jn_int32* ncolor = ncolorinds;

  Debug ((stderr, "convexify: no. of coordinds: %ld\n", (long) numcoordinds));

  for (i = 0; i < numcoordinds; i++)
  {
    /* new face; determine 2D projection plane (built from coordinate axis) */
    if (vi < 0)
    {
      dir = 0;
      normalmaxcomp = fabs (fnormal->x);
      if (fabs (fnormal->y) > normalmaxcomp)
      {
        dir = 1;
        normalmaxcomp = fabs (fnormal->y);
      }
      if (fabs (fnormal->z) > normalmaxcomp)
        dir = 2;

      Debug ((stderr, "convexify: new face. normal (%g, %g, %g), direction: %d\n",
        fnormal->x, fnormal->y, fnormal->z, dir));
    }

    if (texcoordinds)
      ti = texcoordinds[i];
    vi = coordinds[i];  /* vertex index */
    if (vi >= 0)
    {
      num++;

      if (polygon)
      {
        tmp = (vertex *) malloc (sizeof (vertex));
        polygon->prev->next = tmp;
        tmp->prev = polygon->prev;
        polygon->prev = tmp;
        tmp->next = polygon;
      }
      else
      {
        tmp = (vertex*) malloc (sizeof (vertex));
        tmp->prev = tmp;
        tmp->next = tmp;
        polygon = tmp;
      }

      switch (dir)
      {
        case 0: tmp->pos.x = coords[vi].z;
                tmp->pos.y = coords[vi].y;
                break;
        case 1: tmp->pos.x = coords[vi].z;
                tmp->pos.y = coords[vi].x;
                break;
        case 2: tmp->pos.x = coords[vi].x;
                tmp->pos.y = coords[vi].y;
                break;
      }
      tmp->index = vi;
      tmp->texindex = ti;
    }
    else  /* complete polygon */
    {
      n = triangulate (polygon, num, ncoordinds, &nnum, ntexcoordinds);
      fnum += n;
      for (j = 0; j < n; j++)
      {
        init3D (*nnormal, fnormal->x, fnormal->y, fnormal->z);
        nnormal++;
        if (matb)
        {
          if (matb == 2)  /* per face */
            *ncolor = orgfn;
          else  /* per face indexed */
            *ncolor = colorinds[orgfn];
          /* printf ("matb = %d, face = %d, colorindex = %d\n", matb, orgfn, *ncolor); */
          ncolor++;
        }
      }
      tmp = polygon->prev;
      while (tmp != polygon)
      {
        tmp = tmp->prev;
        free (tmp->next);
      }
      free (polygon);
      polygon = NULL;
      num = 0;
      fnormal++;
      orgfn++;
    }
  } /* for each vertexindex */

  Debug ((stderr, "confexify: last polygon.\n"));

  if (num)  /* last polygon */
  {
    n = triangulate (polygon, num, ncoordinds, &nnum, ntexcoordinds);
    fnum += n;
    for (j = 0; j < n; j++)
    {
      init3D (*nnormal, fnormal->x, fnormal->y, fnormal->z);
      nnormal++;
      if (matb)
      {
        if (matb == 2)  /* per face */
          *ncolor = orgfn;
        else  /* per face indexed */
          *ncolor = colorinds[orgfn];
        /* printf ("matb = %d, face = %d, colorindex = %d\n", matb, orgfn, *ncolor); */
        ncolor++;
      }
    }
    tmp = polygon->prev;

    Debug ((stderr, "confexify: freeing last polygon.\n"));

    while (tmp != polygon)
    {
      tmp = tmp->prev;
      free (tmp->next);
    }
    free (polygon);
  }

  Debug ((stderr, "confexify: after last polygon.\n"));

  unhand (ncoordindsh)->count_ = nnum;
  if (ntexcoordindsh)
    unhand (ntexcoordindsh)->count_ = nnum;
  unhand (nfacenormalsh)->count_ = 3 * fnum;
  if (ncolorindsh)
    unhand (ncolorindsh)->count_ = fnum;

  Debug ((stderr, "confexify: finished.\n"));

} /* convexify */
