/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPSvrControlModule.cpp

	Contains:	Implementation of module described in RTSPSvrControlModule.h

	$Log: RTSPSvrControlModule.cpp,v $
	Revision 1.2  1999/02/19 23:08:39  ds
	Created
	
	
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#ifndef __MW_
extern "C" {
#import <mach/mach.h>
#import <mach/message.h>
#import <servers/bootstrap.h>
#import <mach/mach_error.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <unistd.h>
#import "QTSCRPI.h"
}
#endif

#include "RTSPSvrControlModule.h"
#include "RTSPLoggingModule.h"	//KLUDGE!!!
#include "RTPAccessLogModule.h"	//KLUDGE!!!

#include "SCRPITypes.h"
#include "OSMutex.h"
#include "OS.h"

#include "RTPServerInterface.h"
#include "RTSPServerInterface.h"

#pragma mark __RTSP_SERVER_CONTROL_MODULE__

RTSPSvrControlModule*	RTSPSvrControlModule::sModule = NULL;

RTSPSvrControlModule::RTSPSvrControlModule()
: 	fHistoryMutex('hism'), fThread(NULL), fCursor(0),
	fBandwidthLo(-1), fBandwidthHi(0), fBandwidthAvg(0),
	fConnectionLo(-1), fConnectionHi(0), fConnectionAvg(0),
	fSampleIndex(0), fDNSName(NULL), fGracefulShutdownInProgress(false),
	fHistoryIntervalInSeconds(0), fAttributeBuffer(NULL)
{
	fStartupTime = ::time(NULL);//store time_t value for startup of the server

	//init history array
	fHistoryArray.numEntries = 0;
	fHistoryArray.entryInterval = 0;
}

bool RTSPSvrControlModule::Initialize()
{
	sModule = this;
	//allocate enough space to store the largest attribute possible
	fAttributeBuffer = new ('atbf') char[sizeof(QTSServerHistoryRec)];	
	fThread = new RTSPSvrControlThread();
	if (fThread->HasErrorOccurred())
	{
		delete fThread;
		fThread = NULL;
		return false;
	}
	return true;
}

void RTSPSvrControlModule::Shutdown()
{
	if (fThread != NULL)
		delete fThread;
	sModule = NULL;
}

kern_return_t RTSPSvrControlModule::StopServer(int inMinutes)
{
	//if time is -1, we're supposed to wait until all clients have disconnected
	printf("Attempting to shut down server\n");
	if (inMinutes == -1)
	{
		RTPServerInterface::SetServerState(RTPServerInterface::kRefusingConnectionsState);
		fGracefulShutdownInProgress = true;
	}
	else
		//just set the server state to shutting down
		RTPServerInterface::SetServerState(RTPServerInterface::kShuttingDownState);

	return SCNoError;
}

kern_return_t RTSPSvrControlModule::CancelStopServer()
{
	//Not yet implemented
	return SCNoError;
}

void RTSPSvrControlModule::CheckShutdown()
{
	if (fGracefulShutdownInProgress)
	{
		QTSServerStatusRec theServerStatus;
		kern_return_t theErr = GetServerStatusRec(&theServerStatus);
		if ((theErr == SCNoError) && (theServerStatus.numCurrentConnections == 0))
			theErr = StopServer(0);
	}
}	

kern_return_t
RTSPSvrControlModule::SetServerAttribute(AttributeType attrib, UInt32 /*tagSize*/,
											void* /*tags*/, unsigned int bufSize, void* buffer)
{
	if ((attrib.attribClass != kServerAttr) || (attrib.version != kCurrentVersion))
	{
		//because the buffer is being passed out of line, make sure to free it up
		vm_deallocate(task_self(), (unsigned int)buffer, bufSize);
		return SCNoError;
	}

	kern_return_t theError = SCNoError;
		
	switch (attrib.attribKind)
	{
		case kRefuseConnectionsAttr:
		{
			if ((buffer != NULL) && (bufSize == sizeof(QTSRefuseConnectionsRec)))
				theError = this->SetRefuseConnections((QTSRefuseConnectionsRec*)buffer);
			else
				theError = SCBufferToSmall;
			break;
		}
		case kRereadPreferencesAttr:
		{
			theError = this->RereadPreferences();
			break;
		}
		case kLogRollAttr:
		{
			if ((buffer != NULL) && (bufSize == sizeof(QTSLogRollRec)))
				theError = this->RollLogNow((QTSLogRollRec*)buffer);
			else
				theError = SCBufferToSmall;
			break;
		}
		default:
			theError = SCUnsupportedAttrib;
	}

	vm_deallocate(task_self(), (unsigned int)buffer, bufSize);
	return theError;	
}

kern_return_t
RTSPSvrControlModule::GetServerAttribute(AttributeType attrib, UInt32 /*tagSize*/,
											void* /*tags*/, unsigned int bufSize,
											unsigned int* attribSize, void** buffer)
{
	Assert(buffer != NULL);
	Assert(fAttributeBuffer != NULL);
	
	if ((attribSize == NULL) || (buffer == NULL))
		return SCParamErr;
	
	*attribSize = 0;
	//use the fAttributeBuffer memory to store the attribute. This buffer
	//should be big enough to store the largest attribute
	*buffer = fAttributeBuffer;
		
	if ((attrib.attribClass != kServerAttr) || (attrib.version != kCurrentVersion))
		return SCUnsupportedAttrib;
	
	kern_return_t theError = SCNoError;
		
	switch (attrib.attribKind)
	{
		case kDNSNameAttr:
		{
			if (bufSize >= sizeof(QTSServerDNSName))
			{
				*attribSize = sizeof(QTSServerDNSName);
				theError = this->GetServerName((QTSServerDNSName*)fAttributeBuffer);
			}
			else
				theError = SCBufferToSmall;
			break;
		}	
		case kProcessInfoAttr:
		{
			if (bufSize >= sizeof(QTSProcessInfoRec))
			{
				*attribSize = sizeof(QTSProcessInfoRec);
				theError = this->GetProcessInfo((QTSProcessInfoRec*)fAttributeBuffer);
			}
			else
				theError = SCBufferToSmall;
			break;
		}
		case kVersionAttr:
		{
			if (bufSize >= sizeof(QTSServerVersionRec))
			{
				*attribSize = sizeof(QTSServerVersionRec);
				theError = this->GetServerVersion((QTSServerVersionRec*)fAttributeBuffer);
			}
			else
				theError = SCBufferToSmall;
			break;
		}	
		case kStatusAttr:
		{
			if (bufSize >= sizeof(QTSServerStatusRec))
			{
				*attribSize = sizeof(QTSServerStatusRec);
				theError = this->GetServerStatusRec((QTSServerStatusRec*)fAttributeBuffer);
			}
			else
				theError = SCBufferToSmall;
			break;
		}	
		case kRefuseConnectionsAttr:
		{
			if (bufSize >= sizeof(QTSRefuseConnectionsRec))
			{
				*attribSize = sizeof(QTSRefuseConnectionsRec);
				theError = this->GetRefuseConnections((QTSRefuseConnectionsRec*)fAttributeBuffer);
			}
			else
				theError = SCBufferToSmall;
			break;
		}	
		case kHistoryAttr:
		{
			if (bufSize >= sizeof(QTSServerHistoryRec))
			{
				*attribSize = sizeof(QTSServerHistoryRec);
				theError = this->GetHistory((QTSServerHistoryRec*)fAttributeBuffer);
			}
			else
				theError = SCBufferToSmall;
			break;
		}
		default:
			theError = SCUnsupportedAttrib;
	}
	return theError;
}

kern_return_t
RTSPSvrControlModule::RereadPreferences()
{
	//Just tell the prefs object to reread. This is totally thread safe.
	RTSPServerInterface::GetRTSPPrefs()->RereadPreferences();
	return SCNoError;
}

kern_return_t
RTSPSvrControlModule::RollLogNow(QTSLogRollRec* theRollLogRec)
{
//KLUDGE!!!  We should not be referencing individual modules directly here...

	if (theRollLogRec->rollTransferLog)
		RTPAccessLogModule::RollTransferLog();
		
	if (theRollLogRec->rollErrorLog)
		RTSPLoggingModule::RollErrorLog();
	return SCNoError;
}

kern_return_t
RTSPSvrControlModule::GetProcessInfo(QTSProcessInfoRec* inProcessInfo)
{
	Assert(NULL != inProcessInfo);
	inProcessInfo->processID = getpid();
	inProcessInfo->startupTime = fStartupTime;
	return SCNoError;
}

kern_return_t RTSPSvrControlModule::GetServerName(QTSServerDNSName* outServerName)
{
	if (RTSPServerInterface::GetDefaultDNSName() != NULL)
		::strncpy(outServerName->dnsName, RTSPServerInterface::GetDefaultDNSName()->Ptr, kDNSNameSize);
	outServerName->dnsName[kDNSNameSize-1] = '\0';
	return SCNoError;
}

kern_return_t RTSPSvrControlModule::GetServerVersion(QTSServerVersionRec* outServerVersion)
{
	Assert(outServerVersion != NULL);

        //public vershun of the server: 0x00010002 = 1.0.2 
	outServerVersion->serverVersion = 0x00010002;
	outServerVersion->serverControlAPIVersion = 0x00010000;
	return SCNoError;
}

kern_return_t RTSPSvrControlModule::GetServerStatusRec(QTSServerStatusRec* outServerStatus)
{
	Assert(outServerStatus != NULL);
	::memset(outServerStatus, 0, sizeof(QTSServerStatusRec));
	
	//Convert the RTPServerInterface state to the server control's state
	if (fGracefulShutdownInProgress)
		outServerStatus->serverState = kSCGoingToShutDown;
	else if (RTPServerInterface::GetServerState() == RTPServerInterface::kRefusingConnectionsState)
		outServerStatus->serverState = kSCRefusingConnections;
	else if (RTPServerInterface::GetServerState() == RTPServerInterface::kStartingUpState)
		outServerStatus->serverState = kSCStartingUp;
	else if (RTPServerInterface::GetServerState() == RTPServerInterface::kShuttingDownState)
		outServerStatus->serverState = kSCShuttingDown;
	else
		outServerStatus->serverState = kSCRunning;
	
	//get the 4 key stats out of the server interface
	outServerStatus->numCurrentConnections = RTPServerInterface::GetRTPSessionMap()->GetNumRefsInTable();
	outServerStatus->connectionsSinceStartup = RTPServerInterface::GetTotalSessions();
	outServerStatus->currentBandwidth = RTPServerInterface::GetCurrentBandwidthInBits();
	outServerStatus->bytesSinceStartup = RTPServerInterface::GetTotalBytes();
	
	return SCNoError;
}

kern_return_t RTSPSvrControlModule::GetRefuseConnections(QTSRefuseConnectionsRec* outRefuseConnections)
{
	if (RTPServerInterface::GetServerState() == RTPServerInterface::kRefusingConnectionsState)
		outRefuseConnections->refuseConnections = true;
	else
		outRefuseConnections->refuseConnections = false;
	return SCNoError;
}

kern_return_t RTSPSvrControlModule::SetRefuseConnections(QTSRefuseConnectionsRec* inRefuseConnections)
{
	if (inRefuseConnections->refuseConnections != 0)
		RTPServerInterface::SetServerState(RTPServerInterface::kRefusingConnectionsState);
	//make sure not to allow people to stop the server from refusing connections when it
	//is in the process of a graceful shutdown. The only way to stop this is to call
	//CancelStopServer
	else if (fGracefulShutdownInProgress)
		return SCServerShuttingDown;
	else
		RTPServerInterface::SetServerState(RTPServerInterface::kRunningState);
	return SCNoError;
}

kern_return_t RTSPSvrControlModule::GetHistory(QTSServerHistoryRec* outHistory)
{
	OSMutexLocker locker(&fHistoryMutex);
	Assert(outHistory != NULL);
	//copy the current state of the array into this parameter

#if DEBUG
	//if we haven't filled up the array yet, make sure that the cursor is
	//one ahead of the size
	if (fHistoryArray.numEntries < kQTSHistoryArraySize)
		Assert(fCursor == (UInt32)fHistoryArray.numEntries);
#endif
	
	UInt32 theDestinationIndex = 0;//marker for where to write into the array next
	
	//if we have filled up the array, we will have to do 2 separate copies to fill outHistory.
	//Start by copying from fCursor to kMaxArraySize
	if (fHistoryArray.numEntries == kQTSHistoryArraySize)
	{
		::memcpy(outHistory->historyArray, &fHistoryArray.historyArray[fCursor],
					(kQTSHistoryArraySize - fCursor) * sizeof(QTSHistoryEntryRec));
		theDestinationIndex += kQTSHistoryArraySize - fCursor;
	}
	
	//There is ALWAYS valid data between 0 -> fCursor - 1. So copy that now
	::memcpy(&outHistory->historyArray[theDestinationIndex], fHistoryArray.historyArray,
				fCursor * sizeof(QTSHistoryEntryRec));

#if DEBUG
	//we should always write out the same number of entries as is in the fHistoryArray!
	theDestinationIndex += fCursor;
	Assert(theDestinationIndex == (UInt32)fHistoryArray.numEntries);
#endif

	//ok, now set the size of the output array
	outHistory->numEntries = fHistoryArray.numEntries;
	outHistory->entryInterval = fHistoryIntervalInSeconds;

	return SCNoError;
}

void RTSPSvrControlModule::AddHistorySample()
{
	OSMutexLocker locker(&fHistoryMutex);
	
	SInt32 theCurrentBandwidth = (SInt32)RTPServerInterface::GetCurrentBandwidthInBits();
	SInt32 theCurrentSessions = (SInt32)RTPServerInterface::GetRTPSessionMap()->GetNumRefsInTable();
	
	//keep track of maximums.
	if (theCurrentBandwidth > fBandwidthHi)
		fBandwidthHi = theCurrentBandwidth;
	if (theCurrentSessions > fConnectionHi)
		fConnectionHi = theCurrentSessions;
		
	//keep track of minimums.
	if ((theCurrentBandwidth < fBandwidthLo) || (fBandwidthLo == -1))
		fBandwidthLo = theCurrentBandwidth;
	if ((theCurrentSessions < fConnectionLo) || (fConnectionLo == -1))
		fConnectionLo = theCurrentSessions;
		
	//keep track of sum for eventual average
	//fBandwidthAvg += theCurrentBandwidth;		<---this was overflowing at high bitrates
	//fConnectionAvg += theCurrentSessions;
	
	// fBandwidthAvg was overflowing, 
	// so now we do it the ugly way
	fBandwidthAvg =(theCurrentBandwidth+(fBandwidthAvg*fSampleIndex))/(fSampleIndex+1);
	fConnectionAvg =(theCurrentSessions+(fConnectionAvg*fSampleIndex))/(fSampleIndex+1);
		
	fSampleIndex++;
}

void RTSPSvrControlModule::UpdateHistoryArray()
{
	OSMutexLocker locker(&fHistoryMutex);

	if (fSampleIndex == 0)
	{
		Assert(false);
		return;
	}
	
	//figure out min, max, average over this period
	fHistoryArray.historyArray[fCursor].bandwidthAvg = fBandwidthAvg;
	fHistoryArray.historyArray[fCursor].numClientsAvg = fConnectionAvg;
	
	fHistoryArray.historyArray[fCursor].bandwidthHi = fBandwidthHi;
	fHistoryArray.historyArray[fCursor].numClientsHi = fConnectionHi;

	fHistoryArray.historyArray[fCursor].bandwidthLo = fBandwidthLo;
	fHistoryArray.historyArray[fCursor].numClientsLo = fConnectionLo;
	
	//ok, increment the cursor for the next write (make sure to reset it if we've hit
	//the array boundary)
	fCursor++;
	if (fCursor == kQTSHistoryArraySize)
		fCursor = 0;
	
	//also update the array size. This only increments until the array is full, of course.
	if (fHistoryArray.numEntries < kQTSHistoryArraySize)
		fHistoryArray.numEntries++;
		
	//reset the sample index & related variables. We're moving onto a new entry now.
	fBandwidthLo = -1;
	fConnectionLo = -1;
	fBandwidthHi = 0;
	fConnectionHi = 0;
	fBandwidthAvg = 0;
	fConnectionAvg = 0;

	fSampleIndex = 0;
}

#pragma mark __RTSP_SERVER_CONTROL_THREAD__

RTSPSvrControlThread::RTSPSvrControlThread()
:	fMessagePort(0),
	fDone(false), fErrorOccurred(false), fDoneStartingUp(false), fThreadsAllocated(false)
{
	kern_return_t r;
	
	r = ::port_allocate(task_self(), &fMessagePort);
	if (r != SCNoError)
	{
		RTSPModuleInterface::LogError(	RTSPModule::kFatal, RTSPMessages::kCantAllocateMachPort, 0);
		fErrorOccurred = true;
		fDoneStartingUp = true;
		return;
	}
	
	for (int x = 0; x < 5; x++)
	{
		r = ::bootstrap_register(bootstrap_port, "QuickTimeStreamingServer", fMessagePort);
		//sometimes when restarting the server right after the server has gone away,
		//this can fail... so let's retry a couple of times
		if (r != SCNoError)
			thread_switch(THREAD_NULL, SWITCH_OPTION_WAIT, 1000);
		else	
			break;			
	}

	if (r != SCNoError)
	{
		RTSPModuleInterface::LogError(	RTSPModule::kFatal, RTSPMessages::kCantRegisterMachPort, 0);
		fErrorOccurred = true;
		fDoneStartingUp = true;
		return;
	}
	
	//I'm just assuming this always succeeds cause the mach documentation doesn't say
	//anything about it failing!
	fThreadID = ::cthread_fork((cthread_fn_t)_Entry, (any_t)this);
	fHistoryThreadID = ::cthread_fork((cthread_fn_t)_HistoryEntry, (any_t)this);
	fThreadsAllocated = true;
	
	while (!fDoneStartingUp)
		::cthread_yield();
}

RTSPSvrControlThread::~RTSPSvrControlThread()
{
	fDone = true;
	port_deallocate(task_self(), fMessagePort);//force SC thread to wakeup
	fMessagePort = 0;
	//wait for thread to terminate... these mach prototypes are very strange...
	//why, for instance, does thread_resume take an INT????
	if (fThreadsAllocated)
	{
		thread_resume((unsigned int)fThreadID);//force a wakeup.	
		cthread_join(fThreadID);
		thread_resume((unsigned int)fHistoryThreadID);	
		cthread_join(fHistoryThreadID);
	}
}

void RTSPSvrControlThread::_Entry(RTSPSvrControlThread *thread)  //static
{
	thread->Entry();
}
void RTSPSvrControlThread::_HistoryEntry(RTSPSvrControlThread *thread)  //static
{
	thread->HistoryEntry();
}

void RTSPSvrControlThread::HistoryEntry()
{
	//compute how often to run this thread.
	RTSPSvrControlModule* theModule = RTSPSvrControlModule::GetModule();
	Assert(theModule != NULL);
	
	UInt32 theSampleInterval = (RTSPServerInterface::GetRTSPPrefs()->GetHistoryUpdateIntervaInSecs() * 1000) /
									RTSPSvrControlModule::kNumSamplesPerEntry;
	theModule->fHistoryIntervalInSeconds = RTSPServerInterface::GetRTSPPrefs()->GetHistoryUpdateIntervaInSecs();
	UInt32 theEntryInterval = theModule->fHistoryIntervalInSeconds * 1000;
	
	//use local time to figure out when we need move onto a new entry. This
	//will eliminate the possibility that we drift off time.
	
	SInt64 theStartTime = OS::Milliseconds();
	
	while (!fDone)
	{
		//sleep for the kHistoryUpdateInterval
		//kHistoryUpdateInterval is in minutes. Convert to msec.
		thread_switch(THREAD_NULL, SWITCH_OPTION_WAIT, theSampleInterval);
		
		//if server is doing a graceful shutdown, this thread is used to periodically
		//poll, checking if all connections are complete
		theModule->CheckShutdown();
		
		//every time we wake up, first thing we want to do is sample the
		//current state of the server for the history
		theModule->AddHistorySample();
	
		SInt64 theCurrentTime = OS::Milliseconds();
		if ((theCurrentTime - theStartTime) > theEntryInterval)
		{
			theModule->UpdateHistoryArray();
			theStartTime += theEntryInterval;
		}
	}
}

void RTSPSvrControlThread::Entry()
{
	kern_return_t r;
		
	msg_header_t* msg = (msg_header_t*)new char[100];
	msg_header_t* reply = (msg_header_t*)new char[1000];	
	//signal 
	fDoneStartingUp = true;

	while(!fDone)
	{
		msg->msg_local_port = fMessagePort;
		msg->msg_size = 100;
		
		r = msg_receive(msg, MSG_OPTION_NONE, 0);
		
		if (r == RCV_INVALID_PORT)
			break;	//break because there's no more port to receive from
		if (r != SCNoError)
		{
			RTSPModuleInterface::LogError(	RTSPModule::kFatal, RTSPMessages::kServerControlFatalErr, 0);
			
			//attempt to stop the server
			RTSPSvrControlModule::GetModule()->StopServer(0);
			break;
		}

		QTSCRPI_server(msg, reply);
		
		reply->msg_local_port = fMessagePort;
		r = msg_send(reply, MSG_OPTION_NONE, 0);
	}

	if (fMessagePort != 0)
		port_deallocate(task_self(), fMessagePort);
}


#pragma mark __RTSP_SERVER_CONTROL_RPC__

//The following are the "modern" server control RPCs that map directly to server
//control interface calls.

kern_return_t _SCRPIStopServer(port_t /*server*/, int numMinutes)
{
	return RTSPSvrControlModule::GetModule()->StopServer(numMinutes);
}

kern_return_t _SCRPICancelStopServer(port_t /*server*/)
{
	return RTSPSvrControlModule::GetModule()->CancelStopServer();
}


kern_return_t _SCRPIGetServerAttribute(	port_t /*server*/, AttributeType attr, int bufSize,
										AttributeValue* buffer, unsigned int* actualSize)
{
	return RTSPSvrControlModule::GetModule()->
		GetServerAttribute(attr, 0, NULL, (unsigned int)bufSize, actualSize, buffer);
}

kern_return_t _SCRPISetServerAttribute(	port_t /*server*/, AttributeType attr,
										AttributeValue buffer, unsigned int size)
{
	return RTSPSvrControlModule::GetModule()->
		SetServerAttribute(attr, 0, NULL, size, buffer);
}

