/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
// $Id: QTRTPFile.cpp,v 1.37 1999/06/15 23:18:17 serenyi Exp $
//
// QTRTPFile:
//   An interface to QTFile for TimeShare.


// -------------------------------------
// Includes
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "OSMutex.h"

#include "QTFile.h"

#include "QTTrack.h"
#include "QTHintTrack.h"

#include "QTRTPFile.h"



// -------------------------------------
// Macros
//
#define DEBUG_PRINT(s) if(fDebug) printf s
#define DEEP_DEBUG_PRINT(s) if(fDeepDebug) printf s



// -------------------------------------
// Protected cache functions and variables.
//
OSMutex							*QTRTPFile::gFileCacheMutex,
								*QTRTPFile::gFileCacheAddMutex;
QTRTPFile::RTPFileCacheEntry	*QTRTPFile::gFirstFileCacheEntry = NULL;

void QTRTPFile::Initialize(void)
{
	QTRTPFile::gFileCacheMutex = NEW('RTcx', OSMutex, ('FLCH'));
	QTRTPFile::gFileCacheAddMutex = NEW('RTax', OSMutex, ('FLAD'));
}


QTRTPFile::ErrorCode QTRTPFile::new_QTFile(const char * FilePath, QTFile ** File, bool Debug, bool DeepDebug)
{
	// Temporary vars
	QTFile::ErrorCode	rcFile;

	// General vars
	OSMutexLocker	FileCacheAddMutex(QTRTPFile::gFileCacheAddMutex);

	QTRTPFile::RTPFileCacheEntry	*FileCacheEntry;
	
		
	//
	// Find and return the QTFile object out of our cache, if it exists.
	if( QTRTPFile::FindAndRefcountFileCacheEntry(FilePath, &FileCacheEntry) ) {
		FileCacheAddMutex.Unlock();
		FileCacheEntry->InitMutex->Lock();	// Guaranteed to block as the mutex
											// is acquired before it is added to
											// the list.
		FileCacheEntry->InitMutex->Unlock();// Because we don't actually need it.
		*File = FileCacheEntry->File;
		return errNoError;
	}


	//
	// Construct our file object.
	*File = NEW('QTFl', QTFile, (Debug, DeepDebug));
	if( *File == NULL )
		return errInternalError;
		
	//
	// Open the specified movie.
	if( (rcFile = (*File)->Open(FilePath)) != QTFile::errNoError ) {
		delete *File;
		switch( rcFile ) {
			case errFileNotFound: return errFileNotFound;
			case errInvalidQuickTimeFile: return errInvalidQuickTimeFile;
			default: return errInternalError;
		}
	}
	

	//
	// Add this file to our cache and release the global add mutex.
	QTRTPFile::AddFileToCache(FilePath, &FileCacheEntry); // Grabs InitMutex.
	FileCacheAddMutex.Unlock();


	//
	// Finish setting up the FileCacheEntry.
	if( FileCacheEntry != NULL ) {	// it may not have been cached..
		FileCacheEntry->File = *File;
		FileCacheEntry->InitMutex->Unlock();
	}
	

	//
	// Return the file object.
	return errNoError;
}

void QTRTPFile::delete_QTFile(QTFile * File)
{

	// General vars
	OSMutexLocker	FileCacheMutex(QTRTPFile::gFileCacheMutex);
	QTRTPFile::RTPFileCacheEntry	*ListEntry;


	//
	// Find the specified cache entry.
	for( ListEntry = QTRTPFile::gFirstFileCacheEntry; ListEntry != NULL; ListEntry = ListEntry->NextEntry ) {
		//
		// Check for matches.
		if( ListEntry->File != File )
			continue;
			
		//
		// Delete the object if the reference count has dropped to zero.
		if( --ListEntry->ReferenceCount == 0 ) {
			//
			// Delete the file.
			if( File != NULL )
				delete File;

			//
			// Free our other vars.
			if( ListEntry->InitMutex != NULL )
				delete ListEntry->InitMutex;

			if( ListEntry->fFilename != NULL )
			    delete [] ListEntry->fFilename;
			
			//
			// Remove this entry from the list.
			if( ListEntry->PrevEntry != NULL )
				ListEntry->PrevEntry->NextEntry = ListEntry->NextEntry;
			if( ListEntry->NextEntry != NULL )
				ListEntry->NextEntry->PrevEntry = ListEntry->PrevEntry;
			if( QTRTPFile::gFirstFileCacheEntry == ListEntry )
				QTRTPFile::gFirstFileCacheEntry = ListEntry->NextEntry;
			delete ListEntry;
		}

		//
		// Return.
		return;
	}

	//
	// The object was not in the cache.  Delete it
	if( File != NULL )
		delete File;
}


void QTRTPFile::AddFileToCache(const char *inFilename, QTRTPFile::RTPFileCacheEntry ** NewListEntry)
{
	// General vars
	OSMutexLocker	FileCacheMutex(QTRTPFile::gFileCacheMutex);
	QTRTPFile::RTPFileCacheEntry	*ListEntry, *LastListEntry;
	
	
	//
	// Add this track object to our track list.
	(*NewListEntry) = NEW('Rcle', QTRTPFile::RTPFileCacheEntry, ());
	if( (*NewListEntry) == NULL )
		return;

	(*NewListEntry)->InitMutex = NEW('RTix', OSMutex, ('INMT'));
	if( (*NewListEntry)->InitMutex == NULL ) {
		delete (*NewListEntry);
		*NewListEntry = NULL;
		return;
	}
	(*NewListEntry)->InitMutex->Lock();
	
	(*NewListEntry)->fFilename = NEW_ARRAY('FNme', char, (::strlen(inFilename) + 2));
	::strcpy((*NewListEntry)->fFilename, inFilename);
	(*NewListEntry)->File = NULL;
	
	(*NewListEntry)->ReferenceCount = 1;

	(*NewListEntry)->PrevEntry = NULL;
	(*NewListEntry)->NextEntry = NULL;

	//
	// Make this the first entry if there are no entries, otherwise we need to
	// find out where this file fits in the list and insert it there.
	if( QTRTPFile::gFirstFileCacheEntry == NULL ) {
		QTRTPFile::gFirstFileCacheEntry = (*NewListEntry);
	} else {
		//
		// Go through the cache list until we find an inode number greater than
		// the one that we have now.  Insert it in the list when we find this.
		for( ListEntry = LastListEntry = QTRTPFile::gFirstFileCacheEntry; ListEntry != NULL; ListEntry = ListEntry->NextEntry ) {
			//
			// This is the last list entry that we saw (useful for later).
			LastListEntry = ListEntry;
			
			//
			// Skip this entry if this inode number is smaller than the one
			// for our new entry.
			if( strcmp(ListEntry->fFilename,inFilename) < 0 )
				continue;
			
			//
			// We've found a larger inode; insert this one in the list.
			if( ListEntry->PrevEntry == NULL )
				QTRTPFile::gFirstFileCacheEntry = (*NewListEntry);
			else
				ListEntry->PrevEntry->NextEntry = (*NewListEntry);
			(*NewListEntry)->PrevEntry = ListEntry->PrevEntry;
			ListEntry->PrevEntry = (*NewListEntry);
			
			(*NewListEntry)->NextEntry = ListEntry;
			
			return;
		}
		
		//
		// We fell out of our loop; this means that we are the largest inode
		// in the list; add ourselves to the end of the list.
		if( LastListEntry == NULL ) { // this can't happen, but..
			QTRTPFile::gFirstFileCacheEntry = (*NewListEntry);
		} else {
			LastListEntry->NextEntry = (*NewListEntry);
			(*NewListEntry)->PrevEntry = LastListEntry;
		}
	}
}

bool QTRTPFile::FindAndRefcountFileCacheEntry(const char *inFilename, QTRTPFile::RTPFileCacheEntry **CacheEntry)
{
	// General vars
	OSMutexLocker	FileCacheMutex(QTRTPFile::gFileCacheMutex);
	QTRTPFile::RTPFileCacheEntry	*ListEntry;


	//
	// Find the specified cache entry.
	for( ListEntry = QTRTPFile::gFirstFileCacheEntry; ListEntry != NULL; ListEntry = ListEntry->NextEntry ) {
		//
		// Check for matches.
	    if(strcmp(ListEntry->fFilename, inFilename) != 0)
			continue;

		//
		// Update the reference count and set the return value.
		ListEntry->ReferenceCount++;
		*CacheEntry = ListEntry;
		
		//
		// Return.
		return true;
	}

	//
	// The search failed.
	return false;
}



// -------------------------------------
// Constructors and destructors
//
QTRTPFile::QTRTPFile(bool Debug, bool DeepDebug)
	: fDebug(Debug), fDeepDebug(DeepDebug),
	  fFile(NULL), fFCB(NULL),
	  fNumHintTracks(0),
	  fFirstTrack(NULL), fLastTrack(NULL),
	  fSDPFile(NULL), fSDPFileLength(0),
	  fRequestedSeekTime(0.0), fSeekTime(0.0),
	  fLastPacketTrack(NULL),
	  fBytesPerSecond(0)
{
	fFCB = NEW('Ffcb', QTFile_FileControlBlock, ());
}

QTRTPFile::~QTRTPFile(void)
{
	//
	// Free our track list (and the associated tracks)
	RTPTrackListEntry *TrackEntry = fFirstTrack,
					  *NextTrackEntry = TrackEntry ? TrackEntry->NextTrack : NULL;
	while( TrackEntry != NULL ) {
		//
		// Delete this track entry and move to the next one.
		if( TrackEntry->HTCB != NULL )
			delete TrackEntry->HTCB;
		delete TrackEntry;
		
		TrackEntry = NextTrackEntry;
		if( TrackEntry != NULL )
			NextTrackEntry = TrackEntry->NextTrack;
	}

	//
	// Free our variables
	if( fSDPFile != NULL )
		delete[] fSDPFile;
	
	delete_QTFile(fFile);

	if( fFCB != NULL )
		delete fFCB;
}



// -------------------------------------
// Initialization functions.
//
QTRTPFile::ErrorCode QTRTPFile::Initialize(const char * FilePath)
{
	// Temporary vars
	QTRTPFile::ErrorCode	rc;

	// General vars
	QTTrack		*Track;
	
	
	//
	// Create our file object.
	rc = new_QTFile(FilePath, &fFile, fDebug, fDeepDebug);
	if( rc != errNoError ) {
		fFile = NULL;
		return rc;
	}


	//
	// Iterate through all of the tracks, adding hint tracks to our list.
	for( Track = NULL; fFile->NextTrack(&Track, Track); ) {
		// General vars
		QTHintTrack			*HintTrack;
		RTPTrackListEntry	*ListEntry;


		//
		// Skip over anything that's *not* a hint track.
		if( !fFile->IsHintTrack(Track) )
			continue;
		HintTrack = (QTHintTrack *)Track;
		
		//
		// Add this track object to our track list.
		ListEntry = NEW('Rtle', RTPTrackListEntry, ());
		if( ListEntry == NULL )
			return errInternalError;

		ListEntry->TrackID = Track->GetTrackID();
		ListEntry->HintTrack = HintTrack;
		ListEntry->HTCB = NEW('Hfcb', QTHintTrack_HintTrackControlBlock, (fFCB));
		ListEntry->IsTrackActive = false;
		ListEntry->IsPacketAvailable = false;
		ListEntry->QualityLevel = kAllPackets;
		
		ListEntry->Cookie = NULL;
		ListEntry->SSRC = 0;

		ListEntry->BaseSequenceNumberRandomOffset = 0;
		ListEntry->FileSequenceNumberRandomOffset = 0;
		ListEntry->LastSequenceNumber = 0;
		ListEntry->SequenceNumberAdditive = 0;

		ListEntry->BaseTimestampRandomOffset = 0;
		ListEntry->FileTimestampRandomOffset = 0;
		
		ListEntry->CurSampleNumber = 0;
		ListEntry->NumPacketsInThisSample = 0;
		ListEntry->CurPacketNumber = 0;
		
		ListEntry->CurPacketTime = 0.0;
		ListEntry->CurPacketLength = 0;

		ListEntry->NextTrack = NULL;

		if( fFirstTrack == NULL ) {
			fFirstTrack = fLastTrack = ListEntry;
		} else {
			fLastTrack->NextTrack = ListEntry;
			fLastTrack = ListEntry;
		}
		
		// One more track..
		fNumHintTracks++;
	}
	
	// If there aren't any hint tracks, there's no way we can stream this movie,
	// so notify the caller
	if (fNumHintTracks == 0)
		return errNoHintTracks;
		
	// The RTP file has been initialized.
	return errNoError;
}



// -------------------------------------
// Accessors
//
Float64 QTRTPFile::GetMovieDuration(void)
{
	return fFile->GetDurationInSeconds();
}

UInt64 QTRTPFile::GetAddedTracksRTPBytes(void)
{
	// Temporary vars
	RTPTrackListEntry	*CurEntry;
	
	// General vars
	UInt64		TotalRTPBytes = 0;
	
	
	//
	// Go through all of the tracks, adding up the size of the RTP bytes
	// for all active tracks.
	for( CurEntry = fFirstTrack;
		 CurEntry != NULL;
		 CurEntry = CurEntry->NextTrack
	) {
		//
		// We only want active tracks.
		if( !CurEntry->IsTrackActive )
			continue;

		//
		// Add this length to our count.
		TotalRTPBytes += CurEntry->HintTrack->GetTotalRTPBytes();
	}
	
	//
	// Return the size.
	return TotalRTPBytes;
}

char * QTRTPFile::GetSDPFile(int * SDPFileLength)
{
	// Temporary vars
	RTPTrackListEntry	*CurEntry;
	UInt32		tempAtomType;

	// General vars
	QTFile::AtomTOCEntry *globalSDPTOCEntry;
	bool		haveGlobalSDPAtom = false;
	
	char		sdpRangeLine[255];
	char		*pSDPFile;
	
	
	//
	// Return our cached SDP file if we have one.
	if( fSDPFile != NULL ) {
		*SDPFileLength = fSDPFileLength;
		return fSDPFile;
	}
	

	//
	// Build our range header.
	sprintf(sdpRangeLine, "a=range:npt=0-%10.5f\r\n", GetMovieDuration());


	//
	// Figure out how long the SDP file is going to be.
	fSDPFileLength = strlen(sdpRangeLine);
	for( CurEntry = fFirstTrack;
		 CurEntry != NULL;
		 CurEntry = CurEntry->NextTrack
	) {
		// Temporary vars
		int			TrackSDPLength;


		//
		// Get the length of this track's SDP file.
		if( CurEntry->HintTrack->GetSDPFileLength(&TrackSDPLength) != QTTrack::errNoError )
			continue;
		
		//
		// Add it to our count.
		fSDPFileLength += TrackSDPLength;
	}

	//
	// See if this movie has a global SDP atom.
	if( fFile->FindTOCEntry("moov:udta:hnti:rtp ", &globalSDPTOCEntry, NULL) ) {
		//
		// Verify that this is an SDP atom.
		fFile->Read(globalSDPTOCEntry->AtomDataPos, (char *)&tempAtomType, 4);
		if( ntohl(tempAtomType) == 'sdp ' ) {
			haveGlobalSDPAtom = true;
			fSDPFileLength += globalSDPTOCEntry->AtomDataLength - 4;
		}
	}

	//
	// Build the SDP file.
	fSDPFile = pSDPFile = NEW_ARRAY('Qsdp', char, fSDPFileLength);
	if( fSDPFile == NULL )
		return NULL;
	
	if( haveGlobalSDPAtom ) {
		fFile->Read(globalSDPTOCEntry->AtomDataPos + 4, pSDPFile, globalSDPTOCEntry->AtomDataLength - 4);
		pSDPFile += globalSDPTOCEntry->AtomDataLength - 4;
	}
	
	memcpy(pSDPFile, sdpRangeLine, strlen(sdpRangeLine));
	pSDPFile += strlen(sdpRangeLine);
	
	for( CurEntry = fFirstTrack;
		 CurEntry != NULL;
		 CurEntry = CurEntry->NextTrack
	) {
		// Temporary vars
		char		*TrackSDP;
		int			TrackSDPLength;
		
		
		//
		// Get this track's SDP file and add it to our buffer.
		TrackSDP = CurEntry->HintTrack->GetSDPFile(&TrackSDPLength);
		if( TrackSDP == NULL )
			continue;
		
		memcpy(pSDPFile, TrackSDP, TrackSDPLength);
		delete [] TrackSDP;//ARGH! GetSDPFile allocates the pointer that is being returned.
		pSDPFile += TrackSDPLength;
	}
	
	
	//
	// Return the (cached) SDP file.
	*SDPFileLength = fSDPFileLength;
	return fSDPFile;
}



// -------------------------------------
// Track functions
//
QTRTPFile::ErrorCode QTRTPFile::AddTrack(UInt32 TrackID, bool UseRandomOffset)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	
	
	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return errTrackIDNotFound;

	{
		OSMutexLocker locker(fFile->GetMutex());
		//
		// Initialize this track.
		if( TrackEntry->HintTrack->Initialize() != QTTrack::errNoError )
			return errInternalError;
	}
	
	//
	// Set up the sequence number and timestamp offsets.
	if( UseRandomOffset ) {
		TrackEntry->BaseSequenceNumberRandomOffset = (UInt16)rand();
		TrackEntry->BaseTimestampRandomOffset = (UInt32)rand();
	} else {
		TrackEntry->BaseSequenceNumberRandomOffset = 0;
		TrackEntry->BaseTimestampRandomOffset = 0;
	}

	TrackEntry->FileSequenceNumberRandomOffset = TrackEntry->HintTrack->GetRTPSequenceNumberRandomOffset();
	TrackEntry->FileTimestampRandomOffset = TrackEntry->HintTrack->GetRTPTimestampRandomOffset();
	
	TrackEntry->LastSequenceNumber = 0;
	TrackEntry->SequenceNumberAdditive = 0;

	//
	// This track is now active.
	TrackEntry->IsTrackActive = true;
	
	//
	// The track has been added.
	return errNoError;
}

Float64 QTRTPFile::GetTrackDuration(UInt32 TrackID)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	
	
	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return 0;
	
	//
	// Return the duration.
	return TrackEntry->HintTrack->GetDurationInSeconds();
}

UInt32 QTRTPFile::GetTrackTimeScale(UInt32 TrackID)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	
	
	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return 0;
	
	//
	// Return the duration.
	return (UInt32)TrackEntry->HintTrack->GetTimeScale();
}

void QTRTPFile::SetTrackSSRC(UInt32 TrackID, UInt32 SSRC)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	
	
	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return;
	
	//
	// Set the SSRC.
	TrackEntry->SSRC = SSRC;
}

void QTRTPFile::SetTrackCookie(UInt32 TrackID, void * Cookie)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	
	
	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return;
	
	//
	// Set the cookie.
	TrackEntry->Cookie = Cookie;
}

void QTRTPFile::SetTrackQualityLevel(UInt32 TrackID, UInt32 inNewQuality)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	
	
	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return;
	
	//
	// Set the option.
	TrackEntry->QualityLevel = inNewQuality;
}



// -------------------------------------
// Packet functions
//
QTRTPFile::ErrorCode QTRTPFile::Seek(Float64 Time, Float64 MaxBackupTime)
{
	// General vars
	RTPTrackListEntry	*ListEntry;
	UInt32				BytesPerSecond;

	Float64				SyncToTime = Time;


	//
	// Adjust the size of our QTFile buffer to reflect the current bitrate of
	// the movie.
	BytesPerSecond = 0;
	for( ListEntry = fFirstTrack; ListEntry != NULL; ListEntry = ListEntry->NextTrack ) {
		//
		// Skip in active tracks.
		if( !ListEntry->IsTrackActive )
			continue;
		
		//
		// Calculate the average bytes/second for this track.
		BytesPerSecond += (UInt32)(ListEntry->HintTrack->GetTotalRTPBytes() / fFile->GetDurationInSeconds());
	}
	if( BytesPerSecond != fBytesPerSecond )
		fFCB->AdjustDataBuffer((fBytesPerSecond = BytesPerSecond) * 8);

	//
	// Find the earliest sync sample and sync all of the tracks to that
	// sample.
	for( ListEntry = fFirstTrack; ListEntry != NULL; ListEntry = ListEntry->NextTrack ) {
		// General vars
		SInt32			MediaTime;
		UInt32			NewSampleNumber, NewSyncSampleNumber;
		UInt32			NewSampleMediaTime;
		Float64			NewSampleTime;
	
	
		//
		// Only consider active tracks.
		if( !ListEntry->IsTrackActive )
			continue;
		
		//
		// Compute the media time and get the sample at that time.
		MediaTime = (SInt32)(Time * ListEntry->HintTrack->GetTimeScale());
		MediaTime -= ListEntry->HintTrack->GetFirstEditMediaTime();
		if( MediaTime < 0 )
			MediaTime = 0;
		if( !ListEntry->HintTrack->GetSampleNumberFromMediaTime(MediaTime, &NewSampleNumber, ListEntry->HTCB->fsttsSTCB) )
			continue;	// This track is probably done playing.
		
		//
		// Find the nearest (moving backwards in time) keyframe.
		ListEntry->HintTrack->GetPreviousSyncSample(NewSampleNumber, &NewSyncSampleNumber);
		if( NewSampleNumber == NewSyncSampleNumber )
			continue;

		//
		// Figure out what time this sample is at.
		if( !ListEntry->HintTrack->GetSampleMediaTime(NewSyncSampleNumber, &NewSampleMediaTime) )
			return errInvalidQuickTimeFile;
		NewSampleMediaTime += ListEntry->HintTrack->GetFirstEditMediaTime();
		NewSampleTime = (Float64)NewSampleMediaTime * ListEntry->HintTrack->GetTimeScaleRecip();

		//
		// Figure out if this is the time that we need to sync to.
		if( NewSampleTime < SyncToTime )
			SyncToTime = NewSampleTime;
	}

	//
	// Evaluate/Store the seek time
	fRequestedSeekTime = Time;
	if( (Time - SyncToTime) <= MaxBackupTime )
		fSeekTime = SyncToTime;
	else
		fSeekTime = Time;

	//
	// Prefetch a packet for all active tracks.
	for( ListEntry = fFirstTrack; ListEntry != NULL; ListEntry = ListEntry->NextTrack ) {
		//
		// Only fetch packets on active tracks.
		if( !ListEntry->IsTrackActive )
			continue;
		
		//
		// Reset the sample table caches.
		ListEntry->HTCB->fstscSTCB->Reset();

		//
		// Prefetch a packet.
		ListEntry->IsPacketAvailable = false;
		if( !PrefetchNextPacket(ListEntry, true, fSeekTime) )
			continue;
		
		//
		// We've got a packet..
		ListEntry->IsPacketAvailable = true;
	}

	//
	// 'Forget' that we had a previous packet.
	fLastPacketTrack = NULL;
	
	return errNoError;
}

UInt32 QTRTPFile::GetSeekTimestamp(UInt32 TrackID)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	UInt32				MediaTime, RTPTimestamp;
	
	
	DEEP_DEBUG_PRINT(("Calculating RTP timestamp for track #%lu at time %.2f.\n", TrackID, fRequestedSeekTime));

	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return 0;
	
	//
	// Calculate the timestamp at this seek time.
	MediaTime = (UInt32)(fRequestedSeekTime * TrackEntry->HintTrack->GetTimeScale());
	if( TrackEntry->HintTrack->GetRTPTimescale() == TrackEntry->HintTrack->GetTimeScale() )
		RTPTimestamp = MediaTime;
	else
		RTPTimestamp = (UInt32)(MediaTime * (TrackEntry->HintTrack->GetRTPTimescale() * TrackEntry->HintTrack->GetTimeScaleRecip()) );
	
	//
	// Add the appropriate offsets.
	RTPTimestamp += TrackEntry->BaseTimestampRandomOffset + TrackEntry->FileTimestampRandomOffset;

	//
	// Return the RTP timestamp.
	DEEP_DEBUG_PRINT(("..timestamp=%lu\n", RTPTimestamp));
	return RTPTimestamp;
}

UInt16 QTRTPFile::GetNextTrackSequenceNumber(UInt32 TrackID)
{
	// General vars
	RTPTrackListEntry	*TrackEntry;
	UInt16				RTPSequenceNumber;
	
	
	//
	// Find this track.
	if( !FindTrackEntry(TrackID, &TrackEntry) )
		return 0;
	
	//
	// Read the sequence number right out of the packet.
	memcpy(&RTPSequenceNumber, (char *)TrackEntry->CurPacket + 2, 2);
	return ntohs(RTPSequenceNumber);
}

Float64 QTRTPFile::GetNextPacket(char ** Packet, int * PacketLength, void ** Cookie)
{
	// General vars
	RTPTrackListEntry	*ListEntry;

	bool				HaveFirstPacketTime = false;
	Float64				FirstPacketTime = 0.0;
	RTPTrackListEntry	*FirstPacket = NULL;


	//
	// Clear the input.
	*Packet = NULL;
	*PacketLength = 0;
	*Cookie = NULL;
	
	//
	// Prefetch the next packet of the track that the *last* packet came from.
	if( fLastPacketTrack != NULL ) {
		if( !PrefetchNextPacket(fLastPacketTrack) )
			fLastPacketTrack->IsPacketAvailable = false;
	}
	
	//
	// Figure out which track is going to produce the next packet.
	for( ListEntry = fFirstTrack; ListEntry != NULL; ListEntry = ListEntry->NextTrack ) {
		//
		// Only look at active tracks that have packets.
		if( !ListEntry->IsTrackActive || !ListEntry->IsPacketAvailable)
			continue;
		
		//
		// See if this track has a time earlier than our initial time.
		if( (ListEntry->CurPacketTime <= FirstPacketTime) || !HaveFirstPacketTime ) {
			HaveFirstPacketTime = true;
			FirstPacketTime = ListEntry->CurPacketTime;
			FirstPacket = ListEntry;
		}
	}
	
	//
	// Abort if we didn't find a packet.  Either the movie is over, or there
	// weren't any packets to begin with.
	if( FirstPacket == NULL )
		return 0.0;
	
	//
	// Remember the sequence number of this packet.
	FirstPacket->LastSequenceNumber = ntohs(*(UInt16 *)((char *)FirstPacket->CurPacket + 2));
	FirstPacket->LastSequenceNumber -= FirstPacket->BaseSequenceNumberRandomOffset + FirstPacket->FileSequenceNumberRandomOffset + FirstPacket->SequenceNumberAdditive;

	//
	// Return this packet.
	fLastPacketTrack = FirstPacket;
	
	*Packet = FirstPacket->CurPacket;
	*PacketLength = FirstPacket->CurPacketLength;
	*Cookie = FirstPacket->Cookie;
	
	return FirstPacket->CurPacketTime;
}



// -------------------------------------
// Protected member functions
//
bool QTRTPFile::FindTrackEntry(UInt32 TrackID, RTPTrackListEntry **TrackEntry)
{
	// General vars
	RTPTrackListEntry	*ListEntry;


	//
	// Find the specified track.
	for( ListEntry = fFirstTrack; ListEntry != NULL; ListEntry = ListEntry->NextTrack ) {
		//
		// Check for matches.
		if( ListEntry->TrackID == TrackID ) {
			*TrackEntry = ListEntry;
			return true;
		}
	}

	//
	// The search failed.
	return false;
}

bool QTRTPFile::PrefetchNextPacket(RTPTrackListEntry * TrackEntry, bool DoSeek, Float64 Time)
{
	// General vars
	UInt16			*pSequenceNumber;
	UInt32			*pTimestamp;


	//
	// Do a seek if requested.
	if( DoSeek ) {
		// General vars
		SInt32			MediaTime;
	
	
		//
		// Compute the media time and get the sample at that time.
		MediaTime = (SInt32)(Time * TrackEntry->HintTrack->GetTimeScale());
		MediaTime -= TrackEntry->HintTrack->GetFirstEditMediaTime();
		if( MediaTime < 0 )
			MediaTime = 0;
		if( !TrackEntry->HintTrack->GetSampleNumberFromMediaTime(MediaTime, &TrackEntry->CurSampleNumber, TrackEntry->HTCB->fsttsSTCB) )
			return false;
		
		//
		// Clear our current packet information.
		TrackEntry->NumPacketsInThisSample = 0;
		TrackEntry->CurPacketNumber = 0;
	}
	
	// Temporary vars
	QTTrack::ErrorCode	rcTrack = QTTrack::errIsBFrame;
	
	// If we are dropping b-frames, QTHintTrack::GetPacket will return the errIsBFrame error to us.
	// If we get that error, we should fetch another packet. So, we have this loop here.
	while (rcTrack == QTTrack::errIsBFrame)
	{
		//
		// Next packet (or first packet, since CurPacketNumber starts at 0 above)
		TrackEntry->CurPacketNumber++;

		//
		// Do we need to move to the next sample?
		if(    (TrackEntry->CurPacketNumber > TrackEntry->NumPacketsInThisSample)
			&& (TrackEntry->NumPacketsInThisSample != 0)
		) {
			//
			// If we're only reading sync samples, then we need to find the next
			// one to send out, otherwise just increment the sample number and
			// move on.
			if( TrackEntry->QualityLevel >= kKeyFramesOnly ) {
				//Use the quality level to determine how many key frames to skip over.
				for (UInt32 keyFramesSkipCount = (kKeyFramesOnly - 1); keyFramesSkipCount < TrackEntry->QualityLevel; keyFramesSkipCount++)
					TrackEntry->HintTrack->GetNextSyncSample(TrackEntry->CurSampleNumber, &TrackEntry->CurSampleNumber);
			} else {
				TrackEntry->CurSampleNumber++;
			}
			
			//
			// We'll need to recompute the number of samples in this packet.
			TrackEntry->NumPacketsInThisSample = 0;
		}

		//
		// Do we know how many packets are in this sample?  If not, figure it out.
		while( TrackEntry->NumPacketsInThisSample == 0 ) {
			if( TrackEntry->HintTrack->GetNumPackets(TrackEntry->CurSampleNumber, &TrackEntry->NumPacketsInThisSample, TrackEntry->HTCB) != QTTrack::errNoError )
				return false;
			if( TrackEntry->NumPacketsInThisSample == 0 )
				TrackEntry->CurSampleNumber++;
			TrackEntry->CurPacketNumber = 1;
		}
		
		
		//
		// Fetch this packet.
		TrackEntry->CurPacketLength = QTRTPFILE_MAX_PACKET_LENGTH;
		rcTrack = TrackEntry->HintTrack->GetPacket(TrackEntry->CurSampleNumber, TrackEntry->CurPacketNumber,
												   TrackEntry->CurPacket, &TrackEntry->CurPacketLength,
												   &TrackEntry->CurPacketTime,
												   (TrackEntry->QualityLevel == kNoBFrames),
												   TrackEntry->SSRC,
												   TrackEntry->HTCB);
	}

	if( rcTrack != QTTrack::errNoError )
		return false;
		
	//
	// Update our sequence number and timestamp.  If we seeked to get here,
	// then we need to adjust the additive to account for the shift in
	// sequence numbers.
	pSequenceNumber = (UInt16 *)((char *)TrackEntry->CurPacket + 2);
	pTimestamp = (UInt32 *)((char *)TrackEntry->CurPacket + 4);
	
	if( DoSeek || (TrackEntry->QualityLevel > kAllPackets) )
		TrackEntry->SequenceNumberAdditive += (TrackEntry->LastSequenceNumber + 1) - ntohs(*pSequenceNumber);
	
	*pSequenceNumber = htons(ntohs(*pSequenceNumber) + TrackEntry->BaseSequenceNumberRandomOffset + TrackEntry->FileSequenceNumberRandomOffset + TrackEntry->SequenceNumberAdditive);
	*pTimestamp = htonl(ntohl(*pTimestamp) + TrackEntry->BaseTimestampRandomOffset + TrackEntry->FileTimestampRandomOffset);
	
	//
	// Return the packet.
	return true;
}
