/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		osfile.cpp

	Contains:	simple file abstraction
					
	$Log: OSFileSource.cpp,v $
	Revision 1.1  1999/06/02 18:16:35  serenyi
	Created
	
	Revision 1.1  1999/05/27 00:21:52  serenyi
	
	Revision 1.2  1999/02/19 23:06:12  ds
	Created
	
	
*/

#include <stdio.h>
#include <string.h>

#ifndef __MW_
#include <sys/stat.h>
#include <sys/types.h>
#if __MacOSX__
#include <mach/cthreads.h>
#endif
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#endif

#include "OSFileSource.h"

void OSFileSource::Set(const char *inPath)
{
	Close();
#if __MacOSX__
	fFile = open(inPath, O_RDONLY | O_NO_MFS);
#else
	fFile = open(inPath, O_RDONLY);
#endif

	if (fFile != -1)
	{
		struct stat buf;
		if (::fstat(fFile, &buf) >= 0)
			fLength = buf.st_size;
		else
			Close();
	}	
}

void OSFileSource::Advise(UInt64 advisePos, UInt32 adviseAmt)
{
#if __MacOSX__
	//if the caller wants us to, issue an advise. Advise for the file area
	//immediately following the current read area
	struct radvisory radv;
	memset(&radv, 0, sizeof(radv));
	radv.ra_offset = (off_t)advisePos;
	radv.ra_count = (int)adviseAmt;
	::fcntl(fFile, F_RDADVISE, &radv);
#endif
}

QTSS_ErrorCode	OSFileSource::Read(void* inBuffer, UInt32 inLength, UInt32* outRcvLen)
{
	if (!this->IsValid())
		return QTSS_RequestFailed;
	
	//optimization: avoid the extra kernel call. Only do an lseek if the file pointer really
	//doesn't match where we want to read from
	if (fPosition != fReadPos)
		if (lseek(fFile, fPosition, SEEK_SET) == -1)
			return QTSS_RequestFailed;
		
	int rcvLen = ::read(fFile, (char*)inBuffer, inLength);
	if (rcvLen == -1)
#if __MacOSX__
		return (QTSS_ErrorCode)cthread_errno();
#else
		return errno;
#endif

	if (outRcvLen != NULL)
		*outRcvLen = rcvLen;

	fPosition += rcvLen;
	fReadPos = fPosition;
	return QTSS_NoErr;
}

QTSS_ErrorCode	OSFileSource::Read(UInt64 inPosition, void* inBuffer, UInt32 inLength, UInt32* outRcvLen)
{
    Seek(inPosition);
    return Read(inBuffer,inLength,outRcvLen);
}


QTSS_ErrorCode	OSFileSource::Close()
{
	if (fFile != -1)
	{
		::close(fFile);
		fFile = -1;
	}

	fLength = 0;
	fPosition = 0;
	fReadPos = 0;
	return QTSS_NoErr;
}

