/* pipeline.c */
/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *        Copyright (c) 1990, ..., 1996 Bellcore            *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *   dfs@research.att.com       dicook@iastate.edu          *
 *      (973) 360-8423    www.public.iastate.edu/~dicook/   *
 *                                                          *
 *                    Andreas Buja                          *
 *                andreas@research.att.com                  *
 *              www.research.att.com/~andreas/              *
 *                                                          *
 ************************************************************/

#include <math.h>
#include <stdlib.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"
#include "xgobiexterns.h"

#define FUDGE_FACTOR (xg->numvars_t+1)*2/7

/* ------------ Dynamic allocation, freeing section --------- */

void
alloc_pipeline_arrays(xgobidata *xg)
/*
 * Dynamically allocate arrays.
*/
{
  Cardinal nc = (Cardinal) xg->ncols, nr = (Cardinal) xg->nrows;
  register Cardinal i;

  xg->sphered_data = (float **) XtMalloc(nr*sizeof(float *));
  for (i=0; i<nr; i++)
    xg->sphered_data[i] = (float *)XtMalloc(nc * sizeof(float));

  xg->tform_data = (float **) XtMalloc(nr * sizeof(float *));
  for (i=0; i<nr; i++)
    xg->tform_data[i] = (float *) XtMalloc(nc * sizeof(float));

  xg->world_data = (long **) XtMalloc(nr * sizeof(long *));
  for (i=0; i<nr; i++)
    xg->world_data[i] = (long *) XtMalloc(nc * sizeof(long));

  /* XtCalloc initializes to zero */
  xg->jitter_data = (long **) XtMalloc(nr * sizeof(long *));
  for (i=0; i<nr; i++)
    xg->jitter_data[i] = (long *) XtCalloc(nc, sizeof(long));

  xg->planar = (lcoords *) XtMalloc(nr * sizeof(lcoords));
  xg->screen = (icoords *) XtMalloc(nr * sizeof(icoords));

  xg->rows_in_plot = (int *) XtMalloc(
    (Cardinal) xg->nrows * sizeof(int));

  xg->lim0 = (lims *) XtMalloc(nc * sizeof(lims));
  xg->lim = (lims *) XtMalloc(nc * sizeof(lims));
  xg->lim_tform = (lims *) XtMalloc(nc * sizeof(lims));
}

void
free_pipeline_arrays(xgobidata *xg)
/*
 * Dynamically free arrays used in data pipeline.
*/
{
  int i;

  for (i=0; i<xg->nrows; i++)
    XtFree((XtPointer) xg->sphered_data[i]);
  XtFree((XtPointer) xg->sphered_data);

  for (i=0; i<xg->nrows; i++)
    XtFree((XtPointer) xg->tform_data[i]);
  XtFree((XtPointer) xg->tform_data);

  for (i=0; i<xg->nrows; i++)
    XtFree((XtPointer) xg->world_data[i]);
  XtFree((XtPointer) xg->world_data);

  for (i=0; i<xg->nrows; i++)
    XtFree((XtPointer) xg->jitter_data[i]);
  XtFree((XtPointer) xg->jitter_data);

  XtFree((XtPointer) xg->planar);
  XtFree((XtPointer) xg->screen);

  XtFree((XtPointer) xg->rows_in_plot);

  XtFree((XtPointer) xg->lim0);
  XtFree((XtPointer) xg->lim);
  XtFree((XtPointer) xg->lim_tform);
}

/* ------------ End of dynamic allocation section --------- */

/* ------------ Data pipeline section --------- */

void
copy_raw_to_tform(xgobidata *xg)
{
/*
 * Copy raw_data to tform_data.
*/
  int i, j;

  for (i=0; i<xg->nrows; i++)
    for (j=0; j<xg->ncols_used; j++)
      xg->tform_data[i][j] = xg->raw_data[i][j];
}

void
copy_tform_to_sphered(xgobidata *xg)
{
/*
 * Copy tform_data to sphered_data.
*/
  int i, j;

  for (i=0; i<xg->nrows; i++)
    for (j=0; j<xg->ncols_used; j++)
      xg->sphered_data[i][j] = xg->tform_data[i][j];
}

/*
 * This is called in three places:  when a transformation is done,
 * when the dummy variable is changed by brushing, and when points
 * are deleted.  In the transformation case, only the columns in
 * the current variable group are affected; in the brushing case,
 * only the last variable is affected.  When points are deleted,
 * all variables are affected, and everything needs to be done.
 * Let this be the kernel of a routine that can be called whenever
 * we need to push the data pipeline.
*/
void
update_sphered(xgobidata *xg, int *cols, int ncols)
{
  /*
   * Update variance-covariance matrix
  */
  int j;

  if (ncols == xg->ncols_used && xg->ncols_used > 2)
    for (j=0; j<xg->ncols_used; j++)
      recalc_vc(j, xg);
  else
    for (j=0; j<ncols; j++)
      recalc_vc(cols[j], xg);
  /*
   * Update sphered_data[]
  */
  if (xg->is_princ_comp)
  {
    if (update_vc_active_and_do_svd(xg))
      spherize_data(xg);
    else
      copy_tform_to_sphered(xg);
  }
}

void
min_max(xgobidata *xg, float **data, int *cols, int ncols,
float *min, float *max)
/*
 * Find the minimum and maximum values of each column or variable
 * group using using the min-max scaling.
*/
{
  int i, j, k, n;
/*
 * Choose an initial value for *min and *max
*/
  *min = *max = data[xg->rows_in_plot[0]][cols[0]];

  for (n=0; n<ncols; n++)
  {
    j = cols[n];
    for (i=0; i<xg->nrows_in_plot; i++) {
      k = xg->rows_in_plot[i];
      if (data[k][j] < *min)
        *min = data[k][j];
      else if (data[k][j] > *max)
        *max = data[k][j];
    }
  }
}

void
mean_stddev(xgobidata *xg, float **data, int *cols, int ncols, float *min,
float *max, float *mean, float *stddev)
/*
 * Find the minimum and maximum values of each column or variable
 * group scaling by mean and std_width standard deviations.
*/
{
  int i, j, n;
  double dx;
  double sumxi = 0.0;
  double sumxisq = 0.0;
  double dmean, dvar, dstddev;
  double dn = (double) (ncols * xg->nrows_in_plot);

  for (n=0; n<ncols; n++) {
    j = cols[n];
    for (i=0; i<xg->nrows_in_plot; i++) {
      dx = (double) data[ xg->rows_in_plot[i] ][j];
      sumxi = sumxi + dx;
      sumxisq = sumxisq + dx * dx;
    }
  }
  dmean = sumxi / dn;
  dvar = (sumxisq / dn) - (dmean * dmean);
  dstddev = sqrt(dvar);

  *mean = (float) dmean;
  *stddev = (float) dstddev;

  *min = dmean - xg->std_width * dstddev;
  *max = dmean + xg->std_width * dstddev;
}

int
icompare(int *x1, int *x2)
{
  int val = 0;

  if (*x1 < *x2)
    val = -1;
  else if (*x1 > *x2)
    val = 1;

  return(val);
}

/* Also called by texture.c */
int
fcompare(const void *x1, const void *x2)
{
  int val = 0;
  float *f1 = (float *) x1;
  float *f2 = (float *) x2;

  if (*f1 < *f2)
    val = -1;
  else if (*f1 > *f2)
    val = 1;

  return(val);
}

/* static void */
void
med_mad(xgobidata *xg, float **data, int *cols, int ncols,
float *min, float *max, float *median, float *mad)
/*
 * Find the minimum and maximum values of each column or variable
 * group scaling by median and std_width absolute deviations.
*/
{
  int i, j, k, n, np;
  float *x;
  double dmedian = 0, dmad = 0;

  np = ncols * xg->nrows_in_plot;
  x = (float *) XtMalloc((Cardinal) np * sizeof(float));
  for (n=0; n<ncols; n++)
  {
    j = cols[n];
    for (i=0; i<xg->nrows_in_plot; i++)
    {
      k = xg->rows_in_plot[i];
      x[n*xg->nrows_in_plot + i] = data[k][j];
    }
  }

  qsort((void *) x, np, sizeof(float), fcompare);
  if ((np % 2) != 0)
    dmedian = x[(np-1)/2];
  else
    dmedian = (x[np/2-1] + x[np/2])/2.;

  for (j=0; j<np; j++)
  {
    x[j] -= dmedian;
    x[j] = (float) fabs((double) x[j]);
  }
  qsort((void *) x, np, sizeof(float), fcompare);
  if ((np % 2) != 0)
    dmad = x[(np-1)/2];
  else
    dmad = (x[np/2-1] + x[np/2])/2.;
  /*
   * The 0.675 makes med-mad standardization equivalent to med-stddev
   * standardization for normal data.
  */
  dmad = dmad / .675;
  *min = dmedian - xg->std_width * dmad;
  *max = dmedian + xg->std_width * dmad;

  *median = (float) dmedian;
  *mad = (float) dmad;

  XtFree((XtPointer) x);
}

/*
 * This code is also used in the "transpose plots", the logical
 * zooming plots done in identification.
*/
void
adjust_limits(float *min, float *max)
/*
 * This test should be cleverer.  It should test the ratios
 * lim[i].min/rdiff and lim[i].max/rdiff for overflow or
 * rdiff/lim[i].min and rdiff/lim[i].max for underflow.
 * It should probably do it inside a while loop, too.
 * See Advanced C, p 187.  Set up floation point exception
 * handler which alters the values of lim[i].min and lim[i].max
 * until no exceptions occur.
*/
{
  if (*max - *min == 0)
  {
    if (*min == 0.0)
    {
      *min = -1.0;
      *max = 1.0;
    }
    else
    {
      *min = .9 * *min;
      *max = 1.1 * *max;
    }
  }

  /* This is needed to account for the case that max == min < 0 */
  if (*max < *min)
  {
    float ftmp = *max;
    *max = *min;
    *min = ftmp;
  }
}

void
mean_lgdist(xgobidata *xg, float **data)
/*
 * Find the minimum and maximum values of each column or variable
 * group scaling by mean and std_width standard deviations.
*/
{
  int i, j, n;
  float min, max;
  double dx;
  double sumxi;
  double sumdist = 0.0;
  double *mean, lgdist = 0.0;

  mean = (double *) XtMalloc((Cardinal) xg->numvars_t * sizeof(double));

  for (j=0; j<xg->numvars_t; j++)
  {
    sumxi = 0.0;
    for (i=0; i<xg->nrows_in_plot; i++)
    {
      dx = (double) data[xg->rows_in_plot[i]][xg->tour_vars[j]];
      sumxi += dx;
    }
    mean[j] = sumxi / xg->nrows_in_plot;
  }

  for (i=0; i<xg->nrows_in_plot; i++)
  {
    sumdist = 0.0;
    for (j=0; j<xg->numvars_t; j++)
    {
      dx = (data[xg->rows_in_plot[i]][xg->tour_vars[j]]-mean[j]);
      dx *= dx;
      sumdist += dx;
    }
    if (sumdist > lgdist)
      lgdist = sumdist;
  }
  
  lgdist = sqrt(lgdist);
  for (j=0; j<xg->numvars_t; j++)
  {
    min = mean[j] - lgdist;
    max = mean[j] + lgdist;
    adjust_limits(&min, &max);
    n = xg->tour_vars[j];
    xg->lim[n].min = xg->lim0[n].min = min;
    xg->lim[n].max = xg->lim0[n].max = max;
  }

  XtFree((XtPointer) mean);
}

/*
 * This only gives the correct result if the
 * vgroups vector is of the form {0,1,2,...,ngroups-1}
*/
int
numvargroups(xgobidata *xg)
{
  int j, nvgroups = 0;

  for (j=0; j<xg->ncols_used; j++)
     if (xg->vgroup_ids[j] > nvgroups)
       nvgroups = xg->vgroup_ids[j];
  nvgroups++;

  return(nvgroups);
}

void
init_lim0(xgobidata *xg)
{
  int j, *cols, ncols;
  float min, max;
  int k;
  int nvgroups = numvargroups(xg);

  cols = (int *) XtMalloc((Cardinal) xg->ncols * sizeof(int));
  for (k=0; k<nvgroups; k++) {
    ncols = 0;
    for (j=0; j<xg->ncols_used; j++) {
      if (xg->vgroup_ids[j] == k)
        cols[ncols++] = j;
    }
    min_max(xg, xg->tform_data, cols, ncols, &min, &max);
    adjust_limits(&min, &max);
    for (j=0; j<ncols; j++)
    {
      xg->lim0[cols[j]].min = min;
      xg->lim0[cols[j]].max = max;
    }
  }

  XtFree((XtPointer) cols);
}

void
update_lims(xgobidata *xg)
{
  int j, k, n;
  float min, max;
  float mean, stddev, median, mad;
  int *cols, ncols;
  int nvgroups = numvargroups(xg);


/* 
 * First update the limits for the tform_data.  Then 
 * override lim and lim0 if necessary.
*/

  init_lim0(xg);

  /*
   * Take tform_data[], one variable group at a time, and generate
   * the min and max for each variable group (and thus for each
   * column).
  */
  cols = (int *) XtMalloc((Cardinal) xg->ncols * sizeof(int));
  for (k=0; k<nvgroups; k++) {
    ncols = 0;
    for (j=0; j<xg->ncols_used; j++) {
      if (xg->vgroup_ids[j] == k)
        cols[ncols++] = j;
    }

    switch(xg->std_type)
    {
      case 0:
        min_max(xg, xg->tform_data, cols, ncols, &min, &max);
        break;
      case 1:
        mean_stddev(xg, xg->tform_data, cols, ncols,
          &min, &max, &mean, &stddev);
        break;
      case 2:
        med_mad(xg, xg->tform_data, cols, ncols,
          &min, &max, &median, &mad);
        break;
    }

    adjust_limits(&min, &max);

    for (n=0; n<ncols; n++)
    {
      xg->lim[cols[n]].min = min;
      xg->lim[cols[n]].max = max;
    }
  }
  XtFree((XtPointer) cols);

/*
 * Set the limits that are based exclusively on tform data;
 * they need to be kept separately for parallel coordinate
 * plots.
*/
  for (j=0; j<xg->ncols_used; j++) {
    xg->lim_tform[j].min = xg->lim[j].min;
    xg->lim_tform[j].max = xg->lim[j].max;
  }


/*
 * Now override the values in lim and lim0 if necessary;
 * the values in lim_tform will remain the same.
*/
  if (xg->is_princ_comp && xg->is_touring)
  {
  /*
   * Take sphered_data[], one column at a time, and generate
   * each column min and max.
  */
    mean_lgdist(xg, xg->sphered_data);
  }

}

void
tform_to_world(xgobidata *xg)
{
/*
 * Take tform_data[], one column at a time, and generate
 * world_data[]
*/
  int i, j, m;
  float rdiff, ftmp;
  float precis = PRECISION1;  /* 32768 */

  for (j=0; j<xg->ncols_used; j++) {
    rdiff = xg->lim[j].max - xg->lim[j].min;
    for (i=0; i<xg->nrows_in_plot; i++) {
      m = xg->rows_in_plot[i];
      ftmp = -1.0 + 2.0*(xg->tform_data[m][j] - xg->lim[j].min)/rdiff;
      xg->world_data[m][j] = (long) (precis * ftmp);

      /* Add in the jitter values */
      xg->world_data[m][j] += xg->jitter_data[m][j];
    }
  }
}

static void
sphered_to_world(xgobidata *xg)
{
/*
 * Take sphered_data[], one column at a time, and generate
 * world_data[]
*/
  int i, j, k, m;
  float rdiff, ftmp;
  float precis = PRECISION1;

  for (m=0; m<xg->numvars_t; m++) {
    j = xg->tour_vars[m];
    rdiff = xg->lim[j].max - xg->lim[j].min;
    for (i=0; i<xg->nrows_in_plot; i++) {
      k = xg->rows_in_plot[i];
      ftmp = -1.0 + 2.0*(xg->sphered_data[k][j] - xg->lim[j].min)/rdiff;
      xg->world_data[k][j] = (long) (precis * ftmp);

      /* Add in the jitter values */
      xg->world_data[k][j] += xg->jitter_data[k][j];
    }
  }
}

void
update_world(xgobidata *xg)
{
/*
 * Keep world_data[] up to date.
*/
  if (xg->is_princ_comp && xg->is_touring)
    sphered_to_world(xg);
  else
    tform_to_world(xg);

  /*
   * span_planes() operates on world_data[], so it is called
   * when world_data[] changes.
  */
  if (xg->is_touring)
    span_planes(xg);
  if (xg->is_corr_touring)
    corr_span_planes(xg);
}

void
world_to_plane(xgobidata *xg)
/*
 * Using the current view, project the data from world_data[],
 * the data expressed in 'world coordinates,' to planar[], the
 * data expressed in 'projection coordinates.'
*/
{
  if (xg->is_touring)
    all_tour_reproject(xg);
  else if (xg->is_corr_touring)
    corr_reproject(xg);
  else if (xg->is_spinning)
  {
    if (xg->is_spin_type.yaxis || xg->is_spin_type.xaxis)
      ax_rot_reproject(xg);
    else if (xg->is_spin_type.oblique)
      ob_rot_reproject(xg);
  }
  else if (xg->is_xyplotting)
    xy_reproject(xg);

  else if (xg->is_dotplotting)
    dotplot_reproject(xg);
}

void
plane_to_screen(xgobidata *xg)
/*
 * Use the data in 'projection coordinates' and rescale it to the
 * dimensions of the current plotting window, writing it into screen.
 * At the same time, update segs.
*/
{
  int j, k;
  long nx, ny;
  float scale_x = (xg->is_touring) ? xg->tour_scale.x : xg->scale.x;
  float scale_y = (xg->is_touring) ? xg->tour_scale.y : xg->scale.y;
  scale_x /= 2;
  scale_y /= 2;

  /*
   * Calculate is, a scale factor.  Either force the plot to be
   * square or scale so as to use the entire plot window.  (Or
   * as much of the plot window as scale.x and scale.y permit.)
  */
  xg->is.x = (xg->plot_square) ?
    (long) (xg->minxy * scale_x) :
    (long) (xg->max.x * scale_x);
  xg->is.y = (xg->plot_square) ?
    (long) (-1 * xg->minxy * scale_y) :
    (long) (-1 * xg->max.y * scale_y);

  if (xg->is_princ_comp && xg->is_touring) {
    /* multiply by fudge factor so that dimension is 
     * accounted for in screen scaling
    */
     xg->is.x = (long)((float)xg->is.x*(float)FUDGE_FACTOR);
     xg->is.y = (long)((float)xg->is.y*(float)FUDGE_FACTOR);
  }
  /*
   * Calculate new coordinates.
  */
  for (k=0; k<xg->nrows_in_plot; k++) {
    j = xg->rows_in_plot[k];

  /*
   * shift in world coords
  */
    nx = xg->planar[j].x + xg->shift_wrld.x;
    ny = xg->planar[j].y - xg->shift_wrld.y;

  /*
   * scale from world to plot window and expand-contract as desired
  */
    xg->screen[j].x = (int) ((nx * xg->is.x) >> EXP1);
    xg->screen[j].y = (int) ((ny * xg->is.y) >> EXP1);

  /*
   * shift into middle of plot window 
  */
    xg->screen[j].x += xg->mid.x;
    xg->screen[j].y += xg->mid.y;
  }
}
