/* $Id: cfgfile.cpp,v 1.8 1999/12/04 23:50:15 xfmail Exp $ */

#include "cfgfile.h"
#include <stdio.h>

extern struct _xfdef xfdefs;

cfgfile::cfgfile() {
    cfile = NULL;
    strcpy(fname,"");
    changed = 0;
    hasht = NULL;
}

unsigned int cfgfile::hash(char * str)
{
    unsigned int key = 0;
    unsigned char c;

    while((c = *str++) && (c != '=')) {
        key += c;
        key ^= (key << 24) | (key >> 8);
    }

    return key % hashtsize;
}

void cfgfile::add(struct __tagcfl * tag)
{
    unsigned int key;

    key = hash(tag->str);
    tag->next = hasht[key];
    hasht[key] = tag;
}

struct __tagcfl * cfgfile::find(char * str)
{
    unsigned int key;
    struct __tagcfl *tag;
    int len;

    if(hashtsize == 0)
        return NULL;

    len = strlen(str);
    key = hash(str);
    tag = hasht[key];

    while(tag) {
        if(!strncmp(tag->str, str, len) && (tag->str[len] == '='))
            return tag;

        tag = tag->next;

    }

    return NULL;
}

int cfgfile::del(char * str)
{
    unsigned int key;
    struct __tagcfl *tag, *tag1;

    key = hash(str);
    if((tag = find(str)) == NULL)
        return -1;

    if(hasht[key] == tag) {
        hasht[key] = tag->next;
        free(tag);
        return 0;
    }

    tag1 = hasht[key];
    while(tag1->next != tag)
        tag1 = tag1->next;

    tag1->next = tag->next;
    free(tag);

    return 0;
}

void cfgfile::clear()
{
    int i;
    struct __tagcfl *tag, *tag1;

    for(i = 0; i < hashtsize; i++) {
        tag = hasht[i];
        while(tag) {
            tag1 = tag->next;
            free(tag);
            tag = tag1;
        }
        hasht[i] = NULL;
    }

    return;
}

void cfgfile::destroy()
{
    clear();

    if(hasht)
        free(hasht);
    hasht = NULL;
    hashtsize = 0;

    return;
}

void cfgfile::lock(char * fn)
{
    if(cfile != NULL) {
#ifdef HAVE_FLOCK
        flock(fileno(cfile), LOCK_UN);
#else
    #ifdef HAVE_LOCKF
        lockf(fileno(cfile), F_ULOCK, 0);
    #endif
#endif
        fclose(cfile);
    }

#if defined(HAVE_LOCKF) && !defined(HAVE_FLOCK)
    if((cfile=fopen(fn,"r+"))==NULL) {
#else
    if((cfile=fopen(fn,"r"))==NULL) {
#endif
        display_msg(MSG_WARN, "Can not open", "configuration file %s", fn);
        return;
    }

#ifdef HAVE_FLOCK
    if(flock(fileno(cfile), LOCK_EX | LOCK_NB) == -1)
#else
    #ifdef HAVE_LOCKF
    if(lockf(fileno(cfile), F_TLOCK, 0) == -1)
    #endif
#endif
    {
        fprintf(stderr, "Can not lock %s\nProbably XFMail is already running\n", fn);
        if(readonly)
            fprintf(stderr, "Proceeding in readonly mode\n");
        else
            exit(1);
    } else {
        if(readonly) {
#ifdef HAVE_FLOCK
            flock(fileno(cfile), LOCK_UN);
#else
    #ifdef HAVE_LOCKF
            lockf(fileno(cfile), F_ULOCK, 0);
    #endif
#endif
        }
    }
}

void cfgfile::unlock()
{
    if(cfile != NULL) {
#ifdef HAVE_FLOCK
        flock(fileno(cfile), LOCK_UN);
#else
    #ifdef HAVE_LOCKF
        lockf(fileno(cfile), F_ULOCK, 0);
    #endif
#endif
        fclose(cfile);
    }

    cfile = NULL;
}

int cfgfile::save_file(char * file, int nopwd)
{
    FILE *cfd;
    char buf[SLEN];
    int i;
	__cfl *tmpptr;

    // Return if there's no changes to be written out
    if (!changed) {
        return 0;
    }

    if(!file)
        return -1;

    if((cfd = fopen(file, "w")) == NULL) {
        display_msg(MSG_WARN, "save config", "Can not open %s", file);
        return -1;
    }

    for(i = 0; i < hashtsize; i++) {
        tmpptr = hasht[i];
        while(tmpptr) {
            if(nopwd && (tmpptr->flags & CF_UNSECURE)) {
                tmpptr=tmpptr->next;
                continue;
            }

            snprintf(buf, sizeof(buf), "%s\n", tmpptr->str);
            if(fputs(buf, cfd) == EOF) {
                if(errno == ENOSPC)
                    display_msg(MSG_WARN, "save config", "DISK FULL!");
                else
                    display_msg(MSG_WARN, "save config", "Failed to write %s", file);
                return -1;
            }

            tmpptr=tmpptr->next;
        }
    }

    fclose(cfd);
    return 0;
}

/*
 * This one flushes current config
 */
int cfgfile::save(int ask)
{
    FILE *f;
    char buf[SLEN], ctmp[SLEN];
    int i;
    __cfl *tmpptr;

    if(readonly) {
        changed = 0;
        return 0;
    }

    if(ask && changed) {
        if(!display_msg(MSG_QUEST, "Configuration has been changed", "Do you want to save it?"))
            return 0;
    }

    if(fname[0] == '\0')
        return -1;

    snprintf(ctmp, sizeof(ctmp), "%s___temp", fname);

    if((f=fopen(ctmp,"w"))==NULL) {
        display_msg(MSG_WARN, "Can not write to temporary config file", "%s", ctmp);
        return -1;
    }

    for(i = 0; i < hashtsize; i++) {
        tmpptr = hasht[i];
        while(tmpptr) {
            if(tmpptr->flags & DONT_STORE) {
                tmpptr=tmpptr->next;
                continue;
            }

            snprintf(buf, sizeof(buf), "%s\n", tmpptr->str);
            if(fputs(buf,f) == EOF) {
                if(errno == ENOSPC)
                    display_msg(MSG_WARN, "save config", "DISK FULL!");
                else
                    display_msg(MSG_WARN, "save config", "Failed to write %s", ctmp);
                fclose(f);
                unlink(ctmp);
                return -1;
            }
            tmpptr=tmpptr->next;
        }
    }

    if(fclose(f) == EOF) {
        if(errno == ENOSPC)
            display_msg(MSG_WARN, "save config", "DISK FULL!");
        else
            display_msg(MSG_WARN, "save config", "Failed to write %s", ctmp);
        unlink(ctmp);
        return -1;
    }

    unlock();
#ifdef __EMX__ /* Under OS/2 the file will not be deleted during rename() */
    if(access(fname, 0)==0) {
        if(unlink(fname)!=0) {
            display_msg(MSG_WARN, "unlink", "delete %s before moving", fname);
        }
    }
#endif
    if(rename(ctmp, fname) == -1)
        display_msg(MSG_WARN, "can not rename", "%s to %s", ctmp, fname);
    lock(fname);
    chmod(fname,SECFILEMODE);

    changed = 0;

    return 1;
}

/*
 * This function preloads configuration file to be
 * changed in memory
 */

int cfgfile::load(char * fn)
{
    FILE *f;
    char buf[SLEN + 1];
    int i;
    struct __tagcfl *tmpptr, *tmpptr1, *cfgl;
    char *p;

    destroy();
    cfgl = NULL;

    unlock();

    if((f = fopen(fn,"r")) == NULL) {
        if((f = fopen(fn, "w")) == NULL) {
            display_msg(MSG_FATAL, "cfgload", "Can not create %s", fn);
            return -1;
        }

        goto endcfgload;
    }

    while(fgets(buf,SLEN,f) != NULL) {
        if(buf[0] == '#')
            continue;

        if(strchr(buf, '=') == NULL)
            continue;

        strip_newline(buf);

        if(!(tmpptr=(struct __tagcfl *)malloc(sizeof(struct __tagcfl)))) {
            display_msg(MSG_FATAL, "cfgload", "malloc failed");
            return -1;
        }

        strcpy(tmpptr->str,buf);

        tmpptr->next = cfgl;
        tmpptr->flags = 0;
        cfgl = tmpptr;

        hashtsize++;
    }

    endcfgload:
    strcpy(fname,fn);
    if(fclose(f) == EOF) {
        display_msg(MSG_WARN, "cfgload", "Failed to load config file");
        return -1;
    }

    lock(fname);

    if(hashtsize < HASHMINSIZE)
        hashtsize = HASHMINSIZE;

    hasht = (struct __tagcfl **)malloc(sizeof(struct __tagcfl *) * hashtsize);
    if(hasht == NULL) {
        display_msg(MSG_FATAL, "cfgload", "malloc failed");
        return -1;
    }

    for(i = 0; i < hashtsize; i++)
        hasht[i] = NULL;

    tmpptr = cfgl;
    while(tmpptr) {
        tmpptr1 = tmpptr->next;
        add(tmpptr);
        tmpptr = tmpptr1;
    }

    if((f = fopen(GLOBAL_XFMAILRC, "r")) != NULL) {
        while(fgets(buf, SLEN, f) != NULL) {
            if(buf[0] == '#')
                continue;

            if((p = strchr(buf, '=')) == NULL)
                continue;
            *p = '\0';

            if(find(buf) != NULL)
                continue;
            *p = '=';
            strip_newline(buf);

            if((tmpptr = (__cfl *)malloc(sizeof(__cfl))) == NULL) {
                display_msg(MSG_FATAL, "config", "malloc failed");
                return -1;
            }

            strcpy(tmpptr->str, buf);
            tmpptr->flags = 0;
            tmpptr->next = cfgl;
            cfgl = tmpptr;

            add(tmpptr);
        }
        fclose(f);
    }

    return 0;
}

int cfgfile::check_version()
{
	char buf[512];

    if(hasht && strcmp(VERSION, getString(conf_name, "xfversion", ""))) {
#ifndef _PATH_RM
			display_msg(MSG_WARN,"upgrade","Error While Trying to Remove Cache Directory, you will have to do this manually.");
			exit(0);
#else
		sprintf(buf,"%s -rf %s/.cache",_PATH_RM, configdir);
		system(buf);
#endif
		/*
		 if(!remove(buf)) {
			display_msg(MSG_WARN,"upgrade","Error While Trying to Remove Cache Directory, you will have to do this manually.");
		} else {
			display_msg(MSG_WARN,"upgrade","Cache Directory removed successfully.");
		}
		*/

        if(!display_msg(MSG_QUEST|MSG_DEFNO, "Configuration file belongs", "to different version of XFMail, use it anyway?")) {
            clear();
		}

        putString(conf_name, "xfversion", VERSION);
        return 1;
    }

    return 0;
}

void cfgfile::setFlags(char * ln,int flags)
{
    __cfl *tmpptr;

    if((tmpptr = find(ln)) == NULL) {
        if((tmpptr=(__cfl *)malloc(sizeof(__cfl)))==NULL) {
            display_msg(MSG_FATAL, "config", "malloc  failed!");
            return;
        }

        snprintf(tmpptr->str, sizeof(tmpptr->str), "%s=", ln);
        tmpptr->next = NULL;
        tmpptr->flags = flags;
        if(!(flags & DONT_STORE) &&
           !(flags & CF_NOTCHANGED)) {
            changed = 1;
        }
        add(tmpptr);
    }

    tmpptr->flags = flags;

    return;
}

/*
 * This function deletes string from configuration file
 */
int cfgfile::delString(char * fn, char * ln)
{
    if(find(ln) == NULL)
        return 0;

    changed = 1;
    return del(ln);
}

/*
 * This function adds string to configuration file
 */
int cfgfile::putString(char * fn,char * ln,char * b)
{
    __cfl *tmpptr;
    char *p;

    if(!ln || !b)
        return -1;

    if((strlen(ln) + strlen(b) + 2) >= SLEN) {
        display_msg(MSG_WARN, "config", "Line too long");
        return -1;
    }

    if((tmpptr = find(ln)) != NULL) {
        if(!(tmpptr->flags & DONT_STORE) &&
           !(tmpptr->flags & CF_NOTCHANGED)) {
            if((p = strchr(tmpptr->str, '=')) == NULL)
                return -1;
            p++;
            if(strcmp(p, b)) {
                changed = 1;
            }
        }

        strcpy(tmpptr->str, ln);
        strcat(tmpptr->str, "=");
        strcat(tmpptr->str, b);
        return 0;
    }

    if((tmpptr = (__cfl *)malloc(sizeof(__cfl))) == NULL) {
        display_msg(MSG_FATAL, "config", "malloc failed");
        return -1;
    }

    tmpptr->flags = 0;
    tmpptr->next = NULL;
    snprintf(tmpptr->str, sizeof(tmpptr->str), "%s=%s", ln, b);

    add(tmpptr);
    changed = 1;

    return 0;
}

/*
 * Check if specific config entry exists
 */
int cfgfile::exist(char * ln)
{
    return find(ln) ? 1 : 0;
}

/*
 * This function gets configuration text string from
 * named file.
 *
 */

char * cfgfile::getString(char * fn,char * ln,char * def)
{
    struct __tagcfl *cfg;
    char *p;

    if((cfg = find(ln)) != NULL) {
        p = strchr(cfg->str, '=');
        p++;
        return p;
    }

    putString(fn, ln, def);
    if((cfg = find(ln)) == NULL)
        return NULL;

    p = strchr(cfg->str, '=');
    p++;

    return p;
}

char * cfgfile::getStringDefault(char *fn, char *ln, char *def, int return_default)
{
    if(return_default)
        return def;
    else
        return getString(fn,ln,def);
}

int cfgfile::putInt(char * fn,char * ln,int i)
{
    char buf[10];

    sprintf(buf, "%d", i);
    return putString(fn,ln,buf);
}

int cfgfile::getInt(char * fn,char * ln,int def)
{
    char *buf;
    char ibuf[10];

    sprintf(ibuf, "%d", def);
    if(!(buf=getString(fn,ln,ibuf)))
        return def;

    return atoi(buf);
}

int cfgfile::getIntDefault(char *fn, char *ln, int def, int return_default)
{
    if(return_default)
        return def;
    else
        return getInt(fn,ln,def);
}

void cfgfile::read_xfdefaults()
{
    FL_IOPT cntl;
    u_long mask;
    char xfdef[255], *p;
    FILE *xf;
    int cind, r, g, b, checkglobal = 1;

    mask = FL_PDBorderWidth;
    cntl.borderWidth = 2;

    startxfdef:
    if((checkglobal &&
        ((xf = fopen(GLOBAL_XFDEFS, "r")) != NULL)))
        checkglobal = 2;
    else {
        snprintf(xfdef, sizeof(xfdef), "%s/.xfdefaults", configdir);
        if((xf = fopen(xfdef, "r")) == NULL)
            goto xfdef_int;
        checkglobal = 0;
    }

    while(fgets(xfdef, 255, xf)) {
        if((xfdef[0] == '#') ||
           (xfdef[0] == '!'))
            continue;

        strip_newline(xfdef);
        if((p = strchr(xfdef, '=')) == NULL)
            continue;

        *p = '\0';
        p++;
        if(!strcmp(xfdef, "Depth")) {
            mask |= FL_PDDepth;
            sscanf(p, "%d", &cntl.depth);
        } else
            if(!strcmp(xfdef, "VisualClass")) {
            mask |= FL_PDClass;
            sscanf(p, "%d", &cntl.vclass);
        } else
            if(!strcmp(xfdef, "DoubleBuffer")) {
            mask |= FL_PDDouble;
            sscanf(p, "%d", &cntl.doubleBuffer);
        } else
            if(!strcmp(xfdef, "ButtonFontSize")) {
            mask |= FL_PDButtonFontSize;
            sscanf(p, "%d", &cntl.buttonFontSize);
        } else
            if(!strcmp(xfdef, "SliderFontSize")) {
            mask |= FL_PDSliderFontSize;
            sscanf(p, "%d", &cntl.sliderFontSize);
        } else
            if(!strcmp(xfdef, "MenuFontSize")) {
            mask |= FL_PDMenuFontSize;
            sscanf(p, "%d", &cntl.menuFontSize);
        } else
            if(!strcmp(xfdef, "ChoiceFontSize")) {
            mask |= FL_PDChoiceFontSize;
            sscanf(p, "%d", &cntl.choiceFontSize);
        } else
            if(!strcmp(xfdef, "BrowserFontSize")) {
            mask |= FL_PDBrowserFontSize;
            sscanf(p, "%d", &cntl.browserFontSize);
        } else
            if(!strcmp(xfdef, "InputFontSize")) {
            mask |= FL_PDInputFontSize;
            sscanf(p, "%d", &cntl.inputFontSize);
        } else
            if(!strcmp(xfdef, "LabelFontSize")) {
            mask |= FL_PDLabelFontSize;
            sscanf(p, "%d", &cntl.labelFontSize);
        } else
            if(!strcmp(xfdef, "PopUpFontSize")) {
            sscanf(p, "%d", &cntl.pupFontSize);
            xfdefs.popfsize = cntl.pupFontSize;
        } else
            if(!strcmp(xfdef, "PopUpFontStyle")) {
            sscanf(p, "%d", &cntl.pupFontStyle);
            xfdefs.popfstyle = cntl.pupFontStyle;
        } else
            if(!strcmp(xfdef, "PopUpCursor"))
            sscanf(p, "%d", &xfdefs.popcursor);
        else
            if(!strcmp(xfdef, "PopUpBgCol"))
            sscanf(p, "%d", &xfdefs.popbgcol);
        else
            if(!strcmp(xfdef, "PopUpFgCol"))
            sscanf(p, "%d", &xfdefs.popfgcol);
        else
            if(!strcmp(xfdef, "PrivateMap")) {
            mask |= FL_PDPrivateMap;
            sscanf(p, "%d", &cntl.privateColormap);
        } else
            if(!strcmp(xfdef, "SharedMap")) {
            mask |= FL_PDSharedMap;
            sscanf(p, "%d", &cntl.sharedColormap);
        } else
            if(!strcmp(xfdef, "StandardMap")) {
            mask |= FL_PDStandardMap;
            sscanf(p, "%d", &cntl.standardColormap);
        } else
            if(!strcmp(xfdef, "BorderWidth")) {
            mask |= FL_PDBorderWidth;
            sscanf(p, "%d", &cntl.borderWidth);
        } else
            if(!strcmp(xfdef, "MapColor")) {
            if(sscanf(p, "%d %d %d %d", &cind, &r, &g, &b) == 4)
                fl_set_icm_color(cind, r, g, b);
        }

    }
    fclose(xf);

    if(checkglobal) {
        checkglobal = 0;
        goto startxfdef;
    }

    xfdef_int:
    if(exist("InputFontFSize")) {
        mask |= FL_PDInputFontSize;
        cntl.inputFontSize = getInt(conf_name, "InputFontFSize", 0);
    }

    if(exist("LabelFontFSize")) {
        mask |= FL_PDLabelFontSize;
        cntl.labelFontSize = getInt(conf_name, "LabelFontFSize", 0);
    }

    if(exist("ChoiceFontFSize")) {
        mask |= FL_PDChoiceFontSize;
        cntl.choiceFontSize = getInt(conf_name, "ChoiceFontFSize", 0);
    }

    if(exist("MenuFontFSize")) {
        mask |= FL_PDMenuFontSize;
        cntl.menuFontSize = getInt(conf_name, "MenuFontFSize", 0);
    }

    if(exist("BrowserFontFSize")) {
        mask |= FL_PDBrowserFontSize;
        cntl.browserFontSize = getInt(conf_name, "BrowserFontFSize", 0);
    }

    if(exist("ButtonFontFSize")) {
        mask |= FL_PDButtonFontSize;
        cntl.buttonFontSize = getInt(conf_name, "ButtonFontFSize", 0);
    }

    if(exist("PopUpFontFSize")) {
        xfdefs.popfsize = getInt(conf_name, "PopUpFontFSize", 0);
    }

    if(exist("SliderFontFSize")) {
        mask |= FL_PDSliderFontSize;
        cntl.sliderFontSize = getInt(conf_name, "SliderFontFSize", 0);
    }

    if(mask != 0L)
        fl_set_defaults(mask, &cntl);

    return;
}
