/*
 * X-Mame generic video code
 *
 */
#define __VIDEO_C_
#include "xmame.h"
#include "driver.h"
#include "profiler.h"
#include "input.h"
#include <math.h>

extern int bitmap_dirty;

static unsigned char current_palette[256*3];
static unsigned char color_dirty[256];
static int color_start, color_end, palette_dirty = FALSE;

/* Create a bitmap. Also calls osd_clearbitmap() to appropriately initialize */
/* it to the background color. */
/* VERY IMPORTANT: the function must allocate also a "safety area" 16 pixels wide all */
/* around the bitmap. This is required because, for performance reasons, some graphic */
/* routines don't clip at boundaries of the bitmap. */
struct osd_bitmap *osd_new_bitmap(int width,int height,int depth)       /* ASG 980209 */
{
	struct osd_bitmap *bitmap;

	if (!width) width++;
	if (!height) height++;

	if (Machine->orientation & ORIENTATION_SWAP_XY)
	{
		int temp;

		temp = width;
		width = height;
		height = temp;
	}

	if ((bitmap = malloc(sizeof(struct osd_bitmap))) != 0)
	{
		int i,rowlen,rdwidth;
		unsigned char *bm;
		int safety;


		if (width > 64) safety = 16;
		else safety = 0;        /* don't create the safety area for GfxElement bitmaps */

		if (depth != 8 && depth != 16) depth = 8;

		bitmap->depth = depth;
		bitmap->width = width;
		bitmap->height = height;

		rdwidth = (width + 7) & ~7;     /* round width to a quadword */
		if (depth == 16)
			rowlen = 2 * (rdwidth + 2 * safety) * sizeof(unsigned char);
		else
			rowlen =     (rdwidth + 2 * safety) * sizeof(unsigned char);

		if ((bm = malloc((height + 2 * safety) * rowlen)) == 0)
		{
			free(bitmap);
			return 0;
		}

		/* also take safety at the end of the lines array since we now may access
		   lines above height since the dirty codes works with block of 8 */
		if ((bitmap->line = malloc((height+safety) * sizeof(unsigned char *))) == 0)
		{
			free(bm);
			free(bitmap);
			return 0;
		}

		for (i = 0;i < (height+safety);i++)
			bitmap->line[i] = &bm[(i + safety) * rowlen + safety];

		bitmap->_private = bm;

		osd_clearbitmap(bitmap);
	}

	return bitmap;
}

void osd_free_bitmap(struct osd_bitmap *bitmap)
{
	if (bitmap)
	{
		free(bitmap->line);
		free(bitmap->_private);
		free(bitmap);
		bitmap = NULL;
	}
}

/* set the bitmap to black */
void osd_clearbitmap(struct osd_bitmap *bitmap)
{
	int i;
	int safety;

	if (bitmap->width > 64) safety = 16;
	 else safety = 0;
	
	for (i = 0; i < (bitmap->height + safety); i++)
	{
		if (bitmap->depth == 16)
			memset(bitmap->line[i], 0, 2 * (bitmap->width + safety));
		else
			memset(bitmap->line[i], 0, bitmap->width + safety);
	}

	if (bitmap == Machine->scrbitmap)
	{
		bitmap_dirty = 1;
		osd_mark_dirty (0,0,bitmap->width-1,bitmap->height-1,1);
	}
}

struct osd_bitmap *osd_create_display(int _width, int _height, int attributes)
{
   int i;
   
   /* make copies here so that we can pass unmodified versions if we recall
      our selves in case 16 bpp fails */
   int height = _height;
   int width  = _width;
   
   if (Machine->drv->video_attributes & VIDEO_TYPE_VECTOR)
   {
      int vector_width, vector_height;
      if(vector_res &&
         sscanf(vector_res, "%dx%d", &vector_width, &vector_height) == 2)
      {
         widthscale_f  = (float)vector_width  / width;
         heightscale_f = (float)vector_height / height;
         if (widthscale_f > heightscale_f)
            widthscale_f  = heightscale_f;
         else
            heightscale_f = widthscale_f;
         fprintf(stderr_file, "Vectorres %dx%d, scale = %f\n",
            vector_width, vector_height, heightscale_f);
      }
      width  *= widthscale_f;
      height *= heightscale_f;
      widthscale = heightscale = 1;
      /* round with & heigth to a multiple of 8 */
      width  &= ~7;
      height &= ~7;
      /* store info into a visual struct */
      visual.min_x = 0;
      visual.min_y = 0;
      visual.max_x = width - 1;
      visual.max_y = height - 1;
   }
   else
      /* we use the same struct as the core, but under a different name,
         to avoid header conflicts */
      memcpy(&visual, &Machine->drv->visible_area, sizeof(struct my_rectangle));
     
   if (video_16bit && (Machine->drv->video_attributes & VIDEO_SUPPORTS_16BIT))
      bitmap = osd_new_bitmap(width,height,16);
   else
   {
      bitmap = osd_new_bitmap(width,height,8);
      video_16bit = FALSE;
   }
   if (bitmap==NULL) return NULL;
   
   if( Machine->orientation & ORIENTATION_SWAP_XY) {
      i=visual.max_x; visual.max_x=visual.max_y; visual.max_y=i;
      i=visual.min_x; visual.min_x=visual.min_y; visual.min_y=i;
      i=width; width=height; height=i;
   } 

   if (Machine->orientation & ORIENTATION_FLIP_X)
   {
      i            = width - visual.min_x - 1;
      visual.min_x = width - visual.max_x - 1;
      visual.max_x = i;
   }
   
   if (Machine->orientation & ORIENTATION_FLIP_Y)
   {
      i            = height - visual.min_y - 1;
      visual.min_y = height - visual.max_y - 1;
      visual.max_y = i;
   }
   
   /* Can we do dirty? First check if its a vector game */
   if (Machine->drv->video_attributes & VIDEO_TYPE_VECTOR)
   {
      if (use_dirty) use_dirty  = 2;
   }
   else if ( (Machine->drv->video_attributes & VIDEO_SUPPORTS_DIRTY) == 0 )
   {
      use_dirty = FALSE;
   }
   
   if(use_auto_double &&
      (Machine->drv->video_attributes & VIDEO_PIXEL_ASPECT_RATIO_MASK) ==
      VIDEO_PIXEL_ASPECT_RATIO_1_2)
   {
      if (Machine->orientation & ORIENTATION_SWAP_XY)
         widthscale  *= 2;
      else
         heightscale *= 2;
   }
  
   /* round to 8, since the new dirty code works with 8x8 blocks,
      and we need to round to sizeof(long) for the long copies anyway */
   if (visual.min_x & 7)
   {
      if((visual.min_x - (visual.min_x & ~7)) < 4)
         visual.min_x &= ~7;
       else
         visual.min_x = (visual.min_x + 7) & ~7;
   }
   if ((visual.max_x+1) & 7)
   {
      if(((visual.max_x+1) - ((visual.max_x+1) & ~7)) > 4)
         visual.max_x = ((visual.max_x+1 + 7) & ~7) - 1;
       else
         visual.max_x = ((visual.max_x+1) & ~7) - 1;
   }
   
   /* rounding of the y-coordinates is only nescesarry when we are doing dirty */
   if (use_dirty)
   {
      if (visual.min_y & 7)
      {
         if((visual.min_y - (visual.min_y & ~7)) < 4)
            visual.min_y &= ~7;
          else
            visual.min_y = (visual.min_y + 7) & ~7;
      }
      if ((visual.max_y+1) & 7)
      {
         if(((visual.max_y+1) - ((visual.max_y+1) & ~7)) > 4)
            visual.max_y = ((visual.max_y+1 + 7) & ~7) - 1;
          else
            visual.max_y = ((visual.max_y+1) & ~7) - 1;
      }
   }
   
   /* now calculate the visual width / height */
   visual_width  = visual.max_x - visual.min_x + 1;
   visual_height = visual.max_y - visual.min_y + 1;

#if !defined openstep && !defined xgl
   if (osd_dirty_init()!=OSD_OK) return NULL;
#endif
   
   if (sysdep_create_display() != OSD_OK)
   {
      if(video_16bit)
      {
         fprintf(stderr_file, "Video initialisation for 16bpp failed, trying again with 8bpp\n");
         /* clean up */
         osd_close_display();
         video_16bit = FALSE;
         return osd_create_display(_width, _height, attributes);
      }
      return NULL;
   }
   
   /* a lott of display_targets need to have the display initialised before
      initialising any input devices */
   if (osd_input_initpost()!=OSD_OK) return NULL;
   
   set_ui_visarea (visual.min_x, visual.min_y, visual.max_x, visual.max_y);
   
   /* Clear the current palette */
   memset((void *)current_palette, 0, 3*256*sizeof(unsigned char));
   memset((void *)color_dirty, FALSE,   256*sizeof(unsigned char));
   
   if (use_dirty)   fprintf(stderr_file,"Using dirty_buffer strategy\n");
   if (video_16bit) fprintf(stderr_file,"Using 16bpp video mode\n");
   
   /* for debugging only */
   fprintf(stderr_file, "width= %d, height = %d, viswidth = %d, visheight = %d,"
           "visstartx= %d, visstarty= %d\n",
            width, height, visual_width, visual_height, visual.min_x,
            visual.min_y);
   
   return bitmap;
}   

void osd_allocate_colors(unsigned int _totalcolors,const unsigned char *palette,unsigned short *pens)
{
   int a;
   
   totalcolors = _totalcolors;
   
   if (video_16bit)
   {
      sysdep_alloc_palette_16bpp(pens);
      Machine->uifont->colortable[0] = 0;
      Machine->uifont->colortable[1] = pens[32767];
      Machine->uifont->colortable[2] = pens[32767];
      Machine->uifont->colortable[3] = 0;
      return;
   }

   if (totalcolors>256)
   {
      fprintf(stderr_file, "Warning: More than 256 colors (%d) are needed for this emulation,\n"
         "some parts of the screen may be corrupted\n", totalcolors);
      /* fill the remainder of the pens array with 0's to make sure */
      /* nothing strange happens                                    */
      for (a=256;a<totalcolors;a++) pens[a]=0;
      totalcolors=256;
   }
   else
      fprintf(stderr_file, "Game uses %d colors\n", totalcolors);
      
   /* can we use color 0 & 255 for our own purposes ? */
   switch (totalcolors)
   {
      case 256:
         color_start  = 0;
         color_end    = totalcolors;
         break;
      case 255:
         color_start  = 1;
         color_end    = 256;
         totalcolors  = 256;
         break;
      default:
         color_start  = 1;
         color_end    = totalcolors+1;
         totalcolors += 2;
   }
   
   /* alloc the palette */
   sysdep_alloc_palette();
   
   /* init the palette */
   for (a=color_start; a<color_end; a++)
   {
      current_palette[a*3   ] = palette[(a-color_start)*3  ];
      current_palette[a*3 +1] = palette[(a-color_start)*3+1];
      current_palette[a*3 +2] = palette[(a-color_start)*3+2];
      pens[a-color_start]     = a;
      color_dirty[a]          = TRUE;
   }
   
   Machine->uifont->colortable[0] = 0;
   Machine->uifont->colortable[1] = totalcolors - 1;
   Machine->uifont->colortable[2] = totalcolors - 1;
   Machine->uifont->colortable[3] = 0;

   /* if we use color 0 & 255 initialise them */
   if (color_start)
   {
      current_palette[0] = current_palette[1] = current_palette[2] = 0x00;
      color_dirty[0] = TRUE;
   }
   if (color_end != totalcolors)
   {
      a = color_end * 3;
      current_palette[a] = current_palette[a+1] = current_palette[a+2] = 0xFF;
      color_dirty[color_end] = TRUE;
   }
      
   palette_dirty = TRUE;
}

void osd_get_pen(int pen,unsigned char *red, unsigned char *green, unsigned char *blue)
{
    if (video_16bit)
       sysdep_get_pen_16bpp(pen, red, green, blue);
    else
    {
       *red   = current_palette[pen*3   ];
       *green = current_palette[pen*3 +1];
       *blue  = current_palette[pen*3 +2];
    }
}

void osd_modify_pen(int pen, unsigned char red,unsigned char green,unsigned char blue) 
{
	if (video_16bit)
	{
		if (errorlog)
		   fprintf(errorlog,
		      "error: osd_modify_pen() doesn't work with 16 bit video modes.\n");
		return;
	}
	
	/* Optimise out operation if no palette change is required */
	if (current_palette[pen*3   ] == red &&
	    current_palette[pen*3 +1] == green &&
	    current_palette[pen*3 +2] == blue)
	    return;
	    
	current_palette[pen*3   ] = red;
	current_palette[pen*3 +1] = green;
	current_palette[pen*3 +2] = blue;
	
	color_dirty[pen] = TRUE;
	palette_dirty    = TRUE;
}

void sysdep_mark_palette_dirty(void)
{
	int i;
	
	for(i=0; i<totalcolors; i++) color_dirty[i] = TRUE;
	palette_dirty = TRUE;
}

#ifdef HAVE_GETTIMEOFDAY
/* Standard UNIX clock() is based on CPU time, not real time.
   Here is a real-time drop in replacement for UNIX systems that have the
   gettimeofday() routine.  This results in much more accurate timing for
   throttled emulation.
*/
uclock_t uclock()
{
  static long init_sec = 0;

  struct timeval tv;
  gettimeofday(&tv, 0);
  if (init_sec == 0) init_sec = tv.tv_sec;
  return (tv.tv_sec - init_sec) * 1000000 + tv.tv_usec;
}
#endif

static int skip_next_frame = 0;

typedef int (*skip_next_frame_func)(int show_fps_counter);
static skip_next_frame_func skip_next_frame_functions[FRAMESKIP_DRIVER_COUNT] =
{
   dos_skip_next_frame,
   barath_skip_next_frame
};

int osd_skip_this_frame(void)
{   
   return skip_next_frame;
}

/* Update the display. */
void osd_update_video_and_audio(void)
{
   int i;
   static int showfps=0, showfpstemp=0; 
   int skip_this_frame;
   int need_to_clear_bitmap=0;
   
   if (input_ui_pressed(IPT_UI_FRAMESKIP_INC))
   {
      if (autoframeskip)
      {
	 autoframeskip = 0;
	 frameskip = 0;
      }
      else
      {
	 if (frameskip == FRAMESKIP_LEVELS-1)
	 {
	    frameskip = 0;
	    autoframeskip = 1;
	 }
	 else frameskip++;
      }

      if (showfps == 0) showfpstemp = 2 * Machine->drv->frames_per_second;
   }

   if (input_ui_pressed(IPT_UI_FRAMESKIP_DEC))
   {
      if (autoframeskip)
      {
	 autoframeskip = 0;
	 frameskip = FRAMESKIP_LEVELS-1;
      }
      else
      {
	 if (frameskip == 0) autoframeskip = 1;
	 else frameskip--;
      }
      
      if (showfps == 0)	showfpstemp = 2 * Machine->drv->frames_per_second;
   }
   
   if (!keyboard_pressed(KEYCODE_LSHIFT) && !keyboard_pressed(KEYCODE_RSHIFT)
       && !keyboard_pressed(KEYCODE_LCONTROL) && !keyboard_pressed(KEYCODE_RCONTROL)
       && input_ui_pressed(IPT_UI_THROTTLE))
   {
      throttle ^= 1;
   }
   
   if (input_ui_pressed(IPT_UI_THROTTLE) && (keyboard_pressed(KEYCODE_RSHIFT) || keyboard_pressed(KEYCODE_LSHIFT)))
   {
      sleep_idle ^= 1;
   }
   
   if (!keyboard_pressed(KEYCODE_LSHIFT) && !keyboard_pressed(KEYCODE_RSHIFT)
       && !keyboard_pressed(KEYCODE_LCONTROL) && !keyboard_pressed(KEYCODE_RCONTROL)
       && input_ui_pressed(IPT_UI_SHOW_FPS))
   {
      if (showfpstemp)
      {
	 showfpstemp = 0;
	 need_to_clear_bitmap = 1;
      }
      else
      {
	 showfps ^= 1;
	 if (showfps == 0)
	 {
	    need_to_clear_bitmap = 1;
	 }
      }
   }
   
   if (keyboard_pressed (KEYCODE_LCONTROL))
   { 
      if (keyboard_pressed_memory (KEYCODE_INSERT))
         frameskipper = 0;
      if (keyboard_pressed_memory (KEYCODE_HOME))
         frameskipper = 1;
   }
   
   if (showfpstemp)         /* MAURY_BEGIN: nuove opzioni */
   {
      showfpstemp--;
      if (showfpstemp == 0) need_to_clear_bitmap = 1;
   }

   skip_this_frame = skip_next_frame;
   skip_next_frame =
      (*skip_next_frame_functions[frameskipper])(showfps || showfpstemp);
   
   if (skip_this_frame == 0)
   {
      if (palette_dirty)
      {
	 for (i=0; i<totalcolors; i++)
	 {
	    if(color_dirty[i])
	    {
	       int r, g, b;
	       r = 255 * brightness * pow(current_palette[i*3  ] / 255.0, 1 / gamma_correction) / 100;
	       g = 255 * brightness * pow(current_palette[i*3+1] / 255.0, 1 / gamma_correction) / 100;
	       b = 255 * brightness * pow(current_palette[i*3+2] / 255.0, 1 / gamma_correction) / 100;
	       sysdep_modify_pen(i, r, g ,b);
	       color_dirty[i] = FALSE;
	    }
	 }
	 palette_dirty = FALSE;
      }
      
      profiler_mark(PROFILER_BLIT);
      sysdep_update_display();
      profiler_mark(PROFILER_END);
   }

   if (need_to_clear_bitmap) osd_clearbitmap(bitmap);

#ifndef USE_TIMER	
   osd_audio_update();
#endif
}

void osd_set_gamma(float _gamma)
{
	int i;
	gamma_correction = _gamma;
	for(i=color_start; i<color_end; i++) color_dirty[i] = TRUE;
	palette_dirty = TRUE;
}

float osd_get_gamma(void)
{
	return gamma_correction;
}

/* brightess = percentage 0-100% */
void osd_set_brightness(int _brightness)
{
	int i;
	brightness = _brightness;
	for(i=color_start; i<color_end; i++) color_dirty[i] = TRUE;
	palette_dirty = TRUE;
}

int osd_get_brightness(void)
{
	return brightness;
}

void osd_save_snapshot(void)
{
	save_screen_snapshot();
}

void osd_pause(int paused)
{
	static int current_paused = FALSE;
	static int orig_brightness;
	
	if (bitmap->depth == 8)
	{
		int i;

		if (paused && !current_paused)
		{
		   current_paused  = TRUE;
		   orig_brightness = brightness;
		   brightness     *= 0.65;
		   
		}
		else if (current_paused)
		{
		   current_paused = FALSE;
		   brightness     = orig_brightness;
		}

		for (i = 0;i < 256;i++)
		{
			/* don't touch the user interface text */
			if (i != Machine->uifont->colortable[1])
				color_dirty[i] = TRUE;
		}
		palette_dirty = TRUE;
	}
}
