/*
 * X-Mame video specifics code
 *
 */
#ifdef x11
#define __X11_C_

/*
 * Include files.
 */

#include "xmame.h"
#include "x11.h"
#include "input.h"
#include <math.h>

static void x11_alloc_palette_16bpp(unsigned short *pens);
static void x11_getcolor(int pen, unsigned char* red, unsigned char *green, unsigned char *blue);

struct x_func_struct {
   int  (*init)(void);
   int  (*create_display)(void);
   void (*close_display)(void);
   void (*update_display)(void);
   void (*alloc_palette)(void);
   void (*alloc_palette_16bpp)(unsigned short *pens);
   void (*modify_pen)(int pen, unsigned char red, unsigned char green, unsigned char blue);
   void (*get_pen_16bpp)(int pen, unsigned char *red, unsigned char *green, unsigned char *blue);
};

static struct x_func_struct x_func[X11_MODE_COUNT] = {
{ NULL,
  x11_window_create_display,
  x11_window_close_display,
  x11_window_update_display,
  x11_window_alloc_palette,
  x11_alloc_palette_16bpp,
  x11_window_modify_pen,
  x11_getcolor },
#ifdef USE_DGA
{ xf86_dga_init,
  xf86_dga_create_display,
  xf86_dga_close_display,
  xf86_dga_update_display,
  xf86_dga_alloc_palette,
  x11_alloc_palette_16bpp,
  xf86_dga_modify_pen,
  x11_getcolor }
#else
{ NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
#endif
};

int sysdep_init (void)
{
   int i;
   
   if(!(display = XOpenDisplay (NULL)))
   {
      /* Don't use stderr_file here it isn't assigned a value yet ! */
      fprintf (stderr, "Could not open display\n");
      return OSD_NOT_OK;
   }
   
   for (i=0;i<X11_MODE_COUNT;i++)
   {
      if(x_func[i].create_display)
         mode_available[i] = TRUE;
      else
         mode_available[i] = FALSE;
      
      if(x_func[i].init && (*x_func[i].init)() != OSD_OK)
         return OSD_NOT_OK;
   }
   
   return OSD_OK;
}

void sysdep_close(void)
{
   if(display)
      XCloseDisplay (display);
}

/* This name doesn't really cover this function, since it also sets up mouse
   and keyboard. This is done over here, since on most display targets the
   mouse and keyboard can't be setup before the display has. */
int sysdep_create_display (void)
{
   int i;
   
   /* first setup the keyboard that's the same for all X video modes */
   local_key=xkey;
   memset((void *)&xkey[0], FALSE, 128*sizeof(unsigned char) );
   
   if (x11_video_mode >= X11_MODE_COUNT)
   {
      fprintf (stderr_file,
         "X11-mode %d does not exist, falling back to normal window code\n",
         x11_video_mode);
      x11_video_mode = X11_WINDOW;
   }

   if (!mode_available[x11_video_mode])
   {
      fprintf (stderr_file,
         "X11-mode %d not available, falling back to normal window code\n",
         x11_video_mode);
      x11_video_mode = X11_WINDOW;
   }

   return (*x_func[x11_video_mode].create_display)();
}

void osd_close_display (void)
{
   (*x_func[x11_video_mode].close_display)();
   osd_dirty_close ();
   /* free the bitmap after cleaning the dirty stuff as it uses the bitmap */
   osd_free_bitmap (bitmap);
}

static int shift_red = -1, shift_green = -1, shift_blue = -1;

static void x11_init_colorshifts(void)
{
   unsigned long p;
   
   if(shift_red == -1)
   {
      for(shift_red   = 0,p = 1 << (8 * sizeof(p) - 1);
         p && (!(p & xvisual->red_mask));   p >>= 1,shift_red++);

      for(shift_green = 0,p = 1 << (8 * sizeof(p) - 1);
         p && (!(p & xvisual->green_mask)); p >>= 1,shift_green++);

      for(shift_blue  = 0,p = 1 << (8 * sizeof(p) - 1);
         p && (!(p & xvisual->blue_mask));  p >>= 1,shift_blue++);
   }
}

unsigned long x11_mkcolor(unsigned char red, unsigned char green,
   unsigned char blue)
{
   unsigned long p;
	
   x11_init_colorshifts();
   
   p  = xvisual->red_mask   & ((red   << 24) >> shift_red);
   p |= xvisual->green_mask & ((green << 24) >> shift_green);
   p |= xvisual->blue_mask  & ((blue  << 24) >> shift_blue);

   return p;
}

static void x11_getcolor(int pen, unsigned char* red, unsigned char *green,
   unsigned char *blue)
{
   x11_init_colorshifts();
   
   *red   = ((pen & xvisual->red_mask  ) << shift_red  ) >> 24;
   *green = ((pen & xvisual->green_mask) << shift_green) >> 24;
   *blue  = ((pen & xvisual->blue_mask ) << shift_blue ) >> 24;
}

void sysdep_alloc_palette (void)
{
   (*x_func[x11_video_mode].alloc_palette) ();
}

void sysdep_alloc_palette_16bpp (unsigned short *pens)
{
   (*x_func[x11_video_mode].alloc_palette_16bpp) (pens);
}

static void x11_alloc_palette_16bpp(unsigned short *pens)
{
   int i;
   int r1,g1,b1;
   int r,g,b;
   unsigned long *my_xpixel = xpixel;
      
   for (r1 = 0; r1 < 32; r1++)
   {
      for (g1 = 0; g1 < 32; g1++)
      {
         for (b1 = 0; b1 < 32; b1++)
         {
            r = 255 * brightness * pow(r1 / 31.0, 1 / gamma_correction) / 100;
            g = 255 * brightness * pow(g1 / 31.0, 1 / gamma_correction) / 100;
            b = 255 * brightness * pow(b1 / 31.0, 1 / gamma_correction) / 100;
            if (depth == 16)
               *pens++      = x11_mkcolor(r, g, b);
            else
               *my_xpixel++ = x11_mkcolor(r, g, b);
         }
      }
   }
   if (depth != 16)
      for(i=0; i<32768;i++)
         pens[i] = i;
}

   
void sysdep_modify_pen (int pen, unsigned char red, unsigned char green,
   unsigned char blue)
{
   (*x_func[x11_video_mode].modify_pen) (pen, red, green, blue);
}

void sysdep_get_pen_16bpp(int pen, unsigned char *red, unsigned char *green,
   unsigned char *blue)
{
   (*x_func[x11_video_mode].get_pen_16bpp) (pen, red, green, blue);
}

void sysdep_update_display (void)
{
   extern unsigned short *shrinked_pens;
   int new_video_mode = x11_video_mode;

   if (keyboard_pressed (KEYCODE_LALT))
   { 
      if (keyboard_pressed_memory (KEYCODE_INSERT))
         new_video_mode = X11_WINDOW;
      if (keyboard_pressed_memory (KEYCODE_HOME))
         new_video_mode = X11_DGA;
   }

   if (new_video_mode != x11_video_mode && mode_available[new_video_mode])
   {
      (*x_func[x11_video_mode].close_display)();
      if ((*x_func[new_video_mode].create_display)() != OSD_OK)
      {
         fprintf(stderr_file,
            "X11: Warning: Couldn't create display for new x11-mode\n"
            "   Trying again with the old x11-mode\n");
         (*x_func[new_video_mode].close_display)();
         if ((*x_func[x11_video_mode].create_display)() != OSD_OK)
         {
            osd_close_display();   /* This cleans up and must be called to
                                      restore the videomode with dga */
            osd_exit();
            sysdep_close();
            fprintf (stderr_file,
               "X11: Error: couldn't create new display while switching display modes\n");
            exit (1);              /* ugly, anyone know a better way ? */
         }
      }
      else
         x11_video_mode = new_video_mode;

      if (video_16bit)
         sysdep_alloc_palette_16bpp(shrinked_pens);
      else
      {
         sysdep_alloc_palette ();
         sysdep_mark_palette_dirty ();
      }
      memset((void *)&xkey[0], FALSE, 128*sizeof(unsigned char) );
      osd_mark_dirty (0, 0, bitmap->width - 1, bitmap->height - 1, 1);
      /* poll mouse twice to clear internal vars */
      if (use_mouse)
      {
         sysdep_mouse_poll ();
         sysdep_mouse_poll ();
      }
   }

   (*x_func[x11_video_mode].update_display) ();
}

/* these aren't nescesarry under x11 since we have both a graphics window and
   a textwindow (xterm) */
int sysdep_set_video_mode (void)
{
   return OSD_OK;
}

void sysdep_set_text_mode (void)
{
}

#endif /* ifdef x11 */
