/*
 * OpenStep specific file for XMAME. Here we define OpenStep specific
 * versions of all the MAME functions necessary to get it running under
 * OpenStep. 
 *
 * -bat. 18/12/1998
 */

#import <AppKit/AppKit.h>
#import <libc.h>
#import "xmame.h"
#import "sound.h"
#import "osdepend.h"
#import "driver.h"
#import "keyboard.h"

/*
 * Interface to sound file.
 */

extern int soundkit_init(void);
extern void soundkit_exit(void);

/*
 * Some defines to control various code parameters.
 */

#define BORDER 8
#define FLUSH 30

/*
 * Variables used by command-line DPS window.
 */

static NSView *thisView = nil;
static NSWindow *thisWin = nil;
static NSBitmapImageRep *thisBitmap = nil;
static DPSContext this_dps = NULL;

/*
 * Keyboard variables
 */

static unsigned char ibm_keymap[256];	/* map ascii to ibm keycodes */
static NSDate *the_past = nil;

/*
 * Screen and palette variables
 */

static unsigned short *screen12bit = NULL;
static unsigned short colour_shorts[256];
static unsigned char *new_dirty = NULL;
unsigned char *all_dirty = NULL;	/* not static as inlines need it */
static int palette_changed = 0;

/*
 * Autorelease pool variables.
 */



int sysdep_init(void)
{
   return OSD_OK;
}

void sysdep_close(void)
{
}

/*
 * Shut down the display. We close the dps window and exit.
 */

void
osd_close_display(void)
{
	[thisView retain];
	[thisView lockFocus];

	DPSPrintf(this_dps,"currentwindow termwindow\n");
	DPSPrintf(this_dps,"nulldevice\n");
	PSWait();

	[thisView unlockFocus];
	[thisView release];

	this_dps = NULL;
	thisView = nil;
	thisWin = nil;
}

/*
 * We initialise a few constant objects here and set up the first autorelease
 * pool. This function is also responsible for setting up the sound output
 * devices if enabled by calling the soundkit_init function.
 */

int
sysdep_audio_init(void)
{
	pool = [NSAutoreleasePool new];
	the_past = [NSDate distantPast];
	[the_past retain];

	if(play_sound)
		return soundkit_init();

	return OSD_OK;
}


/*
 * Release everything and finish.
 */

void
sysdep_audio_close(void)
{
	soundkit_exit();
	[the_past release];
	[pool release];
}

/*
 * Keyboard init - all I really do here is set up the keymapping array
 * between ASCII values and MAME keycodes. Yes, it was all done by
 * hand, though there aren't actually that many of them.
 */

void
keyboard_init(void)
{
	int i;

	/* create local array */
	local_key = malloc(256*sizeof(unsigned char));	/* only use first 128 */

	/* zero certain arrays */
	for(i=0;i<256;i++) {
		ibm_keymap[i]=0;
		local_key[i]=0;
	}

	/* and now we set up this big tedious array	*/
	ibm_keymap['0']=KEY_0;
	ibm_keymap['1']=KEY_1;
	ibm_keymap['2']=KEY_2;
	ibm_keymap['3']=KEY_3;
	ibm_keymap['4']=KEY_4;
	ibm_keymap['5']=KEY_5;
	ibm_keymap['6']=KEY_6;
	ibm_keymap['7']=KEY_7;
	ibm_keymap['8']=KEY_8;
	ibm_keymap['9']=KEY_9;

	ibm_keymap['-']=KEY_MINUS;
	ibm_keymap['_']=KEY_MINUS;
	ibm_keymap['+']=KEY_EQUALS;
	ibm_keymap['=']=KEY_EQUALS;
	ibm_keymap['\t']=KEY_TAB;

	ibm_keymap['=']=KEY_EQUALS;
	ibm_keymap['\t']=KEY_TAB;
	ibm_keymap['\r']=KEY_ENTER;
	ibm_keymap['\n']=KEY_ENTER;

	ibm_keymap['q']=KEY_Q;
	ibm_keymap['w']=KEY_W;
	ibm_keymap['e']=KEY_E;
	ibm_keymap['r']=KEY_R;
	ibm_keymap['t']=KEY_T;
	ibm_keymap['y']=KEY_Y;
	ibm_keymap['u']=KEY_U;
	ibm_keymap['i']=KEY_I;
	ibm_keymap['o']=KEY_O;
	ibm_keymap['p']=KEY_P;
	ibm_keymap['[']=KEY_OPENBRACE;
	ibm_keymap[']']=KEY_CLOSEBRACE;

	ibm_keymap['a']=KEY_A;
	ibm_keymap['s']=KEY_S;
	ibm_keymap['d']=KEY_D;
	ibm_keymap['f']=KEY_F;
	ibm_keymap['g']=KEY_G;
	ibm_keymap['h']=KEY_H;
	ibm_keymap['j']=KEY_J;
	ibm_keymap['k']=KEY_K;
	ibm_keymap['l']=KEY_L;
	ibm_keymap[';']=KEY_COLON;
	ibm_keymap[':']=KEY_COLON;
	ibm_keymap['\'']=KEY_QUOTE;
	ibm_keymap['@']=KEY_QUOTE;
	ibm_keymap['~']=KEY_TILDE;
	ibm_keymap['#']=KEY_TILDE;

	ibm_keymap['z']=KEY_Z;
	ibm_keymap['x']=KEY_X;
	ibm_keymap['c']=KEY_C;
	ibm_keymap['v']=KEY_V;
	ibm_keymap['b']=KEY_B;
	ibm_keymap['n']=KEY_N;
	ibm_keymap['m']=KEY_M;
	ibm_keymap[',']=KEY_COMMA;
	ibm_keymap['<']=KEY_COMMA;
	ibm_keymap['.']=KEY_STOP;
	ibm_keymap['>']=KEY_STOP;
	ibm_keymap['/']=KEY_SLASH;
	ibm_keymap['?']=KEY_SLASH;

	ibm_keymap['*']=KEY_ASTERISK;
	ibm_keymap[' ']=KEY_SPACE;

	ibm_keymap[8]=KEY_BACKSPACE;
	ibm_keymap[27]=KEY_ESC;
	ibm_keymap[96]=KEY_NUMLOCK;
	ibm_keymap[127]=KEY_DEL;
}


/*
 * We do nothing with the keyboard to close it
 */

void
sysdep_keyboard_close(void)
{
}

/*
 * We do nothing when asked to poll for mouse movements either.
 */
void
sysdep_mouse_poll(void)
{
}

/*
 * Here we are passed an NSEvent from the keyboard and expected to pass
 * back the MAME keycode associated with it. For ASCII characters there
 * is a simple lookup table, for Unicode characters we use a switch statement
 * with the constants defined in NSEvent.h. This function also deals with
 * using the command key to emulate the function keys.
 */

static inline int
event_to_keycode(NSEvent *keyevent)
{
	int keycode;
	unichar buf[2];	/* just in case theres more than 1 */
	NSString *string = [keyevent charactersIgnoringModifiers];
	[string getCharacters:buf range:NSMakeRange(0,1)];
	keycode = buf[0];

	/* check to see if string is ASCII */
	if([string canBeConvertedToEncoding:NSASCIIStringEncoding]) {
		keycode=ibm_keymap[keycode];
	} else {
		 switch(keycode) {
			 case NSUpArrowFunctionKey:
				keycode=KEY_UP;
				break;
			 case NSDownArrowFunctionKey:
				keycode=KEY_DOWN;
				break;
			 case NSLeftArrowFunctionKey:
				keycode=KEY_LEFT;
				break;
			 case NSRightArrowFunctionKey:
				keycode=KEY_RIGHT;
				break;
			 case NSF1FunctionKey:
				keycode=KEY_F1;
				break;
			 case NSF2FunctionKey:
				keycode=KEY_F2;
				break;
			 case NSF3FunctionKey:
				keycode=KEY_F3;
				break;
			 case NSF4FunctionKey:
				keycode=KEY_F4;
				break;
			 case NSF5FunctionKey:
				keycode=KEY_F5;
				break;
			 case NSF6FunctionKey:
				keycode=KEY_F6;
				break;
			 case NSF7FunctionKey:
				keycode=KEY_F7;
				break;
			 case NSF8FunctionKey:
				keycode=KEY_F8;
				break;
			 case NSF9FunctionKey:
				keycode=KEY_F9;
				break;
			 case NSF10FunctionKey:
				keycode=KEY_F10;
				break;
			 case NSF11FunctionKey:
				keycode=KEY_F11;
				break;
			 case NSF12FunctionKey:
				keycode=KEY_F12;
				break;
			 case NSInsertFunctionKey:
				keycode=KEY_INSERT;
				break;
			 case NSDeleteFunctionKey:
				keycode=KEY_DEL;
				break;
			 case NSHomeFunctionKey:
				keycode=KEY_HOME;
				break;
			 case NSEndFunctionKey:
				keycode=KEY_END;
				break;
			 case NSPageUpFunctionKey:
				keycode=KEY_PGUP;
				break;
			 case NSPageDownFunctionKey:
				keycode=KEY_PGDN;
				break;
			 default:
				keycode=0;
		 }
	 }

	/* deal with command key */
	if([keyevent modifierFlags] & NSCommandKeyMask)
		switch(keycode) {
			case KEY_1:
				keycode=KEY_F1;
				break;
			case KEY_2:
				keycode=KEY_F2;
				break;
			case KEY_3:
				keycode=KEY_F3;
				break;
			case KEY_4:
				keycode=KEY_F4;
				break;
			case KEY_5:
				keycode=KEY_F5;
				break;
			case KEY_6:
				keycode=KEY_F6;
				break;
			case KEY_7:
				keycode=KEY_F7;
				break;
			case KEY_8:
				keycode=KEY_F8;
				break;
			case KEY_9:
				keycode=KEY_F9;
				break;
			case KEY_0:
				keycode=KEY_F10;
				break;
			case KEY_MINUS:
				keycode=KEY_F11;
				break;
			case KEY_EQUALS:
				keycode=KEY_F12;
				break;
		}

	return keycode;
}

/*
 * This function is used to mark blocks of the screen as dirty. We use it to
 * mark dirty lines for the sake of speed.
 */

void
osd_mark_dirty(int xmin, int ymin, int xmax, int ymax, int ui)
{
	int y;
	for(y=ymin;y<=ymax;y++)
		new_dirty[y] = all_dirty[y] = 1;
}

/*
 * This loop collect all events from the queue and deals with them by
 * updating the local_key array for key up and down events. Anything we
 * dont use gets passed on the whoever might want it.
 */

void
sysdep_update_keyboard(void)
{
	NSEvent *event=nil;

	for(;;) {
		/* get the next event */
		event= [NSApp nextEventMatchingMask:NSAnyEventMask
				untilDate:the_past inMode:NSDefaultRunLoopMode
				dequeue:YES];
		/* break out of the loop if there are no more events */
		if(event==nil)
			break;

		/* deal with the event */
		switch([event type]) {
			case NSKeyDown:
				local_key[event_to_keycode(event)]=1;
				break;
			case NSKeyUp:
				local_key[event_to_keycode(event)]=0;
				break;
			default:
				[NSApp sendEvent:event]; break;
		}
	}
}

/*
 * We don't implement remapping of keys, but we return OK anyway...
 */

int
sysdep_mapkey(char *arg)
{
	return OSD_OK;
}

/*
 * Allocate the palette - in this case we set the whole lot to black.
 */

void
sysdep_alloc_palette(void)
{
	int i;
	for(i=0;i<256;i++)
		colour_shorts[i] = 0;
}

void sysdep_alloc_palette_16bpp(unsigned short *pens)
{
}

/*
 * This converts an 8 bit value to it's nearest 4 bit equivalent.
 */

static inline unsigned char
byte_to_4(unsigned int x)
{
	float approx= (x/17.0)+0.5;
	int value= approx;
	if(value>15)
		value=15;
	return value;
}

/*
 * Set a particular pen colour. We take the R, G and B colours, create a
 * 12 bit equivalent and then insert this into the palette array. If the
 * palette has been changed then we set a flag to indicate this so we know
 * that we need to undaoe the whole display.
 */

void
sysdep_modify_pen(int pen, unsigned char r ,unsigned char g, unsigned char b)
{
	unsigned char *ptr= (unsigned char*)&(colour_shorts[pen]);
	unsigned char b1 = (byte_to_4(r) << 4) | byte_to_4(g);
	unsigned char b2 = (byte_to_4(b) << 4);

	/* check byte 1 */
	if(b1 != *ptr) {
		*ptr = b1;
		palette_changed = 1;
	}

	/* check byte 2 */
	ptr++;
	if(b2 != *ptr) {
		*ptr = b2;
		palette_changed = 1;
	}
}

void sysdep_get_pen_16bpp(int pen, unsigned char* red, unsigned char *green,
   unsigned char *blue)
{
}

/*
 * Create the display. We create a window of the appropriate size, taking into
 * account any scaling. The keyboard is now initialised here as well...
 */

int
sysdep_create_display(void)
{
	NSRect content_rect = { {100,100}, {0,0} };

	if(video_16bit)
	{
	   fprintf(stderr_file, "%s doesn't support 16bpp video modes\n", title);
	   return OSD_NOT_OK;
	}

	keyboard_init();

	/* set the size of the view */
	content_rect.size.width = visual_width * widthscale;
	content_rect.size.height = visual_height * heightscale;
	content_rect.size.width += (BORDER*2) - 1;
	content_rect.size.height += (BORDER*2) - 1;

	/* allocate memory for dirty buffers */
	new_dirty=malloc(bitmap->height);
	all_dirty=malloc(bitmap->height);
	if(!new_dirty || !all_dirty) {
		fprintf(stderr,"dirty memory allocate failed\n");
		return OSD_NOT_OK;
	}
	memset(new_dirty, 0, bitmap->height);
	memset(all_dirty, 0, bitmap->height);

	/* allocate memory for 12 bit colour version */
	screen12bit=malloc(2*visual_width*visual_height);
	if(!screen12bit) {
		fprintf(stderr,"12 bit memory allocate failed\n");
		return OSD_NOT_OK;
	}

	/* create bitmap object  */
	thisBitmap = [[NSBitmapImageRep alloc]
		initWithBitmapDataPlanes:(void*)&screen12bit
		pixelsWide:visual_width pixelsHigh:visual_height
		bitsPerSample:4 samplesPerPixel:3
		hasAlpha:NO isPlanar:NO
		colorSpaceName:NSDeviceRGBColorSpace
		bytesPerRow:2*visual_width bitsPerPixel:16];
	if(!thisBitmap) {
		fprintf(stderr,"Bitmap creation failed\n");
		return OSD_NOT_OK;
	}

	/* create an application object and a window */
	NSApp = [[NSApplication sharedApplication] retain];
	thisWin = [[NSWindow alloc] initWithContentRect:content_rect
		styleMask:(NSTitledWindowMask | NSMiniaturizableWindowMask)
		backing:NSBackingStoreRetained defer:NO];
	[thisWin setTitle:[NSString
		stringWithCString:Machine->gamedrv->description]];
	puts(Machine->gamedrv->description);

	/* bring to front and set the view variable */
	[thisWin makeKeyAndOrderFront:nil];
	thisView = [thisWin contentView];

	/* initialise it */
	[thisView lockFocus];
	this_dps = DPSGetCurrentContext();
	DPSPrintf(this_dps,"initgraphics 0 setgray\n");
	DPSFlushContext(this_dps);
	PSrectfill(0,0,content_rect.size.width+1,content_rect.size.height+1);
	PSWait();
	[thisView unlockFocus];

	return OSD_OK;
}

/*
 * Dump a copy of the screen to a file in /tmp. The file number increments
 * by one very time to give unique names.
 */

void
screen_dump(void)
{
	static int dumpno = 1;
	char outname[MAXPATHLEN]; /* enough space for whole filename */
	NSData *tiff_data;

	/* make the name */
	sprintf(outname,"/tmp/xmame%03d.tiff",dumpno++);
	printf("Writing screen dump to %s\n",outname);

	/* this object will be tidied up by the autorelease mechanism */
	tiff_data = [thisBitmap
		TIFFRepresentationUsingCompression:NSTIFFCompressionLZW
		factor:1.0];

	[tiff_data writeToFile:[NSString stringWithCString:outname]
		atomically:NO];
}

/*
 * Update the display.  We create the bitmapped data for the current frame
 * and draw it into the window.
 */

void
sysdep_update_display(void)
{
	static int flush_frame = FLUSH;
	unsigned short *row12= screen12bit;
	int colstart=visual.min_x, colend=visual.max_x;
	int rowstart=visual.min_y, rowend=visual.max_y;
	int row, col, col_len = 1 + colend - colstart;

	/* copy any dirty rows */
	for(row=rowstart;row<=rowend;row++) {
		if(palette_changed || !use_dirty || all_dirty[row]) {
			unsigned char *row8 = bitmap->line[row] + colstart;
			for(col=colstart;col<=colend;col++)
				*row12++ = colour_shorts[*row8++];
			/* reset the masks */
			all_dirty[row] = new_dirty[row];
			new_dirty[row] = 0;
		} else
			row12 += col_len;
	}
	palette_changed = 0;

	/* lock focus and draw it */
	[thisView lockFocus];
	[thisBitmap drawInRect:(NSRect){ {BORDER,BORDER},
		{visual_width*widthscale,visual_height*heightscale}}];

	/* deal with autorelease pool every FLUSH frames */
	if(flush_frame-- <= 0) {
		[pool release];
		pool=[[NSAutoreleasePool alloc] init];
		flush_frame=FLUSH;
	}

	/* wait for all PS operations too finish and unlock */
	PSWait();
	[thisView unlockFocus];
}

void osd_led_w(int led,int on) 
{
}
