;==============================================================================

; file: "_nonstd.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(##include "header.scm")

;------------------------------------------------------------------------------

; Non-standard procedures

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(##define-macro (define-runtime-macro pattern . rest)

  (define (form-size parms) ; this definition must match the one in "_eval.scm"
    (let loop ((l parms) (n 1))
      (if (pair? l)
        (loop (cdr l) (+ n 1))
        (if (null? l) n (- 0 n)))))

  `(##top-cte-add-macro!
     ##interaction-cte
     ',(car pattern)
     (##cons ',(form-size (cdr pattern))
             (lambda ,(cdr pattern) ,@rest))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (exit #!optional (status (absent-obj)))
  (if (##eq? status (absent-obj))
    (##exit)
    (force-vars (status)
      (##exit status))))

(define (error msg . args)
  (##call-with-current-continuation
    (lambda (cont) (##sequentially (lambda ()
      (##identify-error
       "ERROR"
       #f
       (##frame-locat (##continuation->first-frame cont))
       msg
       args
       '())
      (##debug-repl cont #t))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define-runtime-macro (include filename)
  `(##include ,filename))

(define-runtime-macro (define-macro pattern body . rest)
  `(##define-macro ,pattern ,body ,@rest))

(define-runtime-macro (declare . rest)
  `(##declare ,@rest))

(define-runtime-macro (namespace . rest)
  `(##namespace ,@rest))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define-runtime-macro (dynamic-define var val)
  `(##dynamic-define ',var ,val))

(define-runtime-macro (dynamic-ref var)
  `(##dynamic-ref ',var))

(define-runtime-macro (dynamic-set! var val)
  `(##dynamic-set! ',var ,val))

(define-runtime-macro (dynamic-let bindings body . rest)
  (##dynamic-let-build
   (##cons 'dynamic-let (##cons bindings (##cons body rest)))))

(define (##dynamic-let-build src)

  (define (build src bindings)
    (if (##pair? bindings)
      (let ((binding (##car bindings)))
        (##shape src (##sourcify binding src) 2)
        (let ((x (##car binding)))
          (if (##not (##symbol? x))
            (##signal '##signal.syntax-error src "Identifier expected"))
          (let ((rest (build src (##cdr bindings))))
            (##cons (##list '##cons (##list 'quote x) (##cadr binding))
                    rest))))
      (if (##null? bindings)
        '()
        (##signal '##signal.syntax-error src "Ill-terminated bindings"))))

  (##list '##dynamic-let
          (##cons '##list
                  (build (##sourcify src (##make-source #f #f))
                         (##cadr src)))
          (##cons 'lambda (##cons '() (##cddr src)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define-runtime-macro (define-structure name . field-defs)

  (define (err)
    (##signal '##signal.syntax-error
              (##sourcify
               (##cons 'define-structure (##cons name field-defs))
               (##make-source #f #f))
              "Ill-formed special form:"
              'define-structure))

  (define (sym . strings)
    (##string->symbol (##apply ##string-append strings)))

  (if (##symbol? name)
    (let ((name-str (##symbol->string name)))

      (define (generate-accessors fields i tag)
        (if (##null? fields)
          '()
          (let* ((field (##car fields))
                 (field-str (##symbol->string field))
                 (field-ref (sym name-str "-" field-str))
                 (field-set! (sym name-str "-" field-str "-set!")))
            (##cons `(define ,field-set!
                       (lambda (x y)
                         (##declare (extended-bindings))
                         (##structure-set! x ,i y ',tag)))
                    (##cons `(define ,field-ref
                               (lambda (x)
                                 (##declare (extended-bindings))
                                 (##structure-ref x ,i ',tag)))
                            (generate-accessors (##cdr fields)
                                                (##fixnum.+ i 1)
                                                tag))))))

      (define (generate-definitions rev-all-fields rev-printed-fields)
        (let* ((all-fields (##reverse rev-all-fields))
               (printed-fields (##reverse rev-printed-fields))
               (tag (##list->vector
                      (##cons name
                              (##cons (##length printed-fields)
                                      all-fields)))))
          `(begin
             ,@(generate-accessors all-fields 1 tag)
             (define ,(sym name-str "?")
               (lambda (x)
                 (##declare (extended-bindings))
                 (and (##structure? x)
                      (##eq? (##vector-ref x 0) ',tag))))
             (define ,(sym "make-" name-str)
               (lambda (,@all-fields)
                 (##declare (extended-bindings))
                 (##subtype-set! (##vector ',tag ,@all-fields)
                                 ,(subtype-structure)))))))

      (let loop1 ((l1 field-defs) (l2 '()))
        (if (##pair? l1)
          (let ((rest (##cdr l1)) (field (##car l1)))
            (cond ((##symbol? field)
                   (loop1 rest (##cons field l2)))
                  ((and (or (##null? field) (##pair? field))
                        (##not (##pair? rest)))
                   (let ((printed-fields l2))
                     (let loop2 ((l1 field) (l2 l2))
                       (if (##pair? l1)
                         (let ((rest (##cdr l1)) (field (##car l1)))
                           (cond ((##symbol? field)
                                  (loop2 rest (##cons field l2)))
                                  (else
                                   (err))))
                         (generate-definitions l2 printed-fields)))))
                  (else
                   (err))))
          (generate-definitions l2 l2))))
    (err)))

(define (##structure-ref obj i tag)
  (if (and (##structure? obj)
           (##eq? (##vector-ref obj 0) tag))
    (##vector-ref obj i)
    (##structure-error tag i #f (##list obj))))

(define (##structure-set! obj i v tag)
  (if (and (##structure? obj)
           (##eq? (##vector-ref obj 0) tag))
    (begin (##vector-set! obj i v) (##void))
    (##structure-error tag i #t (##list obj v))))

(define (##structure-error tag i set!? args)
  (let ((name-str (##symbol->string (##vector-ref tag 0)))
        (field-str (##symbol->string (##vector-ref tag (##fixnum.+ i 1)))))
    (##runtime-error (##string-append "Structure of type `"
                                      name-str
                                      "' expected")
                     (##string->symbol
                       (if set!?
                         (##string-append name-str "-" field-str "-set!")
                         (##string-append name-str "-" field-str)))
                     args)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (set-gc-report! report?)
  (set! ##gc-report report?)
  (##void))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (void)
  (##void))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (runtime)
  (let* ((v (##cpu-time))
         (user (##vector-ref v 0))
         (sys (##vector-ref v 1)))
    (##* (##+ user sys) 1e-9)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (cpu-time)
  (##cpu-time))

(define (real-time)
  (##real-time))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define-runtime-macro (time expr)
  `(##time (lambda () ,expr) ',expr))

(define (##time thunk expr)
  (let ((at-start (##process-statistics)))
    (let ((result (thunk)))
      (let ((at-end (##process-statistics)))
        (let* ((out
                (##repl-out))
               (rt
                (##repl-readtable))
               (real-nsecs
                (##- (##vector-ref at-end 0) (##vector-ref at-start 0)))
               (user-nsecs
                (##- (##vector-ref at-end 1) (##vector-ref at-start 1)))
               (sys-nsecs
                (##- (##vector-ref at-end 2) (##vector-ref at-start 2)))
               (gc-user-nsecs
                (##- (##vector-ref at-end 3) (##vector-ref at-start 3)))
               (gc-sys-nsecs
                (##- (##vector-ref at-end 4) (##vector-ref at-start 4)))
               (nb-gcs
                (##- (##vector-ref at-end 5) (##vector-ref at-start 5)))
               (minflt
                (##- (##vector-ref at-end 9) (##vector-ref at-start 9)))
               (majflt
                (##- (##vector-ref at-end 10) (##vector-ref at-start 10)))
               (bytes-allocated
                (##- (##- (##vector-ref at-end 6) (##vector-ref at-start 6))
                     (##+ (if (##interp-procedure? thunk)
                            (##vector-ref at-end 7) ; thunk call frame space
                            0)
                          (##vector-ref at-end 8))))) ; at-end structure space

          (define (pluralize n msg)
            (##write-string "    " out)
            (if (##= n 0)
              (##write-string "no" out)
              (##write n out rt #f))
            (##write-string msg out)
            (if (##not (##= n 1))
              (##write-string "s" out)))

          (##write (##list 'time expr) out rt #f)
          (##newline out)

          (##write-string "    " out)
          (##write (##round (##/ real-nsecs 1000000)) out rt #f)
          (##write-string " ms real time" out)
          (##newline out)

          (##write-string "    " out)
          (##write (##round (##/ (##+ user-nsecs sys-nsecs) 1000000)) out rt #f)
          (##write-string " ms cpu time (" out)
          (##write (##round (##/ user-nsecs 1000000)) out rt #f)
          (##write-string " user, " out)
          (##write (##round (##/ sys-nsecs 1000000)) out rt #f)
          (##write-string " system)" out)
          (##newline out)

          (pluralize nb-gcs " collection")
          (if (##not (##= nb-gcs 0))
            (begin
              (##write-string " accounting for " out)
              (##write (##round (##/ (##+ gc-user-nsecs gc-sys-nsecs) 1000000)) out rt #f)
              (##write-string " ms cpu time (" out)
              (##write (##round (##/ gc-user-nsecs 1000000)) out rt #f)
              (##write-string " user, " out)
              (##write (##round (##/ gc-sys-nsecs 1000000)) out rt #f)
              (##write-string " system)" out)))
          (##newline out)

          (pluralize bytes-allocated " byte")
          (##write-string " allocated" out)
          (##newline out)

          (pluralize minflt " minor fault")
          (##newline out)

          (pluralize majflt " major fault")
          (##newline out)

          result)))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define ##gensym-count -1)

(define (gensym #!optional (p (absent-obj)))
  (force-vars (p)
    (let ((prefix (if (##eq? p (absent-obj)) 'g p)))
      (check-symbol prefix (gensym p)
        (##make-uninterned-symbol
          (##string-append (##symbol->string prefix)
                           (begin
                             (set! ##gensym-count (##+ ##gensym-count 1))
                             (##number->string ##gensym-count 10))))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (will? x)
  (force-vars (x)
    (##will? x)))

(define (make-will testator #!optional (action (absent-obj)))
  (if (##eq? action (absent-obj))
    (##make-will testator)
    (force-vars (action)
      (check-procedure action (make-will testator action)
        (##make-will testator action)))))

(define (will-testator x)
  (force-vars (x)
    (check-will x (will-testator x)
      (##will-testator x))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (argv)
  ##processed-argv)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (getenv name)
  (force-vars (name)
    (check-string name (getenv name)
      (##getenv name))))

;------------------------------------------------------------------------------
