package biss.awt;

import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;

/**
 * An IndexColorModel which can be shrink-wrapped when all required
 * colors are known. The standard jawa.awt.image.IndexColorModel always
 * allocates storage for 256 color entries (even if just a single color
 * is used). Just here to optimize the memory footprint of biss.awt apps.
 *
 * BEWARE: the standard JDK (1.0.2 at least) seems to have the standard
 * ColorModels hardwired in its native layer. Therefore you cannot 
 * use MutableIndexColorModel as a general replacement of
 * java.awt.image.IndexColorModel (you will get SIGSEGVs, no exceptions!)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.M.Mehlitz
 */
public class MutableIndexClrModel
  extends ColorModel
{
	int Rgb[];
	int Transparent;

public MutableIndexClrModel ( IndexColorModel icm ) {
	super(8);

	int i, n = icm.getMapSize();
	Rgb = new int[n];
	Transparent = icm.getTransparentPixel();

	for ( i=0; i<n; i++ )
		Rgb[i] = icm.getRGB( i);
}

public MutableIndexClrModel ( int bits, int size, byte rgb[], int start,
                       boolean hasAlpha, int transparent ) {
	super(bits);
	int i, j;

	Rgb = new int[size];
	Transparent = transparent;

	if ( hasAlpha ) {
		for ( i=0, j=start; i<size; i++ ) {
			Rgb[i] = ((rgb[j++] & 0xff) << 16) |
			((rgb[j++] & 0xff) << 8) | (rgb[j++] & 0xff) |
			((rgb[j++] & 0xff) << 24);
		}
	}
	else {
		for ( i=0, j=start; i<size; i++ ) {
			Rgb[i] = ((rgb[j++] & 0xff) << 16) |
			((rgb[j++] & 0xff) << 8) | (rgb[j++] & 0xff) |
			0xff000000;
		}	
	}
}

public MutableIndexClrModel ( int rgb[], int transparent ) {
	super(8);

	Rgb = rgb;
	Transparent = transparent;
}

public int getAlpha ( int pixel ) {
	return ( Rgb[pixel] >> 24) & 0xff;
}

public int getBlue ( int pixel ) {
	return Rgb[pixel] & 0xff;
}

public int getGreen ( int pixel ) {
	return (Rgb[pixel] >> 8) & 0xff;
}

public int getMapSize () {
	return Rgb.length;
}

public int getRGB ( int pixel ) {
	return Rgb[pixel];
}

public int getRed ( int pixel ) {
	return (Rgb[pixel] >> 16) & 0xff;
}

public void setRgbs ( int[] newRgbs ) {
	Rgb = newRgbs;
}

public void setTransparentPixel ( int idx ) {
	Transparent = idx;
}

public void shrinkRgbs ( int maxIdx ) {
	int n = maxIdx + 1;
	int[] newRgb = new int[n];

	System.arraycopy( Rgb, 0, newRgb, 0, n);
	Rgb = newRgb;
}
}
