// 
// $Id: qaregexp.cpp,v 1.14 1999/08/16 00:49:44 amos Exp $
//
// Implementation of QaRegExp class
//
// Jan Borsodi <amos@abn.hibu.no>
// Created on: <14-Jul-1999 19:14:57 amos>
//


#include "qaregexp.hpp"
#include "qaregexperror.hpp"
#include <qstring.h>
#include <qstringlist.h>
#include <regex.h>

/*!
  \class QaRegExp qaregexp.hpp
  \brief Matches Regular Expressions against strings.

  \warning This version does not support unicode strings, they are converted to ascii before being used.

  This class uses the system calls reg_comp and reg_exec, see the man pages for regex for more information
  on regular expressions.

  Example:
  \code
  // Match for <text> = <text>
  QaRegExp reg( "([a-zA-Z]+)[ \t]*=[ \t]*([a-zA-Z]+)" );

  // Now match the string
  if ( reg.match( "RegExpTest = True" ) )
  {
      QaRegExpMatch m = reg.matches();

      // Now print out all matches
      for ( QStringList::ConstIterator it = m.toString().begin(); it != m.toString().end(); ++it )
      {
          qDebug( "\"%s\"", (*it).ascii() );
      }
  }

  // Or use it like this
  if ( reg.match( "RegExpTest = True", 2 ) )
      qDebug( "%s", reg.matchString( 2 ).ascii() );

  \endcode
*/

/*!
  Creates a regexp query.
  \param r The string to use for regular expression matching.
  \param c True means match case sensitive.
*/

QaRegExp::QaRegExp( const QString &r, bool c )
{
    Case = c;
    Alloced = false;
    Subs = true;
    Flags = REG_EXTENDED;
    RegString = r;
    compileRegExp();
}

QaRegExp::QaRegExp( const QaRegExp &r )
{
    RegString = r.RegString;
    String = r.String;
    Flags = r.Flags;
    Case = r.Case;
    Subs = r.Subs ;
    Matches = r.Matches;
    compileRegExp();
}

/*!
  Destroys the object
*/

QaRegExp::~QaRegExp()
{
    regfree( &RegComp );
}

/*!
  \return True if matching is done case sensitive.
*/

bool QaRegExp::caseSensitive() const
{
    return Case;
}

/*!
  Sets the case sensitivity flag.
*/

void QaRegExp::setCaseSensitive( bool s )
{
    if ( s == Case )
        return;
    if ( s == false )
        Flags |= REG_ICASE;
    else
        Flags -= Flags & REG_ICASE;
    Case = s;
    compileRegExp();
}

void QaRegExp::setSubReg( bool b )
{
    if ( b == Subs )
        return;
    if ( b == false )
        Flags |= REG_NOSUB;
    else
        Flags -= Flags & REG_NOSUB;
    Subs = b;
    compileRegExp();
}


/*!
  Matches a string against the regexp on the given sub expression.
  \param s The string to match on.
  \param sub Which sub expression should be checked, 0 is default and means the whole regexp.
  \return True if the subexpression matched.
*/

bool QaRegExp::match( const QString &s, int sub = 0 )
{
    setSubReg( true );
    regmatch_t *tmp = new regmatch_t[NumSub + 1];
    for ( int i = 0; i < NumSub + 1; i++ )
    {
        tmp[i].rm_so = -1;
        tmp[i].rm_eo = -1;
    }
    String = s;
    const char *str = String.ascii();
    if ( str == 0 )
        str = "";
    int b = regexec( &RegComp, str, NumSub + 1, tmp, 0 );
    Matches = QaRegExpMatch( String, tmp, NumSub + 1 );
    delete []tmp;
    if ( sub == 0 )
        return b == 0;
    else
    {
        return ( tmp[sub].rm_so != -1 );
    }
}

/*!
  Matches a string against the regexp on the given sub expression, will match as
  many times as possible and extracting the wanted subexpression every time it
  matches.
  \param s The string to match on.
  \param sub Which sub expression should be checked, 0 is default and means the whole regexp.
  \return Number of matches.
*/

int QaRegExp::matchMultiple( const QString &s, int sub = 0 )
{
    setSubReg( true );
    regmatch_t *tmp = new regmatch_t[NumSub + 1];
    for ( int i = 0; i < NumSub + 1; i++ )
    {
        tmp[i].rm_so = -1;
        tmp[i].rm_eo = -1;
    }
    String = s;
    QString right = s;

    int err, start, count;
    start = 0;
    count = 0;
    bool ok = true;
    Matches = QaRegExpMatch( String );
    while ( ok )
    {
        const char *str = right.ascii();
        if ( str == 0 )
            str = "";
        if ( (err = regexec( &RegComp, str, NumSub + 1, tmp, 0 ) ) == 0 )
        {
            Matches.appendMatch( QaRegExpRange( tmp[sub].rm_so + start, tmp[sub].rm_eo + start ) );
            if ( tmp[0].rm_eo == tmp[0].rm_so )
                start++;
            else
                start += tmp[0].rm_eo;
            if ( tmp[0].rm_eo >= 0 )
            {
                if ( tmp[0].rm_eo == tmp[0].rm_so )
                {
                    if ( (unsigned int)tmp[0].rm_eo + 1 < right.length() )
                        right = right.mid( tmp[0].rm_eo + 1 );
                    else
                        ok = false;
                }
                else
                {
                    if ( (unsigned int)tmp[0].rm_eo < right.length() )
                        right = right.mid( tmp[0].rm_eo );
                    else
                        ok = false;
                }
            }
            else
                ok = false;
            count++;
        }
        else
            ok = false;
    }
    delete []tmp;
    return count;
}

/*!
  Splits up the string using the regexp string as the delimiter.
  \param s The string to split.
*/

void QaRegExp::split( const QString &s )
{
    setSubReg( true );
    regmatch_t *tmp = new regmatch_t[NumSub + 1];
    String = s;
    QString right = s;
    Matches = QaRegExpMatch( String );
    int err, count, start;
    start = 0;
    count = 0;
    while ( ( err = regexec( &RegComp, right.ascii(), NumSub + 1, tmp, 0 ) ) == 0 )
    {
        Matches.appendMatch( QaRegExpRange( start, tmp[0].rm_so + start ) );
        start += tmp[0].rm_eo;
        count++;
        right = right.mid(tmp[0].rm_eo);
    }
    Matches.appendMatch( QaRegExpRange( start, String.length()) );
    delete []tmp;
}

/*!
  \return The string that matched the sub expression.
*/

QString QaRegExp::matchString( int sub = 0 )
{
    return Matches.toString(sub);
}

/*!
  \return A list of sub expression strings.
*/

QStringList QaRegExp::matchStrings()
{
    return Matches.toString();
}

/*!
  \return The start offset of the matched sub expression.
*/

int QaRegExp::matchBeginning( int sub = 0 ) const
{
    return Matches[sub].start();
}

/*!
  \return The start offset of the matched sub expression.
*/

int QaRegExp::matchEnd( int sub = 0 ) const
{
    return Matches[sub].end();
}

/*!
  \return A class with all matching information.
*/

const QaRegExpMatch &QaRegExp::matches() const
{
    return Matches;
}

void QaRegExp::compileRegExp()
{
    if ( Alloced )
    {
        regfree( &RegComp );
        Alloced = true;
    }
    int err = regcomp( &RegComp, RegString.ascii(), Flags );
    if ( err != 0 )
    {
        char *tmp;
        int size = regerror( err, &RegComp, 0, 0 );
        tmp = new char[size];
        regerror( err, &RegComp, tmp, size );
//          qDebug( "Error compiling regexp: %s", tmp );
        QString s;
        s.sprintf( "RegExp error: %s", tmp );
        delete []tmp;
        throw QaRegExpError( err, s );
    }
    NumSub = RegComp.re_nsub;
}

/*!
  \return The subexpression count, excluding the master expression.
*/

int QaRegExp::subCount() const
{
    return NumSub;
}

QaRegExp &QaRegExp::operator =( const QaRegExp &r )
{
    RegString = r.RegString;
    String = r.String;
    Flags = r.Flags;
    Case = r.Case;
    Subs = r.Subs;
    Matches = r.Matches;
    compileRegExp();
    return *this;
}

void QaRegExp::setExpression( const QString &r )
{
    RegString = r;
    compileRegExp();
}
