/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

/* Monitor stuff from /proc/stat.
*/

#include "gkrellm.h"

#include <math.h>
#include <utmp.h>

#define	MAX_CPUS	5

gint		smp_cpus;


static GtkWidget
			*cpu_vbox[MAX_CPUS],
			*disk_vbox,
			*proc_vbox;

static gint
			cpu_enabled[MAX_CPUS],
			disk_enabled,
			proc_enabled;

static gint
			cpu_height[MAX_CPUS],
			disk_height,
			proc_height;

static gint	proc_extra,
			disk_extra,
			cpu_extra[MAX_CPUS];

  /* Do a quick 4 cpu thing, later make it general with a linked list of cpu's
  |  Make room for 5 to include the cumulative cpu monitor.
  */
static Chart	cpu[MAX_CPUS];
static Chart	proc;
Chart			disk;			/* Used in misc tab in gui.c */

static gint		n_cpus;

static gint		current_processes;

gint
draw_chart_label(Chart *cp, GdkFont *font, gint x, gint y, gchar *s)
	{
	gint	w, l, r, a, d, e;

	gdk_string_extents(font, s, &l, &r, &w, &a, &d);
	e = effect_string_value(GK.alt2_effect_string);

#if 0
	gdk_draw_pixmap(cp->drawing_area->window, GK.draw1_GC,cp->bg_grided_pixmap,
			x - l, y - a, x - l, y - a, l + r, a + d + e);
    draw_string(cp->drawing_area->window, font, &GK.alt2_color, e, x, y, s);
#endif

	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC,cp->bg_grided_pixmap,
			x - l, y - a, x - l, y - a, l + r, a + d + e);
    draw_string(cp->pixmap, font, &GK.alt2_color, e, x, y, s);
	gdk_draw_pixmap(cp->drawing_area->window, GK.draw1_GC, cp->pixmap,
			x - l, y - a, x - l, y - a, l + r, a + d + e);

	return x + r;
	}

void
draw_proc_extra()
	{
	struct utmp		*ut;
	gint			n_users  = 0;
	gint			x, x1, y;
	gchar			buf[32];

	sprintf(buf, "%d", current_processes);
	y = 4 + label_font_ascent;
	x1 = draw_chart_label(&proc, GK.label_font, 4, y, buf);
	draw_chart_label(&proc, GK.alt_font, x1 + 2, y, "procs");

	setutent();
	while ((ut = getutent()) != NULL)
		if (   ut->ut_type == USER_PROCESS
			&& ut->ut_name[0] != '\0'
		   )
		++n_users;
	endutent();
	sprintf(buf, "%d", n_users);
	y += 2 + label_font_ascent;
	x = draw_chart_label(&proc, GK.label_font, 4, y, buf);
	if (x < x1)
		x = x1;
	draw_chart_label(&proc, GK.alt_font, x + 2, y, "users");
	}

void
draw_disk_extra(unsigned long l)
	{
	gchar	buf[32];

	sprintf(buf, "%d", (gint) l);
	draw_chart_label(&disk, GK.label_font, 4, 4 + label_font_ascent, buf);
	}

void
draw_cpu_extra(Chart *cp, unsigned long l, unsigned long total)
	{
	gint	n;
	gchar	buf[32];

	if (total == 0)
		return;
	n = ((200 * l / total) + 1) / 2;
	if (n > 100)
		n = 100;
	sprintf(buf, "%d%%", n);
	draw_chart_label(cp, GK.label_font, 4, label_font_ascent + 4, buf);
	}

void
draw_temperature_decal(Chart *cp)
	{
	Panel	*p;
	Decal	*d;
	gchar	*name, buf[16];
	gint	t;

	p = &cp->panel;
	name = strcmp(cp->name, "Proc") ? cp->name : "mb";
	if (   p->decal == NULL
		|| p->decal->pixmap == NULL
		|| p->decal->type == DECAL_ZOMBIE
		|| (t = read_temperature(name)) == 0
	   )
		return;
	d = p->decal;
	if (t == d->value)
		return;

	sprintf(buf, "%d%c", t, UC.sensor_temp_fahrenheit ? 'F' : 'C');

	gdk_draw_pixmap(d->pixmap, GK.draw1_GC, p->background,
			d->x, d->y,  0, 0,  d->w, d->h);

    draw_string(d->pixmap, d->text_style.font, &d->text_style.color,
					d->text_style.effect, 0, d->y_baseline, buf);
	d->value = t;
	d->modified = TRUE;
	}

  /* /proc/stat has cpu entries like:
  |		cpu		total_user	total_nice	total_sys	total_idle
  |		cpu0	cpu0_user	cpu0_nice	cpu0_sys	cpu0_idle
  |			...
  |		cpuN	cpuN_user	cpuN_nice	cpuN_sys	cpuN_idle
  |  where ticks for cpu are jiffies * smp_num_cpus
  |  and ticks for cpu[i] are jiffies
  |  and jiffies is 1/100 second.
  */
static void
register_cpus()
	{
	FILE	*f;
	gchar	buf[160], *s;

	n_cpus = 0;

#ifdef SIMULATE_SMP
		f = fopen("/ram/stat", "r");
#else
		f = fopen("/proc/stat", "r");
#endif
	if (f == NULL)
		return;

	while (fgets(buf, sizeof(buf), f))
		{
		if (strncmp(buf, "cpu", 3) != 0)
			continue;
		s = strtok(buf, " \t\n");
		cpu[n_cpus].name = g_strdup(s);
		if (++n_cpus >= MAX_CPUS)		/* XXX */
			break;
		}
	fclose(f);

	if (GK.debug)
		printf("register_cpus() found %d cpu entries\n", n_cpus);

	/* Use private to flag if cpu[0] is a composite cpu.  If so, its value
	|  is smp_cpus and is used to set scale_min.
	*/
	smp_cpus = n_cpus - 1;
	cpu[0].private = smp_cpus;
	}


  /* /proc/stat contents:
  |  cpux USER NICE SYS IDLE
  |  disk n n n n
  |  disk_rio n n n n
  |  disk_wio n n n n
  |  disk_rblk n n n n
  |  disk_wblk n n n n
  |  page n n
  |  swap n n
  |  intr (bunch of n's)
  |  ...
  */
void
update_stat()
	{
	FILE			*f;
	Chart			*cp;
	gchar			*item, *arg, buf[160];
	unsigned long	lr, lw, l0, l1, l2, l3;
	unsigned long	user, nice, sys, idle;
	gint			i, processes, load;
	float			fload;

	if ((f = fopen("/proc/stat", "r")) == NULL)
		return;

	GK.cpu_sys_activity = 0;
	lr = 0;
	while (fgets(buf, sizeof(buf), f))
		{
		item = strtok(buf, " \t\n");
		arg = strtok(NULL, "\n");
		if (item == NULL || arg == NULL)
			continue;
		if (*item == 'c' && *(item+1) == 'p')		/* Presort for "cpu" */
			{
			for (i = 0; i < n_cpus; ++i)
				{
				cp = &cpu[i];
#ifdef SIMULATE_SMP
				if (strncmp(cp->name, item, 3) == 0)  /* This hosed me */
#else
				if (strcmp(cp->name, item) == 0)
#endif
					{
					sscanf(arg,"%lu %lu %lu %lu", &user, &nice, &sys, &idle);
					if (UC.smp_mode == 0)
						GK.cpu_sys_activity += (int)(sys - cp->prevOut);
					else if (i == 0)
						GK.cpu_sys_activity = (int)(sys - cpu[0].prevOut);
					user += nice;
					if (GK.second_tick)
						{
						l0 = (sys - cp->prevOut) + (user - cp->prevIn);
						l1 = cp->prevTotal;
						store_chart_data(cp, sys, user, user + sys + idle);
						draw_chart(cp);
						if (cpu_extra[i])
							draw_cpu_extra(cp, l0, cp->prevTotal - l1);
						if (GK.sensor_temp_files)
							draw_temperature_decal(cp);
						}
					update_krell(&cp->panel, cp->panel.krell, sys + user);
					draw_layers(&cp->panel);
#ifndef SIMULATE_SMP
					break;
#endif
					}
				}
			}
		else if (strcmp("disk_rio", item) == 0)
			{
			sscanf(arg,"%lu %lu %lu %lu", &l0, &l1, &l2, &l3);
			lr = l0 + l1 + l2 + l3;
			continue;
			}
		else if (strcmp("disk_wio", item) == 0)
			{
			sscanf(arg,"%lu %lu %lu %lu", &l0, &l1, &l2, &l3);
			lw = l0 + l1 + l2 + l3;
			if (disk_enabled)
				{
				if (GK.second_tick)
					{
					l0 = (lr - disk.prevIn) + (lw - disk.prevOut);
					store_chart_data(&disk, lw, lr, 0);
					draw_chart(&disk);
					if (disk_extra)
						draw_disk_extra(l0);
					}
				update_krell(&disk.panel, disk.panel.krell, lw + lr);
				draw_layers(&disk.panel);
				}
			continue;
			}
		else if (strcmp("processes", item) == 0)
			{
			sscanf(arg,"%u", &processes);
			if (proc_enabled)
				{
				update_krell(&proc.panel, proc.panel.krell, processes);
				draw_layers(&proc.panel);
				}
			break;		/* last entry in /proc/stat		*/
			}
		else
			continue;
		/* Maybe put page and swap activity LED's here?? */
		}
	fclose(f);
	if (   GK.second_tick
		&& proc_enabled
		&& (f = fopen("/proc/loadavg", "r"))
	   )
		{
		proc.prevOut = 0;
		fgets(buf, sizeof(buf), f);
		fclose(f);
		i = sscanf(buf,"%f %*f %*f %*d/%d", &fload, &current_processes);
		load = (int) (100.0 * fload);

		/* Scale the forks number.  See setup_proc_scaling().
		*/
		store_proc_chart_data(&proc, load, proc.private * processes);
		draw_proc_chart(&proc);
		if (GK.sensor_temp_files)	/* I put mb temp on proc panel */
			draw_temperature_decal(&proc);
		}
	if (GK.second_tick && proc_extra && proc_enabled)
		draw_proc_extra();
	}


static gint
stat_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	gint	i;

	for (i = 0; i < n_cpus; ++i)
		{
		if (widget == cpu[i].drawing_area)
			  gdk_draw_pixmap(widget->window, GK.draw1_GC, cpu[i].pixmap,
					  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					  ev->area.width, ev->area.height);
		}
	if (widget == disk.drawing_area)
	  gdk_draw_pixmap(widget->window, GK.draw1_GC, disk.pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
	if (widget == proc.drawing_area)
	  gdk_draw_pixmap(widget->window, GK.draw1_GC, proc.pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
	return FALSE;
	}

static int
panel_area_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	gint	i;

	for (i = 0; i < n_cpus; ++i)
		{
		if (widget == cpu[i].panel.drawing_area)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC,
					cpu[i].panel.pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
			break;
			}
		}
	if (widget == disk.panel.drawing_area)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, disk.panel.pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
	if (widget == proc.panel.drawing_area)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, proc.panel.pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
	return FALSE;
	}

  /* If lm_sensors is installed, there will be some sensor_temp_files.
  |  If so, create a text decal for this panel regardless of whether
  |  a tempX file is linked to us or not.  If no link, then make the
  |  decal a zombie.  Then, if a link is later made via configuration,
  |  the decal simply can be unzombied.
  */
static void
create_temperature_decal(Chart *cp, Style *style, gchar *label)
	{
	Panel	*p;
	Decal	*d;
	gchar	*sensor_label;
	gint	w, l, r, a, dd, w_used;

	if (   GK.sensor_temp_files == 0
		|| (strcmp(cp->name, "cpu") == 0 && cp->private) /* composite CPU */
	   )
		return;

	/* The text decal must fit next to the panel label, so first get the
	|  label width so I can decide on which font to use for the decal.
	|  configure_panel() is responsible for repositioning/re-fonting the
	|  panel label if more work is needed to make everything fit.
	*/
	gdk_string_extents(GK.label_font, label, &l, &r, &w, &a, &dd);
	w_used = l + r + style->border_panel.left + style->border_panel.right;

	p = &cp->panel;
	d = (Decal *) g_new0(Decal, 1);
	p->decal = d;
	default_textstyle(&d->text_style, TEXTSTYLE_ALT1);

	/* Try for a label_font decal and fall back to alt_font if no room.
	*/
	gdk_string_extents(GK.label_font, "188F", &l, &r, &w, &a, &dd);
	if (w_used + l + r + 8 > UC.chart_width)
		{
		d->text_style.font = GK.alt_font;
		gdk_string_extents(GK.alt_font, "188F", &l, &r, &w, &a, &dd);
		}

	d->y_baseline = a;
	d->w = l + r;
	d->h = a + d->text_style.effect;		/* Add 1 if shadow style */
	d->x = UC.chart_width - d->w - style->border_panel.right;
	d->y = style->border_panel.top;
	d->pixmap = gdk_pixmap_new(top_window->window, d->w, d->h, -1);

	/* I have no clean place to put a motherboard temperature.  So, for
	|  now I choose to put it (if linked) on the Proc panel.
	*/
	sensor_label = strcmp(cp->name, "Proc") ? cp->name : "mb";

	d->type = read_temperature(sensor_label) ? DECAL_OPAQUE : DECAL_ZOMBIE;
#if 0
	if (d->type == DECAL_OPAQUE)
		style->label_position = 30;		/* Make room for temperature decal */
#endif
	if (GK.debug)
		printf("Temp decal: %s type=%d\n", sensor_label, d->type);
	}

  /* Move the position and possibly use a smaller font for a panel label
  |  to make room for a decal.  Used for temperature decals which can
  |  change zombied state as user changes config.
  */
void
adjust_label_for_decal(Panel *p)
	{
	Decal		*d;
	Label		*lbl;
	TextStyle	*ts;
	gint		w, l, r, a, dd;

	d = p->decal;		/* XXX I know temp decal is only/first decal */
	lbl = &p->label;
	ts  = &lbl->textstyle;
	gdk_string_extents(GK.label_font, lbl->string, &l, &r, &w, &a, &dd);
	if (d->x - w < 8)
		{
		ts->font = GK.alt_font;
		gdk_string_extents(GK.label_font, lbl->string, &l, &r, &w, &a, &dd);
		}
	lbl->position = (d->x - w / 2 - 6) * 100 / UC.chart_width;
	if (lbl->position > 50)
		lbl->position = 50;
	if (lbl->position < 0)
		lbl->position = 0;
	}

static gint
cb_cpu_extra(GtkWidget *widget, GdkEventButton *event)
	{
	gint	i;

	if (event->button == 1)
		for (i = 0; i < n_cpus; ++i)
			{
			if (widget == cpu[i].drawing_area)
				cpu_extra[i] = 1 - cpu_extra[i];
			}
	return TRUE;
	}

static void
setup_cpu_scaling(Chart *cp)
	{
	Krell	*krell;
	gint	grids;
	gint	smp_adjust;

	krell = cp->panel.krell;
	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	smp_adjust = cp->private ? cp->private : 1;

	/* scale_min has been seeded with 1 or n_cpus for the composite cpu.
	*/
	cp->scale_min = smp_adjust * CPU_TICKS_PER_SECOND / grids;
	cp->scale_max = 0;	/* Force a chart rescale */
	krell->full_scale = smp_adjust * CPU_TICKS_PER_SECOND / UC.update_HZ;

	/* krell->full_scale can be small (as low as 10), so map it to a bigger
	|  number so ema calc will be smooth.   This is accounted for in
	|  update_krell()
	*/
	krell->full_scale_expand = 10;
	krell->full_scale *= krell->full_scale_expand;
	}

  /* I want cpu labels in upper case.
  */
static gchar	mapname[5]	= { 'C', 'P', 'U', 'x', '\0' };

void
create_cpu(GtkWidget *vbox)
	{
	Chart	*cp;
	Style	*style;
	gint	i;

	if (GK.trace)
		printf("create_cpu()\n");
	register_cpus();

	for (i = 0; i < n_cpus; ++i)
		{
		cpu_vbox[i] = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox), cpu_vbox[i]);
		gtk_widget_show(cpu_vbox[i]);
		cpu_enabled[i] = TRUE;

		cp = &cpu[i];

		style = GK.chart_style[CPU_IMAGE];
		create_krell(cp->name, GK.krell_panel_image[CPU_IMAGE],
						&cp->panel.krell, style);

		setup_cpu_scaling(cp);

		if (GK.debug)
			printf("creating cpu <%s>, scale_min=%d krell->full_scale=%d\n",
					cp->name, cp->scale_min, cp->panel.krell->full_scale);

		cp->h = UC.chart_height[MON_CPU];
		create_chart(cpu_vbox[i], cp, CPU_IMAGE);

		mapname[3] = (cp->name)[3];
		create_temperature_decal(cp, style, mapname);
		default_textstyle(&cp->panel.label.textstyle, TEXTSTYLE_LABEL);
		configure_panel(&cp->panel, g_strdup(mapname), style);
		if (cp->panel.decal && cp->panel.decal->type != DECAL_ZOMBIE)
			adjust_label_for_decal(&cp->panel);
		create_panel_area(cpu_vbox[i], &cp->panel,
										GK.bg_panel_image[CPU_IMAGE]);
		cpu_height[i] = cp->h + cp->panel.h;
		GK.monitor_height += cpu_height[i];

		gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
				(GtkSignalFunc) stat_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
					"button_press_event", (GtkSignalFunc) cb_cpu_extra, NULL);

		gtk_signal_connect(GTK_OBJECT (cp->panel.drawing_area),
				"expose_event",
				(GtkSignalFunc) panel_area_expose_event, NULL);

		alloc_chart_data(cp);
		clear_chart(cp);
		cpu_extra[i] = UC.enable_extra_info;
		}
	if (GK.trace)
		printf("  <-\n");
	}



static gint
cb_proc_extra(GtkWidget *widget, GdkEventButton *event)
	{
	if (event->button == 1)
		proc_extra = 1 - proc_extra;
	return TRUE;
	}

static void
setup_proc_scaling()
	{
	gint	grids;

	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	proc.scale_min = UC.load_resolution;

	/* Since scale_min is set for load, set krell_full_scale explicitely
	|  to get what I want, which is 10 forks full scale.
	|  Use private field to scale forks/sec to get hardwired 50 forks/sec
	|  max on the charts.
	*/
	proc.panel.krell->full_scale = 10;
	proc.private = grids * UC.load_resolution / 50;
	proc.scale_max = 0;
	}

  /* Proc monitor shows processes (forks) per second and load average.
  |  The scaling goes like this.  Load is read from /proc/loadavg as a
  |  float ranging from 0 - n where n may be 1, 2, 3, ...  Basically,
  |  something small.  So, I scale load by 100 and set scale_min to
  |  100.  Now grid lines have units of 1.
  */
void
create_proc(GtkWidget *vbox)
	{
	Style	*style;

	if (GK.trace)
		printf("create_proc()\n");
	proc_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), proc_vbox);
	gtk_widget_show(proc_vbox);
	proc_enabled = TRUE;

	proc.name = "Proc";
	proc.scale_min = 100;		/* For load */

	style = GK.chart_style[PROC_IMAGE];
	create_krell(proc.name, GK.krell_panel_image[PROC_IMAGE], 
									&proc.panel.krell, style);
	setup_proc_scaling();

	proc.h = UC.chart_height[MON_PROC];

	create_chart(proc_vbox, &proc, PROC_IMAGE);

	/* If lm_sensors installed, I put motherboard temp on Proc panel.
	*/
	create_temperature_decal(&proc, style, proc.name);
	default_textstyle(&proc.panel.label.textstyle, TEXTSTYLE_LABEL);
	configure_panel(&proc.panel, proc.name, style);
	if (proc.panel.decal && proc.panel.decal->type != DECAL_ZOMBIE)
		adjust_label_for_decal(&proc.panel);
	create_panel_area(proc_vbox, &proc.panel, GK.bg_panel_image[PROC_IMAGE]);
	proc_height = proc.h + proc.panel.h;
	GK.monitor_height += proc_height;

	gtk_signal_connect(GTK_OBJECT (proc.drawing_area), "expose_event",
			(GtkSignalFunc) stat_expose_event, NULL);
	gtk_signal_connect(GTK_OBJECT(proc.drawing_area), "button_press_event",
			(GtkSignalFunc) cb_proc_extra, NULL);

	gtk_signal_connect(GTK_OBJECT (proc.panel.drawing_area), "expose_event",
			(GtkSignalFunc) panel_area_expose_event, NULL);

	alloc_chart_data(&proc);
	proc.type = 1;
	clear_chart(&proc);
	proc_extra = UC.enable_extra_info;
	if (GK.trace)
		printf("  <-\n");
	}


static gint
cb_disk_extra(GtkWidget *widget, GdkEventButton *event)
	{
	if (event->button == 1)
		disk_extra = 1 - disk_extra;
	return TRUE;
	}

static void
setup_disk_scaling()
	{
	gint	grids;

	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	disk.scale_min = UC.disk_resolution;
	disk.panel.krell->full_scale = disk.scale_min * grids / UC.update_HZ;
	disk.scale_max = 0;
	}

void
create_disk(GtkWidget *vbox)
	{
	Style	*style;

	if (GK.trace)
		printf("create_disk()\n");
	disk_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), disk_vbox);
	gtk_widget_show(disk_vbox);
	disk_enabled = TRUE;

	disk.name = "Disk";
	style = GK.chart_style[DISK_IMAGE];
	create_krell(disk.name, GK.krell_panel_image[DISK_IMAGE],
								&disk.panel.krell, style);
	setup_disk_scaling();

	disk.h = UC.chart_height[MON_DISK]; 
	create_chart(disk_vbox, &disk, DISK_IMAGE);
	default_textstyle(&disk.panel.label.textstyle, TEXTSTYLE_LABEL);
	configure_panel(&disk.panel, disk.name, style);
	create_panel_area(disk_vbox, &disk.panel, GK.bg_panel_image[DISK_IMAGE]);
	disk_height = disk.h + disk.panel.h;
	GK.monitor_height += disk_height;

	gtk_signal_connect(GTK_OBJECT (disk.drawing_area), "expose_event",
			(GtkSignalFunc) stat_expose_event, NULL);
	gtk_signal_connect(GTK_OBJECT(disk.drawing_area), "button_press_event",
			(GtkSignalFunc) cb_disk_extra, NULL);

	gtk_signal_connect(GTK_OBJECT (disk.panel.drawing_area), "expose_event",
			(GtkSignalFunc) panel_area_expose_event, NULL);

	alloc_chart_data(&disk);
	clear_chart(&disk);
	disk_extra = UC.enable_extra_info;
	if (GK.trace)
		printf("  <-\n");
	}

  /* Detect if sensor linking has changed.  If so, redraw the appropriate
  |  panel with the label in its appropriate spot.  If lm_sensors is
  |  installed, I confiscate control of label positioning.
  */
static void
sync_cpu_sensors_config()
	{
	Decal	*d;
	gint	t, i,
			new_label	= FALSE;

	/* The test for state change is decal state vs success at reading
	|  a temperature.
	*/
	for (i = 0; i < n_cpus; ++i)
		{
		if ((d = cpu[i].panel.decal) == NULL)
			continue;
		new_label = FALSE;
		t = read_temperature(cpu[i].name);
		if (d->type == DECAL_ZOMBIE && t > 0)	/* Change to display temp */
			{
			d->type = DECAL_OPAQUE;
			adjust_label_for_decal(&cpu[i].panel);
			new_label = TRUE;
			}
		if (d->type == DECAL_OPAQUE && t == 0)	/* Change to no temp display */
			{
			d->type = DECAL_ZOMBIE;
			cpu[i].panel.label.position = 50;
			cpu[i].panel.label.textstyle.font = GK.label_font;
			new_label = TRUE;
			}
		if (new_label)
			{
			draw_panel_label(&cpu[i].panel, GK.bg_panel_image[CPU_IMAGE]);
			gdk_draw_pixmap(cpu[i].panel.drawing_area->window, GK.draw1_GC,
				cpu[i].panel.pixmap, 0, 0, 0, 0, proc.panel.w, proc.panel.w);
			}
		d->value = 0;	/* Force redraw */
		}
	if ((d = proc.panel.decal) == NULL)
		return;
	new_label = FALSE;
	t = read_temperature("mb");
	if (d->type == DECAL_ZOMBIE && t > 0)	/* Change to display temp */
		{
		d->type = DECAL_OPAQUE;
		adjust_label_for_decal(&proc.panel);
		new_label = TRUE;
		}
	if (d->type == DECAL_OPAQUE && t == 0)	/* Change to no temp display */
		{
		d->type = DECAL_ZOMBIE;
		proc.panel.label.position = 50;
		proc.panel.label.textstyle.font = GK.label_font;
		new_label = TRUE;
		}
	if (new_label)
		{
		draw_panel_label(&proc.panel, GK.bg_panel_image[PROC_IMAGE]);
		gdk_draw_pixmap(proc.panel.drawing_area->window, GK.draw1_GC,
				proc.panel.pixmap, 0, 0, 0, 0, proc.panel.w, proc.panel.w);
		}
	d->value = 0;	/* Force redraw */
	}


void
apply_stat_config()
	{
	gint	i, cpu_enable;

	for (i = 0; i < n_cpus; ++i)
		{
		cpu_enable = UC.enable[MON_CPU];
		if (cpu_enable && smp_cpus > 0)
			{
			if (   (i == 0 && UC.smp_mode == 0)	/* Separate cpu's only */
				|| (i > 0 && UC.smp_mode == 1)	/* Composite cpu only */
			   )
				cpu_enable = FALSE;
			}
		enable_visibility(cpu_enable, &cpu_enabled[i],
						  cpu_vbox[i], cpu_height[i]);
		setup_cpu_scaling(&cpu[i]);
		}

	if (   enable_visibility(UC.enable[MON_DISK],
					&disk_enabled, disk_vbox, disk_height)
		&& disk_enabled == FALSE
	   )
		reset_chart(&disk);

	if (   enable_visibility(UC.enable[MON_PROC],
					&proc_enabled, proc_vbox, proc_height)
		&& proc_enabled == FALSE
	   )
		reset_chart(&proc);

	setup_disk_scaling();
	setup_proc_scaling();

	sync_cpu_sensors_config();
	}



