/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

/* Do the memory and swap meter stuff from /proc/meminfo.
*/

#include "gkrellm.h"


#define PIPE_SIZE	4

static Panel	mem,
				swap;

static gint		mem_pipe[PIPE_SIZE],
				swap_pipe[PIPE_SIZE];

static void
record_activity(gint *pipe, gint modified)
	{
	gint	i;

	for (i = PIPE_SIZE - 1; i > 0; --i)
		pipe[i] = pipe[i-1];
	pipe[0] = modified;
	}

  /* Updating the /proc/meminfo meters is _very_ expensive, so I do some
  |  dynamic adjustments on how often I do the updates.
  |  I increase the update rate if system activity is detected.
  |  This is an effort to get good meter response and to
  |  not contribute to cpu chart activity during quiet times.
  */
gint
force_meminfo_update()
	{
	gint	i;

	if (GK.second_tick)
		{
		for (i = 1; i < PIPE_SIZE; ++i)
			if (mem_pipe[i] || swap_pipe[i])
				{
				return TRUE;
				}
		if (GK.cpu_sys_activity > 3)
			{
			return TRUE;
			}
		}
	return FALSE;
	}

void
update_meminfo(void)
	{
	FILE	*f;
	gchar	buf[160];
	glong	ltotal, lused, lfree, lshared, lbuffers, lcached;

	/* Once every 10 seconds is default update period, unless we force.
	*/
	if (! GK.ten_second_tick)
		if (! force_meminfo_update())
			return;

	if ((f = fopen("/proc/meminfo", "r")) != NULL)
		{		/* total:    used:    free:  shared: buffers:  cached: */
		while ((fgets(buf, sizeof(buf), f)) != NULL)
			{
			if (strncmp(buf, "Mem:", 4) == 0)
				{
				sscanf(buf,"Mem: %ld %ld %ld %ld %ld %ld",
					&ltotal, &lused, &lfree, &lshared, &lbuffers, &lcached);
				lused -= lbuffers + lcached;

				mem.krell->full_scale = ltotal >> 12;
				mem.krell->previous = 0;
				update_krell(&mem, mem.krell, lused >> 12);
				record_activity(mem_pipe, mem.krell->modified);
				draw_layers(&mem);
				}
			if (strncmp(buf, "Swap:", 5) == 0)
				{
				sscanf(buf,"Swap: %ld %ld", &ltotal, &lused);
				swap.krell->full_scale = ltotal >> 12;
				swap.krell->previous = 0;
				update_krell(&swap, swap.krell, lused >> 12);
				record_activity(swap_pipe, swap.krell->modified);
				draw_layers(&swap);
				break;
				}
			}
		fclose(f);
		}
	}


static gint
meminfo_expose_event (GtkWidget *widget, GdkEventExpose *event)
	{
	if (widget == mem.drawing_area)
		{
		gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
				mem.pixmap,
				event->area.x, event->area.y,
				event->area.x, event->area.y,
				event->area.width, event->area.height);
		}
	else if (widget == swap.drawing_area)
		{
		gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
				swap.pixmap,
				event->area.x, event->area.y,
				event->area.x, event->area.y,
				event->area.width, event->area.height);
		}
	return FALSE;
	}


void
create_meminfo(GtkWidget *vbox)
	{
	Style		*style;

	if (GK.trace)
		printf("create_meminfo()\n");

	style = GK.meter_style[MEM_IMAGE];
	create_krell("Mem", GK.krell_meter_image[MEM_IMAGE], &mem.krell, style);
	default_textstyle(&mem.label.textstyle, TEXTSTYLE_METER);
	configure_panel(&mem, "Mem", style);
	create_panel_area(vbox, &mem, GK.bg_meter_image[MEM_IMAGE]);
	GK.monitor_height += mem.h;

	style = GK.meter_style[SWAP_IMAGE];
	create_krell("Swap", GK.krell_meter_image[SWAP_IMAGE], &swap.krell, style);
	default_textstyle(&swap.label.textstyle, TEXTSTYLE_METER);
	configure_panel(&swap, "Swap", style);
	create_panel_area(vbox, &swap, GK.bg_meter_image[MEM_IMAGE]);
	GK.monitor_height += swap.h;


	/* Signals used to handle backing pixmap
	*/
	gtk_signal_connect(GTK_OBJECT (mem.drawing_area), "expose_event",
			(GtkSignalFunc) meminfo_expose_event, NULL);
	gtk_signal_connect(GTK_OBJECT (swap.drawing_area), "expose_event",
			(GtkSignalFunc) meminfo_expose_event, NULL);

	if (GK.trace)
		printf("  <-\n");
	}

