/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"
#include <X11/Xlib.h>
#include <X11/Xmd.h>
#include <X11/Xatom.h>

struct ThemeConfig	GK;
struct UserConfig	UC;

struct
	{
	GtkWidget	*window;		/* Top level window	*/
	GtkWidget	*vbox;

	GtkWidget	*top0_event_box;	/* Top frame event box */
	GtkWidget	*top0_vbox;			/* Top frame			*/

	GtkWidget	*middle_hbox;
	  GtkWidget	*left_event_box;	/* Holds left vbox.		*/
	  GtkWidget	*left_vbox;			/* Holds left frame.	*/

	  GtkWidget	*middle_vbox;		/* A handle for sliding shut */
	  GtkWidget	*monitor_vbox;		/* All monitors go in here.	*/
		GtkWidget	*top1_event_box;	/* First spacer event box */
		GtkWidget	*top1_vbox;			/* First spacer/hostname	*/

	  GtkWidget	*right_event_box;	/* Holds right vbox.	*/
	  GtkWidget	*right_vbox;		/* Holds right frame.	*/

	GtkWidget	*bottom_vbox;	/* Bottom frame	*/

	GtkWidget	*l_pixmapwid,
				*r_pixmapwid;
	GdkPixmap	*frame_left_pixmap;
	GdkPixmap	*frame_right_pixmap;
	}
	gtree;

GtkWidget		*top_window;
GtkWidget		*popup_menu;

gchar			*exec_theme_path;
gchar			*exec_gkrellm[20];
gint			gk_argc;

gint			label_font_ascent;

static gint		w_display, h_display;
static gint		window_decorations;
static gint		top_frame_height,
				total_frame_height;

static gchar	*fail_large_font[2] =
{
"-adobe-courier-medium-r-*-*-*-120-*-*-*-*-*-*",
"-misc-fixed-medium-r-*-*-*-120-*-*-*-*-*-*"
};

static gchar	*fail_label_font[2] =
{
"-adobe-courier-medium-r-*-*-*-100-*-*-*-*-*-*",
"-misc-fixed-medium-r-*-*-*-100-*-*-*-*-*-*"
};

static gchar	*fail_alt_font[2] =
{
"-adobe-helvetica-medium-r-*-*-*-80-*-*-*-*-*-*",
"-misc-fixed-medium-r-*-*-*-80-*-*-*-*-*-*"
};

static void
setup_colors_and_fonts(GtkWidget *win)
	{
	GdkBitmap	*dummy_bitmap;
	GdkColormap	*colormap;
	gint		i, w, h;

	colormap = gtk_widget_get_colormap(win);

	gdk_color_parse(GK.label_color_string, &GK.label_color);
	gdk_color_alloc(colormap, &GK.label_color);
	gdk_color_parse(GK.meter_color_string, &GK.meter_color);
	gdk_color_alloc(colormap, &GK.meter_color);
	gdk_color_parse(GK.time_color_string, &GK.time_color);
	gdk_color_alloc(colormap, &GK.time_color);
	gdk_color_parse(GK.alt1_color_string, &GK.alt1_color);
	gdk_color_alloc(colormap, &GK.alt1_color);
	gdk_color_parse(GK.alt2_color_string, &GK.alt2_color);
	gdk_color_alloc(colormap, &GK.alt2_color);

	gdk_color_parse(GK.chart_in_color, &GK.in_color);
	gdk_color_alloc(colormap, &GK.in_color);
	gdk_color_parse(GK.chart_in_color_grid, &GK.in_color_etch);
	gdk_color_alloc(colormap, &GK.in_color_etch);

	gdk_color_parse(GK.chart_out_color, &GK.out_color);
	gdk_color_alloc(colormap, &GK.out_color);
	gdk_color_parse(GK.chart_out_color_grid, &GK.out_color_etch);
	gdk_color_alloc(colormap, &GK.out_color_etch);

	gdk_color_parse("#000000", &GK.background_color);
	gdk_color_alloc(colormap, &GK.background_color);

	gdk_color_parse("#FFFFFF", &GK.white_color);
	gdk_color_alloc(colormap, &GK.white_color);

	GK.draw1_GC = gdk_gc_new( win->window );
	gdk_gc_copy( GK.draw1_GC, win->style->white_gc );

	GK.draw2_GC = gdk_gc_new( win->window );
	gdk_gc_copy( GK.draw2_GC, win->style->white_gc );

	GK.draw3_GC = gdk_gc_new( win->window );
	gdk_gc_copy( GK.draw3_GC, win->style->white_gc );

	GK.draw4_GC = gdk_gc_new( win->window );
	gdk_gc_copy( GK.draw4_GC, win->style->white_gc );

	GK.text_GC = gdk_gc_new( win->window );
	gdk_gc_copy( GK.text_GC, win->style->white_gc );

	/* Set up the depth 1 GCs
	*/
	dummy_bitmap = gdk_pixmap_new(top_window->window, 16, 16, 1);
	GK.bit1_GC = gdk_gc_new(dummy_bitmap);
	GK.bit0_GC = gdk_gc_new(dummy_bitmap);
	gdk_gc_set_foreground(GK.bit1_GC, &GK.white_color);
	gdk_gc_set_foreground(GK.bit0_GC, &GK.background_color);
	gdk_pixmap_unref(dummy_bitmap);

	if ((GK.large_font = gdk_font_load(GK.large_font_string)) == NULL)
		{
		printf("Could not load large font\n  \"%s\"\nFalling back...\n",
					GK.large_font_string);
		for (i = 0; i < 2; ++i)
			if ((GK.large_font = gdk_font_load(fail_large_font[i])) != NULL)
				break;
		}

	if ((GK.label_font = gdk_font_load(GK.label_font_string)) == NULL)
		{
		printf("Could not load label font\n  \"%s\"\nFalling back...\n",
					GK.label_font_string);
		for (i = 0; i < 2; ++i)
			if ((GK.label_font = gdk_font_load(fail_label_font[i])) != NULL)
				break;
		}

	if ((GK.alt_font = gdk_font_load(GK.alt_font_string)) == NULL)
		{
		printf("Could not load alt font\n  \"%s\"\nFalling back...\n",
					GK.alt_font_string);
		for (i = 0; i < 2; ++i)
			if ((GK.alt_font = gdk_font_load(fail_alt_font[i])) != NULL)
				break;
		}
	if (GK.large_font == NULL || GK.label_font == NULL || GK.alt_font == NULL)
		{
		printf("Could not load a label or alt font.\n");
		exit(0);
		}
	label_font_ascent = gdk_char_height(GK.label_font, '8');

	w = UC.chart_width;
	h = GK.max_chart_height;
	if (GK.data_in_pixmap == NULL)
		{
		GK.data_in_pixmap = gdk_pixmap_new(top_window->window, w, h, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.in_color);
		gdk_draw_rectangle(GK.data_in_pixmap, GK.draw1_GC, TRUE, 0, 0, w, h);
		}
	if (GK.data_out_pixmap == NULL)
		{
		GK.data_out_pixmap = gdk_pixmap_new(top_window->window, w, h, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.out_color);
		gdk_draw_rectangle(GK.data_out_pixmap, GK.draw1_GC, TRUE, 0, 0, w, h);
		}
	if (GK.data_in_etch_pixmap == NULL)
		{
		GK.data_in_etch_pixmap = gdk_pixmap_new(top_window->window, w, 1, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.in_color_etch);
		gdk_draw_rectangle(GK.data_in_etch_pixmap, GK.draw1_GC,
					TRUE, 0, 0, w, 2);
		}
	if (GK.data_out_etch_pixmap == NULL)
		{
		GK.data_out_etch_pixmap = gdk_pixmap_new(top_window->window, w, 1, -1);
		gdk_gc_set_foreground(GK.draw1_GC, &GK.out_color_etch);
		gdk_draw_rectangle(GK.data_out_etch_pixmap, GK.draw1_GC,
					TRUE, 0, 0, w, 2);
		}
	}

gint
update_monitors()
	{
	GK.second_tick = ((GK.timer_ticks % UC.update_HZ) == 0) ? TRUE : FALSE;
	GK.two_second_tick =
			((GK.timer_ticks % (2 * UC.update_HZ)) == 0) ? TRUE : FALSE;

	update_clock();			/* Sets minute, hour, and day ticks	*/
	update_uptime();
	update_stat();			/* CPU, Proc, and disk		*/
	update_inet();			/* tcp ports - http/ftp...	*/
	update_net();			/* ppp, plip, ethx			*/
	update_meminfo();		/* MEM & SWAP				*/
	update_fs();
	update_apm();
	update_mailcheck();

	++GK.timer_ticks;

	/* I hate to do this, but if gkrellm is started from .xinitrc or
	|  .xsession, it will be running before a window manager runs and I
	|  don't know when it is OK to check for window manager hints.
	|  So I try every two secs for the first 20 seconds.
	*/
	if (   GK.on_top_hint_ok
		&& GK.ten_second_tick && GK.timer_ticks / UC.update_HZ < 40
	   )
		set_on_top(UC.on_top);

	if (   GK.on_top_hint_ok == FALSE
		&& GK.two_second_tick && GK.timer_ticks / UC.update_HZ < 40
	   )
		init_on_top_hint();

	return TRUE;			/* This restarts timeout */
	}

  /* Overhead of updates and restarting the timer makes the interval longer
  |  than specified. So I fudge a little. I should dynamically correct - later
  */
void
start_timer()
	{
	static gint	timeout_id	= 0;
	gint		interval;

	interval = 1000 / UC.update_HZ;
	interval = interval * 60 / 63;		/* Compensate for overhead XXX Fixme */
	if (timeout_id)
		gtk_timeout_remove(timeout_id);
	timeout_id = gtk_timeout_add(interval, (GtkFunction) update_monitors,NULL);
	}


  /* Nice set of effects I have here...  Either shadow effect or none.
  |  Returning 1 means shadow effect and the return value is also used by
  |  callers to increment height fields to allow for the offset shadow draw.
  */
gint
effect_string_value(gchar *effect_string)
	{
	if (effect_string)
		return (strcmp(effect_string, "shadow") == 0) ? 1 : 0;
	return 0;
	}


void
default_textstyle(TextStyle *ts, gint textstyle)
	{
	if (textstyle == TEXTSTYLE_LABEL)
		{
		ts->font = GK.label_font;
		ts->color = GK.label_color;
		ts->effect = effect_string_value(GK.label_effect_string);
		}
	else if (textstyle == TEXTSTYLE_METER)
		{
		ts->font = GK.label_font;
		ts->color = GK.meter_color;
		ts->effect = effect_string_value(GK.meter_effect_string);
		}
	if (textstyle == TEXTSTYLE_TIME)
		{
		ts->font = GK.label_font;
		ts->color = GK.time_color;
		ts->effect = effect_string_value(GK.time_effect_string);
		}
	else if (textstyle == TEXTSTYLE_ALT1)
		{
		ts->font = GK.label_font;
		ts->color = GK.alt1_color;
		ts->effect = effect_string_value(GK.alt1_effect_string);
		}
	else if (textstyle == TEXTSTYLE_ALT2)
		{
		ts->font = GK.label_font;
		ts->color = GK.alt2_color;
		ts->effect = effect_string_value(GK.alt2_effect_string);
		}
	}

  /* Calculate panel height required for given border, label height,
  |  krell heights, and decal heights.  Also calculate label extents
  |  and load all this info into a Label structure.
  |  After panel height is calculated, calling routine may want to reset
  |  krell or decal y offsets for centering.  
  */
void
configure_panel(Panel *p, gchar *string, Style *style)
	{
	Krell			*k;
	Decal			*d;
	Label			*lbl;
	GdkImlibBorder	*border;
	gint			h_panel,
					h_decal;

	if (p == NULL)
		return;
	p->style = style;
	lbl = &p->label;
	lbl->string = string;
	lbl->position = style ? style->label_position : LABEL_CENTER;
	border = style ? &style->border_panel : NULL;

	if (lbl->string && lbl->textstyle.font)
		gdk_string_extents(lbl->textstyle.font, string, &lbl->lbearing,
				&lbl->rbearing, &lbl->width, &lbl->ascent, &lbl->descent);
	else
		{
		lbl->lbearing = 0;
		lbl->rbearing = 0;
		lbl->width = 0;
		lbl->ascent = 0;
		lbl->descent = 0;
		}
	h_panel = border ? (border->top + border->bottom) : 0;
	h_panel += lbl->ascent + lbl->descent + lbl->textstyle.effect;

	for (k = p->krell; k; k = k->next)
		if (k->h_stencil > h_panel)
			h_panel = k->h_stencil;
	for (d = p->decal; d; d = d->next)
		{
		/* If a decal y is given, assume wanting to fit it inside borders.
		*/
		h_decal = (d->y == 0) ? d->h : d->y + d->h + border->bottom;
		if (h_decal > h_panel)
			h_panel = h_decal;
		}

	if (h_panel <= 0)
		h_panel = 1;

	lbl->y_baseline = border->top + lbl->ascent;
	lbl->h_panel = h_panel;
	}


void
draw_panel_label(Panel *p, GdkImlibImage *image)
	{
	Label		*lbl	= &p->label;
	TextStyle	*ts;
	gint		xdst, lborder, rborder;

	lbl = &p->label;
	ts  = &lbl->textstyle;

	/* Render the background pixmap, get a copy for us to work on and a
	|  copy to use as a background for stenciled drawings and such.
	|  Create the stencil bitmap as well.
	*/
	if (p->pixmap)
		gdk_imlib_free_pixmap(p->pixmap);
	if (p->background)
		gdk_imlib_free_pixmap(p->background);
	gdk_imlib_render(image, p->w, p->h);
	p->pixmap = gdk_imlib_copy_image(image);
	p->background = gdk_imlib_copy_image(image);

	if (lbl->string)
		{
		lborder = p->style ? p->style->border_panel.left : 0;
		rborder = p->style ? p->style->border_panel.right : 0;
		xdst = UC.chart_width * lbl->position / LABEL_MAX;
		xdst -= lbl->width / 2;
		if (xdst > UC.chart_width - lbl->width - rborder)
			xdst = UC.chart_width - lbl->width - rborder;
		if (xdst < lborder)
			xdst = lborder;
		lbl->x_panel = xdst;
		if (ts->effect)
			{
			gdk_gc_set_foreground(GK.text_GC, &GK.background_color);
			gdk_draw_string(p->pixmap, ts->font, GK.text_GC,
						xdst+1, lbl->y_baseline + 1, lbl->string);
			gdk_draw_string(p->background, ts->font, GK.text_GC,
						xdst+1, lbl->y_baseline + 1, lbl->string);
			}
		gdk_gc_set_foreground(GK.text_GC, &ts->color);
		gdk_draw_string(p->pixmap, ts->font, GK.text_GC,
					xdst, lbl->y_baseline, lbl->string);
		gdk_draw_string(p->background, ts->font, GK.text_GC,
					xdst, lbl->y_baseline, lbl->string);
		}
	}

/* ---------------------------------------------------------------------- */
void
destroy_panel(Panel *p)
	{
	gdk_pixmap_unref(p->stencil);
	gdk_gc_destroy(p->stencil_gc);

	gtk_signal_handlers_destroy(GTK_OBJECT(p->drawing_area));
	gtk_widget_destroy(p->hbox);
	}

  /* Every chart has a panel area under it.  Other monitors may create them
  |  to put their own stuff on.  Krells, LED's, and labels are drawn on the
  |  panel area drawing_area.
  */
void
create_panel_area(GtkWidget *vbox, Panel *p, GdkImlibImage *image)
	{
	GtkWidget	*hbox;
	GtkWidget	*fixed;

	if (image == NULL)
		{
		printf("create_panel_area: no image?\n");
		exit(0);
		}

	if (GK.trace)
		printf("create_panel_area()\n");
	p->h = p->label.h_panel;
	p->w = UC.chart_width;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add (GTK_CONTAINER(vbox), hbox);

	/* And create the drawing area that the middle panel lives in and put
	|  it into a fixed widget.
	*/
	fixed = gtk_fixed_new();
	gtk_box_pack_start(GTK_BOX(hbox), fixed, TRUE, TRUE, 0);
	gtk_widget_show(fixed);
	p->fixed = fixed;
	p->drawing_area = gtk_drawing_area_new();
	gtk_widget_set_events (p->drawing_area, GDK_EXPOSURE_MASK
             | GDK_LEAVE_NOTIFY_MASK
             | GDK_BUTTON_RELEASE_MASK);
	gtk_drawing_area_size(GTK_DRAWING_AREA (p->drawing_area),
				p->w, p->h);
	gtk_fixed_put(GTK_FIXED(fixed), p->drawing_area, 0, 0);
	gtk_widget_show(p->drawing_area);

	p->stencil = gdk_pixmap_new(top_window->window, p->w, p->h, 1);
	p->stencil_gc = gdk_gc_new(p->stencil);

	draw_panel_label(p, image);

	p->hbox = hbox;
	gtk_widget_show(hbox);
	gtk_widget_realize(hbox);
	gtk_widget_realize(p->fixed);
	gtk_widget_realize(p->drawing_area);

	if (GK.trace)
		printf("  <-\n");
	}

void
destroy_chart(Chart *cp)
	{
	g_free(cp->pDataIn);
	g_free(cp->pDataOut);

	gdk_pixmap_unref(cp->pixmap);
	gdk_pixmap_unref(cp->bg_grided_pixmap);

	gdk_pixmap_unref(cp->in_grided_pixmap);
	gdk_pixmap_unref(cp->out_grided_pixmap);
	gdk_pixmap_unref(cp->in_data_bitmap);
	gdk_pixmap_unref(cp->out_data_bitmap);

	gdk_imlib_free_pixmap(cp->bg_clean_pixmap);
	gdk_imlib_free_pixmap(cp->grid_pixmap);

	gtk_signal_handlers_destroy(GTK_OBJECT(cp->drawing_area));
	gtk_widget_destroy(cp->hbox);
	}

void
create_chart(GtkWidget *vbox, Chart *cp, gint type)
	{
	GtkWidget	*hbox;
	GdkWindow	*window;

	if (GK.trace)
		printf("create_chart() - %s\n", cp->name);
	cp->x = 0;
/*	cp->y = 0; */
	cp->w = UC.chart_width;
	if (cp->h == 0)
		cp->h = UC.chart_height[0];

	if (GK.bg_chart_image[type] == NULL)
		printf("bg_chart_image[%d] is NULL.\n", type);

	window = top_window->window;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add (GTK_CONTAINER(vbox), hbox);

	/* Create the drawing area and the various pixmap layers to draw on.
	|  Must gtk_widget_set_events() immediately.
	*/
	cp->drawing_area = gtk_drawing_area_new();
	gtk_widget_set_events (cp->drawing_area,
		GDK_EXPOSURE_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_BUTTON_PRESS_MASK);
	gtk_drawing_area_size(GTK_DRAWING_AREA (cp->drawing_area), cp->w, cp->h);
	gtk_box_pack_start (GTK_BOX(hbox), cp->drawing_area, FALSE, FALSE, 0);
	gtk_widget_show(cp->drawing_area);

	cp->pixmap = gdk_pixmap_new(window, cp->w, cp->h, -1);
	cp->in_grided_pixmap = gdk_pixmap_new(window, cp->w, cp->h, -1);
	cp->out_grided_pixmap = gdk_pixmap_new(window, cp->w, cp->h, -1);

	/* In/out chart data is drawn on data_bitmaps which are then used
	|  as stencils for pushing in/out_grided_pixmap pixels onto expose pixmap.
	*/
	cp->in_data_bitmap = gdk_pixmap_new(window, cp->w, cp->h, 1);
	cp->out_data_bitmap = gdk_pixmap_new(window, cp->w, cp->h, 1);

	/* Get the chart background pixmaps. bg_grided_pixmap will
	|  have grid lines drawn on it.  I use gdk_imlib_move_image()
	|  into the bg_clean_pixmap because that will not be drawn on and maybe
	|  imlib provides a cached copy of that pixmap.
	*/
	cp->bg_grided_pixmap = gdk_pixmap_new(window, cp->w, cp->h, -1);
	gdk_imlib_render(GK.bg_chart_image[type], cp->w, cp->h);
	cp->bg_clean_pixmap = gdk_imlib_move_image(GK.bg_chart_image[type]);
	gdk_imlib_render(GK.chart_grid_image[type], cp->w, 2);	/* XXX */
	cp->grid_pixmap = gdk_imlib_copy_image(GK.chart_grid_image[type]);

	cp->hbox = hbox;
	gtk_widget_show(hbox);

	gtk_widget_realize(hbox);
	gtk_widget_realize(cp->drawing_area);

	if (GK.trace)
		printf("  <-\n");
	}


static gint		moving_gkrellm	= FALSE;
static gint		x_press_event, y_press_event;

static void
gkrellm_motion(GtkWidget *widget, GdkEventMotion *ev, gpointer data)
	{
	gint			x_pointer, y_pointer;
	GdkModifierType	m;

	if (moving_gkrellm)
		{
		gdk_window_get_pointer(NULL, &x_pointer, &y_pointer, &m);

		/* Moves to x,y position relative to root.  Subtract the press
		|  event coordinates to account for pointer offset into top_window.
		*/
		gdk_window_move(top_window->window,
			x_pointer - x_press_event, y_pointer - y_press_event);
		}
	}

static void
top_frame_button_release(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	moving_gkrellm = FALSE;
	}

static void
top_frame_button_press(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	gint			x_pointer, y_pointer;
	GdkModifierType	m;

	gdk_window_get_pointer(NULL, &x_pointer, &y_pointer, &m);

	if (ev->button == 3)
		{
		gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL, NULL, NULL,
					ev->button, ev->time);
		return;
		}
	gdk_window_raise(top_window->window);

	/* I need pointer coords relative to top_window.  So, add in offsets
	|  to spacer window if that is where we pressed.
	*/
	x_press_event = ev->x;
	y_press_event = ev->y;
	if (widget == gtree.top1_event_box)
		{
		x_press_event += GK.frame_width;
		y_press_event += top_frame_height;
		}
	moving_gkrellm = TRUE;
	}

#define	CLOSE_LEFT	0
#define	CLOSE_RIGHT	1

static void
side_frame_button_press(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	static gint		monitors_visible	= TRUE;
	static gint		direction;
	gint			x_gkrell, y_gkrell;

	if (ev->button == 3)
		{
		gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL, NULL, NULL,
					ev->button, ev->time);
		return;
		}
	gdk_window_get_root_origin(gtree.window->window, &x_gkrell, &y_gkrell);
	direction = (x_gkrell < w_display / 2) ? CLOSE_LEFT : CLOSE_RIGHT;

	monitors_visible = 1 - monitors_visible;
	if (monitors_visible)
		{
		gtk_widget_show(gtree.middle_vbox);
		gtk_widget_show(gtree.top0_vbox);
		gtk_widget_show(gtree.bottom_vbox);
		if (direction == CLOSE_RIGHT)
			gdk_window_move(gtree.window->window,
							x_gkrell - UC.chart_width, y_gkrell);
		}
	else
		{
		gtk_widget_hide(gtree.top0_vbox);
		gtk_widget_hide(gtree.bottom_vbox);
		gtk_widget_hide(gtree.middle_vbox);
		if (direction == CLOSE_RIGHT)
			gdk_window_move(gtree.window->window,
							x_gkrell + UC.chart_width, y_gkrell);
		}
	}

void
create_spacer(GtkWidget *vbox, gint height)
	{
	if (GK.trace)
		printf("create_spacer()\n");
	GK.spacer_pixmap = gdk_pixmap_new(top_window->window,
							UC.chart_width, height, -1);
	gdk_imlib_paste_image(GK.bg_spacer_image, GK.spacer_pixmap,
			0, 0, UC.chart_width, height);
	if (GK.trace)
		printf("  <-\n");
	}

void
insert_spacer(GtkWidget *vbox)
	{
	GtkWidget	*pixmapwid;

	if (GK.trace)
		printf("insert_spacer()\n");
	pixmapwid = gtk_pixmap_new(GK.spacer_pixmap, NULL);
	gtk_box_pack_start (GTK_BOX(vbox), pixmapwid, FALSE, FALSE, 0);
	gtk_widget_show (pixmapwid);
	GK.monitor_height += ((GdkWindowPrivate *) GK.spacer_pixmap)->height;
	if (GK.trace)
		printf("  <-\n");
	}


  /* Create a top or bottom frame out of 1/2 of frame_horizontal or out
  |  of frame_top and frame_bottom if they exist.   Probably should have
  |  gone with frame_top /bottom only from the beginning because this sure
  |  junks up my code...
  */
void
create_frame_horizontal(GtkWidget *vbox, gint top)
	{
	GtkWidget		*pixmapwid;
	GdkImlibImage	*im;
	GdkPixmap		*src_pixmap, *dst_pixmap;
	gint			w, h, h_render, y_off;

	if (GK.trace)
		printf("create_frame_horizontal()\n");

	/* Settup for frame_horizontal and then check for optional top/bottom.
	*/
	im = GK.frame_horizontal_image;
	w = UC.chart_width + 2 * GK.frame_width;
	h = im->rgb_height / 2;			/* Handle odd heights */
	if (h < 1)
		h = 1;
	h_render = 2 * h;
	y_off = top ? 0 : h;

	if (top && GK.frame_top_image)
		{
		im = GK.frame_top_image;
		h = im->rgb_height;
		h_render = h;
		}
	if (! top && GK.frame_bottom_image)
		{
		im = GK.frame_bottom_image;
		h = im->rgb_height;
		h_render = h;
		y_off = 0;
		}

	if (top)
		top_frame_height = h;
	total_frame_height += h;
	gdk_imlib_render(im, w, h_render);
	src_pixmap = gdk_imlib_move_image(im);

	dst_pixmap = gdk_pixmap_new(top_window->window, w, h, -1);
	gdk_draw_pixmap(dst_pixmap, GK.draw1_GC, src_pixmap,
		0, y_off,  0, 0, w, h);

	pixmapwid = gtk_pixmap_new(dst_pixmap, NULL);
	gtk_box_pack_start (GTK_BOX(vbox), pixmapwid,
				/* expand */ FALSE, /* fill */ FALSE, /* padding */ 0);
	gtk_widget_show (pixmapwid);

	if (GK.trace)
		printf("  <-\n");
	}

  /* Return TRUE if visibility is changed, FALSE if not.
  */
int
enable_visibility(int new_visibility, gint *current_visibility,
						GtkWidget *vbox, gint height)
	{
	if (new_visibility  && ! *current_visibility)
		{
		gtk_widget_show(vbox);
		*current_visibility  = TRUE;
		GK.monitor_height += height;
		pack_side_frames();
		return TRUE;
		}
	if (! new_visibility  && *current_visibility )
		{
		gtk_widget_hide(vbox);
		*current_visibility  = FALSE;
		GK.monitor_height -= height;
		pack_side_frames();
		return TRUE;
		}
	return FALSE;
	}

void
pack_side_frames()
	{
	static gint		previous_height	= -1;
	gint			x_gkrell, y_gkrell, y_bottom;
	gint			y, w, h;
	gint			was_on_bottom;

	if (!GK.initialized || previous_height == GK.monitor_height)
		return;

	gdk_window_get_root_origin(gtree.window->window, &x_gkrell, &y_gkrell);
	gdk_window_get_size(gtree.window->window, &w, &h);
	was_on_bottom = (y_gkrell + h >= h_display) ? TRUE : FALSE;

	if (gtree.frame_left_pixmap)
		gdk_imlib_free_pixmap(gtree.frame_left_pixmap);
	if (gtree.frame_right_pixmap)
		gdk_imlib_free_pixmap(gtree.frame_right_pixmap);


	gdk_imlib_render(GK.frame_left_image, GK.frame_width,
					GK.monitor_height);
	gtree.frame_left_pixmap = gdk_imlib_copy_image(GK.frame_left_image);
	if (gtree.l_pixmapwid)
		gtk_container_remove(GTK_CONTAINER(gtree.left_vbox),
						gtree.l_pixmapwid);
	gtree.l_pixmapwid = gtk_pixmap_new(gtree.frame_left_pixmap, NULL);
	gtk_box_pack_start (GTK_BOX(gtree.left_vbox), gtree.l_pixmapwid,
				FALSE, FALSE, 0);
	gtk_widget_show(gtree.l_pixmapwid);


	gdk_imlib_render(GK.frame_right_image, GK.frame_width,
					GK.monitor_height);
	gtree.frame_right_pixmap = gdk_imlib_copy_image(GK.frame_right_image);
	if (gtree.r_pixmapwid)
		gtk_container_remove(GTK_CONTAINER(gtree.right_vbox),
						gtree.r_pixmapwid);
	gtree.r_pixmapwid = gtk_pixmap_new(gtree.frame_right_pixmap, NULL);
	gtk_box_pack_start (GTK_BOX(gtree.right_vbox), gtree.r_pixmapwid,
				FALSE, FALSE, 0);	
	gtk_widget_show(gtree.r_pixmapwid);


	/* If GKrellM grows in height and bottom is moved off screen, move so
	|  that all of GKrellM is visible.
	*/
	h = GK.monitor_height + total_frame_height;
	y_bottom = h_display - (y_gkrell + h);
	if (GK.debug)
		printf("pack_side_frames: y_bottom=%d h_display=%d y_gkrell=%d h=%d\n",
				y_bottom, h_display, y_gkrell, h);
	if (y_bottom < 0)
		{
		if ((y = y_gkrell + y_bottom) < 0)
			y = 0;
		gdk_window_move(gtree.window->window, x_gkrell, y);
		}
	/* If GKrellM bottom edge was <= screen bottom, then move to make
	|  resized GKrellM still on bottom whether window has shrunk or grown.
	*/
	else if (was_on_bottom)
		{
		if ((y = h_display - h) < 0)
			y = 0;
		gdk_window_move(gtree.window->window, x_gkrell, y);
		}
	previous_height = GK.monitor_height;
	}

  /* My own parsing of +x+y, -x-y geometry placements.
  */
void
place_gkrellm()
	{
	gchar	*sx, *sy, x_sign, y_sign;
	gint	x, y, w_gkrell, h_gkrell;
	gint	offset;

	w_gkrell = UC.chart_width + 2 * GK.frame_width;
	h_gkrell = GK.monitor_height + total_frame_height;
	sx = GK.geometry;
	if (sx == NULL || *sx == '\0')
		{
		printf("Cannot parse geometry.\n");
		return;
		}
	if (*sx != '+' && *sx != '-')
		x_sign = '+';
	else
		x_sign = *sx++;
	for (sy = sx; *sy != '\0'; ++sy)
		if (*sy == '+' || *sy == '-')
			break;
	if (*sy == '\0')
		{
		printf("Cannot parse geometry.\n");
		return;
		}
	y_sign = *sy;
	*sy++ = '\0';

	offset = (x_sign == '-') ? w_display - w_gkrell : 0;
	x = atoi(sx) + offset;

	offset = (y_sign == '-') ? h_display - h_gkrell : 0;
	y = atoi(sy) + offset;
	if (x > w_display - w_gkrell)
		x = w_display - w_gkrell;
	if (x < 0)
		x = 0;
	if (y > h_display - h_gkrell)
		y = h_display - h_gkrell;
	if (y < 0)
		y = 0;
	gdk_window_move(gtree.window->window, x, y);
	}

void
create_widget_tree()
	{
	if (GK.trace)
		printf("create_widget_tree()\n");
	gtree.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_name(gtree.window, "gkrellm");
	gtk_window_set_title(GTK_WINDOW (gtree.window), "gkrellm");

	gtree.vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.window), gtree.vbox);

	gtree.top0_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.vbox), gtree.top0_event_box);
	gtree.top0_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.top0_event_box), gtree.top0_vbox);

	/* The middle hbox has left frame, monitors & a right frame.
	*/
	gtree.middle_hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.vbox), gtree.middle_hbox);

	gtree.left_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.middle_hbox), gtree.left_event_box);
	gtree.left_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.left_event_box), gtree.left_vbox);

	gtree.middle_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.middle_hbox), gtree.middle_vbox);

	/* Top spacer will go in an event box for moving gkrellm */
	gtree.top1_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.middle_vbox), gtree.top1_event_box);
	gtree.top1_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.top1_event_box), gtree.top1_vbox);

	gtree.monitor_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.middle_vbox), gtree.monitor_vbox);

	gtree.right_event_box = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(gtree.middle_hbox), gtree.right_event_box);
	gtree.right_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.right_event_box), gtree.right_vbox);

	gtree.bottom_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(gtree.vbox), gtree.bottom_vbox);

	gtk_widget_realize(gtree.window);
	gtk_window_set_policy(GTK_WINDOW(gtree.window),
		FALSE /*allow_shrink*/ , FALSE /*allow_grow*/, TRUE /*auto_shrink*/);
	if (!window_decorations)
		gdk_window_set_decorations (gtree.window->window, 0);

	if (GK.test)
		gtk_widget_show_all(gtree.window);
	else
		gtk_widget_show_all(gtree.vbox);

	/* Probably don't need to realize all these here. Just a little paranoia.
	*/
	gtk_widget_realize(gtree.vbox);
	gtk_widget_realize(gtree.top0_vbox);
	gtk_widget_realize(gtree.middle_hbox);
	gtk_widget_realize(gtree.left_vbox);
	gtk_widget_realize(gtree.middle_vbox);
	gtk_widget_realize(gtree.monitor_vbox);
	gtk_widget_realize(gtree.top1_vbox);
	gtk_widget_realize(gtree.right_vbox);
	gtk_widget_realize(gtree.bottom_vbox);

	if (GK.trace)
		printf("  <-\n");
	}

static void
set_or_save_position(gint save)
	{
	FILE	*f;
	gchar	path[128];
	gint	x, y;

	snprintf(path, sizeof(path), "%s/%s/%s",
			homedir(), GKRELLM_DIR, GKRELLM_DATA_DIR);
	if (!isdir(path))
		mkdir(path, 0755);
	snprintf(path, sizeof(path), "%s/%s/%s/startup_position", homedir(),
		GKRELLM_DIR, GKRELLM_DATA_DIR);
	if (save)
		{
		gdk_window_get_position(gtree.window->window, &x, &y);
		if ((f = fopen(path, "w")) != NULL)
			{
			fprintf(f, "%d %d\n", x, y);
			fclose(f);
			}
		}
	else
		{
		if ((f = fopen(path, "r")) != NULL)
			{
			x = y = 0;
			fscanf(f, "%d %d", &x, &y);
			gdk_window_move(gtree.window->window, x, y);
			}
		}
	}

#define	WIN_LAYER_NORMAL	4
#define	WIN_LAYER_ONTOP		6

static Atom	xa_check;
static Atom	xa_layer;

  /* I based the following two on_top routines on code from xmms.
  */
void
set_on_top(gint on_top)
	{
	XEvent				xev;
	GdkWindowPrivate	*win;
	gint				prev_error;
	gint				layer;
	glong				data[1];

	layer = on_top ? WIN_LAYER_ONTOP : WIN_LAYER_NORMAL;
	prev_error = gdk_error_warnings;
	win = (GdkWindowPrivate *) (GTK_WIDGET(top_window)->window);
	if (GTK_WIDGET_MAPPED(top_window))
		{
		xev.type = ClientMessage;
		xev.xclient.type = ClientMessage;
		xev.xclient.window = win->xwindow;
		xev.xclient.message_type = xa_layer;
		xev.xclient.format = 32;
		xev.xclient.data.l[0] = (CARD32) layer;
		xev.xclient.data.l[1] = gdk_time_get();
		XSendEvent(GDK_DISPLAY(), GDK_ROOT_WINDOW(), False,
			SubstructureNotifyMask, (XEvent *) & xev);
		}
	else
		{
		data[0] = layer;
		XChangeProperty(GDK_DISPLAY(), win->xwindow, xa_layer,
			XA_CARDINAL, 32, PropModeReplace, (guchar *) data, 1);
		}
	gdk_error_warnings = prev_error;
	}

void
init_on_top_hint()
	{
	Window	win;
	Atom	type;
	gint	format;
	gulong	count, pad;
	guchar	*prop;
	gint	prev_error;

	prev_error = gdk_error_warnings;
	xa_check = XInternAtom(GDK_DISPLAY(), "_WIN_SUPPORTING_WM_CHECK", False);
	xa_layer = XInternAtom(GDK_DISPLAY(), "_WIN_LAYER", False);
	if (XGetWindowProperty(GDK_DISPLAY(), GDK_ROOT_WINDOW(), xa_check,
			0, 1, False, XA_CARDINAL, &type, &format, &count, &pad, &prop)
			== Success && prop)
		{
		win = *(long *) prop;
		XFree(prop);
		if (type == XA_CARDINAL && format == 32 && count == 1)
			{
			if (XGetWindowProperty(GDK_DISPLAY(), win, xa_check, 0, 1, False,
					XA_CARDINAL, &type, &format, &count, &pad, &prop)
					== Success && prop)
				{
				XFree(prop);
				if (type == XA_CARDINAL && format == 32 && count == 1)
					GK.on_top_hint_ok = TRUE;
				}
			}
		}
	gdk_error_warnings = prev_error;
	if (GK.on_top_hint_ok && UC.on_top)
		set_on_top(UC.on_top);
	}

static void
gkrellm_cleanup()
	{
	save_inet_data();
	if (UC.save_position)
		set_or_save_position(1);
	}

gint
cb_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
	{
	/* From Gtk tutorial:
	|  If you return FALSE in the "delete_event" signal handler,
	|  GTK will emit the "destroy" signal. Returning TRUE means
	|  you don't want the window to be destroyed.
	|  This is useful for popping up 'are you sure you want to quit?'
	|  type dialogs.
	*/
	return(FALSE);
	}

void
cb_destroy_event()
	{
	gtk_main_quit();
	}

void
usage()
	{
	printf("usage: gkrellm [options]\n");
	printf("options:\n");
	printf("    -theme theme_dir\n");
	printf("    -geometry +x+y\n");
	printf("    -wm\n");
	printf("    -debug\n");
	printf("\n");
	printf("-wm allows window manager decorations.\n");
	printf("\n");
	}


int
main (int argc, char *argv[])
	{
	gint		i;
	gchar		*s,
				*geometry		= NULL;

	gtk_init(&argc, &argv);		/* Will call gdk_init() */
	gdk_imlib_init();

	GK.initialized = FALSE;
	GK.theme_path = "";

	exec_gkrellm[gk_argc++] = argv[0];
	for (i = 1; i < argc; ++i)
		{
		s = argv[i];
		if (*s == '-')
			{
			++s;
			if (*s == '-')
				++s;
			}
		if (strcmp(s, "theme") == 0)
			{
			GK.theme_path = g_strdup(argv[++i]);
			exec_theme_path = GK.theme_path;
			}
		else if (strcmp(s, "geometry") == 0 || strcmp(s, "g") == 0)
			geometry = argv[++i];
		else if (strcmp(s, "wm") == 0)
			{
			window_decorations = TRUE;
			exec_gkrellm[gk_argc++] = "-wm";
			}
		else if (strcmp(s, "debug") == 0)
			GK.debug = TRUE;
		else if (strcmp(s, "trace") == 0)
			GK.trace = TRUE;
		else if (strcmp(s, "test") == 0)
			GK.test = TRUE;
		else if (strcmp(s, "help") == 0 || strcmp(s, "h") == 0)
			{
			usage();
			exit(0);
			}
		else
			{
			printf("Unknown arg: %s\n", argv[i]);
			usage();
			exit(0);
			}
		}
	gdk_window_get_geometry(NULL /*gdk_root_parent*/, &i, &i, &w_display,
				&h_display, &i);
	gkrellmrc_config();
	read_user_config();

	if (geometry)
		GK.geometry = geometry;

	create_widget_tree();

	top_window = gtree.window;


	gtk_signal_connect (GTK_OBJECT(gtree.window), "delete_event",
			GTK_SIGNAL_FUNC (cb_delete_event), NULL);
	gtk_signal_connect (GTK_OBJECT(gtree.window), "destroy",
				GTK_SIGNAL_FUNC (cb_destroy_event), NULL);

	load_images();
	setup_colors_and_fonts(gtree.window);

	create_spacer(gtree.vbox, 3);

	create_frame_horizontal(gtree.top0_vbox, 1);

	/* Create the monitors.  Each monitor must increment GK.monitor_height
	|  by vertical space it consumes.
	*/
	GK.monitor_height = 0;

	create_hostname(gtree.top1_vbox);
	create_clock(gtree.monitor_vbox);

	create_cpu(gtree.monitor_vbox);
	create_proc(gtree.monitor_vbox);
	create_inet(gtree.monitor_vbox);
	create_disk(gtree.monitor_vbox);
	create_net(gtree.monitor_vbox);

	insert_spacer(gtree.monitor_vbox);

	create_meminfo(gtree.monitor_vbox);

	insert_spacer(gtree.monitor_vbox);

	create_fs(gtree.monitor_vbox);
	create_mailcheck(gtree.monitor_vbox);
	create_apm(gtree.monitor_vbox);
	create_uptime(gtree.monitor_vbox);

	create_frame_horizontal(gtree.bottom_vbox, 0);

	GK.initialized = TRUE;
	pack_side_frames();

	gtk_signal_connect(GTK_OBJECT(gtree.top0_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(top_frame_button_press), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.top1_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(top_frame_button_press), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.top0_event_box),
			"motion_notify_event", (GtkSignalFunc) gkrellm_motion, NULL);
	gtk_signal_connect(GTK_OBJECT(gtree.top1_event_box),
			"motion_notify_event", (GtkSignalFunc) gkrellm_motion, NULL);
	gtk_signal_connect(GTK_OBJECT(gtree.top0_event_box), "button_release_event",
			GTK_SIGNAL_FUNC(top_frame_button_release), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.top1_event_box), "button_release_event",
			GTK_SIGNAL_FUNC(top_frame_button_release), NULL );

	gtk_signal_connect(GTK_OBJECT(gtree.left_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(side_frame_button_press), NULL );
	gtk_signal_connect(GTK_OBJECT(gtree.right_event_box), "button_press_event",
			GTK_SIGNAL_FUNC(side_frame_button_press), NULL );

	popup_menu = create_popup_menu();

	if (GK.test)
		gtk_widget_hide(gtree.window);
	gtk_widget_show(gtree.window);

	if (GK.geometry)		/* User wants placement */
		place_gkrellm();
	else if (UC.save_position)
		set_or_save_position(0);

	GK.start_time = time(0);
	current_tm = *(localtime(&GK.start_time));
	load_inet_data();
	apply_stat_config();	/* Fixme: Get the smp cpus/temp display right */

	start_timer();
	gtk_main ();
	gkrellm_cleanup();

	return 0;
	}

void
restart_gkrellm()
	{
	gint	x_gkrell, y_gkrell, w, h;
	gchar	geom[32], xbuf[32], ybuf[32];

	gkrellm_cleanup();
	gdk_window_get_root_origin(top_window->window, &x_gkrell, &y_gkrell);
	gdk_window_get_size(top_window->window, &w, &h);

	if (y_gkrell + h >= h_display)	/* GKrellM was on bottom of screen */
		sprintf(ybuf, "-0");
	else
		sprintf(ybuf, "+%d", y_gkrell);
	if (x_gkrell + w >= w_display)	/* GKrellM was on right of screen */
		sprintf(xbuf, "-0");
	else
		sprintf(xbuf, "+%d", x_gkrell);
	sprintf(geom, "%s%s", xbuf, ybuf);
	exec_gkrellm[gk_argc++] = "-g";
	exec_gkrellm[gk_argc++] = geom;
	if (exec_theme_path && *exec_theme_path != '\0')
		{
		exec_gkrellm[gk_argc++] = "-theme";
		exec_gkrellm[gk_argc++] = exec_theme_path;
		}
	exec_gkrellm[gk_argc] = NULL;
	execvp(exec_gkrellm[0], exec_gkrellm);
	printf("exec failed\n");
	exit(0);
	}

