/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"

typedef struct
	{
	GtkWidget		*vbox;

	gchar			*name;
	gint			idx;
	Chart			*chart;
	Chart			chart_minute;
	Chart			chart_hour;
	gint			inet_extra;
	gint			height;

	gint			scale_min_minute;
	gint			scale_min_hour;

	gshort			*mark_data;		/* Draw marks if hits for any second */
	gint			mark_position,
					mark_prev_hits;

	gchar			*label0;
	gint			active0;
	gint			prev_active0;
	gulong			hits0;
	gulong			port0_0,
					port0_1;

	gchar			*label1;
	gint			active1;
	gint			prev_active1;
	gulong			hits1;
	gulong			port1_0,
					port1_1;
	}
	InetMon;


  /* Values for flag.
  */
#define	TCP_DEAD	0
#define	TCP_ALIVE	1

typedef struct
	{
	gint	state;
	gint	local_port;
	gulong	remote_addr;
	gint	remote_port;
	}
	ActiveTCP;

static GList		*inet_mon_list;

static GList		*active_tcp_list,
					*free_tcp_list;

static GtkWidget	*inet_vbox;
static GdkImage		*grid;

static gint			n_inet_monitors;

static ActiveTCP *
tcp_alloc()
	{
	ActiveTCP	*tcp;

	if (free_tcp_list)
		{
		tcp = free_tcp_list->data;
		free_tcp_list = g_list_remove(free_tcp_list, tcp);
		}
	else
		tcp = g_new0(ActiveTCP, 1);
	return tcp;
	}

static gint
log_active_port(ActiveTCP *tcp)
	{
	GList		*list;
	ActiveTCP	*active_tcp, *new_tcp;

	for (list = active_tcp_list; list; list = list->next)
		{
		active_tcp = (ActiveTCP *) (list->data);
		if (   active_tcp->remote_addr == tcp->remote_addr
			&& active_tcp->remote_port == tcp->remote_port
			&& active_tcp->local_port == tcp->local_port
		   )
			{
			active_tcp->state = TCP_ALIVE;
			return 0;		/* Old hit still alive, not a new hit	*/
			}
		}
	tcp->state = TCP_ALIVE;
	new_tcp = tcp_alloc();
	*new_tcp = *tcp;
	active_tcp_list = g_list_prepend(active_tcp_list, (gpointer) new_tcp);
	return 1;		/* A new hit	*/
	}

void
scan_for_active_port(gchar *buf)
	{
	InetMon		*in;
	ActiveTCP	tcp;
	GList		*list;
	gint		tcp_status;

	sscanf(buf, "%*d: %*x:%x %lx:%x %x", &tcp.local_port,
						&tcp.remote_addr, &tcp.remote_port, &tcp_status);
	if (tcp_status == 1)
		for (list = inet_mon_list; list; list = list->next)
			{
			in = (InetMon *) list->data;
			if (in->port0_0 == tcp.local_port || in->port0_1 == tcp.local_port)
				{
				++in->active0;
				in->hits0 += log_active_port(&tcp);
				}
			if (in->port1_0 == tcp.local_port || in->port1_1 == tcp.local_port)
				{
				++in->active1;
				in->hits1 += log_active_port(&tcp);
				}
			}
	}

void
draw_inet_extra(InetMon *in)
	{
	gint	x0, x1, y;
	gchar	buf[16];

	y = 2;
	x0 = 4;
	if (in->label0 && in->label0[0] != '\0')
		{
		sprintf(buf, "%d", in->active0);
		y += 2 + label_font_ascent;
		x0 = draw_chart_label(in->chart, GK.label_font, x0, y, buf);
		draw_chart_label(in->chart, GK.alt_font, x0 + 4, y, in->label0);
		}

	if (in->label1 && in->label1[0] != '\0')
		{
		sprintf(buf, "%d", in->active1);
		y += 2 + label_font_ascent;
		x1 = draw_chart_label(in->chart, GK.label_font, 4, y, buf);
		if (x1 < x0)
			x1 = x0;
		draw_chart_label(in->chart, GK.alt_font, x1 + 4, y, in->label1);
		}
	}

  /* Use the reserved area below the main chart to draw marks if any
  |  hits in second intervals.
  */
void
draw_inet_mark_data(InetMon *in, gint minute_mark)
	{
	Chart	*cp;
	gint	hits, x, y, n;

	cp = in->chart;
	in->mark_position = (in->mark_position + 1) % cp->w;
	if (minute_mark)
		{
		in->mark_data[in->mark_position] = -1;	/* minute flag in the data */
		return;
		}
	hits = in->hits0 + in->hits1;
	in->mark_data[in->mark_position] = hits - in->mark_prev_hits;
	in->mark_prev_hits = hits;

	/* Clear out the area and redraw the marks.
	*/
	y = cp->h - cp->y;
	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cp->bg_clean_pixmap,
			0, y,  0, y,  cp->w, cp->y);
	gdk_gc_set_foreground(GK.draw1_GC, &GK.out_color);
	gdk_gc_set_foreground(GK.draw2_GC, &GK.in_color);
	gdk_gc_set_foreground(GK.draw3_GC, &GK.white_color);
	for (n = 0; n < cp->w; ++n)
		{
		x = (in->mark_position + n + 1) % cp->w;
		if (in->mark_data[x] > 0)
			gdk_draw_line(cp->pixmap, GK.draw1_GC,
						cp->x + n, cp->h - 1, cp->x + n, y);
		else if (in->mark_data[x] == -1)	/* Minute tick	*/
			gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n, cp->h - 1, cp->x + n, y);
		}
	gdk_draw_pixmap(cp->drawing_area->window, GK.draw1_GC, cp->pixmap,
			0, y,  0, y,  cp->w, cp->y);
	}

static gchar	*weekday_char[]	= { "S", "M", "T", "W", "T", "F", "S" };

void
draw_inet_chart(InetMon *in)
	{
	Chart		*cp;
	GdkColor	tmp_color;
	guint32		pixel0, pixel1;
	gint		y0, h4, n, wday, hour, minute;

	cp = in->chart;

	/* Each draw of the hour chart must do a total redraw because the
	|  in/out_data_bitmaps are scrolled when the minute inet chart data
	|  is stored.  I need separate bitmaps for hour/minute XXX.
	*/
	if (cp == &in->chart_hour || GK.hour_tick)
		cp->scale_max = 0;

	draw_chart(cp);

	y0 = cp->h - cp->y;
	h4 = y0 / 4;
	if (grid == NULL)
		grid = gdk_image_get(in->chart->grid_pixmap, 0, 0, UC.chart_width, 2);

	gdk_gc_set_foreground(GK.draw3_GC, &GK.white_color);
	if (cp == &in->chart_hour)
		{
		hour = current_tm.tm_hour;
		wday = current_tm.tm_wday;
		for (n = cp->w - 1; n >= 0; --n)
			{
			/* When hour ticked to 0, 23rd hour data was stored and a slot
			|  was skipped.
			*/
			if (hour == 0)	/* Draw day mark at midnight	*/
				{
				pixel0 = gdk_image_get_pixel(grid, cp->x + n, 0);
				pixel1 = gdk_image_get_pixel(grid, cp->x + n, 1);

				tmp_color.pixel = pixel0;
				gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
				gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, 3);
				gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, 3);

				tmp_color.pixel = pixel1;
				gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
				gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n, y0 - 3, cp->x + n, 3);
				gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
						cp->x + n, y0 - 3, cp->x + n, 3);
				}
			if (hour == 1 && n < cp->w - 5)
				draw_chart_label(in->chart, GK.alt_font,
						cp->x + n, label_font_ascent + 3, weekday_char[wday]);
			if (--hour < 0)
				{
				hour = 24;		/* Extra hour for skipped slot	*/
				if (--wday < 0)
					wday = 6;
				}
			}
		}
	if (cp == &in->chart_minute)
		{
		minute = current_tm.tm_min;
		for (n = cp->w - 1; n >= 0; --n)
			{
			/* When minute ticked to 0, 59 minute data was stored and a slot
			|  was skipped.
			*/
			if (minute == 0)	/* Draw hour mark	*/
				{
				pixel0 = gdk_image_get_pixel(grid, cp->x + n, 0);
				pixel1 = gdk_image_get_pixel(grid, cp->x + n, 1);

				tmp_color.pixel = pixel0;
				gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
				gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, y0 - h4);
				gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, y0 - h4);

				tmp_color.pixel = pixel1;
				gdk_gc_set_foreground(GK.draw3_GC, &tmp_color);
				gdk_draw_line(cp->pixmap, GK.draw3_GC,
						cp->x + n, y0 - 3, cp->x + n, y0 - h4);
				gdk_draw_line(cp->drawing_area->window, GK.draw3_GC,
						cp->x + n, y0 - 3, cp->x + n, y0 - h4);
				}
			if (--minute < 0)
				minute = 60;	/* extra minute for skipped slot */
			}
		}
	if (in->inet_extra)
		draw_inet_extra(in);
	in->prev_active0 = in->active0;
	in->prev_active1 = in->active1;
	}

void
update_inet(void)
	{
	FILE		*f;
	InetMon		*in;
	Chart		*cp;
	ActiveTCP	*tcp;
	GList		*list;
	gchar		buf[160];

	if (inet_mon_list == NULL)
		return;

	if (GK.second_tick || GK.minute_tick || GK.hour_tick)
		{
		for (list = inet_mon_list; list; list = list->next)
			{
			in = (InetMon *) list->data;
			in->active0 = 0;
			in->active1 = 0;
			}
		/* Assume all connections are dead, then scan_active_ports() will set
		|  still alive ones back to alive.  Then I can prune really dead ones.
		*/
		for (list = active_tcp_list; list; list = list->next)
			{
			tcp = (ActiveTCP *)(list->data);
			tcp->state = TCP_DEAD;
			}
		f = fopen("/proc/net/tcp", "r");
		if (f)
			{
			fgets(buf, sizeof(buf), f);		/* Waste first line */
			while (fgets(buf, sizeof(buf), f))
				scan_for_active_port(buf);
			fclose(f);
			}
		for (list = active_tcp_list; list; )
			{
			tcp = (ActiveTCP *)(list->data);
			if (tcp->state == TCP_DEAD)
				{
				if (list == active_tcp_list)
					active_tcp_list = active_tcp_list->next;
				list = g_list_remove(list, tcp);
				free_tcp_list = g_list_prepend(free_tcp_list, tcp);
				}
			else
				list = list->next;
			}
		}

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		cp = in->chart;
		if (GK.hour_tick)
			{
			store_chart_data(&in->chart_hour, in->hits1, in->hits0, 0);

			if (GK.day_tick)	/* Make room for vertical day grid */
				{
				store_chart_data(&in->chart_hour, in->hits1, in->hits0, 0);
				store_chart_data(&in->chart_hour, in->hits1, in->hits0, 0);
				}
			if (cp == &in->chart_hour)
				draw_inet_chart(in);
			}
		if (GK.minute_tick)
			{
			store_chart_data(&in->chart_minute, in->hits1, in->hits0, 0);

			if (GK.hour_tick)	/* Make room for vertical hour grid */
				{
				store_chart_data(&in->chart_minute, in->hits1, in->hits0, 0);
				store_chart_data(&in->chart_minute, in->hits1, in->hits0, 0);
				}
			if (cp == &in->chart_minute)
				draw_inet_chart(in);
			draw_inet_mark_data(in, 1);
			}
		else if (   GK.second_tick
				 && (   in->prev_active0 != in->active0
					 || in->prev_active1 != in->active1
					)
				)
			draw_inet_chart(in);	/* Just to update extra info draw */

		if (GK.second_tick)
			draw_inet_mark_data(in, 0);

		update_krell(&in->chart_minute.panel, in->chart_minute.panel.krell,
					in->hits0 + in->hits1);
		draw_layers(&in->chart_minute.panel);
		}
	}


  /* Real pixmaps are always in chart_minute even if chart_hour is displayed.
  */
static gint
inet_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	InetMon		*in;
	GList		*list;
	Chart		*cp;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		cp = &in->chart_minute;
		if (widget == cp->panel.drawing_area)
			gdk_draw_pixmap(widget->window,GK.draw1_GC, cp->panel.pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
		if (widget == cp->drawing_area)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC, cp->pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
			if (in->inet_extra)
				draw_inet_extra(in);
			}
		}
	return FALSE;
	}

static gint
cb_inet_extra(GtkWidget *widget, GdkEventButton *event)
	{
	InetMon		*in;
	GList		*list;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (widget != in->chart->drawing_area)
			continue;
		if (event->button == 1)
			{
			in->inet_extra = 1 - in->inet_extra;
			in->prev_active0 = -1;	/* Force a redraw */
			draw_inet_chart(in);
			}
		if (event->button == 2)
			{
			if (in->chart == &in->chart_minute)
				in->chart = &in->chart_hour;
			else
				in->chart = &in->chart_minute;
			in->chart->scale_max = 0;			/* Force a rescale */
			draw_inet_chart(in);
			}
		}
	return TRUE;
	}

static void
create_inet_monitor(GtkWidget *vbox1, InetMon *in, gint index)
	{
	GtkWidget	*vbox;
	Chart		*cp, *cp_hour;
	Style		*style;
	gchar		buf[16];

	if (GK.debug)
		printf("Creating inet monitor %d %s %s\n",
			index, in->label0, in->label1);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox1), vbox);
	in->vbox = vbox;

	sprintf(buf, "inet%d", index);
	in->name = g_strdup(buf);
	in->chart = &in->chart_minute;
	cp = in->chart;
	cp_hour = &in->chart_hour;

	cp->name = in->name;
	in->idx = index;
	++n_inet_monitors;

	cp->scale_min = in->scale_min_minute;
	cp_hour->scale_min = in->scale_min_hour;

	in->chart->y = 3;
	style = GK.chart_style[INET_IMAGE];
	create_krell(in->name, GK.krell_panel_image[INET_IMAGE],
					&cp->panel.krell, style);

	/* Inet krells are not related to chart scale_max.  Just give a constant
	|  full scale of 5.
	*/
	cp->panel.krell->full_scale = 5;

	cp->h = UC.chart_height[MON_INET];
	create_chart(vbox, cp, INET_IMAGE);
	default_textstyle(&cp->panel.label.textstyle, TEXTSTYLE_LABEL);
	configure_panel(&cp->panel, cp->name, style);
	create_panel_area(vbox, &cp->panel, GK.bg_panel_image[INET_IMAGE]);
	in->height = cp->h + cp->panel.h;
	GK.monitor_height += in->height;

	gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
			(GtkSignalFunc) inet_expose_event, NULL);
	gtk_signal_connect(GTK_OBJECT(cp->drawing_area),"button_press_event",
			(GtkSignalFunc) cb_inet_extra, NULL);

	gtk_signal_connect(GTK_OBJECT(cp->panel.drawing_area),"expose_event",
			(GtkSignalFunc) inet_expose_event, NULL);

	alloc_chart_data(cp);
	clear_chart(cp);

	/* Setup the hour chart.  Panel and krell are in the minute chart, so
	|  The chart_hour is going to be a skeleton chart with only stuff
	|  needed for plotting the hourly data.  The same pixmaps are used for
	|  minute or hour draws.
	*/
	cp_hour = &in->chart_hour;
	cp_hour->name = cp->name;
	cp_hour->drawing_area = cp->drawing_area;
	cp_hour->pixmap = cp->pixmap;
	cp_hour->in_grided_pixmap = cp->in_grided_pixmap;
	cp_hour->out_grided_pixmap = cp->out_grided_pixmap;
	cp_hour->in_data_bitmap = cp->in_data_bitmap;
	cp_hour->out_data_bitmap = cp->out_data_bitmap;

	cp_hour->bg_clean_pixmap = cp->bg_clean_pixmap;
	cp_hour->bg_grided_pixmap = cp->bg_grided_pixmap;
	cp_hour->grid_pixmap = cp->grid_pixmap;
	cp_hour->x = cp->x;
	cp_hour->y = cp->y;
	cp_hour->w = cp->w;
	cp_hour->h = cp->h;
	alloc_chart_data(cp_hour);
	in->mark_data = g_new0(gshort, cp->w);

	/* Don't want to waste an hour priming the pump, and don't need to
	|  because data always starts at zero.
	*/
	in->chart_hour.primed = TRUE;
	in->chart_minute.primed = TRUE;
	
	in->inet_extra = UC.enable_extra_info;
	gtk_widget_show_all(vbox);
	}

void
destroy_inet_monitor(InetMon *in)
	{
	in->chart = &in->chart_minute;	/* Everything was allocate here */
	g_free(in->name);
	g_free(in->label0);
	g_free(in->label1);
	GK.monitor_height -= in->height;
	destroy_panel(&in->chart->panel);
	destroy_krell(in->chart->panel.krell);
	destroy_chart(in->chart);
	g_free(in->chart_hour.pDataIn);
	g_free(in->chart_hour.pDataOut);
	g_free(in->mark_data);
	gtk_widget_destroy(in->vbox);
	g_free(in);
	--n_inet_monitors;
	}

void
create_inet(GtkWidget *vbox)
	{
	GList		*list;
	gint		i;

	inet_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), inet_vbox);
	gtk_widget_show(inet_vbox);

	for (i = 0, list = inet_mon_list; list; ++i, list = list->next)
		create_inet_monitor(inet_vbox, (InetMon *)list->data, i);
	}


void
write_inet_config(FILE *f)
	{
	GList	*list;
	InetMon	*in;
	gchar	*l0, *l1;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		l0 = (in->label0[0] == '\0') ? "NONE" : in->label0;
		l1 = (in->label1[0] == '\0') ? "NONE" : in->label1;
		fprintf(f, "inet %s %lu %lu %s %lu %lu %d %d\n",
				l0, in->port0_0, in->port0_1,
				l1, in->port1_0, in->port1_1,
				in->scale_min_minute, in->scale_min_hour);

		}
	}

void
load_inet_config(gchar *arg)
	{
	InetMon	*in;
	gchar	label0[16], label1[16];

	in = g_new0(InetMon, 1);
	sscanf(arg, "%s %lu %lu %s %lu %lu %d %d",
			label0, &in->port0_0, &in->port0_1,
			label1, &in->port1_0, &in->port1_1,
			&in->scale_min_minute, &in->scale_min_hour);

	if (strcmp(label0, "NONE") == 0)
		{
		label0[0] = '\0';
		in->port0_0 = 0;
		in->port0_1 = 0;
		}
	if (strcmp(label1, "NONE") == 0)
		{
		label1[0] = '\0';
		in->port1_0 = 0;
		in->port1_1 = 0;
		}
	in->label0 = g_strdup(label0);
	in->label1 = g_strdup(label1);
	inet_mon_list = g_list_append(inet_mon_list, in);
	}


/* --------------------------------------------------------------------- */
/* Read / write inet historical data so it won't be lost at restarts.	*/

  /* Read saved inet data (from a previous gkrellm process).  Return the
  |  number of missing data slots (skew).
  */
static gint
read_inet_data(Chart *cp, FILE *f, gint minute_chart,
			gint min, gint hour, gint yday, gint width)
	{
	gchar	data[64];
	gint	n, in, out, cur_slot, skew, day;

	day = current_tm.tm_yday - yday;

	/* Check for new years wrap around. I don't handle leap year here, will
	|  get some email, then be safe for four more years...
	*/
	if (day < 0)
		day = current_tm.tm_yday + ((yday < 365) ? 365 - yday : 0);

	cur_slot = day * 24 + current_tm.tm_hour;
	n = hour;
	if (minute_chart)
		{
		cur_slot = cur_slot * 60 + current_tm.tm_min;
		n = n * 60 + min;
		}
	skew = cur_slot - n;
	if (GK.debug)
		printf("read_inet_data() %d cur_slot=%d skew=%d\n", minute_chart,
					cur_slot, skew);

	for (n = 0; n < width; ++n)
		{
		if (fgets(data, sizeof(data), f) == NULL)
			break;

		if (skew >= cp->w)	/* All stored data is off the chart	*/
			continue;

		/* Use chart data storing routines to load in data so I don't care
		|  if current chart width is less or greater than stored data width.
		|  Charts will circular buff fill until data runs out.
		*/
		sscanf(data, "%d %d", &in, &out);
		cp->prevIn  = 0;
		cp->prevOut = 0;
		store_chart_data(cp, out, in, 0);
		}
	/* Need to store zero data for time slots not in read data to bring
	|  the chart up to date wrt current time.  As in update_inet() I need
	|  to skip slots for hour or minute ticks.
	|  Warning: skew can be negative if quit gkrellm, change system clock
	|  to earlier time, then restart gkrellm.
	*/
	if ((n = skew) < cp->w)		/* Do this only if some data was stored  */
		{
		while (n-- > 0)
			{
			store_chart_data(cp, out, in, 0);
			if (minute_chart && min++ == 0)
				{
				store_chart_data(cp, out, in, 0);
				store_chart_data(cp, out, in, 0);
				if (min == 60)
					min = 0;
				}
			else if (!minute_chart && hour++ == 0)
				{
				store_chart_data(cp, out, in, 0);
				store_chart_data(cp, out, in, 0);
				if (hour == 24)
					hour = 0;
				}
			}
		}
	return skew;
	}

static void
write_inet_data(Chart *cp, FILE *f)
	{
	gchar	data[64];
	gint	n, x;

	for (n = 0; n < cp->w; ++n)
		{
		x = computed_index(cp, n);
		sprintf(data, "%d %d\n", cp->pDataIn[x], cp->pDataOut[x]);
		fputs(data, f);
		}
	}

static gchar *
make_inet_data_fname(InetMon *in)
	{
	static gchar	fname[128];

	sprintf(fname, "%s/%s/%s/inet_%ld_%ld_%ld_%ld", homedir(),
		GKRELLM_DIR, GKRELLM_DATA_DIR,
		in->port0_0, in->port0_1, in->port1_0, in->port1_1);
	return fname;
	}

void
save_inet_data()
	{
	FILE			*f;
	GList			*list;
	InetMon			*in;
	gchar			buf[128], *fname;

	sprintf(buf, "%s/%s/%s", homedir(), GKRELLM_DIR, GKRELLM_DATA_DIR);
	if (!isdir(buf))
		{
		if (mkdir(buf, 0755) < 0)
			printf("Cannot create gkrellm data directory %s\n", buf);
		}
	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		fname = make_inet_data_fname(in);
		if ((f = fopen(fname, "w")) == NULL)
			continue;

		fputs("minute hour yday width\n", f);
		sprintf(buf, "%d %d %d %d\n", current_tm.tm_min, current_tm.tm_hour,
					current_tm.tm_yday, in->chart_minute.w);
		fputs(buf, f);

		/* Save any accumulated hits which have not been stored into the
		|  chart data array, and then save the chart data.
		*/
		fputs("hits0 hits1 min.prev0 min.prev1 hr.prev0 hr.prev1\n", f);
		sprintf(buf, "%ld %ld %ld %ld %ld %ld\n", in->hits0, in->hits1,
					in->chart_minute.prevIn, in->chart_minute.prevOut,
					in->chart_hour.prevIn, in->chart_hour.prevOut);
		fputs(buf, f);
		write_inet_data(&in->chart_minute, f);
		write_inet_data(&in->chart_hour, f);
		fclose(f);
		}
	}

void
load_inet_data()
	{
	FILE	*f;
	GList	*list;
	InetMon	*in;
	gchar	buf[96], *fname;
	gint	min, hour, yday, len, skew;
	gulong	min_prevIn, min_prevOut, hr_prevIn, hr_prevOut;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		fname = make_inet_data_fname(in);
		if ((f = fopen(fname, "r")) == NULL)
			{
			draw_inet_chart(in);
			continue;
			}
		if (GK.debug)
			printf("Loading %s\n", fname);
		fgets(buf, sizeof(buf), f);		/* Comment line */
		fgets(buf, sizeof(buf), f);
		sscanf(buf, "%d %d %d %d", &min, &hour, &yday, &len);
		fgets(buf, sizeof(buf), f);		/* Comment line */
		fgets(buf, sizeof(buf), f);
		sscanf(buf, "%ld %ld %ld %ld %ld %ld", &in->hits0, &in->hits1,
					&min_prevIn, &min_prevOut, &hr_prevIn, &hr_prevOut);

		skew = read_inet_data(&in->chart_minute, f, 1, min, hour, yday, len);
		if (skew > 0)  /* Current minute slot is different from saved */
			{
			in->chart_minute.prevIn = in->hits0;
			in->chart_minute.prevOut = in->hits1;
			}
		else  /* saved hit0 and hit1 data is valid for current min slot */
			{
			in->chart_minute.prevIn = min_prevIn;
			in->chart_minute.prevOut = min_prevOut;
			}

		skew = read_inet_data(&in->chart_hour, f, 0, min, hour, yday, len);
		if (skew > 0)  /* Current hour slot is different from saved */
			{
			in->chart_hour.prevIn = in->hits0;
			in->chart_hour.prevOut = in->hits1;
			}
		else  /* saved hit0 and hit1 data is valid for current hour slot */
			{
			in->chart_hour.prevIn = hr_prevIn;
			in->chart_hour.prevOut = hr_prevOut;
			}
		fclose(f);
		in->chart_minute.scale_max = 0;
		in->chart_hour.scale_max = 0;
		draw_inet_chart(in);
		}
	}

/* --------------------------------------------------------------------- */

GtkWidget			*inet_clist;
static gint			selected_row,
					inet_list_modified;


static GtkWidget	*label0_entry,
					*label1_entry;
static GtkWidget	*port0_0_entry,
					*port0_1_entry,
					*port1_0_entry,
					*port1_1_entry;

static GtkWidget	*minute_spinbutton,
					*hour_spinbutton;

static gint		min_res_temp,
				hr_res_temp;

static gint	inet_map[] =
	{
	2, 5,
	10, 20, 50,
	100, 200, 500,
	1000, 2000, 5000,
	10000, 20000, 50000
	};


static void
cb_inet_resolution(GtkWidget *widget, GtkSpinButton *spin)
	{
	gchar	smbuf[16], shbuf[16];
	gint	sm, sh;
	gint	value, *ref;

	value = gtk_spin_button_get_value_as_int(spin);
	if (spin == GTK_SPIN_BUTTON(minute_spinbutton))
		ref = &min_res_temp;
	else
		ref = &hr_res_temp;
	if (value != *ref)		/* Avoid recursion */
		{
		value = map_1_2_5(value, inet_map, sizeof(inet_map)/sizeof(int));
		*ref = value;
		gtk_spin_button_set_value(spin, (gfloat) value);

		if (selected_row >= 0)
			{
			sm = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(minute_spinbutton));
			sh = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(hour_spinbutton));
			sprintf(smbuf, "%d", sm);
			sprintf(shbuf, "%d", sh);
			gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 8, smbuf);
			gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 9, shbuf);
			inet_list_modified = TRUE;
#if 0
			/* If I can correspond selected row to an existing monitor, go
			|  ahead and let the charts track??
			*/
			in->chart_hour.scale_min = sm;
			in->chart_minute.scale_min = sh;
			in->chart->scale_max = 0;
			draw_inet_chart(in);
#endif
			}
		}
	}

static void
reset_inet_entries()
	{
	gtk_entry_set_text(GTK_ENTRY(label0_entry), "");
	gtk_entry_set_text(GTK_ENTRY(port0_0_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(port0_1_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(label1_entry), "");
	gtk_entry_set_text(GTK_ENTRY(port1_0_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(port1_1_entry), "0");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(minute_spinbutton), 2);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(hour_spinbutton), 2);
	min_res_temp = 0;
	hr_res_temp = 0;
	}

static void
cb_inet_clist_selected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	gchar	*s;

	gtk_clist_get_text(GTK_CLIST(clist), row, 0, &s);
	gtk_entry_set_text(GTK_ENTRY(label0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 1, &s);
	gtk_entry_set_text(GTK_ENTRY(port0_0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 2, &s);
	gtk_entry_set_text(GTK_ENTRY(port0_1_entry), s);

	gtk_clist_get_text(GTK_CLIST(clist), row, 4, &s);
	gtk_entry_set_text(GTK_ENTRY(label1_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 5, &s);
	gtk_entry_set_text(GTK_ENTRY(port1_0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 6, &s);
	gtk_entry_set_text(GTK_ENTRY(port1_1_entry), s);

	gtk_clist_get_text(GTK_CLIST(clist), row, 8, &s);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(minute_spinbutton), atoi(s));
	gtk_clist_get_text(GTK_CLIST(clist), row, 9, &s);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(hour_spinbutton), atoi(s));

	selected_row = row;
	}

static void
cb_inet_clist_unselected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	selected_row = -1;
	reset_inet_entries();
	}

static void
cb_enter_inet(GtkWidget *widget, gpointer data)
	{
	gchar	*buf[11], smbuf[16], shbuf[16];
	gint	sm, sh;

	buf[0] = gtk_entry_get_text(GTK_ENTRY(label0_entry));
	buf[1] = gtk_entry_get_text(GTK_ENTRY(port0_0_entry));
	buf[2] = gtk_entry_get_text(GTK_ENTRY(port0_1_entry));
	buf[3] = "";
	buf[4] = gtk_entry_get_text(GTK_ENTRY(label1_entry));
	buf[5] = gtk_entry_get_text(GTK_ENTRY(port1_0_entry));
	buf[6] = gtk_entry_get_text(GTK_ENTRY(port1_1_entry));
	buf[7] = "";
	sm = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(minute_spinbutton));
	sh = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(hour_spinbutton));
	sprintf(smbuf, "%d", sm);
	sprintf(shbuf, "%d", sh);
	buf[8] = smbuf;
	buf[9] = shbuf;
	buf[10] = NULL;

	/* Validate the values
	*/
	if (   (*buf[0] == '\0' && *buf[4] == '\0')
		|| ((*buf[0] != '\0' && atoi(buf[1]) == 0 && atoi(buf[2]) == 0))
		|| ((*buf[4] != '\0' && atoi(buf[5]) == 0 && atoi(buf[6]) == 0))
	   )
		return;

	if (selected_row >= 0)
		{
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 0, buf[0]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 1, buf[1]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 2, buf[2]);

		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 4, buf[4]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 5, buf[5]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 6, buf[6]);

		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 8, buf[8]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 9, buf[9]);
		gtk_clist_unselect_row(GTK_CLIST(inet_clist), selected_row, 0);
		selected_row = -1;
		}
	else
		gtk_clist_append(GTK_CLIST(inet_clist), buf);
	reset_inet_entries();
	inet_list_modified = TRUE;
	}

static void
cb_delete_inet(GtkWidget *widget, gpointer data)
	{
	reset_inet_entries();
	if (selected_row >= 0)
		{
		gtk_clist_remove(GTK_CLIST(inet_clist), selected_row);
		inet_list_modified = TRUE;
		selected_row = -1;
		}
	}

void
apply_inet_config()
	{
	InetMon	*in;
	GList	*list, *new_inet_list, *old_inet_list;
	gchar	*s;
	gint	row, need_new_monitor;


	if (! inet_list_modified)
		{	/* Rescale and redraw in case grid mode has changed */
		for (list = inet_mon_list; list; list = list->next)
			{
			in = (InetMon *) list->data;
			in->chart_minute.scale_max = 0;
			in->chart_hour.scale_max = 0;
			draw_inet_chart(in);
			}
		return;
		}

	/* Just save all data and then later read it back in.  This avoids
	|  complicated detecting of name changes but ports the same, moving
	|  a inet down or up slots, etc.  Data is lost only if a port number
	|  for a monitor is changed.
	*/
	save_inet_data();
	new_inet_list = NULL;
	old_inet_list = inet_mon_list;
	for (row = 0; row < GTK_CLIST(inet_clist)->rows; ++row)
		{
		if (old_inet_list)
			{
			in = (InetMon *) old_inet_list->data;
			g_free(in->label0);
			g_free(in->label1);
			old_inet_list = old_inet_list->next;
			need_new_monitor = FALSE;
			}
		else
			{
			in = g_new0(InetMon, 1);
			need_new_monitor = TRUE;
			}
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 0, &s);
		in->label0 = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 1, &s);
		in->port0_0 = atoi(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 2, &s);
		in->port0_1 = atoi(s);

		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 4, &s);
		in->label1 = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 5, &s);
		in->port1_0 = atoi(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 6, &s);
		in->port1_1 = atoi(s);

		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 8, &s);
		in->scale_min_minute = atoi(s);
		in->chart_minute.scale_min = in->scale_min_minute;
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 9, &s);
		in->scale_min_hour = atoi(s);
		in->chart_hour.scale_min = in->scale_min_hour;

		new_inet_list = g_list_append(new_inet_list, in);
		if (need_new_monitor)
			{
			create_inet_monitor(inet_vbox, in, n_inet_monitors);
			draw_inet_chart(in);
			}
		}
	while (old_inet_list)	/* Number of inet monitors went down. */
		{
		destroy_inet_monitor((InetMon *) old_inet_list->data);
		old_inet_list = old_inet_list->next;
		}
	while (inet_mon_list)	/* Destroy the old glist */
		inet_mon_list = g_list_remove(inet_mon_list, inet_mon_list->data);
	inet_mon_list = new_inet_list;
	pack_side_frames();
	load_inet_data();
	inet_list_modified = FALSE;
	}

static gchar	*inet_info_text =
"Inet charts show historical TCP port hits on a minute or hourly\n"
"chart. Below the chart there is a strip where marks are drawn for\n"
"port hits in second intervals.   The inet krell has a full scale\n"
"value of 5 hits and samples once per second.  The extra info\n"
"display shows current TCP port connections.\n\n"

"For each internet monitor, you can specify two labeled data sets with\n"
"one or two non-zero port numbers entered for each data set.  Two\n"
"ports are allowed because some internet ports are related and you\n"
"might want to group them.  Check /etc/services for port numbers.\n\n"

"For example, if you created an inet monitor:\n"
"    http 80 8080   ftp 21\n"
"Http hits on the standard http port 80 and www web caching service\n"
"on port 8080 are combined and plotted in the \"in\" color.  Ftp hits\n"
"on the single ftp port 21 are plotted in the \"out\" color.\n\n"

"Set chart grid resolutions for the minute and hour charts with the\n"
"hits/minute and hits/hour spin buttons.  Chart data is saved when\n"
"GKrellM exits and loaded at startup.\n\n"

"    User Interface\n"
" * Left click on an inet chart to toggle the extra info display of\n"
"current TCP port connections.\n"
" * Middle click on an inet chart to toggle the hour/minute charts.\n"
;

static gchar	*inet_titles[10] =
	{"Label", "Port0", "Port1", "    ", "Label", "Port0", "Port1", "    ",
		"Hits/min", "Hits/hr" };

void
create_inet_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*table;
	GtkWidget		*hbox, *vbox;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;
	GtkWidget		*separator;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GtkWidget		*label;
	GtkWidget		*button;
	GList			*list;
	InetMon			*in;
	gchar			*buf[11];
	gchar			p00[16], p01[16], p10[16], p11[16], resm[16], resh[16];
	
	inet_list_modified = FALSE;
	selected_row = -1;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	vbox = create_tab(tabs, "Ports");

	table = gtk_table_new(7, 5, FALSE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 3);

	label = gtk_label_new("Data0");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 0, 1);
	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 0, 3, 1, 2);
	label = gtk_label_new("Data1");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 4, 7, 0, 1);
	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 4, 7, 1, 2);

	separator = gtk_vseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 3, 4, 0, 5);

	label = gtk_label_new("Label");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	label = gtk_label_new("Port0");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 2, 3);
	label = gtk_label_new("Port1");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 2, 3);
	label = gtk_label_new("Label");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 4, 5, 2, 3);
	label = gtk_label_new("Port0");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 5, 6, 2, 3);
	label = gtk_label_new("Port1");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 6, 7, 2, 3);

	label0_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(label0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), label0_entry, 0, 1, 3, 4);
	port0_0_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port0_0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port0_0_entry, 1, 2, 3, 4);
	port0_1_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port0_1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port0_1_entry, 2, 3, 3, 4);

	label1_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(label1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), label1_entry, 4, 5, 3, 4);
	port1_0_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port1_0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port1_0_entry, 5, 6, 3, 4);
	port1_1_entry = gtk_entry_new_with_max_length(5);
	gtk_widget_set_usize(port1_1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port1_1_entry, 6, 7, 3, 4);

	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 0, 7, 4, 5);

	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	adj = (GtkAdjustment *) gtk_adjustment_new(2.0, 2.0, 100000.0,
					1.0 /*step*/, 1.0 /*page*/, 0.0);
	minute_spinbutton = gtk_spin_button_new (adj, 0.95, 0 /*digits*/);
	gtk_widget_set_usize(minute_spinbutton, 50, 0);
	spin = GTK_SPIN_BUTTON(minute_spinbutton);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			GTK_SIGNAL_FUNC (cb_inet_resolution), (gpointer) spin);
	gtk_box_pack_start(GTK_BOX(hbox), minute_spinbutton, FALSE, FALSE, 5);
	label = gtk_label_new("Hits/minute");
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);

	adj = (GtkAdjustment *) gtk_adjustment_new(2.0, 2.0, 100000.0,
					1.0 /*step*/, 1.0 /*page*/, 0.0);
	hour_spinbutton = gtk_spin_button_new (adj, 0.95, 0 /*digits*/);
	gtk_widget_set_usize(hour_spinbutton, 50, 0);
	spin = GTK_SPIN_BUTTON(hour_spinbutton);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			GTK_SIGNAL_FUNC (cb_inet_resolution), (gpointer) spin);
	gtk_box_pack_start(GTK_BOX(hbox), hour_spinbutton, FALSE, FALSE, 5);
	label = gtk_label_new("Hits/hour");
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);

	button = gtk_button_new_with_label("Enter");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_enter_inet, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

	button = gtk_button_new_with_label("Delete");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_delete_inet, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 2);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	inet_clist = gtk_clist_new_with_titles(10, inet_titles);
	gtk_clist_set_shadow_type (GTK_CLIST(inet_clist), GTK_SHADOW_OUT);
	gtk_clist_set_column_width (GTK_CLIST(inet_clist), 0, 32);
	gtk_signal_connect (GTK_OBJECT(inet_clist), "select_row",
			(GtkSignalFunc) cb_inet_clist_selected, NULL);
	gtk_signal_connect (GTK_OBJECT(inet_clist), "unselect_row",
			(GtkSignalFunc) cb_inet_clist_unselected, NULL);
	gtk_container_add (GTK_CONTAINER (scrolled), inet_clist);
	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		sprintf(p00, "%d", (int) in->port0_0);
		sprintf(p01, "%d", (int) in->port0_1);
		sprintf(p10, "%d", (int) in->port1_0);
		sprintf(p11, "%d", (int) in->port1_1);
		sprintf(resm, "%d", in->scale_min_minute);
		sprintf(resh, "%d", in->scale_min_hour);
		buf[0] = in->label0;
		buf[1] = p00;
		buf[2] = p01;
		buf[3] = "";
		buf[4] = in->label1;
		buf[5] = p10;
		buf[6] = p11;
		buf[7] = "";
		buf[8] = resm;
		buf[9] = resh;
		buf[10] = NULL;
		gtk_clist_append(GTK_CLIST(inet_clist), buf);
		}

/* --Info tab */
	vbox = create_tab(tabs, "Info");
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, inet_info_text, -1);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);

	reset_inet_entries();
	}

