/* 
   sitecopy, for managing remote web sites.
   Copyright (C) 1999, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* TODO:
 * - Make the 'remote' and 'local' file state more general,
 *   allow for permissions being being different etc etc.
 *   Necessary for more accurate synchronization.
 * - The 'driver' and 'driver_options' thing is all a bit crap.
 *   I'm not sure how to do this better, ideas welcome.
 */

#ifndef SITES_H
#define SITES_H

#include "config.h"

#include <time.h>

#include "common.h"
#include "protocol.h"

extern const struct proto_driver ftp_driver;
#ifdef USE_DAV
extern const struct proto_driver dav_driver;
#endif /* USE_DAV */


#define MAXASCII 20

#define PERMSMODE(site)					\
(site->perms==sitep_ignore?"Ignored":			\
 (site->perms==sitep_exec?"Maintained for executables":	\
  "Always maintained")					\
    )

#define SYMLINKSMODE(site)				\
(site->symlinks==sitesym_ignore?"Ignored":		\
 (site->symlinks==sitesym_maintain?"Maintained":	\
  "Followed")						\
    )

/* Unimplemented function of protocol driver */
#define SITE_UNIMPL 931

/* Return codes for site_update */
/* updated okay */
#define SITE_OK 0 
/* could not resolve hostname */
#define SITE_LOOKUP 1 
/* could not connect to remote host */
#define SITE_CONNECT 2
/* there were some errors when updating */
#define SITE_ERRORS 3
/* Could not authenticate user on server */
#define SITE_AUTH 4
#define SITE_FAILED 5

struct site_file_t;
struct site_t;

enum file_diff {
    file_unchanged, /* Remote file is same as local file */
    file_changed, /* File has changed locally, and should be uploaded */
    file_new, /* File is new locally, and should be uploaded */
    file_deleted,  /* File deleted locally, and should be deleted remotely */
    file_moved /* File has been moved locally, should be moved remotely */
};

/* Valid file permissions mirroring values */
enum site_perm_modes {
    sitep_ignore, /* Ignore file permissions */
    sitep_exec, /* Maintain execute permissions */
    sitep_all /* Maintain all permissions */
};

/* Valid symlink handling modes */
enum site_symlink_modes {
    sitesym_ignore,
    sitesym_follow,
    sitesym_maintain
};

/* Protocol modes */

enum site_protocol_modes {
    siteproto_ftp,
    siteproto_http
};

struct exclude_t {
    char *pattern;
    bool haspath;
    struct exclude_t *prev;
    struct exclude_t *next;
};

/* A filename is made up of three parts:
   The site root path, the directory, and the name.
   eg, a remote file:

   /html/mainsite/subdir/another/file.html
   [--site root--][--directory--][-name--]

   site root = "html/mainsite/"
   directory = "subdir/another/"
   name = "file.html"

   A directory MUST NOT include a leading slash but MUST include a
   trailing slash, except in the special case of a file in the site
   root directory, in which case the directory part will be
   zero-length.

   Within a given site, the site roots are the same for all files locally
   and remotely. The site roots may be 0-length, eg. for FTP sites where
   the home (login) directory is the site root directory.

   Note that a directory is also classed as any other 'file'.
   
   For convenience, the site_file_t structure contains four other properties
   which are *directly* derived from the three properties, and the site root:
     - The 'full' local filename
     - The 'full' remote filename
     - The 'relative' local filename
     - The 'relative' remote filename
   
   The 'full' filenames are:
      site root + directory + file name
   The 'relative' filenames are:
      "/" + directory + filename
  
   The 'relative' filenames for displaying to users.
   The 'full' filenames are for communicating with the server and use
   with the local file system.

   The site_assignnames() function fills the full_* and rel_* members.

   Splitting the directory from the filename is a legacy of the
   'rename' misfeature, where remote filenames could be different
   from local filenames.

   *** Site Roots ***
   
   The root directories are stored as three members of site_t.
      ->foo_root, ->foo_root_user, ->foo_root_isrel.
   (where foo is remote or local)

   ->foo_root_user is what the user enters as the root in the
   rcfile. This may have a ~/ prefix to indicate the root is to
   be taken relative to the login directory. This is translated
   into a usable version, in ->foo_root. foo_roo_isrel is true
   if this is a relative directory (i.e., ->foo_root_user has
   a ~/ prefix).

   Example:
       ->local_root_user = "~/html/mysite/"
       ->local_root = "/home/ego/html/mysite/"
       ->local_root_isrel = true;

       ->remote_root_user = "/mydir/"
       ->remote_root = "/mydir/"
       ->remote_root_isrel = false;

*/

struct site_file_t {
    enum file_diff diff;
    bool dir; /* This is a directory */
    bool link; /* This is a link */
    bool isascii; /* This is an ASCII file rather than a binary one */

    char *directory; /* the directory, relative to site root */
    char *filename; /* the remote name */

    /* The following two are computed from the above three for
     * simplicity, and are the FULL remote and local file names */
    char *full_remote;
    char *full_local; 
    /* These are for convenience, and is directory cat'ed with filename */
    char *rel_local;
    char *rel_remote;

    struct site_file_t *old; /* the item before it was moved */

    time_t localtime; /* the local last mod time of the file */
    time_t remotetime; /* the remote last mod time of the file */
    off_t localsize; /* the local file size */
    off_t remotesize; /* the remote file size */
    mode_t mode; /* the permissions */
    bool updated; /* whether the remote file has been updated according
		   * to the diff setting */
    bool marked; /* whether the file is marked to be updated or not */

    char *remotelink; /* the remote target of the link */    
    char *locallink; /* the local target of the link */

    /* Linked list nodes */
    struct site_file_t *next;
    struct site_file_t *prev;

};

/* This represents a site */
struct site_t {

    char *name; /* symbolic name for site */
    char *url; /* URL for site - used by flatlist mode */

    char *server; /* server name */
    int port; /* server port */
    char *username; /* username to log in to server */
    char *password; /* password for ditto */

    enum site_protocol_modes protocol;
    const struct proto_driver *driver; /* the protocol driver routines */

    /* ugh - protocol specific options here */
    bool ftp_pasv_mode;
    
/*  UNUSED currently:
    bool http_expect_continue;
    bool http_persistent; */

    char *remote_root; /* root directory of site on server */
    char *remote_root_user; /* what the user gave/sees as the remote root */
    bool remote_isrel; /* is the remote root dir relative to login dir? (~/) */
    char *local_root; /* root directory of site locally */
    char *local_root_user; /* what the user gave/sees as the remote root */
    bool local_isrel; /* is the local root directory relative to home dir */

    char *infofile;  /* local storage file in ~/.sitecopy/  */

    /* Options for the site */
    enum site_perm_modes perms; /* permissions maintenance mode */
    enum site_symlink_modes symlinks; /* symlink handline mode */
    bool nodelete;
    bool checkmoved;
    bool nooverwrite;

    /* Linked list of excludes */
    struct exclude_t *excludes;

    char *ascii[MAXASCII];
    int numascii;

    /* array of renames */
    struct name_mapping_t *renames;

    struct site_file_t *files; /* list of files */
    struct site_file_t *files_tail; /* end of the list */

    /* Some useful counts for the files */
    int numnew; /* number of new files */
    int numchanged; /* number of changed files */
    int numdeleted; /* number of deleted files */
    int nummoved; /* number of moved files */
    int numunchanged; /* number of unchanged files */
    
    size_t totalnew; /* total file size of new files */
    size_t totalchanged; /* total file size of changed files */

    bool is_different; /* whether the remote site is different from the
			* local site */

    bool use_this; /* whether the site is being operated on - handy
		    * for the console FE */

    struct site_t *next;
    struct site_t *prev;

};

/* The list of all sites as read from the rcfile */
extern struct site_t *all_sites;

/* This updates the files information for the given site - both the
 * local and remote ones. Returns 0 on success, or non-zero on failure */
int site_readfiles( struct site_t * );

/* This marks remote files as updated locally */
void site_catchup( struct site_t * );

/* This writes the remote files list back to the file.
 * Returns 0 on success or -1 on failure. */
int site_writefiles( struct site_t * );

/* Initialize the site - pretend there are NO files held remotely */
void site_initialize( struct site_t * );

/* Update the remote site.
 * Only files with marked==true will be updated if onlymarked == true.
 * WARNING: DO NOT USE onlymarked==true LIGHTLY. You will mess the
 * Returns one of:
 *  SITE_OK	  If the site was updated successfully
 *  SITE_AUTH     If the user could not be authenticated on the server
 *  SITE_LOOKUP   If the hostname could not be resolved
 *  SITE_CONNECT  Could not connect to remote server
 *  SITE_ERRORS   Were errors while updating remote site
 */
int site_update( struct site_t *site, bool onlymarked );

/* site_keepgoing:
 *  If true, then site_update will not give up if remote
 *  directories cannot be created etc.
 */
extern bool site_keepgoing;

/* Finds a site with the given name, and returns a pointer to it.
 * If no site of given name is found, returns NULL
 */
struct site_t *site_find( const char *sitename );

/* Syncronizes the local site with the remote copy.
 * Returns:
 *  SITE_OK       if the synch was completed
 *  SITE_AUTH     If the user could not be authenticated on the server
 *  SITE_LOOKUP   If the hostname could not be resolved
 *  SITE_CONNECT  Could not connect to remote server
 *  SITE_ERRORS   if there were errors while completing the synch
 * fe_synch_* will be called during the synchronize.
 */
int site_synch( struct site_t *the_site );

/* Updates the files listing from the remote site.
 * Returns:
 *  SITE_OK       if fetch went okay
 *  SITE_UNIMPL   if fetch mode is not implemented for the driver
 *  SITE_AUTH     If the user could not be authenticated on the server
 *  SITE_LOOKUP   If the hostname could not be resolved
 *  SITE_CONNECT  Could not connect to remote server
 *  SITE_FAILED   if it failed.
 * fe_fetch_found() will be called for each file that is found
 * in the fetch
 */
int site_fetch( struct site_t *the_site );

/* Destroys all the files... use before doing a second
 * site_readfiles on a site. */
void site_destroy( struct site_t *the_site );

/* Outputs the flat listing style output for the given site
 * to the given stream
 */
void site_flatlist( FILE *f, struct site_t *the_site );

/* Returns a pseudo-URL for the given site, in a statically allocated
 * memory location which will be overwritten by subsequent calls to
 * the function. (-> NOT thread-safe) */
const char *site_pseudourl( struct site_t *the_site );

#endif /* SITES_H */
