/* 
   sitecopy, manage remote web sites.
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>.
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* You MUST implement all of the below functions in the front end.
 * After a site_update call, for each file which is being updated
 * the following calls are made:
 *   Optionally: fe_can_update( file )
 *   fe_updating( file );
 *   fe_updated( file, true, NULL ) if the update was successful, or
 *   fe_updated( file, false, "error message" ) otherwise.
 * Similarly for site_synch calls.
 * If fe_can_updated( file ) returns false, the update of that file
 * will not continue.
 *
 * During a file transfer, whenever a block is written to the remote
 * site, fe_transfer_progress( bytes_transferred, bytes_total ) is 
 * called. In other words, for a 10k file, the calls might be:
 *  fe_t_p( 1024, 10240 ), fe_t_p( 2048, 10240 ),
 *  fe_t_p( 3072, 10240 ) ... fe_t_p( 10240, 10240 )
 * File transfers occur in site_update and site_synch, when uploading
 * and downloading files, respectively.
 */

#ifndef FRONTEND_H
#define FRONTEND_H

#include "common.h"
#include "sites.h"

/* Connection Status API.
 *
 * fe_connection() is called to indicate what state the connection is
 * in. Note, the status may bounce between fe_connected and
 * fe_connecting many times during an operation - especially if
 * connected to a WebDAV server which doesn't implement HTTP/1.1
 * persistent connections.
 */

typedef enum {
    fe_namelookup, /* Looking up hostname */
    fe_connecting, /* Connecting to server */
    fe_connected /* Connection established */
} fe_conn_status;

extern void fe_connection( fe_conn_status status );

extern bool fe_can_update( const struct site_file_t *file );
extern void fe_updating( const struct site_file_t *file );
extern void fe_updated( const struct site_file_t *file, const bool success,
			const char *error );

/* Set to true for fe_can_update() calls before each update 
 * in site_update */
extern bool fe_prompting;

/* For synch mode */
extern void fe_synching( const struct site_file_t *file );
extern void fe_synched( const struct site_file_t *file, const bool success,
			const char *error );
/* For synch and update modes */
extern void fe_transfer_progress( size_t progress, size_t total );

/* For fetch mode - called for each file found remotely */
extern void fe_fetch_found( const struct site_file_t *file );

#endif /* FRONTEND_H */
