/*  Screem:  externBrowsers.c,
 *
 *  External Browser plugin:
 *  Set up to 3 external browsers that can be used to preview the page
 *  and set the icon that they use on the toolbar
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gmodule.h>
#include <gnome.h>

#include <glade/glade.h>

#include "site.h"
#include "page.h"

#include "browser.xpm"

typedef struct _Browser {
	gchar     *path;     /* the path of the browser to execute */
	GtkWidget *widget;   /* the widget in the toolbar */
	gchar     *iconFile; /* the icon file name */
	gboolean  use_url;  /* should we view using the sites http url */
	GtkWidget *dialog;   /* the settings dialog for this browser */
} Browser;

static Browser b1;
static Browser b2;
static Browser b3;

extern GtkWidget *app;
extern Site *current_site;
extern Page *current_page;

extern GnomeClient *client;

GladeXML *xml;

static GtkWidget* createToolbar();
static void clicked( GtkWidget *widget, GdkEventButton *event, Browser *b );
static void launch_url( gchar *path );
static void launch( gchar *path );
static void settings( Browser *b );

void extern_browsers_clicked( GtkWidget *widget, gint button, Browser *b );
void extern_browsers_deleted( GtkWidget *widget, GdkEvent *event, Browser *b );
/*********************************************************************/
G_MODULE_EXPORT const gchar*
g_module_check_init( GModule *module )
{
        g_print("externBrowsers: check-init\n");
        return NULL;
}
/*********************************************************************/
G_MODULE_EXPORT void
g_module_unload( GModule *module )
{
        g_print( "externBrowsers: unloaded\n" );
}
/*********************************************************************/
G_MODULE_EXPORT void 
init() 
{
	GtkWidget *toolbar;
	gchar *icon;
	
	/* load the browser settings */
	gnome_config_push_prefix(gnome_client_get_config_prefix(client));
	b1.path = gnome_config_get_string( "External Browsers/path1=not set" );
	b1.use_url = gnome_config_get_int( "External Browsers/url1=0" );
	b2.path = gnome_config_get_string( "External Browsers/path2=not set" );
	b2.use_url = gnome_config_get_int( "External Browsers/url2=0" );
	b3.path = gnome_config_get_string( "External Browsers/path3=not set" );
	b3.use_url = gnome_config_get_int( "External Browsers/url3=0" );

	if( ! strcmp( b1.path, "not set" ) )
		b1.path = NULL;
	if( ! strcmp( b2.path, "not set" ) )
		b2.path = NULL;
	if( ! strcmp( b3.path, "not set" ) )
		b3.path = NULL;

	icon = gnome_config_get_string( "External Browsers/icon1=no icon" );
	if( ! strcmp( "no icon", icon ) )
		icon = NULL;
	b1.iconFile = icon;
  
	icon = gnome_config_get_string( "External Browsers/icon2=no icon" );
	if( ! strcmp( "no icon", icon ) )
		icon = NULL;
	b2.iconFile = icon;

	icon = gnome_config_get_string( "External Browsers/icon3=no icon" );
	if( ! strcmp( "no icon", icon ) )
		icon = NULL;
	b3.iconFile = icon;
	gnome_config_pop_prefix();


	b1.dialog = NULL;
	b2.dialog = NULL;
	b3.dialog = NULL;

	toolbar = createToolbar();

	gnome_app_add_toolbar( GNOME_APP( app ),
			       GTK_TOOLBAR( toolbar ),
			       "external browsers",
			       GNOME_DOCK_ITEM_BEH_NORMAL,
			       GNOME_DOCK_TOP,
			       2, 4, 2 );

	g_print( "externBrowsers: initialised\n" );
}
/*********************************************************************/
static GtkWidget* createToolbar()
{
	GtkWidget *toolbar;
	GtkWidget *widget;
	gchar *tip;

	toolbar = gtk_toolbar_new( GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_ICONS);

	/* create 3 buttons for external browsers */
	if( ! b1.iconFile )
		b1.widget = gnome_pixmap_new_from_xpm_d( browser_xpm );
	else
		b1.widget = gnome_pixmap_new_from_file_at_size( b1.iconFile, 
								24, 24 );

	if( ! b2.iconFile )
		b2.widget =  gnome_pixmap_new_from_xpm_d( browser_xpm );
	else
		b2.widget = gnome_pixmap_new_from_file_at_size( b2.iconFile,
								24, 24 );

	if( ! b3.iconFile )
		b3.widget =  gnome_pixmap_new_from_xpm_d( browser_xpm );
	else
		b3.widget = gnome_pixmap_new_from_file_at_size( b3.iconFile,
								24, 24 );

	tip = b1.path ? b1.path : _( "Not set" );
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "", tip, "",
				 b1.widget, 0, 0 );

	tip = b2.path ? b2.path : _( "Not set" );
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "", tip, "", 
				 b2.widget, 0, 0 );

;	tip = b3.path ? b3.path : _( "Not set" );
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "", tip, "",
				 b3.widget, 0, 0 );

	widget = b1.widget->parent->parent;
	gtk_signal_connect_after( GTK_OBJECT( widget ),"button_press_event",
				  clicked, &b1 );
	widget = b2.widget->parent->parent;
	gtk_signal_connect_after( GTK_OBJECT( widget ),"button_press_event",
				  clicked, &b2 );
	widget = b3.widget->parent->parent;
	gtk_signal_connect_after( GTK_OBJECT( widget ),"button_press_event",
				  clicked, &b3 );

	return toolbar;
}
/*********************************************************************/
static void clicked( GtkWidget *widget, GdkEventButton *event, Browser *b )
{
	if( event->button == 1 ) {
		/* launch the browser */
		if( b->path && ! b->use_url ) 
			launch( b->path );
		else if( b->path )
			launch_url( b->path );
		else
			/* if the browser isn't configured then lets open
			   the dialog to configure it */
			event->button = 3;
	}
	
	if( event->button == 3 ) {
		/* configure the button */
		settings( b );
	}
}
/*********************************************************************/
static void launch_url( gchar *path )
{
	Site *site;
	Page *page;
	const gchar *site_path;
	const gchar *http_url;
	const gchar *page_path;

	gchar *url;

	site = current_site;

	g_return_if_fail( site != NULL );

	page = screem_site_get_current_page( site );

	g_return_if_fail( page != NULL );

	site_path = screem_site_get_pathname( site );
	http_url = screem_site_get_http_url( site );
	page_path = screem_page_get_pathname( page );

	url = g_strconcat( path, " http://", http_url, 
			   page_path + strlen( site_path ), NULL );

	gnome_execute_shell( NULL, url );

	g_free( url );
}

static void launch( gchar *path )
{
	Page *page;
	gchar *file;
	gchar *rest;
	gint pos;

	const gchar *pathname;

	if( current_site )
     		page = screem_site_get_current_page( current_site );
	else
                page = current_page;

        g_return_if_fail( page != NULL );

	pathname = screem_page_get_pathname( page );

	file = strstr( path, "%f" );

	if( file ) {
		pos = file - path;
		rest = file + 2;
		if( rest > ( path + strlen( path ) ) )
			rest = "";
		file = g_strndup( path, pos );
	} else {
		/* if there was no %f then we will assume that we should just
		   pass the path on the end */
		file = g_strdup( path );
		rest = "";
	}

	path = g_strdup_printf( "%s %s %s", file, pathname, rest );
	g_free( file );

	/* path is now the program + args we should execute */
	gnome_execute_shell( NULL, path );

	g_free( path );
}
/*********************************************************************/
static void settings( Browser *b )
{
	GtkWidget *dialog;
	GtkWidget *path;
	GtkWidget *icon;
	GtkWidget *check;

	if( ! b )
		return;

	if( b->dialog ) {
		gdk_window_raise( b->dialog->window );
                gdk_window_show( b->dialog->window );
                return;
	}

	xml = glade_xml_new( GLADE_PATH"/extern_browsers.glade",
			     "extern_browsers_dialog" );

	dialog = glade_xml_get_widget( xml, "extern_browsers_dialog" );
	path = glade_xml_get_widget( xml, "browser_entry" );
	icon = glade_xml_get_widget( xml, "icon" );
	check = glade_xml_get_widget( xml, "url_check" );

	/* set the path/icon */
	if( b->path )
		gtk_entry_set_text( GTK_ENTRY( path ), b->path );

	if( b->iconFile )
		gnome_icon_entry_set_icon(GNOME_ICON_ENTRY(icon), b->iconFile);

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( check ), b->use_url );

	g_print( "b->iconFile: %s\n", b->iconFile );

	gtk_widget_show_all( icon );

	gtk_object_set_data( GTK_OBJECT( dialog ), "path", path );
	gtk_object_set_data( GTK_OBJECT( dialog ), "icon", icon );
	gtk_object_set_data( GTK_OBJECT( dialog ), "check", check );

	/* 
	 * have to do the signals here as glade can't handle the passing of
	 * a data pointer 
	 */
	gtk_signal_connect( GTK_OBJECT( dialog ), "clicked",
			    GTK_SIGNAL_FUNC( extern_browsers_clicked ), b );
	gtk_signal_connect( GTK_OBJECT( dialog ), "delete_event",
			    GTK_SIGNAL_FUNC( extern_browsers_deleted ), b );

	b->dialog = dialog;
}
/*********************************************************************/
void extern_browsers_deleted( GtkWidget *widget, GdkEvent *event, Browser *b )
{

}
/*********************************************************************/
void extern_browsers_clicked( GtkWidget *widget, gint button, Browser *b )
{
	GtkWidget *path;
	GtkWidget *icon;
	GdkImlibImage *image;
	GtkWidget *pixmap;
	GtkTooltipsData *ttd;
	GtkWidget *check;

	gchar *bpath;
	gchar *bicon;

	if( button < 2 ) {
		path = gtk_object_get_data( GTK_OBJECT( widget ), "path" );
		icon = gtk_object_get_data( GTK_OBJECT( widget ), "icon" );
		check = gtk_object_get_data( GTK_OBJECT( widget ), "check" );

		bpath = gtk_entry_get_text( GTK_ENTRY( path ) );
		
		if( b->path )
			g_free( b->path );
		
		b->path = g_strdup( bpath );
		
		bicon = gtk_entry_get_text( GTK_ENTRY( gnome_icon_entry_gtk_entry( GNOME_ICON_ENTRY( icon ) ) ) );

		if( b->iconFile )
			g_free( b->iconFile );

		b->iconFile = g_strdup( bicon );

		b->use_url = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( check ) );

		gnome_pixmap_load_file_at_size( GNOME_PIXMAP( b->widget ), 
						bicon, 24, 24 );

		/* set the tooltip to the programs path */
		ttd = gtk_tooltips_data_get( b->widget->parent->parent );
		gtk_tooltips_set_tip( ttd->tooltips, ttd->widget, b->path, "");

		/* save the settings */
		gnome_config_push_prefix(gnome_client_get_config_prefix(client));
		if( b == &b1 ) {
			gnome_config_set_string( "External Browsers/path1",
						 b->path );
			gnome_config_set_string( "External Browsers/icon1",
						 b->iconFile);
			gnome_config_set_int( "External Browsers/url1",
					      b->use_url );
		} else if( b == &b2 ) {
			gnome_config_set_string( "External Browsers/path2",
						 b->path );
			gnome_config_set_string( "External Browsers/icon2",
						 b->iconFile);
			gnome_config_set_int( "External Browsers/url2",
					      b->use_url );
		} else {
			gnome_config_set_string( "External Browsers/path3",
						 b->path );
			gnome_config_set_string( "External Browsers/icon3",
						 b->iconFile);
			gnome_config_set_int( "External Browsers/url3",
					      b->use_url );
		}
		gnome_config_pop_prefix();
	}

	if( button != 1 ) {
		gtk_widget_destroy( widget );
		b->dialog = NULL;
	}
}
