#include <assert.h>
#include <stdio.h>
#include <string.h>

#define HASH_SIZE 32767
typedef unsigned char uchar_t;

typedef struct yomi {
  uchar_t yomi[24];
  uchar_t kanjis[60];
  int kanjis_idx;
  uchar_t *kanjis_rest;
  int kanjis_rest_max;
  int kanjis_rest_idx;
  struct yomi *link; /* chain hash link. */
} yomi;

int reverse_p;
yomi *yomi_hash[HASH_SIZE];
int n_yomis;
yomi **yomi_sorted;

char *
xmalloc (n) 
    unsigned n;
{
  char *m = (char*)malloc(n);
  if (!m) {
    fprintf(stderr, "malloc : memory full\n");
    exit(1);
  }
  return m;
}

char *
xrealloc (old, n) 
	char *old;
	unsigned n;
{
  char *m = (char*)realloc(old, n);
  if (!m) {
    fprintf(stderr, "realloc : memory full\n");
    exit(1);
  }
  return m;
}

void 
append_char (y, ch) 
     yomi *y;
     uchar_t ch;
{
  if (y->kanjis_idx < 60) {
    y->kanjis[y->kanjis_idx++] = ch;
  } else if (!y->kanjis_rest) {
    assert(y->kanjis_idx == 60);
    y->kanjis_rest = (uchar_t*)xmalloc(y->kanjis_rest_max = 300);
    y->kanjis_rest[0] = ch;
    y->kanjis_rest_idx = 1;
  } else {
    if (y->kanjis_rest_idx >= y->kanjis_rest_max) {
      y->kanjis_rest = (uchar_t*)xrealloc(y->kanjis_rest, 
					  (y->kanjis_rest_max *= 2));
    } 
    y->kanjis_rest[y->kanjis_rest_idx++] = ch;
  } 
}

void 
append_kanji (y, kanji) 
     yomi *y;
     uchar_t *kanji;
{
  append_char(y, '/');
  while (*kanji) {
    append_char(y, *kanji);
    kanji++;
  }
}

yomi *
intern_yomi (s)
     uchar_t *s;
{
  yomi *y;
  unsigned hash = 0;
  uchar_t *sp = s;
  while (*sp) {
    hash = (hash<<1) + *sp++;
  }
  hash %= HASH_SIZE;

  y = yomi_hash[hash];
  while (y) {
    if (!strcmp(y->yomi, s)) return y;
    y = y->link;
  }
  y = (yomi*)xmalloc(sizeof *y);
  strcpy(y->yomi, s);
  y->kanjis_idx = 0;
  y->kanjis_rest = 0;
  y->kanjis_rest_max = 0;
  y->link = yomi_hash[hash];
  yomi_hash[hash] = y;
  n_yomis++;
  return y;
}

void 
read_file (path)
     uchar_t *path;
{
  int linenum = 0;
  char buf[BUFSIZ];
  FILE *f = fopen(path, "r");
  if (!f) {
    perror(path);
    return;
  }
  while (fgets(buf, BUFSIZ, f)) {
    char reverse[30];
    uchar_t *s = strtok(buf, " \t\n");
    uchar_t *kanji = strtok(NULL, " \t\n");
    yomi *y;
    linenum++;
    if (!s || !kanji || s[0] < 127) {
      fprintf(stderr, "%s:%d:ignoring %s", path, linenum,buf);
      continue;
    }
    if (reverse_p) {
      int len = strlen(s);
      int i;
      assert(len % 2 == 0);
      reverse[len] = 0;
      for (i = 0; i < len; i+=2) {
	reverse[len-i-2] = s[i];
	reverse[len-i-1] = s[i+1];
      }
      s = reverse;
    }
    y = intern_yomi(s);
    append_kanji(y, kanji);
  }
  fclose(f);
}

int 
comp (a, b) 
    yomi **a, **b;
{
  return strcmp((*a)->yomi, (*b)->yomi);
}

void
sort_yomi () 
{
  int h, i = 0;
  /* Copy all the pointers stored in YOMI_HASH into
     newly created YOMI_SORTED array. */
  fprintf(stderr, "total %d yomis.\n", n_yomis);
  yomi_sorted = (yomi**)xmalloc(sizeof(yomi*) * n_yomis);
  for (h = 0; h < HASH_SIZE; h++) {
    yomi *y = yomi_hash[h];
    while (y) {
      yomi_sorted[i++] = y;
      assert(i <= n_yomis);
      y = y->link;
    }
  }

  /* QSORT YOMI_SORTED */
  fprintf(stderr, "sorting.\n");
  qsort(yomi_sorted, n_yomis, sizeof(yomi*), comp);
}

void 
output ()
{
  int i;
  fprintf(stderr, "writing.\n");
  for (i = 0; i < n_yomis; i++) {
    int j;
    yomi *y = yomi_sorted[i];
    printf("%s ", y->yomi);
    for (j = 0; j < y->kanjis_idx; j++) putchar(y->kanjis[j]);
    if (y->kanjis_rest) {
      for (j = 0; j < y->kanjis_rest_idx; j++) putchar(y->kanjis_rest[j]);
    }
    fputs("/\n", stdout);
  }
}

int
main (argc, argv)
     int argc;
     char **argv;
{
  int i;
  int c;
  extern int optind;
  while ((c = getopt(argc, argv, "r")) != EOF) {
    switch (c) {
    case 'r' :
      reverse_p = 1;
      break;
    default:
      fprintf(stderr,"%s [-r] files...\n", argv[0]);
      fprintf(stderr,"-r : make mazegaki.rev format file.\n"); 
      exit(1);
    }
  }
  for (i = optind; i < argc; i++) {
    read_file(argv[i]);
  }
  sort_yomi();
  output();
  return 0;
}
