/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 * $RCSfile: ftfmt-tapelib.c,v $
 * $Revision: 1.10 $
 * $Date: 1999/03/17 11:25:42 $
 *
 *      This program contains the tape manipulation functions for the
 *      user level floppy tape formatting stuff for the
 *      QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#ifndef MAP_FAILED
#define MAP_FAILED (void *)(-1)
#endif
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <assert.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <linux/ftape.h>
#include <linux/ftape-vendors.h>
#include <linux/ftape-header-segment.h>

#include "ftformat.h"
#include "ftfmt-options.h"
#include "ftfmt-bsm.h"
#include "ftfmt-tapelib.h"

const ftape_error                 qic117_error[] = QIC117_ERRORS;
const struct qic117_command_table qic117_cmd[]   = QIC117_COMMANDS;
static int api_version = -1;

static int tune_bsm_before(void *hseg, const ftfmt_tpparms_t *tpparms);
static SectorMap tune_bsm_entry(SectorMap map);
static int write_segment(const int tape_fd,
			 const int seg_id, int mode, const void *data);

void *tape_mmap(int tape_fd, size_t dma_size)
{
    void *dma_buffer;
    struct mtop  mtop;

    if (api_version != 0) {
	return (void *)~0; /* fake success */
    }
    /*
     *  First try to set the number of buffers to the desired size:
     */
    mtop.mt_op    = MTSETDRVBUFFER;
    mtop.mt_count = dma_size / FT_BUFF_SIZE;
    if (ioctl(tape_fd, MTIOCTOP, &mtop) == -1) {
	perror(_("Error setting tape drive buffering"));
	return NULL;
    }
    /*
     *  Then try to mmap it
     */
    dma_buffer = mmap(0, dma_size, PROT_READ|PROT_WRITE, MAP_SHARED,tape_fd,0);
    if (dma_buffer == MAP_FAILED) {
	perror(_("Error mapping tape dma buffers"));
	(void)close(tape_fd);
	return NULL;
    }
#if 0
    printf(_("Got mmaped tape buffer @ %p/%d\n"), dma_buffer, dma_size);
#endif
    return dma_buffer;
}

/*
 *  open the tape device, try to determine whether it is a floppy tape,
 *  and store the hardware status in drive_config, resp. tape_config.
 *  mmap() the dma buffers if op_mode != PROBING
 *  
 *  Returns the tape fd on success, -1 otherwise
 *
 *  Note: we first open the tape device read-only, as the tape type
 *  autodetection of some drives doesn't work with damaged cartridges.
 *  Later, after we have told the tape drive which format mode we
 *  need, we re-open it read-write
 *
 */
int tape_reopen_rw(const char *name, int tape_fd)
{
    if (tape_fd == -1) {
	fprintf(stderr, _("%s hasn't been opened yet!\n"), name);
	return -1;
    }
    if (close(tape_fd) == -1) {
	perror(_("Error closing tape dev"));
	return -1; /* is it still open, or not ? */
    }
    if ((tape_fd = open(name, O_RDWR)) == -1) {
	perror(_("Error re-opening tape dev read/write"));
	return -1;
    }
    return tape_fd;
}

int tape_open(const char *name,
	      const struct opt_parms *opts,
	      ftfmt_tpparms_t *tpparms)
{
    int tape_fd;
    struct mtget mtget;
    struct mtop  mtop = { MTNOP, 1 };
    ft_drive_status hw_status;
    vendor_struct *vendor;
    qic117_make_code *make;
    format_parms *fmt_opts = &tpparms->fmt_parms;
    static qic117_make_code make_codes[] = QIC117_MAKE_CODES;
    static vendor_struct vendors[] = QIC117_VENDORS;

    /* open the tape device
     */
    if ((tape_fd = open(name, O_RDONLY)) == -1) {
	perror(_("Error opening tape device read-only"));
	return -1;
    }
    if (opts->read_hseg) {
	/* set its status, might result in reading the header segments.
	 */
	if (ioctl(tape_fd, MTIOCTOP, &mtop) == -1) {
	    perror(_("Error setting tape drive status"));
	    (void)close(tape_fd);
	    return -1;
	}
    }
    /* get its status
     */
    if (ioctl(tape_fd, MTIOCGET, &mtget) == -1) {
	perror(_("Error getting tape drive status"));
	(void)close(tape_fd);
	return -1;
    }
    if (GMT_DR_OPEN(mtget.mt_gstat)) {
	fprintf(stderr, _("Error: No tape cartridge present!\n"));
	(void)close(tape_fd);
	return -1;
    }
    if (GMT_WR_PROT(mtget.mt_gstat)) {
	fprintf(stderr, _("Warning: Write protected cartridge!\n"));
#if 0
	(void)close(tape_fd);
	return -1;
#endif
    }
    if (!GMT_ONLINE(mtget.mt_gstat)) {
	fprintf(stderr, _("Error: Tape drive is offline!\n"));
	(void)close(tape_fd);
	return -1;
    }
    if ((mtget.mt_type & MT_ISFTAPE_FLAG) == 0) {
	fprintf(stderr, _("Error: This is not a floppy tape drive!\n"));
	(void)close(tape_fd);
	return -1;
    }

    mtget.mt_type &= ~MT_ISFTAPE_FLAG; /* clear the flag bit */
    vendor = &vendors[0];
    while (vendor->vendor_id != UNKNOWN_VENDOR &&
	   vendor->vendor_id != mtget.mt_type) {
	vendor++;
    }
    if (vendor->vendor_id == UNKNOWN_VENDOR) {
	fprintf(stderr, _("Unknown floppy tape drive. Bailing out. Contact "
			  THE_FTAPE_MAINTAINER"\n"));
	(void)close(tape_fd);
	return -1;
    }
    if (vendor->flags & FT_CANT_FORMAT) {
	/*  some drives do not support formatting. We support
	 * --verify-only nevertherless, but special care has to be
	 * taken:
	 */
	if (opts->do_format) {		
	    fprintf(stderr,
		    _("Your tape drive (%s -- 0x%04lx) isn't able to format "
		      "tapes.\n"
		      "Read the Ftape-HOWTO and contact the linux-tape mailing"
		      "list if unsure\n"),
		    vendor->name, vendor->vendor_id);
	    (void)close(tape_fd);
	    return -1;
	} else if (!opts->read_hseg) {
	    fprintf(stderr,
		    _("Your tape drive (%s -- 0x%04lx) isn't able to format "
		      "tapes.\n"
		      "Read the Ftape-HOWTO and contact the linux-tape mailing"
		      "list if unsure\n"),
		    vendor->name, vendor->vendor_id);
	    (void)close(tape_fd);
	    return -1;
	}
    }
    make = &make_codes[0];
    while ((unsigned int)make->make != QIC117_UNKNOWN_MAKE &&
	   (unsigned int)make->make != QIC117_MAKE_CODE(vendor->vendor_id)) {
	make ++;
    }
    if ((unsigned int)make->make == QIC117_UNKNOWN_MAKE) {
	fprintf(stderr, _("Warning: Unknown floppy tape drive make code. "
			  "Continuing, but please contact "
			  THE_FTAPE_MAINTAINER"\n"));
    }
    if (opts->verbose > 0) {
	printf(_("Tape drive: %s.\n"
		 "Make code : %s (%d).\n"
		 "Model     : %d.\n"
		 "Vendor id : 0x%04x.\n"),
	       vendor->name,
	       make->name,
	       make->make,
	       QIC117_MAKE_MODEL(vendor->vendor_id),
	       vendor->vendor_id);
    }
    if (UNSET(fmt_opts->ft_ffb)) {
	if (make->rffb != 0x00) { /* recommendation given */
	    fmt_opts->ft_ffb = make->rffb;
	} else {
	    fmt_opts->ft_ffb  = 0x6b; /* somewhat arbitrary */
	}
    } else if (make->rffb != 0x00 && opts->mode != FORCE) {
	/* recommendation given, but user tries to override */
	fprintf(stderr,
		_("Vendor suggest format-filler-byte=0x%02x, "
		  "use \"--mode=force\" to override.\n"),
		make->rffb);
	(void)close(tape_fd);
	return -1;
    }
    hw_status.space = mtget.mt_dsreg;
    tpparms->drive_status   = hw_status.status.drive_status;
    tpparms->drive_config   = hw_status.status.drive_config;
    tpparms->tape_status    = hw_status.status.tape_status;
    tpparms->vendor         = vendor;
    return tape_fd;
}

/*  Close the tape device. I wonder whether we should put it offline,
 *  or reset it straight forward. But why should we? At least this should
 *  ONLY be done after a successful format.
 */
void tape_close(const int tape_fd, void *dma_buffer, const size_t dma_size)
{
    if (tape_fd != -1) {
	/* just in case */
	(void)qic_simple_command(tape_fd,
				 0, QIC_ENTER_PRIMARY_MODE, -1, 0, NULL);
	if (dma_buffer != NULL && api_version == 0) {
	    if (munmap(dma_buffer, dma_size) == -1) {
		perror(_("Error unmapping dma buffer"));
	    }
	}
	if (close(tape_fd) == -1) {
	    perror(_("Error closing tape dev"));
	}
    }
}

int qic_simple_command(const int tape_fd,
		       const int ready_wait,
		       const qic117_cmd_t qic_cmd, const int parameter,
		       const int timeout, u_int8_t *status)
{
    struct mtftcmd cmd;

    memset(&cmd, 0, sizeof(cmd));
    cmd.ft_wait_before = ready_wait * 1000;
    cmd.ft_cmd         = qic_cmd;
    if (!UNSET(parameter)) {
	cmd.ft_parm_cnt    = 1;
	cmd.ft_parms[0]    = parameter;
    }
    cmd.ft_wait_after  = timeout * 1000;

    if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
	fprintf(stderr, _("Ioctl error sending %s command: %s\n"),
		qic117_cmd[qic_cmd].name, strerror(errno));
	return -1;
    }
    if (status) {
	*status = cmd.ft_status & 0xff;
    }
    if (cmd.ft_error) {
	fprintf(stderr, _("Tape drive error sending %s command: %d (%s)\n"),
		qic117_cmd[qic_cmd].name, cmd.ft_error,
		qic117_error[(cmd.ft_error < NR_ITEMS(qic117_error) ?
			      cmd.ft_error : 0)].message);
    }
    return cmd.ft_error ? -1 : 0;
}

int qic_calibrate_tape_length(const int tape_fd)
{	
    int segments;
    struct mtftcmd cmd;

    printf(_("Calibrating tape length ... "));
    memset(&cmd, 0, sizeof(cmd));
    cmd.ft_cmd         = QIC_CALIBRATE_TAPE_LENGTH;
    cmd.ft_wait_after  = 1300*1000;/*FIXME: calculate timeouts in user space */
    if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
	perror(_("Ioctl error sending QIC_CALIBRATE_TAPE_LENGTH"));
	return -1;
    }
    if (cmd.ft_error) {
	fprintf(stderr, _("Calibrate tape length not supported. "
			  "Surprise! Error: %d (%s)\n"),
		cmd.ft_error,
		qic117_error[(cmd.ft_error < NR_ITEMS(qic117_error) ?
			      cmd.ft_error : 0)].message);
	return -1;
    }
    segments = qic_report_command(tape_fd, QIC_REPORT_FORMAT_SEGMENTS, 16);
    printf(_("got %d segments\n"), segments);
    return segments;
}

int qic_report_command(const int tape_fd,
		       const qic117_cmd_t qic_cmd,
		       const int numbits)
{
    struct mtftcmd cmd;

    memset(&cmd, 0, sizeof(cmd));
    cmd.ft_cmd         = qic_cmd;
    cmd.ft_result_bits = numbits;
    if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
	fprintf(stderr, _("Ioctl error sending %s command: %s\n"),
		qic117_cmd[qic_cmd].name, strerror(errno));
	return -1;
    }
    return cmd.ft_result;
}


int write_reference_burst(const int tape_fd)
{
    u_int8_t status;

    printf(_("Writing reference bursts ... "));
    if (qic_simple_command(tape_fd, 0, QIC_ENTER_FORMAT_MODE, -1, 0, NULL)) {
	fprintf(stderr, "\n"__FUNCTION__ " %s", _("failed!\n"));
	return -1;
    }
    if (qic_simple_command(tape_fd, 0, QIC_WRITE_REFERENCE_BURST, -1, 940,
			   &status)) {
	return -1;
    }
    if (!(status & QIC_STATUS_REFERENCED)) {
	fprintf(stderr, "\n"__FUNCTION__ " %s", _("failed!\n"));
	return -1;
    }
    printf(_("done.\n"));
    return 0;
}

/* Erase the entire cartridge, using physical forward/revers in format mode.
 */
int erase_cartridge(const int tape_fd, const struct opt_parms *opt, int tpc)
{
    int i;
    u_int8_t status;

    /* the following could be replaced by a MTREW command */
    if (qic_simple_command(tape_fd,
			   0, QIC_PHYSICAL_REVERSE, -1, 650, &status)) {
	return -1;
    }
    if (!(status & QIC_STATUS_AT_BOT)) {
	fprintf(stderr, _("Unable to rewind the cartridge\n"));
	return -1;
    }
    if (qic_simple_command(tape_fd, 650, QIC_ENTER_FORMAT_MODE, -1, 0, NULL)) {
	return -1;
    }
    if (!opt->notty) {
	printf(_("Erasing track   0"));
    }
    for (i=0; i < tpc; i++) {
	if (opt->notty) {
	    printf("Erasing track %3d\n",i);
	} else {
	    printf("\b\b\b%3d", i);
	}
	if (qic_simple_command(tape_fd,
			       0, QIC_SEEK_HEAD_TO_TRACK, i, 15, &status)) {
	    if (!opt->notty) {
		printf("\n");
	    }
	    return -1;
	}
	if (i&1) {
	    if (!(status & QIC_STATUS_AT_EOT)) {
		fprintf(stderr, _("\ncartridge not a eot\n"));
		return -1;
	    }
	    if (qic_simple_command(tape_fd,
				   0, QIC_PHYSICAL_REVERSE, -1, 650,&status)) {
		printf("\n");
		return -1;
	    }
	} else {
	    if (!(status & QIC_STATUS_AT_BOT)) {
		fprintf(stderr, _("\ncartridge not a bot\n"));
		return -1;
	    }
	    if (qic_simple_command(tape_fd,
				   0, QIC_PHYSICAL_FORWARD, -1, 650,&status)) {
		if (!opt->notty) {
		    printf("\n");
		}
		if (i == 0) {
		    /* tape drive doesn't support this command in format mode
		     */
		    fprintf(stderr,
			    _("Please ignore the error message printed above!"));
		    return 0;
		} else {
		    return -1;
		}
	    }
	}
    }
    if (!opt->notty) {
	printf("\n");
    }
    return 0;
}

int read_header_segment(const int tape_fd,
			ftfmt_tpparms_t *tpparms, u_int8_t *hseg)
{
    struct mtftseg ft_seg;
    int i;
    const unsigned long hseg_magic = FT_HSEG_MAGIC;

    printf(_("Reading old header segment ... "));
    for (i = 0; i < 64; i++) {
	memset(&ft_seg, 0, sizeof(ft_seg));
	ft_seg.mt_data  = hseg;
	ft_seg.mt_segno = i;
	ft_seg.mt_mode  = MT_FT_RD_SINGLE;
	if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
	    fprintf(stderr, _("Ioctl error reading header segment: %s\n"),
		    strerror(errno));
	    return -1;
	}
	if (ft_seg.mt_result == FT_SEGMENT_SIZE) {
	    break;
	}
    }
    if (memcmp(&hseg_magic, hseg, 4)) {
	fprintf(stderr, _("Unable to read the header segment\n"));
	fprintf(stderr, _("Consider using \"--header-discard\"\n"));
	return -1;
    }
    printf(_("done.\n"));
    
    if (hseg[FT_FMT_CODE] == fmt_big) {
	tpparms->frst_header_segment = GET4(hseg, FT_6_HSEG_1);
	tpparms->scnd_header_segment = GET4(hseg, FT_6_HSEG_2);
	tpparms->frst_data_segment   = GET4(hseg, FT_6_FRST_SEG);
	tpparms->last_data_segment   = GET4(hseg, FT_6_LAST_SEG);
    } else {
	tpparms->frst_header_segment = GET2(hseg, FT_HSEG_1);
	tpparms->scnd_header_segment = GET2(hseg, FT_HSEG_2);
	tpparms->frst_data_segment   = GET2(hseg, FT_FRST_SEG);
	tpparms->last_data_segment   = GET2(hseg, FT_LAST_SEG);
    }

    return 0;
}

int check_api_version(const int tape_fd, struct opt_parms *opts)
{
    int result;
    struct mtftformat_virtual arg;
	
    api_version = -1;
    arg.fmt_magic = -1;
    /* first try the most recent API */
    arg.fmt_op    = FTFMT_API_VERSION;
    result = ioctl(tape_fd, MTIOCFTFORMAT, &arg);
    if (result == 0) {
	api_version = arg.fmt_magic;
    } else if (result == -1 && errno == EINVAL) {
	/* this can be either ftape-3.x or ftape-4.x previous format API */
	result = ioctl(tape_fd, MTIOCFTFORMAT_4, &arg);
	if (result == 0) {
	    api_version = arg.fmt_magic;
	} else if (result == -1 && errno == EINVAL) {
	    /* this should be ftape-3.x, or we are out of luck */
	    arg.fmt_op = FTFMT_GET_PARMS;
	    if (ioctl(tape_fd, MTIOCFTFORMAT_3, &arg.fmt_op) == 0) {
		api_version = 0;
	    }
	}
    }
    switch (api_version) {
    case 0:
	if (opts->verbose) {
	    printf(_("ftape-3.x format API\n"));
	}
	return 0;
    case FTFMT_MAGIC_4:
    case FTFMT_MAGIC:
	if (opts->verbose) {
	    printf(_("ftape-4.x format API: 0x%08x\n"), api_version);
	}
	return 0;
    default:
	fprintf(stderr, _("Unknown format API version. Giving up.\n"));
	return -1;
    }
}

int send_format_ioctl(const int tape_fd,
		      struct mtftformat_virtual *fmtc)
{
    int result;

    switch (api_version) {
    case 0:
	return ioctl(tape_fd, MTIOCFTFORMAT_3, &fmtc->fmt_op);
    case FTFMT_MAGIC_4:
	fmtc->fmt_magic = FTFMT_MAGIC_4;
	return ioctl(tape_fd, MTIOCFTFORMAT_4, fmtc);
    case FTFMT_MAGIC:
	fmtc->fmt_magic = FTFMT_MAGIC;
	return ioctl(tape_fd, MTIOCFTFORMAT, fmtc);
    default:
	fprintf(stderr,_("Unknown api version: 0x%08x\n"), api_version);
	return -1;
    }
}	

/* This function informs the tape drive and the kernel driver about
 * the desired format. It also computes segments_per_floppy_head which
 * is needed for fill_dma_buffer() to compute the data to fed the fdc
 * and tape drive with
 */
int set_format_parms(const int tape_fd, const ftfmt_tpparms_t *tpparms)
{
    const struct ftfmtparms_virtual *fmt_opts = &tpparms->fmt_parms;
    int fmt;
    struct mtftcmd cmd;
    struct mtftformat_virtual fmtc;

    if (tpparms->vendor->flags & FT_CANT_FORMAT) {
	return -1;
    }

    /* first the hardware stuff
     */
    fmt = (((fmt_opts->ft_qicstd & QIC_TAPE_STD_MASK) << 2) |
	   (fmt_opts->ft_qicstd & QIC_TAPE_WIDE ? 3 : 1));
    if (qic_simple_command(tape_fd, 0, QIC_SELECT_RATE, fmt, 1, NULL) != 0) {
	fprintf(stderr,
		_("Please ignore the error message printed above if in case\n"
		  "that your cartridge can be formatted successfully!\n"));
    } else {
	/* If the tape drive doesn't support QIC_SELECT_FORMAT
	 * (QIC_SELECT_RATE with param > 4) it doesn't support
	 * QIC_SET_FORMAT_SEGMENTS either.  Even worse:
	 * QIC_SET_FORMAT_SEGMENTS seems to irritate some drives so
	 * much that they aren't useable afterwards.  
	 */
	memset(&cmd, 0, sizeof(cmd));
	cmd.ft_wait_before = 0;
	cmd.ft_cmd         = QIC_SET_FORMAT_SEGMENTS;
	cmd.ft_parm_cnt    = 3;
	cmd.ft_parms[0]    = fmt_opts->ft_spt & 0xf;
	cmd.ft_parms[1]    = (fmt_opts->ft_spt>>4) & 0xf;
	cmd.ft_parms[2]    = (fmt_opts->ft_spt>>8) & 0xf;
	if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
	    fprintf(stderr, _("Ioctl error sending %s command: %s\n"),
		    qic117_cmd[QIC_SET_FORMAT_SEGMENTS].name, strerror(errno));
	    return -1;
	}
	if (cmd.ft_error) {
	    fprintf(stderr, _("Tape drive error sending %s command: %d (%s)\n"),
		    qic117_cmd[QIC_SET_FORMAT_SEGMENTS].name, cmd.ft_error,
		    qic117_error[(cmd.ft_error < NR_ITEMS(qic117_error) ?
				  cmd.ft_error : 0)].message);
	    fprintf(stderr,
		    _("Please ignore the error message printed above if it proves\n"
		      "that your cartridge can be formatted successfully.\n"));
	}
    }
    /* and now the software kernel driver stuff
     */
    memset(&fmtc, 0, sizeof(fmtc));
    fmtc.fmt_op = FTFMT_SET_PARMS;
    fmtc.fmt_arg.fmt_parms = *fmt_opts;
    if (send_format_ioctl(tape_fd, &fmtc) == -1) {
	fprintf(stderr, _("Ioctl error sending format parameters: %s\n"),
		strerror(errno));
	return -1;
    }
    return 0;
}

/* Query the kernel driver about the parms it uses. Needed for --verify-only
 */
int get_format_parms(const int tape_fd,
		     const u_int8_t *hseg,
		     ftfmt_tpparms_t *tpparms)
{
    struct mtftformat_virtual fmtc;
    format_parms *fmt_opts = &tpparms->fmt_parms;

    memset(&fmtc, 0, sizeof(fmtc));
    fmtc.fmt_op = FTFMT_GET_PARMS;
    if (send_format_ioctl(tape_fd, &fmtc) == -1) {
	fprintf(stderr, _("Ioctl error getting format parameters: %s\n"),
		strerror(errno));
	return -1;
    }
    *fmt_opts = fmtc.fmt_arg.fmt_parms;
    return 0;
}

/*  This should actually be even easier than in kernel space.
 *  The dma buffer is one large contiguous region.
 */

/*  *user_pos is always (#<last computed segment> + 1). kernel_pos gives
 *  the segment that is currently being formatted by the kernel driver.
 *  We should not touch that segment (i.e. the part of the dma buffer 
 *  that belongs that segment). At the very begin, kernel_pos is 0. We always 
 *  compute segs_in_buf (approx 256) segments in advance, which is plenty.
 *
 *  When starting a new track, we need to start at the beginning of the dma 
 *  buffer
 *
 *  When calling msync() (to support parallel port tape drives with
 *  their funky mmap() implementation) we need to make sure that the
 *  start-address is page-aligned, otherwise msync() will fail.
 *
 */
static void fill_dma_buffer(format_segment *buf, const size_t dma_size,
			    int *user_pos, const int kernel_pos,
			    const struct ftfmtparms_virtual *parms)
{
    int segs_in_buf         = dma_size / sizeof(format_segment);
    int seg_id              = *user_pos;
    int trk_seg             = seg_id % parms->ft_spt;
    format_segment *buf_end = &buf[segs_in_buf];
    format_segment *buf_pos = &buf[trk_seg % segs_in_buf];
    void *buf_start 	= (void*) buf_pos;
    int i;
    static size_t msync_offset = 0;
    size_t msync_size;

    if (buf_pos == buf) {
	msync_offset = 0; /* initialize to start of buffer, needed in case
			   * we retry the same segment
			   */
    }
    while (seg_id < (kernel_pos + segs_in_buf) && trk_seg < parms->ft_spt) {
	buf_pos->sectors[0].cyl =
	    (seg_id % SPFH(parms->ft_ftm)) / SEGMENTS_PER_FLOPPY_TRACK;
	buf_pos->sectors[0].head =
	    seg_id / SPFH(parms->ft_ftm);
	buf_pos->sectors[0].sect =
	    (seg_id % SEGMENTS_PER_FLOPPY_TRACK) * FT_SECTORS_PER_SEGMENT + 1;
	buf_pos->sectors[0].size = 3;
#if 0
	if (verbose >= 2) {
	    fprintf(stderr, "%d %d\n", seg_id, trk_seg);
	    fprintf(stderr, "0x%08x\n", *((int *)&buf_pos->sectors[0]));
	}
#endif
	for (i = 1; i < FT_SECTORS_PER_SEGMENT; i++) {
	    buf_pos->sectors[i] = buf_pos->sectors[0];
	    buf_pos->sectors[i].sect += i;
#if 0
	    if (verbose >= 2) {
		fprintf(stderr, "0x%08x\n", *((int *)&buf_pos->sectors[i]));
	    }
#endif
	}
	seg_id    ++;
	trk_seg   ++;
	buf_pos   ++; /* pointer arithmetic */

	/* We only sync after computing enough parameters for an entire page,
	 * or when we hit the end of a track.
	 *
	 * Note that end and start of the entire buffer are alway page
	 * aligned as long as PAGE_SIZE < 32k
	 *
	 * Of course, the computation of msync_size is highly inefficient.
	 * We could just treat it as a counter and increment it etc.
	 */
	msync_size = (void*)buf_pos - (void *)buf - msync_offset;
	if (msync_size && (msync_size % getpagesize()) == 0) {
	    if (msync((void *)buf + msync_offset, msync_size, MS_SYNC) == -1) {
		fprintf(stderr, _("msync(%p, %d, MS_SYNC) failed: %s.\n"),
			(void *)buf + msync_offset, msync_size,
			strerror(errno));
		fprintf(stderr, _("Dumping core\n"));
		abort();
	    }
	    msync_offset += msync_size;
	}
	if (buf_pos == buf_end) { /* cycle */
	    msync_offset = 0; /* reset to start of buffer */
	    buf_start = (void*) buf_pos = buf;
	}
    }
    /* We only sync after computing enough parameters for an entire page,
     * or when we hit the end of a track.
     *
     * Note that end and start of the entire buffer are alway page
     * aligned as long as PAGE_SIZE < 32k
     *
     * The following handles the end-of-tape-track condition:
     */
    if (trk_seg == parms->ft_spt) {
	msync_size = (void*)buf_pos - (void *)buf - msync_offset;
	if (msync_size) {
	    if (msync((void *)buf + msync_offset, msync_size, MS_SYNC) == -1) {
		fprintf(stderr, _("msync(%p, %d, MS_SYNC) failed: %s.\n"),
			(void *)buf + msync_offset, msync_size,
			strerror(errno));
		fprintf(stderr, _("Dumping core\n"));
		abort();
	    }
	}
	msync_offset = 0; /* reset to start of buffer */
    }
    *user_pos = seg_id;
}


/*  This does the main work. We don't need to issue raw QIC-117 commands here
 *  because the driver need some information about the track it is located on.
 *  Does it? Maybe change it. First: let's try to make it work again.
 */
int format_cartridge(const int tape_fd, void *dma_buf, const size_t dma_size,
		     const format_parms *fmt_opts,
		     const struct opt_parms *opt)
{
    int user_pos = 0, kernel_pos = 0;
    int trk;
    struct mtftformat_virtual fmtc;

    if (qic_simple_command(tape_fd, 0, QIC_ENTER_FORMAT_MODE, -1, 0, NULL)) {
	fprintf(stderr, __FUNCTION__ " %s", _("failed!\n"));		
	return -1;
    }
    trk = 0;
    if (!opt->notty) {
	printf("\r");
	printf(_("Formatting track %3d, "), trk);
    } else {
	printf("Formatting track %3d\n", trk);
    }
    while (trk < fmt_opts->ft_tpc) {
	user_pos = kernel_pos = trk * fmt_opts->ft_spt;
	if (api_version == 0) {
	    fill_dma_buffer(dma_buf, dma_size, &user_pos, kernel_pos, fmt_opts);
	}
	if (!opt->notty) {
	    printf(_("segment %8d"), kernel_pos);
	} else {
	    printf("segment %8d\n", kernel_pos);
	}
	memset(&fmtc, 0, sizeof(fmtc));
	fmtc.fmt_op = FTFMT_FORMAT_TRACK;
	fmtc.fmt_arg.fmt_track.ft_track = trk;
	fmtc.fmt_arg.fmt_track.ft_gap3  = fmt_opts->ft_gap3;
	if (send_format_ioctl(tape_fd, &fmtc) == -1) {
	    fprintf(stderr, _("\nIoctl error formatting track %d: %s\n"),
		    trk, strerror(errno));
	    return -1;
	}
	memset(&fmtc, 0, sizeof(fmtc));
	fmtc.fmt_arg.fmt_status.ft_segment = trk * fmt_opts->ft_spt;
	do {
	    kernel_pos = fmtc.fmt_arg.fmt_status.ft_segment;
	    if (api_version == 0 && user_pos % fmt_opts->ft_spt) {
		fill_dma_buffer(dma_buf, dma_size, &user_pos,kernel_pos,
				fmt_opts);
	    }
	    if (kernel_pos % fmt_opts->ft_spt) {
				/* don't print the first segment of a track 'cause already
				 * done
				 */
		if (!opt->notty) {
		    printf("\b\b\b\b\b\b\b\b%8d", kernel_pos);
		} else {
		    printf("segment %8d\n", kernel_pos);
		}
	    }
	    fmtc.fmt_op = FTFMT_STATUS;
	    if (send_format_ioctl(tape_fd, &fmtc) == -1) {
		fprintf(stderr, _("\nError getting format status (%d/%d): %s\n"),
			trk, kernel_pos, strerror(errno));
		return -1;
	    }
	} while(fmtc.fmt_arg.fmt_status.ft_segment > kernel_pos);
	if (fmtc.fmt_arg.fmt_status.ft_segment - fmt_opts->ft_spt * (trk + 1)) {
	    if (!opt->notty) {
		printf("\r");
		printf(_("Retrying   track %3d, "), trk);
	    } else {
		printf("Retrying   track %3d\n", trk);
	    }
	} else {
	    trk ++;
	    if (trk < fmt_opts->ft_tpc) {
		if (!opt->notty) {
		    printf("\r");
		    printf(_("Formatting track %3d, "), trk);
		} else {
		    printf("Formatting track %3d\n", trk);
		}
	    }
	}
    }
    printf("\n");
    return 0;
}

/*  This does the main work. We don't need to issue raw QIC-117 commands here
 *  because the driver need some information about the track it is located on.
 *  Does it? Maybe change it. First: let's try to make it work again.
 */
int verify_cartridge(const int tape_fd,
		     const ftfmt_tpparms_t *tpparms,
		     const struct opt_parms *opt,
		     void *hseg)
{
    int trk, seg, seg_id, i;
    struct mtftformat_virtual fmtc;

    if (!(tpparms->vendor->flags & FT_CANT_FORMAT)) {       
	if (qic_simple_command(tape_fd,
			       0, QIC_ENTER_VERIFY_MODE, -1, 1, NULL)) {
	    return -1;
	}
    }
    if (qic_simple_command(tape_fd, 0, QIC_SEEK_LOAD_POINT, -1, 670, NULL)) {
	return -1;
    }

    /* QIC-3020 cartridges have some segments mapped bad a priori
     * As this saves some space in the bad sector map, we mark them as bad
     * before verifying the cartridge.
     */
    if (tune_bsm_before(hseg, tpparms)) {
	return -1;
    }

    /* TODO: if this is a --verify-only pass, optionally erase the
     * cartridge by writing zero bytes to it.
     */

    for (trk = 0; trk < (int)tpparms->fmt_parms.ft_tpc; trk += 2) {
	if (opt->zero_tape) {
	    for (i = 0; i < 2; i++) {
		if (!opt->notty) {
		    printf("\r");
		    printf(_("Erasing    track %3d, segment %8d"),
			   trk + i, (trk + i) * tpparms->fmt_parms.ft_spt);
		} else {
		    printf("Erasing    track %3d\n", trk + i);
		}
		for (seg = 0; seg < (int)tpparms->fmt_parms.ft_spt; seg ++) {
		    static const u_int8_t data[FT_SEGMENT_SIZE] = { '\0', };
		    int mode;

		    seg_id = (trk + i) * tpparms->fmt_parms.ft_spt + seg;

		    if (seg_id <= tpparms->scnd_header_segment) {
			continue; /* DON'T */
		    }
		    if (seg_id == (tpparms->fmt_parms.ft_spt * 
				   tpparms->fmt_parms.ft_tpc - 1)) {
			mode = MT_FT_WR_SINGLE;
		    } else {
			mode = MT_FT_WR_ASYNC;
		    }

		    assert(seg_id >= 2);

		    printf(opt->notty ? "segment %8d\n":"\b\b\b\b\b\b\b\b%8d",
			   seg_id);

		    /* don't care in case of an error */
		    (void)write_segment(tape_fd, seg_id, mode, data);
		}
	    }
	}
	for (i = 0; i < 2; i++) {
	    if (!opt->notty) {
		printf("\r");
		printf(_("Verifying  track %3d, segment %8d"),
		       trk + i, (trk + i) * tpparms->fmt_parms.ft_spt);
	    } else {
		printf("Verifying  track %3d\n", trk + i);
	    }
	    for (seg = 0; seg < (int)tpparms->fmt_parms.ft_spt; seg ++) {

		seg_id = (trk + i) * tpparms->fmt_parms.ft_spt + seg;
		memset(&fmtc, 0, sizeof(fmtc));
		fmtc.fmt_op = FTFMT_VERIFY;
		fmtc.fmt_arg.fmt_verify.ft_segment = seg_id;
		printf(opt->notty ? "segment %8d\n" : "\b\b\b\b\b\b\b\b%8d",
		       seg_id);
		if (send_format_ioctl(tape_fd, &fmtc) == -1) {
		    fprintf(stderr,
			    _("\nIoctl error verifying segment %d: %s\n"),
			    seg_id, strerror(errno));
#if 0
		    /* the kernel driver now handles totally damaged
		     * segments but return -EIO if wait_segment()
		     * fails.
		     */
		    return -1;
#else
		    /*  We treat this is an all bad segment. */
		    put_bsm_entry(seg_id, EMPTY_SEGMENT);
#endif
		} else {
		    SectorMap bsm = fmtc.fmt_arg.fmt_verify.ft_bsm;

		    put_bsm_entry(seg_id,
				  get_bsm_entry(seg_id) | tune_bsm_entry(bsm));
		}
#if 0
		if (fmtc.fmt_arg.fmt_verify.ft_bsm != 0) {
		    fprintf(stderr, _("bsm for segment %d: 0x%08x\n"),
			    seg_id, fmtc.fmt_arg.fmt_verify.ft_bsm);
		}
#endif
	    }
	}
    }
    if (qic_simple_command(tape_fd, 0, QIC_ENTER_PRIMARY_MODE, -1, 0, NULL)) {
	printf("\n");
	return -1;
    }
    printf("\n");
    return 0;
}

unsigned long tape_capacity(const format_parms *fmt_opts)
{
    int i;
    int seg_size;
    unsigned long capacity;

    capacity = 0;
    for (i = 0;
	 i < fmt_opts->ft_spt * fmt_opts->ft_tpc;
	 i ++) {
	seg_size = (FT_SECTORS_PER_SEGMENT - 
		    count_ones(get_bsm_entry(i)) - 
		    FT_ECC_SECTORS);
	if (seg_size > 0) {
	    capacity += seg_size;
	}
    }
    return capacity;
}

/*  tune the bad sector map, i.e. maps some segments as empty etc.
 */
static int tune_bsm_before(void *hseg, const ftfmt_tpparms_t *tpparms)
{
    int i;
    const format_parms *fmt_opts = &tpparms->fmt_parms;   

    /*  map regions with hole imprints bad for QIC-3010/3020
     *
     *  NOTE: if FT_CANT_FORMAT is set, then we don't do this.
     */
    if (((fmt_opts->ft_qicstd & QIC_TAPE_STD_MASK) == QIC_TAPE_QIC3020 ||
	 (fmt_opts->ft_qicstd & QIC_TAPE_STD_MASK) == QIC_TAPE_QIC3010) &&
	!(tpparms->vendor->flags & FT_CANT_FORMAT)) {

	if (fmt_opts->ft_qicstd & QIC_TAPE_WIDE) {
	    /* 0.315in tape */
	    for (i = 17; i <= 37; i += 2) {
		put_bsm_entry(i     * fmt_opts->ft_spt,     EMPTY_SEGMENT);
		put_bsm_entry(i     * fmt_opts->ft_spt + 1, EMPTY_SEGMENT);
		put_bsm_entry(i     * fmt_opts->ft_spt + 2, EMPTY_SEGMENT);
		put_bsm_entry(i     * fmt_opts->ft_spt + 3, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 4, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 3, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 2, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 1, EMPTY_SEGMENT);
	    }
	} else {
	    /* quarterinch tape */
	    for (i = 5; i <= 27; i += 2) {
		put_bsm_entry(i     * fmt_opts->ft_spt,     EMPTY_SEGMENT);
		put_bsm_entry(i     * fmt_opts->ft_spt + 1, EMPTY_SEGMENT);
		put_bsm_entry(i     * fmt_opts->ft_spt + 2, EMPTY_SEGMENT);
		put_bsm_entry(i     * fmt_opts->ft_spt + 3, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 4, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 3, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 2, EMPTY_SEGMENT);
		put_bsm_entry((i+1) * fmt_opts->ft_spt - 1, EMPTY_SEGMENT);
	    }
	}
    }
    /* This might save some space in the already existing bad sector map
     */
    for (i=tpparms->frst_data_segment; i <= tpparms->last_data_segment; i++) {
	put_bsm_entry(i, tune_bsm_entry(get_bsm_entry(i)));
    }
    return 0;
}

/* Segments with more than this many bad sectors are marked as
 * entirely unusable
 */
#define FTFMT_MAX_BAD_SECTORS 8  /*  other suggestions? */

/*  "Tune" a bad sector map entry. Mark a segment with more than, say,
 *  FTFMT_MAX_BAD_SECTORS as empty Also mark sectors where both
 *  neighbours are bad as bad.
 */
static SectorMap tune_bsm_entry(SectorMap map)
{
    if (map != 0) {
	SectorMap newmap = map | ((map >> 1) & (map << 1));
	SectorMap blockmap = (newmap >> 1) & newmap;
	
	/* surround contigous blocks of bad sectors by one extra bad
	 * sector at each side.
	 */
	newmap  |= (blockmap >> 1) | (blockmap << 2);
	
	if (count_ones(newmap) >= FTFMT_MAX_BAD_SECTORS) {
	    newmap = EMPTY_SEGMENT;
	}
	
	map = newmap;
    }
    return map;
}

/* Returns -EIO if cartridge is really damaged.
 */
static int tune_bsm_after(void *hseg, ftfmt_tpparms_t *tpparms)
{
    const format_parms *fmt_opts = &tpparms->fmt_parms;   
    int *first = &tpparms->frst_header_segment;
    int *scnd  = &tpparms->scnd_header_segment;
    int *vtbl  = &tpparms->frst_data_segment;
    int *last  = &tpparms->last_data_segment;
    int i;

    /*  Ok. What remains to do: find three segments without errors
     *  towards the beginning of the tape and write the space between them 
     *  (if any) with deleted data marks.
     */
    for (*first = 0; *first < fmt_opts->ft_spt * fmt_opts->ft_tpc; (*first)++) {
	if (get_bsm_entry(*first) == 0)
	    break;
	put_bsm_entry(*first, EMPTY_SEGMENT);
    }
    /* now for the second header segment: */
    for (*scnd= (*first) + 1; *scnd < fmt_opts->ft_spt*fmt_opts->ft_tpc; (*scnd)++) {
	if (get_bsm_entry(*scnd) == 0)
	    break;
	put_bsm_entry(*scnd, EMPTY_SEGMENT);
    }
    /* and now for the volume table segment */
    for (*vtbl = (*scnd) + 1; *vtbl < fmt_opts->ft_spt*fmt_opts->ft_tpc; (*vtbl)++) {
	if (get_bsm_entry(*vtbl) == 0)
	    break;
	put_bsm_entry(*vtbl, EMPTY_SEGMENT);
    }
    if (*vtbl >= fmt_opts->ft_spt * fmt_opts->ft_tpc) {
	/* totally damaged cartridge */
	fprintf(stderr,
		_("Newly formatted cartride hasn't three error free segments.\n"));
	return -1;
    }
    *last = fmt_opts->ft_spt * fmt_opts->ft_tpc - 1;
    while (*last > *vtbl && get_bsm_entry(*last) == EMPTY_SEGMENT) {
	(*last) --;
    }
    return 0;
}

static u_int32_t create_time_stamp(void)
{
    struct tm *fmt_time;
    time_t raw_time;
	
    time(&raw_time);
    fmt_time = gmtime(&raw_time);
    return FT_TIME_STAMP(fmt_time->tm_year + 1900, 
			 fmt_time->tm_mon, 
			 fmt_time->tm_mday,
			 fmt_time->tm_hour,
			 fmt_time->tm_min,
			 fmt_time->tm_sec);
}

/* Decode the date stored in timestamp in the format described in the
 * various QIC-something standards. This is year 2000 proof, but will
 * overflow in year 2097 :-)
 */
char *decode_date(u_int32_t timestamp)
{
	struct tm tapetm;
	time_t tapetime;
	static char date[18];

	memset(&tapetm, 0, sizeof(tapetm));
	
	tapetm.tm_year  = timestamp >> FT_YEAR_SHIFT;
	tapetm.tm_year += FT_YEAR_0 - 1900;
	timestamp      &= FT_TIME_MASK;

	tapetm.tm_sec   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_min   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_hour  = timestamp % 24;
	timestamp      /= 24;
	tapetm.tm_mday  = timestamp % 31 + 1;
	timestamp      /= 31;
	tapetm.tm_mon   = timestamp % 12;
	tapetm.tm_isdst = -1;
	if ((tapetime = mktime(&tapetm)) == (time_t)(-1)) {
		return _("invalid");
	} else {
		/* maybe we should use locale's time representation here.
		 */
		(void)strftime(date, 18, "%T %D", &tapetm);
		return date;
	}
}

int compose_header_segment(u_int8_t *hseg, const char *tape_label,
			   ftfmt_tpparms_t *tpparms)
{
    u_int32_t time_stamp;
    const format_parms *parms = &tpparms->fmt_parms;   

    /* determine the location of the header segments etc.
     */
    if (tune_bsm_after(hseg, tpparms)) {
	return -1;
    }

    PUT4(hseg, FT_SIGNATURE, FT_HSEG_MAGIC);
    hseg[FT_FMT_CODE] = parms->ft_fmtcode;
    if (parms->ft_fmtcode == fmt_big) {
	memset(hseg + FT_HSEG_1, 0, 8);
	PUT4(hseg, FT_6_HSEG_1,   tpparms->frst_header_segment);
	PUT4(hseg, FT_6_HSEG_2,   tpparms->scnd_header_segment);
	PUT4(hseg, FT_6_FRST_SEG, tpparms->frst_data_segment);
	PUT4(hseg, FT_6_LAST_SEG, tpparms->last_data_segment);
    } else {
	memset(hseg + FT_6_HSEG_1, 0, 16);
	PUT2(hseg, FT_HSEG_1,   tpparms->frst_header_segment);
	PUT2(hseg, FT_HSEG_2,   tpparms->scnd_header_segment);
	PUT2(hseg, FT_FRST_SEG, tpparms->frst_data_segment);
	PUT2(hseg, FT_LAST_SEG, tpparms->last_data_segment);
    }
    time_stamp = create_time_stamp();
    PUT4(hseg, FT_FMT_DATE, time_stamp);
    PUT4(hseg, FT_WR_DATE,  time_stamp);
    PUT2(hseg, FT_SPT, parms->ft_spt);
    hseg[FT_TPC] = parms->ft_tpc;
    hseg[FT_FHM] = parms->ft_fhm;
    hseg[FT_FTM] = parms->ft_ftm;
    hseg[FT_FSM] = MAX_FLOPPY_SECTOR;
    memset(hseg + FT_LABEL, ' ', FT_LABEL_SZ);
    memcpy(hseg + FT_LABEL, tape_label,
	   FT_LABEL_SZ < strlen(tape_label)? FT_LABEL_SZ : strlen(tape_label));
    PUT4(hseg, FT_LABEL_DATE, time_stamp);
    PUT2(hseg, FT_CMAP_START, 0);

    /*  number of segments written, formatted or verified through
     *  lifetime of tape
     */
    PUT4(hseg, FT_SEG_CNT,
	 GET4(hseg, FT_SEG_CNT) + (parms->ft_spt * parms->ft_tpc) * 2 + 
	 tpparms->frst_data_segment);
    PUT2(hseg, FT_FMT_CNT, GET2(hseg, FT_FMT_CNT) + 1);
    PUT2(hseg, FT_FSL_CNT, 0); /* clear the failed sector log */

    return 0;
}

static int write_segment(const int tape_fd,
			 const int seg_id, int mode, const void *data)
{
    struct mtftseg ft_seg;
	
    memset(&ft_seg, 0, sizeof(ft_seg));
    ft_seg.mt_data  = (void *)data;
    ft_seg.mt_segno = seg_id;
    ft_seg.mt_mode  = mode;
    if (ioctl(tape_fd, MTIOCWRFTSEG, &ft_seg) == -1) {
	fprintf(stderr, _("Ioctl error reading header segment: %s\n"),
		strerror(errno));
	return -1;
    }
    if (ft_seg.mt_result < 0) {
	fprintf(stderr, _("Failed to write segment %d, mode %d: %d\n"),
		seg_id, mode, ft_seg.mt_result);
	return -1;
    }
    return 0;
}

/* The tape drive must already be in primary mode when this function
 * is called
 */
int write_header_segments(const int tape_fd,
			  u_int8_t *hseg, ftfmt_tpparms_t *tpparms)
{
    const format_parms *fmt_opts = &tpparms->fmt_parms;
    int seg_id;
    int first, scnd, vtbl, last;

    first = tpparms->frst_header_segment;
    scnd  = tpparms->scnd_header_segment;
    vtbl  = tpparms->frst_data_segment;
    last  = tpparms->last_data_segment;

    /* Ok, now write it out to tape
     */
    seg_id = 0;
    while (seg_id < first) {
	printf(_("Deleting segment %d ... "), seg_id);
	if (write_segment(tape_fd, seg_id++, MT_FT_WR_DELETE, NULL))
	    return -1;
	printf(_("done\n"));
    }
    printf(_("Writing 1st header segment at %d ... "), seg_id);
    if (write_segment(tape_fd, seg_id++, MT_FT_WR_ASYNC, hseg))
	return -1;
    printf(_("done.\n"));
    while (seg_id < scnd) {
	printf(_("Deleting segment %d ... "), seg_id);
	if (write_segment(tape_fd, seg_id++, MT_FT_WR_DELETE, NULL))
	    return -1;
	printf(_("done\n"));
    }
    printf(_("Writing 2nd header segment at %d ... "), seg_id);
    if (write_segment(tape_fd, seg_id++, MT_FT_WR_ASYNC, hseg))
	return -1;
    printf(_("done\n"));
    while (seg_id < vtbl) {
	printf(_("Deleting segment %d ... "), seg_id);
	if (write_segment(tape_fd, seg_id++, MT_FT_WR_DELETE, NULL))
	    return -1;
	printf(_("done\n"));
    }
    memset(hseg, 0, FT_SEGMENT_SIZE);
    printf(_("Initializing volume table  at %d ... "), seg_id);
    if (write_segment(tape_fd, vtbl, MT_FT_WR_SINGLE, hseg))
	return -1;
    printf(_("done.\n"));
    return 0;
}

/*  Dump the header segment in a readable form to stdout.
 */
#undef PRINT1hex
#define PRINT1hex(name) printf(#name": 0x%02x\n", (int)hseg[FT_##name])
#undef PRINT1dec
#define PRINT1dec(name) printf(#name": %d\n", (int)hseg[FT_##name])
#undef PRINT2
#define PRINT2(name) printf(#name": %d\n", GET2(hseg, FT_##name))
#undef PRINT4
#define PRINT4(name) printf(#name": %d\n", GET4(hseg, FT_##name))
#undef PRINTDATE
#define PRINTDATE(name) printf(#name": \"%s\"\n",			\
			       decode_date(GET4(hseg, FT_##name)))
#undef PRINTSTRING
#define PRINTSTRING(name, size)			\
{						\
    char buffer[size + 1];			\
						\
    memcpy(buffer, &hseg[FT_##name], size);	\
    buffer[size] = '\0';			\
    printf(#name": \"%s\"\n", buffer);		\
}

void print_header_segment(const u_int8_t *hseg, ftfmt_tpparms_t *tpparms)
{

    printf("SIGNATURE: 0x%08x\n", GET4(hseg, FT_SIGNATURE));
    PRINT1hex(FMT_CODE);
    PRINT1hex(REV_LEVEL);
    if (hseg[FT_FMT_CODE] != fmt_big) {
	PRINT2(HSEG_1);
	PRINT2(HSEG_2);
	PRINT2(FRST_SEG);
	PRINT2(LAST_SEG);
    }
    PRINTDATE(FMT_DATE);
    PRINTDATE(WR_DATE);
    PRINT2(SPT);
    PRINT1dec(TPC);
    PRINT1dec(FHM);
    PRINT1dec(FTM);
    PRINT1dec(FSM);
    PRINTSTRING(LABEL, FT_LABEL_SZ);
    PRINTDATE(LABEL_DATE);
    PRINT2(CMAP_START);
    PRINT1hex(FMT_ERROR);
    PRINT4(SEG_CNT);
    PRINTDATE(INIT_DATE);
    PRINT2(FMT_CNT);
    PRINT2(FSL_CNT);
    PRINTSTRING(MK_CODE, FT_LOT_CODE - FT_MK_CODE);
    PRINTSTRING(LOT_CODE, FT_6_HSEG_1 - FT_LOT_CODE);
    if (hseg[FT_FMT_CODE] == fmt_big) {
	PRINT4(6_HSEG_1);
	PRINT4(6_HSEG_2);
	PRINT4(6_FRST_SEG);
	PRINT4(6_LAST_SEG);
    }
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 * End:
 */
