#!/bin/env python
#
#  makemake.py - Make makefile
#
#  $Id: makemake.py,v 1.3 1998/10/09 20:55:01 cg Exp $
#
#  SGMLtools - an SGML toolkit.
#  Copyright (C) 1998 Cees A. de Groot
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#

"""
NAME

    makemake.pl - Create a makefile with dependencies

SYNOPSIS

    % perl makemake.pl

DESCRIPTION

    This program reads the package list, and generates a makefile with
    prep/build/install/clean commands for every package that is a) in
    the package list and b) available in source form. The make commands
    typically delegate the actual work to rpmaction.pl, this code is mostly
    used for UI and dependency reasons.

    The script expects to be called from the top-level directory of
    the full SGMLtools distribution. The make commands are put into the
    file "Makerules".

"""

#
#  Load support routines.
#
from readpackagelist import readpackagelist
import os, string

#
#  Read the profile, if it is there. The contents of the profile
#  are put into os.environ.
#
if os.path.isfile('profile'):
    fh = open('profile', 'r')
    cmd = ''
    for line in fh.readlines():
	if line[0] == '#':
	    continue

	#
	#  Split the assignment and put it in the resulting command
	#  a) as a direct assignment, and b) as an assignment to
	#  os.environ
	#
	(lhs, rhs) = string.split(line, '=')
	rhs = string.rstrip(rhs)
	cmd = cmd + '%s = "%s"\n' % (lhs, rhs)
	cmd = cmd + 'os.environ["%s"] = "%s"\n' % (lhs, rhs)

    exec cmd
    fh.close()

if os.environ.has_key('PYTHON'):
    pythoncmd = os.environ['PYTHON']
else:
    pythoncmd = '/usr/bin/python'

#
#  Load the package file.
#
packagelist = readpackagelist ("packages/packagelist")

#
#  Load the list of installed packages. Advanced users may use this
#  to preset installed-ness of packages, regardless of their availability
#  as source code.
#
if os.path.isfile('.installed'):
    fh = open('.installed', 'r')
    for line in fh.readlines():
	packagelist[line]['installed'] = 1
    fh.close()

makerules = open('Makerules', 'w')
makerules.write("#\n# AUTOGENERATED - DO NOT EDIT!!!\n#\n\n")

allpackages = []

#
#  Loop through the packages, generating Makerules.
#
for package in packagelist['KEYS']:

    if not os.path.isdir('packages/cvs/' + package):
	packagelist[package]['installed'] = '1'

    if packagelist[package].has_key('installed'):
	continue

    depexp = ''
    if packagelist[package].has_key('flags'):
	if packagelist[package]['flags'].has_key('noinstall'):
	    continue
	
	if packagelist[package]['flags'].has_key('dep'):
	    deps = split(packagelist[package]['flags']['dep'], '+')
	    for dep in deps:
		depexp = depexp + ' .deps/install-' + dep

    allpackages.append(package)
    makeblock = """

.PHONY: prep-%(package)s build-%(package)s install-%(package)s
prep-%(package)s: .deps/prep-%(package)s
build-%(package)s: .deps/build-%(package)s
install-%(package)s: .deps/install-%(package)s

.deps/prep-%(package)s: %(depexp)s
	%(pythoncmd)s tools/rpmaction.py prep %(package)s
	touch .deps/prep-%(package)s

.deps/build-%(package)s: .deps/prep-%(package)s
	%(pythoncmd)s tools/rpmaction.py build %(package)s
	touch .deps/build-%(package)s

.deps/install-%(package)s: .deps/build-%(package)s
	%(pythoncmd)s tools/rpmaction.py install %(package)s
	%(pythoncmd)s tools/rpmaction.py post %(package)s
	touch .deps/install-%(package)s

clean-%(package)s:
	-cd packages/cvs/%(package)s; make clean
"""
    makerules.write(makeblock % locals())

#
#  Emit blocks with top-level statements
#
def emitall(rulename):
    makerules.write('\n%s-all: ' % rulename)

    for package in allpackages:
	makerules.write(' %s-%s' % (rulename, package))

emitall('prep')
emitall('build')
emitall('install')
emitall('clean')

makerules.close()

#
#  Rewrite .installed.
#
installed = open('.installed', 'w')
for package in packagelist['KEYS']:
    if packagelist[package].has_key('installed'):
	installed.write(package + '\n')
installed.close()

