/*  alsa.c - ALSA output plugin
 *  Copyright (C) 1999 by Andy Lo A Foe <andy@alsa-project.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "config.h"
#include <stdio.h>
#include <sys/asoundlib.h>
#include <stdlib.h>
#include <string.h>
#include "output_plugin.h"


#define LOW_FRAGS	1	
//#define QUEUE_COUNT

#ifdef USE_ALSA_VOID
static void *sound_handle;
#else	// New snd_pcm_t handles
static snd_pcm_t *sound_handle;
#endif

static int alsa_init()
{
	// Always return ok for now
	sound_handle = NULL;
	return 1;
}


static int alsa_open(int card, int device)
{
	int err;

	if ((err = snd_pcm_open(&sound_handle, card, device, SND_PCM_OPEN_PLAYBACK)) < 0) {
#ifdef DEBUG
		fprintf(stderr, "ALSA: %s\n", snd_strerror(err));
#endif
		return 0;
	}
	return 1;
}


static void alsa_close()
{
	snd_pcm_close(sound_handle);
	return;
}

static int alsa_write(void *data, int count)
{
	snd_pcm_write(sound_handle, data, count);
	return 1;
}

static int frag_size = -1;
static int frag_count = -1;

static int alsa_set_buffer(int fragment_size, int fragment_count)
{
	snd_pcm_playback_params_t pp;
	int err;
	
	memset(&pp, 0, sizeof(pp));

	pp.fragment_size = fragment_size;
	pp.fragments_max = fragment_count;
	pp.fragments_room = LOW_FRAGS;

	if ((err = snd_pcm_playback_params(sound_handle, &pp)) < 0) {
		fprintf(stderr, "ALSA: playback parameter error \"%s\"\n", snd_strerror(err));
		return 0;
	}
	frag_size = fragment_size;
	frag_count = fragment_count;

	return 1;
}


static int alsa_set_sample_rate(int rate)
{
	snd_pcm_format_t pf;
	int err;

	memset(&pf, 0, sizeof(pf));
	pf.format = SND_PCM_SFMT_S16_LE;
	pf.rate = rate;
	pf.channels = 2; // Stereo
	if ((err = snd_pcm_playback_format(sound_handle, &pf)) < 0) {
		fprintf(stderr, "ALSA: playback format error \"%s\"\n", snd_strerror(err));
		return 0;
	}
	return 1;
}

#ifdef QUEUE_COUNT
static int alsa_get_queue_count()
{
	struct snd_pcm_playback_status st;
	memset(&st, 0, sizeof(st));

	snd_pcm_playback_status(sound_handle, &st);
	
	return st.queue;
}
#endif

static int alsa_get_latency()
{
	return (frag_size * frag_count);
}

output_plugin alsa_output = {
	OUTPUT_PLUGIN_VERSION,
	{ "ALSA output v1.0" },
	{ "Andy Lo A Foe" },
	alsa_init,
	alsa_open,
	alsa_close,
	alsa_write,
	alsa_set_buffer,
	alsa_set_sample_rate,
#ifdef QUEUE_COUNT
	alsa_get_queue_count,
#else
	NULL,
#endif	
	alsa_get_latency,
};


output_plugin *output_plugin_info(void)
{
	return &alsa_output;
}


