/* Implements directives, pseudo-ops and processor opcodes
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "symbol.h"
#include "opcode.h"
#include "gpasm.tab.h"
#include "lst.h"
#include "gperror.h"

/* Forward declarations */
void execute_macro(struct macro_head *h, int arity, struct pnode *parms);

/* XXXPRO: new class of processor may require new instruction classes */
enum insn_class {
  INSN_CLASS_LIT8,	/* bits 7:0 contain an 8 bit literal		*/
  INSN_CLASS_LIT9,	/* bits 8:0 contain a 9 bit literal		*/
  INSN_CLASS_LIT11,	/* bits 10:0 contain an 11 bit literal		*/
  INSN_CLASS_IMPLICIT,	/* instruction has no variable bits at all	*/
  INSN_CLASS_OPF5,	/* bits 4:0 contain a register address		*/
  INSN_CLASS_OPWF5,	/* as above, but bit 5 has a destination flag	*/
  INSN_CLASS_B5,	/* as for OPF5, but bits 7:5 have bit number	*/
  INSN_CLASS_OPF7,	/* bits 6:0 contain a register address		*/
  INSN_CLASS_OPWF7,	/* as above, but bit 7 has destination flag	*/
  INSN_CLASS_B7,	/* as for OPF7, but bits 9:7 have bit number	*/
  INSN_CLASS_FUNC,	/* instruction is an assembler function		*/
  INSN_CLASS_LIT3_BANK, /* SX: bits 3:0 contain a 3 bit literal, shifted 5 bits */
  INSN_CLASS_LIT3_PAGE, /* SX: bits 3:0 contain a 3 bit literal, shifted 9 bits */
  INSN_CLASS_LIT4,      /* SX: bits 3:0 contain a 4 bit literal         */
};

#define ATTRIB_COND 1

struct insn {
  char *name;
  int mask;
  enum insn_class class;
  int attribs;
};

/************************************************************************/

/* Write a word into the memory image at the current location */

static void emit(unsigned int value)
{
  state.i_memory[state.org++] = MEM_USED_MASK | value;
}

/************************************************************************/

struct amode {
  enum { in_then, in_else } mode;
  int enabled;	/* Are we currently enabled? */
  int prev_enabled;
  struct amode *prev;
};

/************************************************************************/

#define HEAD(L) (L)->value.list.head
#define TAIL(L) (L)->value.list.tail

static int list_length(struct pnode *L)
{
  if (L == NULL) {
    return 0;
  } else {
    return 1 + list_length(TAIL(L));
  }
}

static int can_evaluate(struct pnode *p)
{
  switch (p->tag) {
  case constant:
    return 1;
  case symbol:
    {
      struct symbol *s;

      /* '$' means current org, which we can always evaluate */
      if (strcmp(p->value.symbol, "$") == 0) {
	return 1;
      } else {
	/* Otherwise look it up */
	s = get_symbol(state.stTop, p->value.symbol);
      
	return ((s != NULL) &&
		(get_symbol_annotation(s) != NULL));
      }
    }
  case unop:
    return can_evaluate(p->value.unop.p0);
  case binop:
    return can_evaluate(p->value.binop.p0) && can_evaluate(p->value.binop.p1);
  case string:
    return 0;
  default:
    assert(0);
  }
}

static gpasmVal evaluate(struct pnode *p)
{
  struct variable *var;
  gpasmVal p0, p1;

  switch (p->tag) {
  case constant:
    return p->value.constant;
  case symbol:
    {
      struct symbol *s;

      if (strcmp(p->value.symbol, "$") == 0) {
	return state.org;
      } else {
	s = get_symbol(state.stTop, p->value.symbol);
	var = get_symbol_annotation(s);
	assert(var != NULL);
	return var->value;
      }
    }
  case unop:
    switch (p->value.unop.op) {
    case '!':
      return !evaluate(p->value.unop.p0);
    case '-':
      return -evaluate(p->value.unop.p0);
    case '~':
      return ~evaluate(p->value.unop.p0);
    case HIGH:
      return (evaluate(p->value.unop.p0) & 0xff) >> 8;
    case LOW:
      return evaluate(p->value.unop.p0) & 0xff;
    default:
      assert(0);
    }
  case binop:
    p0 = evaluate(p->value.binop.p0);
    p1 = evaluate(p->value.binop.p1);
    switch (p->value.binop.op) {
    case '+':      return p0 + p1;
    case '-':      return p0 - p1;
    case '*':      return p0 * p1;
    case '/':      return p0 / p1;
    case '%':      return p0 % p1;
    case '&':      return p0 & p1;
    case '|':      return p0 | p1;
    case '^':      return p0 ^ p1;
    case LSH:      return p0 << p1;
    case RSH:      return p0 >> p1;
    case EQUAL:    return p0 == p1;
    case '<':      return p0 < p1;
    case '>':      return p0 > p1;
    case NOT_EQUAL:          return p0 != p1;
    case GREATER_EQUAL:      return p0 >= p1;
    case LESS_EQUAL:         return p0 <= p1;
    case LOGICAL_AND:        return p0 && p1;
    case LOGICAL_OR:         return p0 || p1;
    default:
      assert(0); /* Unhandled binary operator */
    }
  default:
    assert(0); /* Unhandled parse node tag */
  }
  return (0); /* Should never reach here */
}

/* Attempt to evaluate expression 'p'.  Return its value if
 * successful, otherwise generate an error message and return 0.  */

static gpasmVal maybe_evaluate(struct pnode *p)
{
  gpasmVal r;

  if (can_evaluate(p)) {
    r = evaluate(p);
  } else {
    gperror(104, "Can't evaluate expression");
    r = 0;
  }

  return r;
}

static int off_or_on(struct pnode *p)
{
  int had_error = 0, ret = 0;

  if (p->tag != symbol)
    had_error = 1;
  else if (strcasecmp(p->value.symbol, "OFF") == 0)
    ret = 0;
  else if (strcasecmp(p->value.symbol, "ON") == 0)
    ret = 1;
  else
    had_error = 1;

  if (had_error)
     gperror(125, "Option should be \"OFF\" or \"ON\"");

  return ret;
}

static void data(struct pnode *L, int flavor)
{
  if (L) {
    struct pnode *p;

    p = HEAD(L);
    if (p->tag == string) {
      char *pc = p->value.string;
      
      while (*pc) {
	emit(*pc | flavor);
	pc++;
      }
    } else {
      emit(maybe_evaluate(p) | flavor);
    }
    data(TAIL(L), flavor);
  }
}

static int enforce_arity(int arity, int must_be)
{
  char mesg[80];

  if (arity == must_be)
    return 1;
  else {
    sprintf(mesg,
	    "Too %s operands for opcode, expected %d",
	    (arity < must_be) ? "few" : "many",
	    must_be);
    gperror(105, mesg);
    return 0;
  }
}

static int enforce_simple(struct pnode *p)
{
  if (p->tag == symbol) {
    return 1;
  } else {
    gperror(109, "Expression too complex");
    return 0;
  }
}

/* Do the work for beginning a conditional assembly block.  Leave it
   disabled by default.  This is used by do_if, do_ifdef and
   do_ifndef. */

static void enter_if(void)
{
  struct amode *new = malloc(sizeof(*new));

  new->mode = in_then;
  new->prev = state.astack;
  if (state.astack == NULL)
    new->prev_enabled = 1;
  else
    new->prev_enabled = state.astack->enabled && state.astack->prev_enabled;
  new->enabled = 0;	/* Only the default */
  state.astack = new;
}

/* Checking that a macro definition's parameters are correct */

static int macro_parms_simple(struct pnode *parms)
{
  if (parms == NULL)
    return 1;
  else
    return ((HEAD(parms)->tag == symbol) &&
	    macro_parms_simple(TAIL(parms)));
}

static int list_symbol_member(struct pnode *M, struct pnode *L)
{
  if (L == NULL)
    return 0;
  if (STRCMP(M->value.symbol, HEAD(L)->value.symbol) == 0)
    return 1;
  else
    return list_symbol_member(M, TAIL(L));
}

static int macro_parms_unique(struct pnode *parms)
{
  if (parms == NULL)
    return 1;
  else
    return (!list_symbol_member(HEAD(parms), TAIL(parms)) &&
	    macro_parms_unique(TAIL(parms)));
}

static int macro_parms_ok(struct pnode *parms)
{
  if (!macro_parms_simple(parms)) {
    gperror(109, "Expression too complex");
    return 0;
  }
  if (!macro_parms_unique(parms)) {
    gperror(120, "Duplicate macro parameter");
    return 0;
  }
  return 1;
}

typedef gpasmVal opfunc(gpasmVal r,
			char *name,
			int arity,
			struct pnode *parms);

/************************************************************************/

static gpasmVal do_badram(gpasmVal r,
			  char *name,
			  int arity,
			  struct pnode *parms)
{
  struct pnode *p;

  for (; parms != NULL; parms = TAIL(parms)) {
    p = HEAD(parms);
    if ((p->tag == binop) &&
	(p->value.binop.op == '-')) {
      int start, end;

      if (!can_evaluate(p->value.binop.p0) ||
	  !can_evaluate(p->value.binop.p1)) {
	gperror(104, "Can't evaluate expression");
      } else {
	start = evaluate(p->value.binop.p0);
	end = evaluate(p->value.binop.p1);

	if ((end < start) ||
	    (start < 0) ||
	    (MAX_RAM <= end)) {
	  gperror(106, "Invalid RAM address");
	} else {
	  for (; start <= end; start++)
	    state.badram[start] = 1;
	}
      }
    } else {
      if (!can_evaluate(p)) {
	gperror(113, "Can't evaluate expression");
      } else {
	int loc;

	loc = evaluate(p);
	if ((loc < 0) ||
	    (MAX_RAM <= loc))
	  gperror(106, "Invalid RAM address");
	else
	  state.badram[loc] = 1;
      }
    }
  }

  return r;
}

static gpasmVal do_config(gpasmVal r,
			  char *name,
			  int arity,
			  struct pnode *parms)
{
  struct pnode *p;

  if (enforce_arity(arity, 1)) {
    p = HEAD(parms);
    if (can_evaluate(p)) {
      state.i_memory[0x2007] = MEM_USED_MASK | evaluate(p);
    } else
      gperror(104, "Can't evaluate expression");
  }

  return r;
}

static gpasmVal do_data(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  data(parms, 0);

  return r;
}

static gpasmVal do_dt(gpasmVal r,
		      char *name,
		      int arity,
		      struct pnode *parms)
{
  struct symbol *s;
  struct insn *i;

  s = get_symbol(state.stBuiltin, "RETLW");
  assert(s != NULL); /* Every PIC has a RETLW instruction */
  i = get_symbol_annotation(s);
  data(parms, i->mask);

  return r;
}

static gpasmVal do_else(gpasmVal r,
			char *name,
			int arity,
			struct pnode *parms)
{
  if (state.astack == NULL)
    gperror(107, "Unexpected \"else\"");
  else if ((state.astack->mode != in_then))
    gperror(107, "Unexpected \"else\"");
  else
    state.astack->enabled = !state.astack->enabled;

  return r;
}

static gpasmVal do_endif(gpasmVal r,
			 char *name,
			 int arity,
			 struct pnode *parms)
{
  if (state.astack == NULL)
    gperror(108, "Unexpected \"endif\"");
  else if ((state.astack->mode != in_then) &&
	   (state.astack->mode != in_else))
    gperror(108, "Unexpected \"endif\"");
  else
    state.astack = state.astack->prev;

  return r;
}

static gpasmVal do_endm(gpasmVal r,
			char *name,
			int arity,
			struct pnode *parms)
{
  assert(!state.mac_head);
  
  if (state.mac_prev == NULL)
    gperror(116, "Unexpected \"endm\"");
  else
    state.mac_prev = NULL;

  state.mac_body = NULL;

  return r;
}

static gpasmVal do_equ(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  struct pnode *p;

  state.lst.line.linetype = equ;
  if (enforce_arity(arity, 1)) {
    p = HEAD(parms);
    if (can_evaluate(p))
      r = evaluate(p);
    else
      gperror(104, "Can't evaluate expression");
  }

  return r;
}

static gpasmVal do_if(gpasmVal r,
		      char *name,
		      int arity,
		      struct pnode *parms)
{
  struct pnode *p;

  enter_if();

  /* Only evaluate the conditional if it matters... */
  if (state.astack->prev_enabled) {
    if (enforce_arity(arity, 1)) {
      p = HEAD(parms);
      if (can_evaluate(p))
	state.astack->enabled = evaluate(p);
      else
	gperror(104, "Can't evaluate expression");
    }
  }

  return r;
}

static gpasmVal do_ifdef(gpasmVal r,
			 char *name,
			 int arity,
			 struct pnode *parms)
{
  struct pnode *p;

  enter_if();

  /* Only evaluate the conditional if it matters... */
  if (state.astack->prev_enabled) {
    if (enforce_arity(arity, 1)) {
      p = HEAD(parms);
      if (p->tag != symbol) {
	gperror(109, "Expression too complex");
      } else {
	if (get_symbol(state.stDefines, p->value.symbol))
	  state.astack->enabled = 1;
      }
    }
  }

  return r;
}

static gpasmVal do_ifndef(gpasmVal r,
			  char *name,
			  int arity,
			  struct pnode *parms)
{
  struct pnode *p;

  enter_if();

  /* Only evaluate the conditional if it matters... */
  if (state.astack->prev_enabled) {
    if (enforce_arity(arity, 1)) {
      p = HEAD(parms);
      if (p->tag != symbol) {
	gperror(109, "Expression too complex");
      } else {
	if (!get_symbol(state.stDefines, p->value.symbol))
	  state.astack->enabled = 1;
      }
    }
  }

  return r;
}

static gpasmVal do_list(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  struct pnode *p;

  state.lst.enabled = 1;

  for (; parms; parms = TAIL(parms)) {
    p = HEAD(parms);
    if ((p->tag == binop) &&
	(p->value.binop.op == '=')) {
      if (enforce_simple(p->value.binop.p0)) {
	char *lhs;

	lhs = p->value.binop.p0->value.symbol;
	if (STRCMP(lhs, "b") == 0) {
	  int b;

	  b = maybe_evaluate(p->value.binop.p1);
	  if (b != 0)
	    state.lst.tabstop = b;
	} else if (STRCMP(lhs, "c") == 0)
	  ; /* Ignore this for now: column width not used */
	else if (STRCMP(lhs, "mm") == 0)
	  ; /* Ignore this for now: memory map not produced */
	else if (STRCMP(lhs, "n") == 0)
	  state.lst.linesperpage = maybe_evaluate(p->value.binop.p1);
	else if (STRCMP(lhs, "p") == 0) {
	  if (enforce_simple(p->value.binop.p1))
	    select_processor(p->value.binop.p1->value.symbol);
	} else if (STRCMP(lhs, "r") == 0) {
	  if (enforce_simple(p->value.binop.p1))
	    select_radix(p->value.binop.p1->value.symbol);
	} else if (STRCMP(lhs, "st") == 0)
	  state.lst.symboltable = off_or_on(p->value.binop.p1);
	else if (STRCMP(lhs, "t") == 0)
	  ; /* Ignore this for now: always wrap long list lines */
	else if (STRCMP(lhs, "w") == 0)
	  ; /* Ignore this for now: warnings, error always enabled */
	else if (STRCMP(lhs, "x") == 0)
	  ; /* Ignore this for now: macro expansion always off */
	} else {
	  gperror(103, "Syntax error");
	}
    } else {
      if (enforce_simple(p)) {
	if (STRCMP(p->value.symbol, "free") == 0)
	  ; /* Ignore this directive */
	else if (STRCMP(p->value.symbol, "fixed") == 0)
	  ; /* Ignore this directive */
	else
	  gperror(103, "Syntax error");
      }
    }
  }

  return r;
}

static gpasmVal do_local(gpasmVal r,
			 char *name,
			 int arity,
			 struct pnode *parms)
{
  struct pnode *p;

  while (parms) {
    p = HEAD(parms);
    if (p->tag != symbol) {
      gperror(109, "Expression too complex");
    } else if (state.stGlobal == state.stTop) {
      gperror(119, "Attempt to use \"local\" outside macro definition");
    } else {
      /* Looks OK.  Add the symbol to the current macro's local symbol
         table. */
      add_symbol(state.stTop, p->value.symbol);
    }
    parms = TAIL(parms);
  }

  return r;
}

static gpasmVal do_nolist(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  state.lst.enabled = 0;
  return r;
}

static gpasmVal do_maxram(gpasmVal r,
			  char *name,
			  int arity,
			  struct pnode *parms)
{
  struct pnode *p;

  if (enforce_arity(arity, 1)) {
    p = HEAD(parms);
    if (can_evaluate(p))
      state.maxram = evaluate(p);
    else
      gperror(104, "Can't evaluate expression");
  }

  return r;
}

static gpasmVal do_macro(gpasmVal r,
			 char *name,
			 int arity,
			 struct pnode *parms)
{
  struct macro_head *head = malloc(sizeof(*head));

  head->parms = parms;
  head->body = NULL;
  /* Record the line number - it's useful for debugging */
  head->line_number = state.src->line_number;

  if (macro_parms_ok(parms))
    state.mac_head = head;

  state.mac_prev = &(head->body);
  state.mac_body = NULL;

  return r;
}
static gpasmVal do_messg(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  struct pnode *p;

  if (state.pass == 2) {
    state.lst.line.linetype = none;
    if (enforce_arity(arity, 1)) {
      p = HEAD(parms);
      if (p->tag == string) {
	lst_line(p->value.string);
	fprintf(stderr, "%s\n", p->value.string);
      }
      else
	gperror(110, "Expected string");
    }
    state.num.messages++;
  }

  return r;
}

static gpasmVal do_org(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  struct pnode *p;

  state.lst.line.linetype = org;
  if (enforce_arity(arity, 1)) {
    p = HEAD(parms);
    if (can_evaluate(p))
      r = state.org = evaluate(p);
    else
      gperror(104, "Can't evaluate expression");
  }

  return r;
}

static gpasmVal do_page(gpasmVal r,
			char *name,
			int arity,
			struct pnode *parms)
{
  state.lst.line.linetype = org;
  if (enforce_arity(arity, 0))
    lst_throw();

  return r;
}

static gpasmVal do_processor(gpasmVal r,
			     char *name,
			     int arity,
			     struct pnode *parms)
{
  if (enforce_arity(arity, 1)) {
    struct pnode *p = HEAD(parms);
    
    if (enforce_simple(p))
      select_processor(p->value.symbol);
  }

  return r;
}

static gpasmVal do_radix(gpasmVal r,
			 char *name,
			 int arity,
			 struct pnode *parms)
{
  struct pnode *p;

  state.lst.line.linetype = org;
  if (enforce_arity(arity, 1)) {
    p = HEAD(parms);
    if (enforce_simple(p)) {
      select_radix(p->value.symbol);
    }
  }

  return r;
}

static gpasmVal do_res(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  struct pnode *p;

  state.lst.line.linetype = org;
  if (enforce_arity(arity, 1)) {
    p = HEAD(parms);
    if (can_evaluate(p)) {
      r = state.org;
      state.org += evaluate(p);
    }
    else
      gperror(104, "Can't evaluate expression");
  }

  return r;
}

static gpasmVal do_space(gpasmVal r,
		       char *name,
		       int arity,
		       struct pnode *parms)
{
  struct pnode *p;

  if (state.pass == 2) {
    if (enforce_arity(arity, 1)) {
      p = HEAD(parms);
      if (can_evaluate(p)) {
	int i;
	
	for (i = evaluate(p); i > 0; i--)
	  lst_line("");
      } else
	gperror(104, "Can't evaluate expression");
    }
  }

  return r;
}

void next_line(int value)
{
  char l[BUFSIZ], m[BUFSIZ];
  char *e;
  unsigned int emitted = 0;

  if (state.pass == 2) {
    fgets(l, BUFSIZ, state.src->f2);
    l[strlen(l) - 1] = '\0';	/* Eat the trailing newline */

    if (state.mac_prev) {
      state.lst.line.linetype = none;
      if (state.mac_body)
	state.mac_body->src_line = strdup(l);
    }

    if (state.lst.enabled) {
      e = m;
      switch (state.lst.line.linetype) {
      case equ:
	sprintf(e, "  %08X", value);
	e += strlen(e);
	strcpy(e, "     ");
	e += 5;
	break;
      case org:
	sprintf(e, "%04X      ", state.org);
	e += strlen(e);
	strcpy(e, "     ");
	e += 5;
	break;
      case insn:
	sprintf(e, "%04X ", state.lst.line.was_org);
	e += strlen(e);
	emitted = state.org - state.lst.line.was_org;
	if (emitted >= 1)
	  sprintf(e, "%04X ", state.i_memory[state.lst.line.was_org] & 0xffff);
	else
	  sprintf(e, "     ");
	e += strlen(e);
	if (emitted >= 2)
	  sprintf(e, "%04X ", state.i_memory[state.lst.line.was_org + 1] & 0xffff);
	else
	  sprintf(e, "     ");
	e += strlen(e);
	break;
      case none:
	strcpy(e, "               ");
	e += 15;
	break;
      }

      sprintf(e,
	      "%05d ",
	      state.src->line_number);
      e += strlen(e);

      /* Now copy 'l' to 'e', expanding tabs as required */
      {
	int column = 0;
	char *old = l;

	while (*old) {
	  if (*old == '\t') {
	    *e++ = ' ';
	    column++;
	    while ((column % state.lst.tabstop) != 0) {
	      *e++ = ' ';
	      column++;
	    }
	  } else {
	    *e++ = *old;
	    column++;
	  }
	  old++;
	}
	*e = '\0';		/* terminate the new string */
      }

      lst_line(m);

      if (emitted > 2) {
	int i;

	for (i = 2; i < emitted; i += 2) {
	  if ((i + 1) < emitted)
	    sprintf(m, "%04X %04X %04X",
		    state.lst.line.was_org + i,
		    state.i_memory[state.lst.line.was_org + i] & 0xffff,
		    state.i_memory[state.lst.line.was_org + i + 1] & 0xffff);
	  else
	    sprintf(m, "%04X %04X",
		    state.lst.line.was_org + i,
		    state.i_memory[state.lst.line.was_org + i] & 0xffff);
	  lst_line(m);
	}
      }
    }
  }

  state.src->line_number++;
}

int asm_enabled(void)
{
  return ((state.astack == NULL) ||
	  (state.astack->enabled &&
	   state.astack->prev_enabled));
}

/* Check that a register file address is ok */
static int file_ok(unsigned int file)
{
  if ((0 <= file) &&
      (file <= state.maxram) &&
      (!state.badram[file]))
    return 1;
  else {
    gperror(126, "Attempt to use illegal RAM location");
    return 0;
  }
}

gpasmVal do_insn(char *name, struct pnode *parms)
{
  struct symbol *s;
  int arity;
  struct pnode *p;
  int file;		/* register file address, if applicable */
  gpasmVal r;		/* Return value */

  state.lst.line.linetype = insn;

  r = state.org;

  arity = list_length(parms);

  s = get_symbol(state.stBuiltin, name);

  if ((s == NULL) &&
      (state.processor_chosen == 0) &&
      (state.processor != no_processor)) {
    opcode_init(1);	/* General directives */
    opcode_init(2);	/* Processor-specific */
    state.processor_chosen = 1;

    /* Try again */
    return do_insn(name, parms);
  }

  if (s) {
    struct insn *i;

    i = get_symbol_annotation(s);
    /* Interpret the instruction if assembly is enabled, or if it's a
       conditional. */
    if (asm_enabled() || (i->attribs & ATTRIB_COND)) {
      switch (i->class) {
      case INSN_CLASS_LIT3_BANK:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  emit(i->mask | ((maybe_evaluate(p) >> 5) & 0x07));
	  /* should issue warning if out of range */
	}
	break;
      case INSN_CLASS_LIT3_PAGE:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  emit(i->mask | ((maybe_evaluate(p) >> 9) & 0x07));
	  /* should issue warning if out of range */
	}
	break;
      case INSN_CLASS_LIT4:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  emit(i->mask | (maybe_evaluate(p) & 0x0f));
	  /* should issue warning if out of range */
	}
	break;
      case INSN_CLASS_LIT8:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  emit(i->mask | (maybe_evaluate(p) & 0xff));
	}
	break;
      case INSN_CLASS_LIT9:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  emit(i->mask | (maybe_evaluate(p) & 0x1ff));
	}
	break;
      case INSN_CLASS_LIT11:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  emit(i->mask | (maybe_evaluate(p) & 0x7ff));
	}
	break;
      case INSN_CLASS_OPF5:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  file = maybe_evaluate(p);
	  if (file_ok(file))
	    emit(i->mask | (file & 0x1f));
	}
	break;
      case INSN_CLASS_OPWF5:
	{
	  int d = 1; /* Default destination of 1 (file) */
	  struct pnode *p2; /* second parameter */
	    
	  p = HEAD(parms);
	  switch (arity) {
	  case 2:
	    p2 = HEAD(TAIL(parms));
	    /* Allow "w" and "f" as destinations */
	    if ((p2->tag == symbol) &&
		(strcasecmp(p2->value.symbol, "f") == 0))
	      d = 1;
	    else if ((p2->tag == symbol) &&
		     (strcasecmp(p2->value.symbol, "w") == 0))
	      d = 0;
	    else
	      d = maybe_evaluate(p2);
	    break;
	  case 1:
	    d = 1;
	    break;
	  default:
	    enforce_arity(arity, 2);
	  }
	  file = maybe_evaluate(p);
	  if (file_ok(file))
	    emit(i->mask | (d << 5) | (file & 0x1f));
	}
	break;
      case INSN_CLASS_B5:
	{
	  struct pnode *f, *b;
	  int bit;
	    
	  if (enforce_arity(arity, 2)) {
	    f = HEAD(parms);
	    b = HEAD(TAIL(parms));
	    file = maybe_evaluate(f);
	    bit = maybe_evaluate(b);
	    if (!((0 <= bit) && (bit <= 7)))
	      gperror(127, "Bit address out of range");
	    if (file_ok(file))
	      emit(i->mask |
		   ((bit & 7) << 5) |
		   (file & 0x1f));
	  }
	}
	break;
      case INSN_CLASS_OPF7:
	if (enforce_arity(arity, 1)) {
	  p = HEAD(parms);
	  file = maybe_evaluate(p);
	  if (file_ok(file))
	    emit(i->mask | (file & 0x7f));
	}
	break;
      case INSN_CLASS_OPWF7:
	{
	  int d = 1; /* Default destination of 1 (file) */
	  struct pnode *p2; /* second parameter */
	    
	  p = HEAD(parms);
	  switch (arity) {
	  case 2:
	    p2 = HEAD(TAIL(parms));
	    /* Allow "w" and "f" as destinations */
	    if ((p2->tag == symbol) &&
		(strcasecmp(p2->value.symbol, "f") == 0))
	      d = 1;
	    else if ((p2->tag == symbol) &&
		     (strcasecmp(p2->value.symbol, "w") == 0))
	      d = 0;
	    else
	      d = maybe_evaluate(p2);
	    break;
	  case 1:
	    d = 1;
	    break;
	  default:
	    enforce_arity(arity, 2);
	  }
	  file = maybe_evaluate(p);
	  if (file_ok(file))
	    emit(i->mask | (d << 7) | (file & 0x7f));
	}
	break;
      case INSN_CLASS_B7:
	{
	  struct pnode *f, *b;
	  int bit;
	    
	  if (enforce_arity(arity, 2)) {
	    f = HEAD(parms);
	    b = HEAD(TAIL(parms));
	    file = maybe_evaluate(f);
	    bit = maybe_evaluate(b);
	    if (!((0 <= bit) && (bit <= 7)))
	      gperror(127, "Bit address out of range");
	    if (file_ok(file))
	      emit(i->mask |
		   ((bit & 7) << 7) |
		   (file & 0x7f));
	  }
	}
	break;
      case INSN_CLASS_IMPLICIT:
	if (arity != 0) {
	  gperror(105, "Too many parameters for operation, expected none");
	}
	emit(i->mask);
	break;
      case INSN_CLASS_FUNC:
	r = (*(opfunc*)i->mask)(r, name, arity, parms);
	break;
      }
    }
  } else {
    s = get_symbol(state.stMacros, name);
    if (s) {
      /* Found the macro: execute it */
      if (asm_enabled())
	execute_macro(get_symbol_annotation(s), arity, parms);
    } else {
      char mesg[80];
	
      if (asm_enabled()) {
	sprintf(mesg, "Unknown opcode \"%.40s\"", name);
	gperror(112, mesg);
      }
    }
  }

  return r;
}

void execute_body(struct macro_body *b)
{
  /* Execute the body of the macro, line by line */
  for (; b; b = b->next) {
    gpasmVal r;

    if (b->op) {
      /* Easy place to catch 'EXITM' */
      if (strcasecmp(b->op, "EXITM") == 0)
	break;
      else
	r = do_insn(b->op, b->parms);
    } else {
      r = state.org;
    }
    if (b->label)
      set_global(b->label, r);
  }
}

void execute_macro(struct macro_head *h, int arity, struct pnode *parms)
{
  if (enforce_arity(arity, list_length(h->parms))) {
    state.stTop = push_symbol_table(state.stTop, state.case_insensitive);

    /* Now add the macro's declared parameter list to the new local
       symbol table.  Set the value of each expression to the
       corresponding value in the macro call. */
    if (arity > 0) {
      gpasmVal *values, *pvalue;
      struct pnode *pFrom, *pFromH;
      struct pnode *pTo, *pToH;

      /* First stage, evaluate macro parameters.  Do this BEFORE
         creating the macro's local symbol table so that the
         invokation does the right thing with names that appear in
         both the caller and the callee. */
      pvalue = values = malloc(arity * sizeof(gpasmVal));

      for (pFrom = parms; pFrom; pFrom = TAIL(pFrom)) {
	pFromH = HEAD(pFrom);
	*pvalue++ = maybe_evaluate(pFromH);
      }
      assert((pvalue - values) == arity);

      pvalue = values;
      /* 'values' now holds our parameters.  Create the local symbol
	 table and add the parameters to it. */
      for (pTo = h->parms; pTo; pTo = TAIL(pTo)) {
	struct symbol *sym;
	struct variable *var;

	pToH = HEAD(pTo);
	assert(pToH->tag == symbol);
	
	sym = add_symbol(state.stTop, pToH->value.symbol);
	/* This assert will fail if the declared parameter list
	   contains non-unique entries.  do_macro should have already caught
	   this problem, and prevented the declaration. */
	assert(get_symbol_annotation(sym) == NULL);
	var = malloc(sizeof(*var));
	annotate_symbol(sym, var);
	var->value = *pvalue++;
      }
      assert((pvalue - values) == arity);

      free(values);
    }

    if (state.pass == 1) {
      execute_body(h->body);
    } else {
      unsigned int org_copy;

      org_copy = state.org;
      state.pass = 1;
      execute_body(h->body);

      state.org = org_copy;
      state.pass = 2;
      execute_body(h->body);
    }

    state.stTop = pop_symbol_table(state.stTop);
  }
}

/************************************************************************/

/************************************************************************/

/* There are several groups of operations that we handle here.  First
   is op_0: the instructions that can happen before the processor type
   is known.  Second is op_1, the set of instructions that are common
   to all processors, third is processor-family specific: op_XXX */

/* Note that instructions within each group are sorted alphabetically */

static struct insn  op_0[] = {
  { "else",	(int)do_else, 	INSN_CLASS_FUNC,	ATTRIB_COND },
  { "endif",	(int)do_endif,	INSN_CLASS_FUNC,	ATTRIB_COND },
  { "endm",	(int)do_endm,	INSN_CLASS_FUNC,	0 },
  { "equ",	(int)do_equ,  	INSN_CLASS_FUNC,	0 },
  { "if",	(int)do_if,   	INSN_CLASS_FUNC,	ATTRIB_COND },
  { "ifdef",	(int)do_ifdef, 	INSN_CLASS_FUNC,	ATTRIB_COND },
  { "ifndef",	(int)do_ifndef,	INSN_CLASS_FUNC,	ATTRIB_COND },
  { "list",	(int)do_list, 	INSN_CLASS_FUNC,	0 },
  { "local",	(int)do_local, 	INSN_CLASS_FUNC,	0 },
  { "macro",	(int)do_macro,	INSN_CLASS_FUNC,	0 },
  { "messg",	(int)do_messg,	INSN_CLASS_FUNC,	0 },
  { "nolist",	(int)do_nolist, INSN_CLASS_FUNC,	0 },
  { "page",	(int)do_page,   INSN_CLASS_FUNC,	0 },
  { "processor",(int)do_processor,INSN_CLASS_FUNC,	0 },
  { "radix",	(int)do_radix, 	INSN_CLASS_FUNC,	0 }
};

static struct insn  op_1[] = {
  { "__badram", (int)do_badram, INSN_CLASS_FUNC,	0 },
  { "__config", (int)do_config, INSN_CLASS_FUNC,	0 },
  { "__maxram", (int)do_maxram, INSN_CLASS_FUNC,	0 },
  { "data",     (int)do_data, 	INSN_CLASS_FUNC,	0 },
  { "dt",       (int)do_dt, 	INSN_CLASS_FUNC,	0 },
  { "org",	(int)do_org,    INSN_CLASS_FUNC,	0 },
  { "res",	(int)do_res,    INSN_CLASS_FUNC,	0 },
  { "space",	(int)do_space,  INSN_CLASS_FUNC,	0 }
};

/* XXXPRO: Need to add a struct here for any extra processor classes. */

/* PIC 12-bit instruction set */
static struct insn  op_12c5xx[] = {
  { "ADDWF",  0x1c0,	INSN_CLASS_OPWF5 	},
  { "ANDLW",  0xe00,	INSN_CLASS_LIT8 	},
  { "ANDWF",  0x140,	INSN_CLASS_OPWF5 	},
  { "BCF",    0x400,	INSN_CLASS_B5 		},
  { "BSF",    0x500,	INSN_CLASS_B5 		},
  { "BTFSC",  0x600,	INSN_CLASS_B5 		},
  { "BTFSS",  0x700,	INSN_CLASS_B5		},
  { "CALL",   0x900,	INSN_CLASS_LIT8 	},
  { "CLRF",   0x060,	INSN_CLASS_OPF5 	},
  { "CLRW",   0x040,	INSN_CLASS_IMPLICIT 	},
  { "CLRWDT", 0x004, 	INSN_CLASS_IMPLICIT 	},
  { "COMF",   0x240,	INSN_CLASS_OPWF5 	},
  { "DECF",   0x0c0,	INSN_CLASS_OPWF5 	},
  { "DECFSZ", 0x2c0,	INSN_CLASS_OPWF5 	},
  { "GOTO",   0xa00,	INSN_CLASS_LIT9 	},
  { "INCF",   0x280,	INSN_CLASS_OPWF5 	},
  { "INCFSZ", 0x3c0,	INSN_CLASS_OPWF5 	},
  { "IORLW",  0xd00,	INSN_CLASS_LIT8 	},
  { "IORWF",  0x100,	INSN_CLASS_OPWF5 	},
  { "MOVF",   0x200,	INSN_CLASS_OPWF5 	},
  { "MOVLW",  0xc00,	INSN_CLASS_LIT8 	},
  { "MOVWF",  0x020,	INSN_CLASS_OPF5 	},
  { "NOP",    0x000,	INSN_CLASS_IMPLICIT 	},
  { "OPTION", 0x002,	INSN_CLASS_IMPLICIT 	},
  { "RETLW",  0x800,	INSN_CLASS_LIT8 	},
  { "RLF",    0x340,	INSN_CLASS_OPWF5 	},
  { "RRF",    0x300,	INSN_CLASS_OPWF5 	},
  { "SLEEP",  0x003,	INSN_CLASS_IMPLICIT 	},
  { "SUBWF",  0x080,	INSN_CLASS_OPWF5 	},
  { "SWAPF",  0x380,	INSN_CLASS_OPWF5 	},
  { "TRIS",   0x000,	INSN_CLASS_OPF5 	}, /* not optimal */
  { "XORLW",  0xf00,	INSN_CLASS_LIT8 	},
  { "XORWF",  0x180,	INSN_CLASS_OPWF5 	}
};

/* Scenix SX has a superset of the PIC 12-bit instruction set */
/*
 * It would be nice if there was a more elegant way to do this,
 * either by adding a flags field to struct insn, or by allowing a
 * processor to have more than one associated table.
 */
static struct insn  op_sx[] = {
  { "ADDWF",  0x1c0,	INSN_CLASS_OPWF5 	},
  { "ANDLW",  0xe00,	INSN_CLASS_LIT8 	},
  { "ANDWF",  0x140,	INSN_CLASS_OPWF5 	},
  { "BANK",   0x018,    INSN_CLASS_LIT3_BANK    }, /* SX only */
  { "BCF",    0x400,	INSN_CLASS_B5 		},
  { "BSF",    0x500,	INSN_CLASS_B5 		},
  { "BTFSC",  0x600,	INSN_CLASS_B5 		},
  { "BTFSS",  0x700,	INSN_CLASS_B5		},
  { "CALL",   0x900,	INSN_CLASS_LIT8 	},
  { "CLRF",   0x060,	INSN_CLASS_OPF5 	},
  { "CLRW",   0x040,	INSN_CLASS_IMPLICIT 	},
  { "CLRWDT", 0x004, 	INSN_CLASS_IMPLICIT 	},
  { "COMF",   0x240,	INSN_CLASS_OPWF5 	},
  { "DECF",   0x0c0,	INSN_CLASS_OPWF5 	},
  { "DECFSZ", 0x2c0,	INSN_CLASS_OPWF5 	},
  { "GOTO",   0xa00,	INSN_CLASS_LIT9 	},
  { "INCF",   0x280,	INSN_CLASS_OPWF5 	},
  { "INCFSZ", 0x3c0,	INSN_CLASS_OPWF5 	},
  { "IORLW",  0xd00,	INSN_CLASS_LIT8 	},
  { "IORWF",  0x100,	INSN_CLASS_OPWF5 	},
  { "IREAD",  0x041,    INSN_CLASS_IMPLICIT     }, /* SX only */
  { "MODE",   0x050,    INSN_CLASS_LIT4         }, /* SX only */
  { "MOVF",   0x200,	INSN_CLASS_OPWF5 	},
  { "MOVLW",  0xc00,	INSN_CLASS_LIT8 	},
  { "MOVMW",  0x042,    INSN_CLASS_IMPLICIT     }, /* SX only */
  { "MOVWF",  0x020,	INSN_CLASS_OPF5 	},
  { "MOVWM",  0x043,    INSN_CLASS_IMPLICIT     }, /* SX only */
  { "NOP",    0x000,	INSN_CLASS_IMPLICIT 	},
  { "OPTION", 0x002,	INSN_CLASS_IMPLICIT 	},
  { "PAGE",   0x010,    INSN_CLASS_LIT3_PAGE    }, /* SX only */
  { "RETI",   0x00e,    INSN_CLASS_IMPLICIT     }, /* SX only */
  { "RETIW",  0x00f,    INSN_CLASS_IMPLICIT     }, /* SX only */
  { "RETLW",  0x800,	INSN_CLASS_LIT8 	},
  { "RETP",   0x00d,    INSN_CLASS_IMPLICIT     }, /* SX only */
  { "RETURN", 0x00c,    INSN_CLASS_IMPLICIT     }, /* SX only */
  { "RLF",    0x340,	INSN_CLASS_OPWF5 	},
  { "RRF",    0x300,	INSN_CLASS_OPWF5 	},
  { "SLEEP",  0x003,	INSN_CLASS_IMPLICIT 	},
  { "SUBWF",  0x080,	INSN_CLASS_OPWF5 	},
  { "SWAPF",  0x380,	INSN_CLASS_OPWF5 	},
  { "TRIS",   0x000,	INSN_CLASS_OPF5 	}, /* not optimal */
  { "XORLW",  0xf00,	INSN_CLASS_LIT8 	},
  { "XORWF",  0x180,	INSN_CLASS_OPWF5 	}
};

/* PIC 14-bit instruction set */
static struct insn  op_16cxx[] = {
  { "ADDLW",  0x3e00, 	INSN_CLASS_LIT8 	},
  { "ADDWF",  0x0700,	INSN_CLASS_OPWF7 	},
  { "ANDLW",  0x3900,	INSN_CLASS_LIT8 	},
  { "ANDWF",  0x0500,	INSN_CLASS_OPWF7 	},
  { "BCF",    0x1000,	INSN_CLASS_B7 		},
  { "BSF",    0x1400,	INSN_CLASS_B7 		},
  { "BTFSC",  0x1800,	INSN_CLASS_B7 		},
  { "BTFSS",  0x1c00,	INSN_CLASS_B7		},
  { "CALL",   0x2000,	INSN_CLASS_LIT11 	},
  { "CLRF",   0x0180,	INSN_CLASS_OPF7		},
  { "CLRW",   0x0100,	INSN_CLASS_IMPLICIT 	},
  { "CLRWDT", 0x0064, 	INSN_CLASS_IMPLICIT 	},
  { "COMF",   0x0900,	INSN_CLASS_OPWF7 	},
  { "DECF",   0x0300,	INSN_CLASS_OPWF7 	},
  { "DECFSZ", 0x0b00,	INSN_CLASS_OPWF7 	},
  { "GOTO",   0x2800,	INSN_CLASS_LIT11 	},
  { "INCF",   0x0a00,	INSN_CLASS_OPWF7 	},
  { "INCFSZ", 0x0f00,	INSN_CLASS_OPWF7 	},
  { "IORLW",  0x3800,	INSN_CLASS_LIT8 	},
  { "IORWF",  0x0400,	INSN_CLASS_OPWF7 	},
  { "MOVF",   0x0800,	INSN_CLASS_OPWF7 	},
  { "MOVLW",  0x3000,	INSN_CLASS_LIT8 	},
  { "MOVWF",  0x0080,	INSN_CLASS_OPF7 	},
  { "NOP",    0x0000,	INSN_CLASS_IMPLICIT 	},
  { "OPTION", 0x0062,	INSN_CLASS_IMPLICIT 	},
  { "RETFIE", 0x0009,	INSN_CLASS_IMPLICIT 	},
  { "RETLW",  0x3400,	INSN_CLASS_LIT8 	},
  { "RETURN", 0x0008,	INSN_CLASS_IMPLICIT 	},
  { "RLF",    0x0d00,	INSN_CLASS_OPWF7 	},
  { "RRF",    0x0c00,	INSN_CLASS_OPWF7 	},
  { "SLEEP",  0x0063,	INSN_CLASS_IMPLICIT 	},
  { "SUBLW",  0x3c00,	INSN_CLASS_LIT8 	},
  { "SUBWF",  0x0200,	INSN_CLASS_OPWF7 	},
  { "SWAPF",  0x0e00,	INSN_CLASS_OPWF7 	},
  { "TRIS",   0x0060,	INSN_CLASS_OPF7		},
  { "XORLW",  0x3a00,	INSN_CLASS_LIT8 	},
  { "XORWF",  0x0600,	INSN_CLASS_OPWF7 	}
};

/* XXXPRO: Need to add a line here for any extra processors. */

#define NUM_OP_0	(sizeof(op_0) / sizeof(op_0[0]))
#define NUM_OP_1	(sizeof(op_1) / sizeof(op_1[0]))
#define NUM_OP_12C5XX	(sizeof(op_12c5xx) / sizeof(op_12c5xx[0]))
#define NUM_OP_16CXX	(sizeof(op_16cxx) / sizeof(op_16cxx[0]))
#define NUM_OP_SX	(sizeof(op_sx) / sizeof(op_sx[0]))

void opcode_init(int stage)
{
  int i;
  int count;
  struct insn *base;

  switch (stage) {
  case 0:
    base = op_0;
    count = NUM_OP_0;
    break;
  case 1:
    base = op_1;
    count = NUM_OP_1;
    break;
  case 2:

    /* XXXPRO: Need to add a line here for any extra processors. */

    switch (state.processor) {
    case pic12c509:
    case pic12c508:
    case pic16c54:
    case pic16c55:
    case pic16c56:
    case pic16c57:
    case pic16c58:
      base = op_12c5xx;
      count = NUM_OP_12C5XX;
      break;
    case sx18:
    case sx20:
    case sx28:
      base = op_sx;
      count = NUM_OP_SX;
      break;
    case pic16c71:
    case pic16c84:
    case pic16f84:
      base = op_16cxx;
      count = NUM_OP_16CXX;
      break;
    default:
      assert(0);
    }
    break;
  default:
    assert(0);
  }

  for (i = 0; i < count; i++)
    annotate_symbol(add_symbol(state.stBuiltin, base[i].name), &base[i]);
}

/************************************************************************/

void begin_cblock(struct pnode *c)
{
  state.cblock = maybe_evaluate(c);
}

void cblock_expr(char *s)
{
  set_global(s, state.cblock);
  state.cblock++;
}

void cblock_expr_incr(char *s, struct pnode *incr)
{
  set_global(s, state.cblock);
  state.cblock += maybe_evaluate(incr);
}
