/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __UTIL_H__
#define __UTIL_H__

#include <stdio.h>
#include <glib.h>

/**
 * Works like bsearch, except it returns the lower bound index
 * in the array.  This is useful for finding a sorted insertion
 * point or for finding the nearest match.
 *
 * @param key the item to search for
 * @param base the beginning of the sorted array to search
 * @param n number of items in the array
 * @param size the size of each item in the array
 * @param compare standard c compare func
 * @param exact_match (optional): if non-NULL, this is set to TRUE or FALSE if
 *        the lower bound is an exact match of key
 *
 * @return the index of the lower bound of the search
 * @prerequisite base must be sorted with respect to compare
 */
extern int lower_bound (
   const void * key,
   const void * base,
   size_t n,
   size_t size,
   int (*compare)(const void *, const void *),
   gboolean* exact_match );

/**
 * Used to lock the UI thread.
 *
 * @see pan_unlock
 */
extern void pan_lock (void);

/**
 * Used to unlock the UI thread after pan_lock() has been called.
 *
 * @see pan_lock
 */
extern void pan_unlock (void);

/**
 * Adds "Re: " to a subject if it's not already there
 */
extern gchar * add_re (const gchar *);

extern gchar * pan_make_temp (FILE **);

extern void pan_clean_temp_dir (void);

/**
 * Pops up a gnome error dialog with the specified message.
 * This function is threadsafe.
 *
 * @param message or printf-style format string for the message.
 */
extern void pan_error_dialog (const gchar *, ...);

/**
 * Returns the number of bytes in the file, or 0 on error.
 * @return the number of bytes in the file, or 0 on error.
 * @param filename
 */
extern size_t get_filesize (const gchar* filename);

/**
 * Checks to make sure the directory exists.  If it doesn't, directory_check()
 * tries to create it.  If its parent directories don't exist either,
 * directory_check() tries to create them first.
 *
 * @param pathname
 */
extern void directory_check (const gchar* pathname);

/**
 * Formats a number with commas; ie, 10321 becomes "10,321"
 * @param num the number to format
 * @param setme the string where the resulting string is written
 */
extern void commatize_ulong (gulong num, char* setme);


extern void    not_implemented    (void);
extern GArray* read_file ( const char* filename );
extern gchar* wrap_long_lines (gchar *string, gint maxlen);
extern gchar* pan_append (gchar* old, const gchar *appendme, char optional_delimeter);
extern gchar* pan_substitute (const gchar* original, const gchar* search, const gchar* replace);
extern gboolean file_exists (const gchar* filename);
extern char* get_fqdn (const char* host);
extern char* get_host_name (void);
extern void array_shrink (gpointer a, int idx, size_t ele_size, int ele_qty);

extern void msort (
	void *b,
	size_t n,
	size_t s,
	int(*cmp)(const void*, const void*));

#define pan_warn_if_reached()           G_STMT_START{           \
     g_log (G_LOG_DOMAIN,                                       \
            G_LOG_LEVEL_WARNING,                                \
            "file %s: line %d (%s): should not be reached",     \
            __FILE__,                                           \
            __LINE__,                                           \
            __PRETTY_FUNCTION__);       }G_STMT_END

#define pan_warn_if_fail(expr)          G_STMT_START{                   \
     if (!(expr))                                                       \
       {                                                                \
         g_log (G_LOG_DOMAIN,                                           \
                G_LOG_LEVEL_CRITICAL,                                   \
                "file %s: line %d (%s): assertion `%s' failed.",        \
                __FILE__,                                               \
                __LINE__,                                               \
                __PRETTY_FUNCTION__,                                    \
                #expr);                                                 \
       };                               }G_STMT_END

#endif /* __UTIL_H__ */
