/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-popup-menu.h>

#include "acache.h"
#include "articlelist.h"
#include "gui.h"
#include "globals.h"
#include "grouplist.h"
#include "message.h"
#include "message-window.h"
#include "prefs.h"
#include "queue.h"
#include "queue-item-body.h"
#include "text.h"
#include "util.h"


static void realize_text_cb (GtkWidget *text, gpointer data);
static void text_button_press_cb (GtkWidget *widget, GdkEventButton *bevent);
static void text_key_press_cb (GtkWidget *widget, GdkEventKey *event);
static void text_page_down_or_next_article (GtkText *text);
static void show_all_headers_cb (GtkCheckMenuItem* item, gpointer data);

static GnomeUIInfo text_menu_popup[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Open in Window"),
		N_("Open this message for reading."),
		message_read_window
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Followup to newsgroup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply by _E-mail"),
		N_("Create a mail reply to the sender."),
		message_reply_window
	},
#if 0
	{
		GNOME_APP_UI_ITEM,
		N_("Forward"),
		N_("Forward this message to someone."),
		not_implemented
	},
#endif
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show All Headers in Body"),
		N_("Show All Headers in Body"),
		show_all_headers_cb
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Select All"),
		N_("Select all of the message body"),
		text_select_all,
		NULL, NULL, GNOME_APP_PIXMAP_NONE, NULL,
		'A', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};

static gchar welcome[] =
{
"Pan " VERSION "\n"
"Copyright (C) 1999-2000  Pan Development Team (pan@superpimp.org) \n"
"\n"
"--\n"
"\n"
"   Request A Feature: http://www.superpimp.org/contact.html\n"
"   Report A Bug: http://www.superpimp.org/bugreport.html\n"
"   Upgrade: http://www.superpimp.org/download.html\n"
"\n"
"--\n"
"\n"
"This program is free software; you can redistribute it and/or modify \n"
"it under the terms of the GNU General Public License as published by \n"
"the Free Software Foundation; either version 2 of the License, or \n"
"(at your option) any later version. \n"
"\n"
"This program is distributed in the hope that it will be useful, \n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of \n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \n"
"GNU General Public License for more details. \n"
"\n"
"You should have received a copy of the GNU General Public License \n"
"along with this program; if not, write to the Free Software \n"
"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA"
};

static GtkWidget *text_rightclick_popup = NULL;
PanCallback *text_show_all_headers_changed = NULL;
gboolean rot13_enabled = FALSE;
gboolean show_all_headers_in_body = FALSE;


static void
show_all_headers_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_show_all_headers_in_body (item->active);
}

void
text_select_all (void)
{
	pan_lock();
	gtk_editable_select_region (GTK_EDITABLE(Pan.text), 0, -1);
	pan_unlock();
}

void
text_refresh (void)
{
	/*FIXME: this limits rot13'ing to the main window.
	  FIXME: this ties text to the articlelist for server/group. */
	if (Pan.active_message_id)
	{
		text_set_from_message_id (
			articlelist_get_current_server(),
			articlelist_get_current_group(),
			Pan.active_message_id,
			NNTP_ARTICLE_READ,
			FALSE );
	}
}

void
text_set_show_all_headers_in_body (gboolean on)
{
	if (on != show_all_headers_in_body)
	{
		show_all_headers_in_body = on;

		pan_callback_call (
			text_show_all_headers_changed,
			Pan.text,
			GINT_TO_POINTER(show_all_headers_in_body));

		text_refresh ();

		gtk_check_menu_item_set_active (
			GTK_CHECK_MENU_ITEM(text_menu_popup[5].widget),
			show_all_headers_in_body );
	}

}

/* Scroll down one page, or if we're at the bottom, open the next article */
static void
text_page_down_or_next_article (GtkText *text)
{
        gfloat upper;
	gfloat old_val;

	g_return_if_fail (text != NULL);

	old_val = text->vadj->value;
	text->vadj->value += text->vadj->page_increment;

	upper = text->vadj->upper - text->vadj->page_size;
        text->vadj->value = MIN (text->vadj->value, upper);
        text->vadj->value = MAX (text->vadj->value, 0.0);

	if (text->vadj->value != old_val)
		gtk_signal_emit_by_name (GTK_OBJECT (text->vadj), "value_changed");
	else
		articlelist_read_next_article ();
}

/* Called when the widget is first drawn, puts our welcome text into view */
static void
realize_text_cb (GtkWidget *text, gpointer data)
{
	GtkAdjustment* hadj = GTK_ADJUSTMENT(GTK_TEXT(text)->hadj);
	GtkAdjustment* vadj = GTK_ADJUSTMENT(GTK_TEXT(text)->vadj);

	gui_set_headers_default (NULL, NULL);
	gtk_editable_delete_text (GTK_EDITABLE(text), 0, -1);
	gtk_text_insert (
		GTK_TEXT(text), NULL, &text->style->black, NULL, welcome, -1);
	gtk_text_set_point (GTK_TEXT(text), 0);
	gtk_adjustment_set_value (GTK_ADJUSTMENT(vadj), 0);
	gtk_text_set_adjustments (GTK_TEXT(text), hadj, vadj);

	gtk_signal_disconnect_by_func (
		GTK_OBJECT(text), GTK_SIGNAL_FUNC(realize_text_cb), NULL);
}

/* Process a mouse button events inside the text widget */
static void
text_button_press_cb (GtkWidget *widget, GdkEventButton *bevent)
{
	gtk_widget_grab_focus (GTK_WIDGET (Pan.text));

	if (bevent->button != 1)
		gtk_signal_emit_stop_by_name (
			GTK_OBJECT(widget), "button_press_event");
	
	if (bevent->button == 3)
	{
		const gboolean sel = current_article != NULL;

		GList* l = GTK_MENU_SHELL(text_rightclick_popup)->children;
		/* l is "open in window" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel);
		l = l->next; /* l is separator */
		l = l->next; /* l is "followup to group */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel);
		l = l->next; /* l is "reply by e-mail" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel);
		l = l->next; /* l is separator */
		l = l->next; /* l is "show all headers in body" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel);
		l = l->next; /* l is separator */
		l = l->next; /* l is "select all" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel);

		gnome_popup_menu_do_popup (text_rightclick_popup, NULL,
					   NULL, bevent, NULL);
	}
}

/* Process a keyboard input event in the text pane */
static void
text_key_press_cb (GtkWidget *widget, GdkEventKey *event)
{
	switch (event->keyval)
	{
		case GDK_KP_Space:
		case GDK_space:
			text_page_down_or_next_article (GTK_TEXT (Pan.text));
			break;
		case GDK_Up:
			if (event->state & GDK_CONTROL_MASK)
				articlelist_read_prev_article ();
			break;
		case GDK_Down:
			if (event->state & GDK_CONTROL_MASK)
				articlelist_read_next_article ();
		default: 
			break;
	}
}

/**
 * text_create:
 * 
 * Return: a GtkScrolledWindow containing the message body text widget
 **/
GtkWidget *
text_create (void)
{
	GtkWidget *scrolled_window;

	Pan.text = gtk_text_new (NULL, NULL);
	widget_set_font (GTK_WIDGET(Pan.text), message_body_font);
	gtk_text_set_editable (GTK_TEXT(Pan.text), FALSE);
#if 0 /* GtkText doesn't allow horizontal scroll.  Rewrite coming in Gtk 1.4 */
	gtk_text_set_word_wrap (GTK_TEXT(Pan.text), FALSE);
	gtk_text_set_line_wrap (GTK_TEXT(Pan.text), FALSE);
#endif

	text_show_all_headers_changed = pan_callback_new ();

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (
		GTK_SCROLLED_WINDOW (scrolled_window),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_signal_connect (
		GTK_OBJECT (Pan.text), "realize",
		GTK_SIGNAL_FUNC(realize_text_cb), NULL);
	gtk_signal_connect (
		GTK_OBJECT (Pan.text), "button_press_event",
		GTK_SIGNAL_FUNC(text_button_press_cb), NULL);
	gtk_signal_connect (
		GTK_OBJECT (Pan.text), "key_press_event",
		GTK_SIGNAL_FUNC(text_key_press_cb), NULL);

	gtk_container_add (GTK_CONTAINER(scrolled_window), Pan.text);

	text_rightclick_popup = gnome_popup_menu_new (text_menu_popup);

	return scrolled_window;
}

/* Do ROT13 shifting on a message body */
static void
rot13 (gchar* body)
{
	const gchar a [] = "abcdefghijklmnopqrstuvwxyz";
	gchar *pch = NULL;

	for (pch=body; *pch!='\0'; ++pch)
	{
		if (isalpha((int)*pch))
		{
			const gboolean uc = isupper((int)*pch);
			*pch = tolower(*pch);
			*pch = a[((*pch-'a')+13)%26];
			if (uc)
				*pch = toupper(*pch);
		}
	}
}

void
text_update (Message *msg)
{
	gchar *body;
	article_data *adata = NULL;
	article_db db;

        /* sanity checks */
        g_return_if_fail (msg!=NULL);
        g_return_if_fail (msg->message_id!=NULL);
        g_return_if_fail (*msg->message_id!='\0');

        /* update active_message_id */
	g_free (Pan.active_message_id);
	Pan.active_message_id = g_strdup (msg->message_id);
	
        /* load article header */
	db = ahdb_ref (msg->sdata, msg->gdata);
	adata = ahdb_get (db, msg->message_id);
	ahdb_unref (db);

        /* load article body */
	body = acache_load_body (msg->message_id);

	if (body)
	{
		if (rot13_enabled)
			rot13 (body);

		if (show_all_headers_in_body)
		{
			gchar* headers = acache_load_header (msg->message_id);
			if (headers) {
				gchar* tmp = g_strconcat (headers,"\n\n", body, 0);
				g_free (headers);
				g_free (body);
				headers = NULL;
				body = tmp;
			}
		}

		if (use_message_window_always || (msg->needs_window))
		{
			Message* m2 = message_copy (msg);
			message_import_adata (m2, adata);
			message_window_new (m2);
		}
		else
		{
		        /* update the text window */
			update_body_pane (Pan.text, body);
			gui_set_headers_default (NULL, adata);

                        /* update current article */
                        if (current_article)
			        article_free (current_article);
			current_article = article_copy (article_new(), adata);
		}
	}

	g_free (body);

	message_free (msg);
	article_free (adata);
}


void
text_set_from_message_id (const server_data *sdata,
			  group_data *gdata,
			  const gchar *message_id,
			  MessageWindowType type,
			  gboolean needs_window)
{
	Message *msg = NULL;

	/* sanity checks */
	g_return_if_fail (sdata!=NULL);
	g_return_if_fail (gdata!=NULL);
	g_return_if_fail (message_id!=NULL);
	g_return_if_fail (*message_id!='\0');

	/* create message object */
	msg = message_new (sdata, gdata, NULL);
	msg->message_id = g_strdup (message_id);
	msg->type = type;
	msg->needs_window = needs_window;

	queue_add (QUEUE_ITEM(
		queue_item_body_new(
			(server_data*)articlelist_get_current_server(), //FIXME
			articlelist_get_current_group(), msg)));
}
