/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "acache.h"
#include "article.h"
#include "article-db.h"
#include "debug.h"
#include "log.h"
#include "nntp.h"
#include "queue.h"
#include "queue-item-body.h"
#include "text.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint queue_item_body_run (QueueItem* item);

static gchar* queue_item_body_describe (const StatusItem* item);

static int nntp_article_download (QueueItem*, article_data*);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
queue_item_body_destructor (PanObject *o)
{
	QueueItemBody *item = QUEUE_ITEM_BODY(o);

	if (item->checkout) {
		item->checkout = FALSE;
		acache_file_checkin(item->message->message_id);
	}

	if (item->message) {
		message_free (item->message);
		item->message = NULL;
	}

	queue_item_destructor (o);
}

/************
*************  PUBLIC ROUTINES
************/

PanObject*
queue_item_body_new (server_data* sdata, group_data* gdata, Message* message)
{
	QueueItemBody* body = NULL;
	gboolean needs_socket;

	/* sanity checks */
	g_return_val_if_fail (message!=NULL, NULL);
	g_return_val_if_fail (message->message_id!=NULL, NULL);
	g_return_val_if_fail (*message->message_id!='\0', NULL);

	/* do we need to pull this body from the network? */
	needs_socket = TRUE;
	if (acache_body_exists(message->message_id) &&
	    acache_header_exists(message->message_id))
	{
        	debug (DEBUG_PAN_OBJECT, "queue_item_body_new: no socket needed; it's cached");
		acache_file_checkout(message->message_id);
		needs_socket = FALSE;
	}

	/* create the queue item */
       	body = g_new0 (QueueItemBody, 1);
        debug (DEBUG_PAN_OBJECT, "queue_item_body_new: %p", body);
	queue_item_constructor (QUEUE_ITEM(body),
		queue_item_body_destructor,
		queue_item_body_describe,
		queue_item_body_run,
		sdata, TRUE, needs_socket);

	/* initialize the queue-item-body */
	body->gdata = gdata;
	body->message = message;
	body->checkout = !needs_socket;

	return PAN_OBJECT(body);
}

/*****
******
*****/

static gchar*
queue_item_body_describe (const StatusItem* si)
{
	QueueItemBody *item = QUEUE_ITEM_BODY(si);

	if (item->checkout)
	{
		return g_strdup_printf (_("Loading body of cached article ``%s'' from group %s"),
			item->message->message_id,
			item->gdata->name );
	} else {
		return g_strdup_printf (_("Downloading body of ``%s'' from group %s"),
			item->message->message_id,
			item->gdata->name );
	}
}

/*****
******
*****/

static gint
queue_item_body_run (QueueItem* item)
{
	server_data *sdata = item->sdata;
	group_data *gdata = QUEUE_ITEM_BODY(item)->gdata;
	Message *msg = QUEUE_ITEM_BODY(item)->message;
	article_db db = NULL;
	article_data *adata = NULL;

	/* sanity checks */
	g_return_val_if_fail(sdata!=NULL, -1);
	g_return_val_if_fail(gdata!=NULL, -1);
	g_return_val_if_fail(msg!=NULL, -1);
	g_return_val_if_fail(msg->message_id!=NULL, -1);
	g_return_val_if_fail(*msg->message_id!='\0', -1);

	if (QUEUE_ITEM_BODY(item)->checkout)
		status_item_emit_status(STATUS_ITEM(item),
			_("Loading Cached Article '%s'"),
			msg->message_id );

	/* get the article header */
	db = ahdb_ref (sdata, gdata);
	adata = ahdb_get ( db, (const char*)msg->message_id );
	ahdb_unref ( db );

	/* make sure we've got the first header */
	if ( !adata ) {
		status_item_emit_error(STATUS_ITEM(item),
			_("Couldn't find article header for message_id '%s'"),
			msg->message_id );
		return -1;
	}

	article_remove_flag (sdata, gdata, adata, STATE_DOWNLOAD_QUEUED, TRUE);

	/* Check the cache before we download */
	if (!acache_header_exists(adata->message_id) ||
            !acache_body_exists(adata->message_id))
	{
		acache_file_checkout (adata->message_id);
		if ( nntp_article_download(item,adata) ) {
			status_item_emit_status(STATUS_ITEM(item),
				_("Downloading article ``%s'' failed."),
				adata->subject);
			acache_file_checkin (adata->message_id);
			article_free (adata);
			return -1;
		}
	}

	/* Display, et cetera */
	if (item->high_priority)
	{
		message_import_adata (msg, adata);
		article_mark_read (sdata, gdata, adata, TRUE);
		text_update (msg);

		/* check in this file.. */
		if (QUEUE_ITEM_BODY(item)->checkout) {
			QUEUE_ITEM_BODY(item)->checkout = FALSE;
			acache_file_checkin (adata->message_id);
		}

	       	/* text_update frees the message */
		QUEUE_ITEM_BODY(item)->message = msg = NULL;
	}

	/* cleanup */
	acache_file_checkin (adata->message_id);
	article_free (adata);
	adata = NULL;
	return 0;
}

/*****
******
*****/

static int
nntp_article_download (QueueItem *item, article_data* adata)
{
	PanSocket *sock = item->sock;
	group_data *gdata = QUEUE_ITEM_BODY(item)->gdata;
	server_data *sdata = QUEUE_ITEM(item)->sdata;
	gboolean adata_headers_changed = FALSE;
	const char *buffer;
	FILE *f;

	/* sanity checks */
	g_return_val_if_fail (sock!=NULL, -1);

	status_item_emit_status (STATUS_ITEM(item),
		_("Downloading article '%s'"), adata->subject);

	/* set group */
	if (nntp_set_group(sock, gdata)) {
		status_item_emit_error (STATUS_ITEM(item),
			_("Couldn't select group '%s'"), gdata->name);
		return -1;
	}

	/* get the head */
	if (pan_socket_putline_va (sock, "HEAD %s\r\n", adata->number)) {
		queue_item_emit_sockwrite_err(QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (pan_socket_getline (sock, &buffer)) {
		queue_item_emit_sockread_err(QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (strtol(buffer, (char **) NULL, 10) != 221) {
		status_item_emit_error(STATUS_ITEM(item),
			_("NNTP \"HEAD\" command returned %d.\n"
			  "The article may have expired."),
			  buffer);
		return -1;
	}
	if (pan_socket_getline (sock, &buffer)) {
		queue_item_emit_sockread_err(QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}

	f = acache_open_header (adata);
	while (strncmp(buffer, ".\r\n", 3))
	{
		char* pch = NULL;
		const char* name = NULL;

		/* check for exit... */
		if (must_exit) {
			acache_close_header (f);
			acache_delete (adata->message_id);
			return -1;
		}

		/* write to acache... */
		acache_putline_header (f, buffer);

		/* parse it for adata... */
		if ((pch = strrchr (buffer, '\x0D'))) /* trim linefeed */
			*pch = '\0';
		name = "Newsgroups: ";
		if (!strncmp ( buffer, name, strlen(name))) {
			g_free (adata->newsgroups);
			adata->newsgroups = g_strdup ( buffer+strlen(name) );
			adata_headers_changed = TRUE;
		}
		name = "Reply-To: ";
		if (!strncmp (buffer, name, strlen(name))) {
			g_free (adata->reply_to);
			adata->reply_to = g_strdup ( buffer+strlen(name) );
			adata_headers_changed = TRUE;
		}
		name = "Followup-To: ";
		if (!strncmp ( buffer, name, strlen(name))) {
			g_free (adata->followup_to);
			adata->followup_to = g_strdup ( buffer+strlen(name) );
			adata_headers_changed = TRUE;
		}

		/* get the next line... */
		if (pan_socket_getline (sock, &buffer)) {
			queue_item_emit_sockread_err(QUEUE_ITEM(item));
			acache_close_header (f);
			acache_delete (adata->message_id);
			return QUEUE_ITEM_STATUS_ESOCKET;
		}
	}
	acache_close_header (f);

	if (adata_headers_changed)
		article_update (sdata, gdata, adata, TRUE);

	if (pan_socket_putline_va (sock, "BODY %s\r\n", adata->number)) {
		queue_item_emit_sockwrite_err(QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (pan_socket_getline (sock, &buffer)) {
		queue_item_emit_sockread_err(QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	if (atoi(buffer) != 222) {
		return -1;
	}
	
	if (pan_socket_getline (sock, &buffer)) {
		queue_item_emit_sockread_err(QUEUE_ITEM(item));
		return QUEUE_ITEM_STATUS_ESOCKET;
	}
	f = acache_open_body (adata);

	status_item_emit_init_steps (STATUS_ITEM(item), adata->linecount);

	while (strncmp(buffer, ".\r\n", 3))
	{
		if (must_exit)
		{
			acache_close_body (f);
			acache_delete (adata->message_id);
			return -1;
		}
		acache_putline_body (f, buffer);
		status_item_emit_next_step(STATUS_ITEM(item));
		if (pan_socket_getline (sock, &buffer))
		{
			queue_item_emit_sockread_err(QUEUE_ITEM(item));
			acache_close_body (f);
			acache_delete (adata->message_id);
			return QUEUE_ITEM_STATUS_ESOCKET;
		}
	}
	acache_close_body (f);
	acache_file_checkin (adata->message_id);
	status_item_emit_progress(STATUS_ITEM(item), 100);

	return 0;
}
